
// Copyright (c) 2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org

//---------------------------------------------------------------------------

#include "IIR_ArchitectureStatement.hh"
#include "IIR_Declaration.hh"
#include "IIR_Label.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_EnumerationSubtypeDefinition.hh"
#include "IIR_ComponentInstantiationStatement.hh"
#include "IIR_SimpleSimultaneousStatement.hh"
#include "symbol_table.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "StandardPackage.hh"
#include "published_file.hh"
#include <iostream>
#include "sstream-wrap.hh"

IIRScram_ArchitectureStatement::IIRScram_ArchitectureStatement() {
  enclosingRegion = NULL;
  enclosing_path = NULL;
}

IIRScram_ArchitectureStatement::~IIRScram_ArchitectureStatement() {
  if(enclosing_path != NULL) {
    delete [] enclosing_path;
  }
}

void 
IIRScram_ArchitectureStatement::_publish_cc_state( published_file & ){
  _report_undefined_scram_fn("_publish_cc_state( _cc_out )");
}

void 
IIRScram_ArchitectureStatement::_publish_createNetInfo( published_file & ){
  _report_undefined_scram_fn("_publish_createNetInfo( published_file &_cc_out )");
}

IIR_Boolean
IIRScram_ArchitectureStatement::_publish_cc_type_info( published_file &, IIR_Boolean ){
  _report_undefined_scram_fn("_publish_cc_type_info( published_file &, IIR_Boolean )");
  return FALSE;
}

void
IIRScram_ArchitectureStatement::_publish_cc_extern_type_info( published_file & ){
   _report_undefined_scram_fn("_publish_cc_extern_type_info");
}

#ifdef PROCESS_GRAPH
void
IIRScram_ArchitectureStatement::_publish_cc_driver_info( published_file & ){
  _report_undefined_scram_fn("_publish_cc_driver_info");
}
#endif

void 
IIRScram_ArchitectureStatement::_publish_cc_headerfiles_for_cc_generate_statement( published_file &_cc_out ){
  const string temp = _get_current_architecture_name();
  const string old_current_name = _get_current_publish_name();
  IIR_ComponentInstantiationStatement *compInst = NULL;

  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureStatement::_publish_cc_headerfiles_for_cc_generate_statement" );
  
  IIR_ArchitectureStatement *arch_stmt;
  IIR *stmt = _get_statement_list()->first();

  while( stmt != NULL ) {
    ASSERT( stmt->_is_iir_concurrent_statement() == TRUE );
    arch_stmt = (IIR_ArchitectureStatement *)stmt;

    switch( arch_stmt->get_kind() ){
    case IIR_PROCESS_STATEMENT:
      arch_stmt->_publish_cc_headerfiles_for_cc_default( _cc_out );
      break;

    case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    case IIR_CONCURRENT_GENERATE_IF_STATEMENT:
    case IIR_BLOCK_STATEMENT:
      _set_current_publish_name( "SG" );
      arch_stmt->_publish_cc_headerfiles_for_cc_default( _cc_out );
      _set_current_architecture_name( temp );
      _set_current_publish_name( old_current_name );
      break;

    case IIR_COMPONENT_INSTANTIATION_STATEMENT:{
      const string include_file_name = arch_stmt->_get_cc_elaboration_class_name() + ".hh";
      IIRScram::_publish_cc_include( _cc_out, include_file_name );
      compInst = (IIR_ComponentInstantiationStatement *) arch_stmt;
      ASSERT ( compInst->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT );
      if (compInst->_get_configuration() != NULL) {
	compInst->_get_configuration()->_publish_cc_headers( _cc_out );
      }
      break;
    }
    default:
      cerr << "ERROR! IIRScram_ConcurrentGenerateForStatement::"
	   << "_publish_cc_headerfiles_for_cc( _cc_out ): unknown conc_statement "
	   << "type |" << arch_stmt->get_kind_text() << "|" << endl;
    }

    stmt = _get_statement_list()->successor( stmt );
  }
}

void
IIRScram_ArchitectureStatement::_publish_cc_scoping_prefix( ostream &os, 
							  IIR *finalScope, 
							  IIR *currentScope ){
  IIR *outerScope = NULL;
  IIR_Declaration* tempDeclaration = NULL;
  IIR_ArchitectureStatement* tempConcStmt = NULL;

  if(currentScope == finalScope){
    return;
  }

  if((finalScope->_is_iir_package_declaration() == TRUE) ||
     (finalScope->get_kind() == IIR_PACKAGE_BODY_DECLARATION)) {
    return;
  }

  if ((currentScope->get_kind() == IIR_BLOCK_STATEMENT) || (currentScope->get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT) || (currentScope->get_kind() == IIR_CONCURRENT_GENERATE_IF_STATEMENT)) {
    if (finalScope != currentScope) {
      

      if (currentScope->_is_iir_concurrent_statement() == FALSE ) {
	ASSERT ( currentScope->_is_iir_declaration() == TRUE );
	tempDeclaration = (IIR_Declaration *) this;
	ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
	
	outerScope = tempDeclaration->_get_declarative_region();
      }
      else {
	ASSERT ( currentScope->_get_enclosing_scope() != NULL );
	outerScope = currentScope->_get_enclosing_scope();
      }
      
      ASSERT ( outerScope != NULL );
      
      if (outerScope->_is_iir_declaration() == TRUE ) {
	ASSERT ( outerScope->get_kind() == IIR_ARCHITECTURE_DECLARATION );
	os << "enclosingScope->";
      }
      else {
	ASSERT ( outerScope->_is_iir_concurrent_statement() == TRUE );
	tempConcStmt = (IIR_ArchitectureStatement *) outerScope;
	ASSERT ( outerScope->_is_iir_concurrent_statement() == TRUE );
	
	os << "enclosingScope->";
	
	tempConcStmt->_publish_cc_scoping_prefix( os, finalScope, _get_enclosing_scope() );
      }
    }
  }
  else {
    if (currentScope->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
      ASSERT (currentScope->_get_enclosing_scope() != NULL);
      currentScope->_get_enclosing_scope()->_publish_cc_scoping_prefix( os, finalScope, _get_enclosing_scope());
    }
  }
}

IIR * 
IIRScram_ArchitectureStatement::_transmute() {
  return this;
}

void
IIRScram_ArchitectureStatement::_publish_cc( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureStatement::_publish_cc" );
  
  switch(get_kind()) {
  case IIR_COMPONENT_INSTANTIATION_STATEMENT:
    break;
  case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    break;
  default:
    cerr <<  "_publish_cc_lvalue( _cc_out ) not defined for " << get_kind_text() << "\n";
    break;
  }
}

void
IIRScram_ArchitectureStatement::_publish_cc_binding_name(ostream& os) {
  //  SCRAM_CC_REF( os, "IIRScram_ArchitectureStatement::_publish_cc_binding_name" );
  os << *_get_label();
}

void 
IIRScram_ArchitectureStatement::_publish_cc_enclosing_stmt_to_architecture_path(ostream& outstream) {
  //  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureStatement::_publish_cc_enclosing_stmt_to_architecture_path" );
  
  outstream << _get_enclosing_stmt_to_architecture_path();
}

// For example We have the following code:
// architecture a of ent e is
// begin
// Ablk: block
// begin
// Bblk: block
// begin
// Cblk: block
// begin
// end block;
// end block;
// end block;
// end a
// _get_enclosing_stmt_to_architecture_path() of Ablk gives NULL;
// _get_enclosing_stmt_to_architecture_path() of Bblk gives Ablk;
// _get_enclosing_stmt_to_architecture_path() of Cblk gives Ablk_Bblk;
string 
IIRScram_ArchitectureStatement::_get_enclosing_stmt_to_architecture_path() {
  string retval;

  IIR *enclosing_scope_ptr = _get_enclosing_scope();
  ASSERT(enclosing_scope_ptr != NULL);
  if( enclosing_scope_ptr->get_kind() != IIR_ARCHITECTURE_DECLARATION && 
      enclosing_scope_ptr->get_kind() != IIR_ENTITY_DECLARATION ) {
    if( enclosing_path == NULL ){
      ASSERT(enclosing_scope_ptr->_is_iir_concurrent_statement() == TRUE);
      IIR_ArchitectureStatement *arch_stmt = (IIR_ArchitectureStatement*)enclosing_scope_ptr;
      if( arch_stmt->enclosing_path != NULL) {
	retval = retval + arch_stmt->enclosing_path + "_";
      }
      else {
	string outerScope = arch_stmt->_get_enclosing_stmt_to_architecture_path();
	retval = retval + outerScope + "_";
      }
      retval = retval + arch_stmt->_get_label()->_to_string();
    }
  }

  return retval;
}

void
IIRScram_ArchitectureStatement::_set_enclosing_scope( IIR *outerRegion ){
  enclosingRegion = outerRegion;
}

IIR *
IIRScram_ArchitectureStatement::_get_enclosing_scope(){
  return enclosingRegion;
}

void
IIRScram_ArchitectureStatement::_publish_cc_concurrent_stmt_init( published_file &_cc_out,
								  IIR_DeclarationList * ){
  if (_is_simultaneous_statement() != TRUE) {
    ASSERT((_is_concurrent_generate_statement() == TRUE) ||
     	 (_is_block_statement() == TRUE));
    IIR_AssociationList* generic_map = _get_generic_map_aspect();

    SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureStatement::_publish_cc_concurrent_stmt_init" );
  
    //_cc_out << *(_get_label());
    _get_label()->_publish_cc_elaborate( _cc_out.get_stream() );
    _cc_out << "_elab_obj = new "
	  << _get_cc_elaboration_class_name()
	  << OS("(this");
    if(generic_map != NULL) {
      if (generic_map->num_elements() > 0) {
        _cc_out << "," << NL();
        //_current_publish_node = this;
        generic_map->_publish_cc_generic_map_aspect_for_conc_stmts( _cc_out );
        //_current_publish_node = NULL;
      }
    }
    _cc_out << CS(");");
  }
}

void
IIRScram_ArchitectureStatement::_publish_cc_instantiate_call( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureStatement::_publish_cc_instantiate_call" );
  
  if( _is_simultaneous_statement() != TRUE ){
    switch( get_kind() ) {
    case IIR_PROCESS_STATEMENT:
      _cc_out << "Hierarchy *newHier;" << NL()
	      << "newHier = new Hierarchy;" << NL()
	      << "hier->add_block(\"";
      _cc_out.get_stream() << *get_label();
      _cc_out << "\", newHier);" << NL();
      break;
    default:
      break;
    }
    _get_label()->_publish_cc_elaborate( _cc_out.get_stream() );
    _cc_out << "_elab_obj->instantiate(hier);" << NL();
  }
}

void 
IIRScram_ArchitectureStatement::_set_guard_signal( IIR_SignalDeclaration * ){
  _report_undefined_scram_fn("_set_guard_signal( IIR_Declaration * )");
}

void 
IIRScram_ArchitectureStatement::_resolve_guard_signal( symbol_table *sym_tab ){
  // Lookup all signals named "guard"
  IIR_SignalDeclaration *guard_signal = NULL;
  set<IIR_Declaration> *guard_decls;
  guard_decls = new set<IIR_Declaration>(*sym_tab->find_set( "guard" ));
  if( guard_decls == NULL ){
    ostringstream err;
    err << "No signal |guard|, implicit or explicit, has been declared.";
    report_error( this, err.str() );
    goto finish;
  }
  
  guard_decls->reduce_set( &IIR::_is_signal );
  ASSERT( guard_decls->num_elements() == 1 || guard_decls->num_elements() == 0 );

  if( guard_decls->num_elements() == 1 ){
    // Get rid of any that aren't boolean typed.
    guard_signal = (IIR_SignalDeclaration *)guard_decls->get_element();
    if( guard_signal->get_subtype() != StandardPackage::get_boolean_type() ){
      guard_signal = NULL;
    }
  }

  if( guard_signal == NULL ){
    ostringstream err;
    err << "Signal |guard| must be a signal of builtin type |boolean|.";
    report_error( this, err.str() );    
  }
  else{
    _set_guard_signal( guard_signal );
  }

 finish:
  delete guard_decls;
}

void
IIRScram_ArchitectureStatement::_publish_cc_class_generate_stmt( published_file &_cc_out ){
  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureStatement::_publish_cc_class_generate_stmt" );

  _cc_out << "class " << _get_cc_elaboration_class_name() << " : public _savant_entity_elab {\n"
	  << "public:\n\n"
	  << _get_cc_elaboration_class_name() << "(";
  ASSERT ( _get_enclosing_scope() != NULL );
  _get_enclosing_scope()->_publish_cc_class_name( _cc_out.get_stream() );
  _cc_out << " *);\n";

  _cc_out << "~" << _get_cc_elaboration_class_name() << "();\n"
	  << _get_enclosing_scope()->_get_cc_elaboration_class_name()
  	  << "* enclosingScope;\n";
  
  _cc_out << "void instantiate(Hierarchy * hier);\n";
  _cc_out << "void createNetInfo();\n";
  _cc_out << "void connect(int, int, ...);\n";
  _cc_out << "int getGenerateConstant(int);\n";
  _cc_out << "void resetGenerateConstant();\n";

  _publish_cc_signals( _cc_out, _get_declaration_list());
  ASSERT( _get_statement_list()->get_kind() == IIR_ARCHITECTURE_STATEMENT_LIST );
  _publish_cc_object_pointers( _cc_out, (IIR_ArchitectureStatementList *)_get_statement_list());

  if( get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT ){
    _cc_out << "static int generateConstant;\n";
    _cc_out << "static ArrayInfo::ArrayDirn_t generateDirection;\n";
  }
  
  _cc_out << "};\n\n";
}

void
IIRScram_ArchitectureStatement::_publish_cc_characteristic_expressions( IIRScram_ArchitectureStatement::SimultaneousIfPublishingPart,
									published_file & ){
  _report_undefined_scram_fn("_publish_cc_characteristic_expressions()");
}
