
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer 
//          Krishnan Subramani  
//          Umesh Kumar V. Rajasekaran
//          Malolan Chetlur     
//          Narayanan Thondugulam
//          Swaminathan Subramanian
//	    Magnus Danielson	cfmd@swipnet.se

#include "IIRScram.hh"
#include "IIR_AssociationElementByExpression.hh"
#include "IIR_ArrayTypeDefinition.hh"
#include "IIR_CaseStatementAlternativeByChoices.hh"
#include "IIR_Choice.hh"
#include "IIR_ConcurrentStatement.hh"
#include "IIR_InterfaceDeclaration.hh"
#include "IIR_Label.hh"
#include "IIR_RangeAttribute.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "symbol_table.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"

IIRScram_AssociationElementByExpression::~IIRScram_AssociationElementByExpression() {}

void 
IIRScram_AssociationElementByExpression::_publish_vhdl(ostream &_vhdl_out) {
  if (get_formal() != NULL) {
    if (get_formal()->get_kind() == IIR_INTEGER_SUBTYPE_DEFINITION)
      get_formal()->_publish_vhdl_range(_vhdl_out);
    else 
      get_formal()->_publish_vhdl(_vhdl_out);
    _vhdl_out << " => ";
  }

  if( get_actual() != NULL ){
    get_actual()->_publish_vhdl(_vhdl_out);
  }
  else{
    _vhdl_out << "open";
  }
}

void 
IIRScram_AssociationElementByExpression::_publish_vhdl_without_formals(ostream &_vhdl_out) {
  
  if ( get_formal() != NULL ){
    if ( get_formal()->get_kind() == IIR_FUNCTION_CALL ){
      get_formal()->_publish_vhdl(_vhdl_out);
      _vhdl_out << " => ";
    }
  }

  if( get_actual() != NULL ){
    get_actual()->_publish_vhdl(_vhdl_out);
  }
  else{
    _vhdl_out << "open";
  }
}

IIR_Boolean 
IIRScram_AssociationElementByExpression::_is_resolved(){
  IIR_Boolean retval = TRUE;

  if( get_formal() != NULL && get_formal()->_is_resolved() == FALSE ){
    retval = FALSE;
    goto finish;
  }
  
  if( get_actual() != NULL && get_actual()->_is_resolved() == FALSE ){
    retval = FALSE;
  }

 finish:
  return retval;
}

IIR_Boolean 
IIRScram_AssociationElementByExpression::_is_signal(){
  IIR_Boolean retval = TRUE;

  if( get_actual()->_is_signal() == FALSE ){
    retval = FALSE;
  }

  return retval;
}

IIR_Boolean 
IIRScram_AssociationElementByExpression::_is_variable(){
  IIR_Boolean retval = TRUE;

  if( get_actual()->_is_variable() == FALSE ){
    retval = FALSE;
  }

  return retval;
}

IIR_Boolean 
IIRScram_AssociationElementByExpression::_is_readable(){
  ASSERT( _is_resolved() == TRUE );

  return get_actual()->_is_readable();
}

IIR_Boolean 
IIRScram_AssociationElementByExpression::_is_writable(){
  ASSERT( _is_resolved() == TRUE );

  return get_actual()->_is_writable();
}

IIR_Boolean
IIRScram_AssociationElementByExpression::_is_above_attribute_found(){
  IIR_Boolean retval = FALSE;
  if(get_actual() != NULL) {
    retval = retval || get_actual()->_is_above_attribute_found();
  }
  return retval;
}

void
IIRScram_AssociationElementByExpression::
_build_above_attribute_set(set<IIR_AboveAttribute> *to_build) {
  if(get_actual() != NULL) {
   get_actual()->_build_above_attribute_set(to_build);
  }
}

void 
IIRScram_AssociationElementByExpression::_publish_cc_lvalue( published_file &_cc_out ) {
  ASSERT(_is_resolved() == TRUE);

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationElementByExpression::_publish_cc");
  get_actual()->_publish_cc_lvalue( _cc_out );
}

const string
IIRScram_AssociationElementByExpression::_publish_cc_magic_prefix( published_file &_cc_out ){
  ostringstream prefix;

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationElementByExpression::_publish_cc_magic_prefix");

  ASSERT( _get_current_publish_node() != NULL );
  ASSERT( _get_current_publish_node()->_is_iir_concurrent_statement() == TRUE ||
	  _get_current_publish_node()->get_kind() == IIR_LABEL);

  prefix << "enclosingArch->";
  if( _get_current_publish_node()->_is_iir_concurrent_statement() == TRUE ){
    prefix << *(((IIR_ConcurrentStatement*)_get_current_publish_node())->_get_label()->_get_declarator());
  }
  else {
    ASSERT( _get_current_publish_node()->get_kind() == IIR_LABEL );
    IIR_Label *label = (IIR_Label*)_get_current_publish_node();
    if( label->get_statement() == NULL ){
      //The label that we have is already a mangled one
      prefix << *( label->_get_declarator() );
    }
    else {
      //get the mangled label from unmangled one
      prefix << *(label->get_statement()->_get_label()->_get_declarator());
    }
  }
  prefix << "_elab_obj->";

  return prefix.str();
}

void 
IIRScram_AssociationElementByExpression::_publish_cc_generic_map_aspect( published_file &_cc_out ) {

  ASSERT(_is_resolved() == TRUE);

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationElementByExpression::_publish_cc_generic_map_aspect");
  const string prefix_string = _publish_cc_magic_prefix( _cc_out );
  if(get_formal() != NULL) {
    ASSERT ( (get_formal()->get_subtype()) != NULL);
    _cc_out << "(const " << get_formal()->get_subtype()->_get_cc_type_name();
    _cc_out << " &)";
  }
  _set_publish_prefix_string( prefix_string );
  get_actual()->_publish_cc_lvalue( _cc_out );
  _set_publish_prefix_string( "" );
}

void
IIRScram_AssociationElementByExpression::_publish_cc_generic_map_aspect_for_conc_stmts( published_file &_cc_out )
{

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationElementByExpression::_publish_cc_generic_map_aspect_for_conc_stmts");

  ASSERT(_is_resolved() == TRUE);
  if(get_formal() != NULL) {
    ASSERT( (get_formal()->get_subtype()) != NULL);
    _cc_out << "(const " <<
      get_formal()->get_subtype()->_get_cc_type_name() <<
      "&)";
  }
  
  get_actual()->_publish_cc_lvalue( _cc_out );
}

void 
IIRScram_AssociationElementByExpression::_publish_cc_elaborate( published_file &_cc_out ){

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationElementByExpression::_publish_cc_elaborate");

  get_actual()->_publish_cc_elaborate( _cc_out );
}

IIR_Int32
IIRScram_AssociationElementByExpression::_get_number_of_element_associations() {

  if(get_actual()->get_kind() == IIR_OTHERS_INITIALIZATION) {
    return 1;
  }

  if (get_formal() == NULL) {
    return 1;
  }
  
  if(get_formal()->get_kind() == IIR_CASE_STATEMENT_ALTERNATIVE_BY_CHOICES) {
    IIR_CaseStatementAlternativeByChoices* case_choices = (IIR_CaseStatementAlternativeByChoices*)get_formal();
    return case_choices->choices.num_elements();
  }
  else {
    return 1;
  }
  return 1;
}

void
IIRScram_AssociationElementByExpression::_publish_cc_aggregate_iterator( published_file &_cc_out )
{

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationElementByExpression::_publish_cc_aggregate_iterator");

  _cc_out << "( ";
  if (_get_aggregate_iterator_subtype() != NULL) {
    ASSERT(_get_aggregate_iterator_subtype()->_get_base_type_left() != NULL);
    _get_aggregate_iterator_subtype()->_get_base_type_left()->_publish_cc_value( _cc_out );
    if( _get_aggregate_iterator_subtype()->_get_base_type_direction()->_is_ascending_range()){
      _cc_out << " + ";
    }
    else {
      _cc_out << " - ";
    }
  }
  else {
    _cc_out << "1 + ";
  }
  
  _cc_out << _get_aggregate_iterator_counter() << " )";
}

void 
IIRScram_AssociationElementByExpression::_publish_cc_non_aggregate_object_init( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationElementByExpression::_publish_cc_non_aggregate_object_init");
  switch(get_actual()->get_kind()) {
  case IIR_CONSTANT_DECLARATION:
  case IIR_CONSTANT_INTERFACE_DECLARATION:
  case IIR_HIGH_ATTRIBUTE:
  case IIR_INDEXED_NAME:
  case IIR_LOW_ATTRIBUTE:
  case IIR_SIGNAL_DECLARATION:
  case IIR_SIGNAL_INTERFACE_DECLARATION:
  case IIR_SELECTED_NAME:
  case IIR_VARIABLE_DECLARATION:
  case IIR_VARIABLE_INTERFACE_DECLARATION:
    _cc_out << " new " <<
    get_actual()->get_subtype()->_get_cc_type_name();
    _cc_out << "(true, ";
    if (get_actual()->get_subtype()->_is_scalar_type() == TRUE) {
      get_actual()->_publish_cc_rvalue( _cc_out );
    } 
    else {
      if(get_actual()->get_subtype()->_is_array_type() == TRUE) {
	_cc_out << "ObjectBase::VARIABLE, ";
	if (get_actual()->get_subtype()->_get_num_indexes() < 2) {
	  get_actual()->get_subtype()->_publish_cc_range( _cc_out );
	  _cc_out << ", ";
	}
	
	get_actual()->_publish_cc_rvalue( _cc_out );

	if (get_actual()->get_subtype()->_get_num_indexes() < 2) {
	  _cc_out << ", ";
	  get_actual()->get_subtype()->_publish_cc_bounds( _cc_out );
	}
      }
      else {
	if(get_actual()->get_subtype()->_is_record_type() == TRUE){
	  _cc_out << "ObjectBase::VARIABLE, ";
	  get_actual()->_publish_cc_rvalue( _cc_out );
	}
      }
    }
    _cc_out << ")";
    break;
  default:
    //Currently for all other nodes, the same old stuff is done.
    //For any node, anything special has to be done, lets bother that time.
    //Presume IIR_*Attribute nodes needs work like object and interface 
    //declarations 
    get_actual()->_publish_cc_state_object_init( _cc_out );
    break;
  }
}

void 
IIRScram_AssociationElementByExpression::_publish_cc_state_object_init( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationElementByExpression::_publish_cc_state_object_init");

  if(get_actual()->get_kind() == IIR_OTHERS_INITIALIZATION) {
    _cc_out << OS("new ElementAssociation(Others,");
    _publish_cc_non_aggregate_object_init( _cc_out );
    _cc_out << CS(")");
  }
  else{
    // get_actual()->get_kind() != IIR_OTHERS_INITIALIZATION
    if (get_formal() == NULL) {
      _cc_out << OS("new ElementAssociation(");
      _publish_cc_aggregate_iterator( _cc_out );
      _cc_out << "," << NL();
      _publish_cc_non_aggregate_object_init( _cc_out );
      _cc_out << CS(")");
    }
    else {
      // Formal is not null.
      if( get_formal()->get_kind() == IIR_CASE_STATEMENT_ALTERNATIVE_BY_CHOICES ) {
	IIR_CaseStatementAlternativeByChoices *case_choices = 
	  (IIR_CaseStatementAlternativeByChoices*)get_formal();
	IIR_Choice* current_choice = case_choices->choices.first();
	while(current_choice != NULL) {
	  _cc_out << OS("new ElementAssociation(");
	  current_choice->get_value()->_publish_cc_universal_value( _cc_out );
	  _cc_out << "," << NL();
	  _publish_cc_non_aggregate_object_init( _cc_out );
	  _cc_out << CS(")");
	  current_choice = case_choices->choices.successor(current_choice);
	  if(current_choice != NULL) {
	    _cc_out << "," << NL();
	  }
	}
      }
      else if( get_formal()->get_kind() == IIR_RANGE_ATTRIBUTE ){
	_cc_out << OS("new ElementAssociation(");

	if (get_formal()->get_subtype()->_is_anonymous() == FALSE) {
	  get_formal()->get_subtype()->_publish_cc_array_info( _cc_out );
	}
	else {
	  // This is a nonymous range..have to pick up the range from the formal
	  // variable associated with the range attribute...
	  IIR_RangeAttribute *rangeAttribute = (IIR_RangeAttribute *) get_formal();
	  ASSERT ( rangeAttribute->get_kind() == IIR_RANGE_ATTRIBUTE );
	  _cc_out << OS("ArrayInfo(");
	  rangeAttribute->_publish_cc_range( _cc_out );
	  _cc_out << CS(")");
	}
	
	_cc_out << "," << NL();
	_publish_cc_non_aggregate_object_init( _cc_out );
	_cc_out << CS(")");
      }
      else {
	// We've got a formal and it's not "by choices".
	_cc_out << OS("new ElementAssociation(");
	get_formal()->_publish_cc_universal_value( _cc_out );
	_cc_out << "," << NL();
	_publish_cc_non_aggregate_object_init( _cc_out );
	_cc_out << CS(")");
      }
    }
  }
}

// This method publishes the element as an argument to a subprogram
// (function/procudure).  This is different from _publish_cc because this
// publishes just the declarators.
// For example, 
//   "var" will be published by _publish_cc_lvalue( _cc_out ) as 
//     "state.current->var" and as
//     "var" by _publish_cc_subprogram_arguments( _cc_out ).
// Assumption: get_actual() returns one of IIR_Declaration (any), IIR_Literal
// or IIR_IndexedName 
void
IIRScram_AssociationElementByExpression::_publish_cc_subprogram_arguments( published_file &_cc_out ) {
  ASSERT(get_actual()->_is_resolved());

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationElementByExpression::_publish_cc_subprogram_arguments");

  if(get_actual()->_is_iir_declaration()) {
    ((IIR_Declaration *)get_actual())->_get_declarator()->_publish_cc_lvalue( _cc_out.get_stream() );
  } else if(get_actual()->get_kind() == IIR_INDEXED_NAME) {
    get_actual()->_publish_cc_subprogram_arg( _cc_out );
  } else {    
    get_actual()->_publish_cc_lvalue( _cc_out );
  }
}

void
IIRScram_AssociationElementByExpression::_get_signal_source_info( set<IIR> *siginfo) {
  if(get_formal() != NULL) {
    ASSERT(_is_resolved() == TRUE);
    ASSERT(get_formal() != NULL);
    ASSERT(get_formal()->_is_resolved() == TRUE);
    if( get_formal()->_is_signal() ){
      ASSERT(get_formal()->_is_interface() == TRUE);
      IIR_Mode signal_mode = ((IIR_InterfaceDeclaration *)get_formal())->get_mode();
      if(signal_mode == IIR_OUT_MODE || signal_mode == IIR_INOUT_MODE) {
	siginfo->add( get_actual() );
      }
    }
  }
  else {
    ASSERT(get_actual()->_is_resolved() == TRUE);
    get_actual()->_get_signal_source_info(siginfo);
  }
}

IIR*
IIRScram_AssociationElementByExpression::_get_actual() {
  return get_actual();
}

void 
IIRScram_AssociationElementByExpression::_set_actual( IIR *new_actual ) {
  set_actual( new_actual );
}

void
IIRScram_AssociationElementByExpression::_publish_cc_first_objectParameter( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_AssociationElementByExpression::_publish_cc_first_objectParameter");

  get_actual()->_publish_cc_first_objectParameter( _cc_out );
}
  
set<IIR_TypeDefinition> *
IIRScram_AssociationElementByExpression::_get_rval_set( IIR_Boolean (IIR::*constraint_function)()){
  set<IIR_TypeDefinition> *retval = NULL;

  // the actual will be NULL if the VHDL says "open".
  set<IIR_TypeDefinition> *actual_rvals = NULL;
  if( get_actual() != NULL ){
    actual_rvals = get_actual()->_get_rval_set( constraint_function );
    if( actual_rvals == NULL ){
      report_undefined_symbol( get_actual() );
      return NULL;
    }
  }

  if( get_formal() == NULL ){
    retval = actual_rvals;
  }
  else{
    switch( get_formal()->get_kind() ){
    case IIR_INTEGER_LITERAL:
    case IIR_OTHERS_INITIALIZATION:{
      // Then we need to return an list of the array types with elements that match
      // the actuals.
      
      // Get the set of array types...
      set<IIR_ArrayTypeDefinition> *array_types = _get_symbol_table()->get_in_scope_array_types();

      IIR_TypeDefinition *current_array_type; 
      current_array_type = array_types->get_element();

      while( current_array_type != NULL ){
	IIR_TypeDefinition *actual_type = actual_rvals->get_element();
	
	bool one_matched = false;
	while( actual_type != NULL ){
	  if( actual_type->_is_compatible( current_array_type->_get_element_subtype() ) != NULL ){
	    one_matched = true;
	  }
	  actual_type = actual_rvals->get_next_element();
	}

	if( one_matched == true ){
	  if( retval == NULL ){
	    retval = new set<IIR_TypeDefinition>;
	  }
	  retval->add( current_array_type );
	}
	
	current_array_type = array_types->get_next_element();
      }

      delete actual_rvals;

      break;
    }
    default:{
//       set<IIR_TypeDefinition> *formal_rvals = get_formal()->_get_rval_set();
      if( actual_rvals == NULL ){
	return NULL;
      }
      else{
	return actual_rvals;
      }
    }
    }
  }

  return retval;
}

void 
IIRScram_AssociationElementByExpression::_type_check( set<IIR_TypeDefinition> *context_set ){
  IIR *expr = get_actual();

  ASSERT( expr != NULL );

  expr->_type_check( context_set );
}

IIR *
IIRScram_AssociationElementByExpression::_semantic_transform( set<IIR_TypeDefinition> *context_set ){
  IIR *expr = get_actual();

  ASSERT( expr != NULL );

  set_actual( expr->_semantic_transform( context_set ) );

  return this;
}

IIR *
IIRScram_AssociationElementByExpression::_rval_to_decl( IIR_TypeDefinition *my_rval ){

  if( _is_by_others() == TRUE &&
      my_rval->_is_array_type() == TRUE ){
    // Actual is an element of the array.
    IIR_TypeDefinition *element_subtype = my_rval->_get_element_subtype();
    set_actual( get_actual()->_rval_to_decl( element_subtype ) );
  }
  else{    
    set_actual( get_actual()->_semantic_transform( my_rval ) );
    get_actual()->_type_check( my_rval );
    set_actual( get_actual()->_rval_to_decl( my_rval ) );
  }


  return (IIR *)this;
}

ostream &
IIRScram_AssociationElementByExpression::_print( ostream &os ){
  if( get_formal() != NULL ){
    os << *get_formal();
    os << " => ";
  }

  if( get_actual() != NULL ){
    os << *get_actual();
  }

  return os;
}

void
IIRScram_AssociationElementByExpression::_build_sensitivity_list(IIR_DesignatorList* sensitivity_list){
  get_actual()->_build_sensitivity_list(sensitivity_list);
}

void 
IIRScram_AssociationElementByExpression::_add_decl_into_cgen_symbol_table() {
  if(get_actual() != NULL) {
    get_actual()->_add_decl_into_cgen_symbol_table();
  }
}

IIR*
IIRScram_AssociationElementByExpression::_clone() {
  IIR *elem;
  IIR_AssociationElementByExpression *assoc;

  assoc = new IIR_AssociationElementByExpression();
  IIRScram::_clone( assoc );

  if (get_formal() != NULL) {
    elem = get_formal()->_clone();
    assoc->set_formal(elem);
  }
  elem = get_actual()->_clone();
  assoc->set_actual(elem);

  return assoc;
}

IIR_Boolean 
IIRScram_AssociationElementByExpression::_is_globally_static_for_array(){
  IIR_Boolean retval = TRUE;

  if( get_formal() != NULL &&
      get_formal()->_is_iir_scalar_type_definition() == TRUE &&
      get_formal()->_is_globally_static() == FALSE ){
    retval = FALSE;
  }    

  // Can't have an OPEN array element association.
  ASSERT( get_actual() != NULL );
  
  if( get_actual()->_is_globally_static() == FALSE ){
    retval = FALSE;
  }
  
  return retval;
}

IIR_Boolean 
IIRScram_AssociationElementByExpression::_is_globally_static_for_record(){
  // Can't have an OPEN record element association.
  ASSERT( get_actual() != NULL );

  return get_actual()->_is_globally_static();
}

IIR_Boolean 
IIRScram_AssociationElementByExpression::_is_globally_static_for_function(){
  if( get_actual() == NULL || get_actual()->_is_globally_static() == TRUE ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

visitor_return_type *IIRScram_AssociationElementByExpression::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_AssociationElementByExpression(this, arg);
};
