
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Malolan Chetlur     
//          Umesh Kumar V. Rajasekaran
//          Krishnan Subramani  
//          Narayanan Thondugulam
//          Swaminathan Subramanian
//	    Magnus Danielson	cfmd@swipnet.se

//---------------------------------------------------------------------------

#include "IIR_ProcessStatement.hh"
#include "IIR_Identifier.hh"
#include "IIR_Label.hh"
#include "IIR_GenericList.hh"
#include "IIR_PortList.hh"
#include "IIR_AssociationList.hh"
#include "IIR_DeclarationList.hh"
#include "IIR_AssociationElement.hh"
#include "IIR_TypeDefinition.hh"
#include "IIR_ComponentDeclaration.hh"
#include "IIR_ConstantInterfaceDeclaration.hh"
#include "IIR_ComponentConfiguration.hh"
#include "IIR_ComponentInstantiationStatement.hh"
#include "IIR_SignalInterfaceDeclaration.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_AliasDeclaration.hh"
#include "IIR_BlockStatement.hh"
#include "IIR_ConfigurationSpecification.hh"
#include "IIR_Designator.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_EnumerationLiteral.hh"
#include "IIR_IntegerLiteral.hh"
#include "IIR_ConcurrentGenerateForStatement.hh"
#include "IIR_ConcurrentGenerateIfStatement.hh"
#include "IIR_FunctionCall.hh"
#include "IIR_IndexedName.hh"
#include "IIR_SubprogramDeclaration.hh"
#include "IIR_LibraryDeclaration.hh"
#include "set.hh"
#include "resolution_func.hh"
#include "symbol_table.hh"
#include "published_header_file.hh"
#include "published_cc_file.hh"
#include "sstream-wrap.hh"


IIR*
IIRScram_BlockStatement::guard_expr = NULL;

IIRScram_BlockStatement::IIRScram_BlockStatement() {
  my_implicit_guard_signal = NULL;
}

IIRScram_BlockStatement::~IIRScram_BlockStatement(){}

void 
IIRScram_BlockStatement::_set_implicit_guard_signal(IIR_SignalDeclaration *implicit_guard_signal ){
  my_implicit_guard_signal = implicit_guard_signal;
}

IIR_SignalDeclaration *
IIRScram_BlockStatement::_get_implicit_guard_signal(  ){
  return my_implicit_guard_signal;
}

IIR_GenericList *
IIRScram_BlockStatement::_get_generic_list(){
  return &generic_clause;
}

void 
IIRScram_BlockStatement::_publish_vhdl(ostream &_vhdl_out) {

  _publish_vhdl_stmt_label(_vhdl_out);

  _vhdl_out << "  block ";
  if( get_guard_expression() != NULL) {
    _vhdl_out << "(";
    get_guard_expression()->_publish_vhdl(_vhdl_out);
    _vhdl_out << ")";
    _vhdl_out << " is\n";
  }

  if(generic_clause.num_elements() != 0) {
    _vhdl_out << "   generic ( ";
    generic_clause._publish_vhdl(_vhdl_out);
    _vhdl_out << " );\n";

    if(generic_map_aspect.num_elements() != 0) {
      _vhdl_out << "   generic map ( ";
      generic_map_aspect._publish_vhdl(_vhdl_out);
      _vhdl_out << " );\n";
    }
  }

  if(port_clause.num_elements() != 0) {
    _vhdl_out << "   port ( ";
    port_clause._publish_vhdl(_vhdl_out);
    _vhdl_out << " );\n";

    if(port_map_aspect.num_elements() != 0) {
      _vhdl_out << "   port map ( ";
      port_map_aspect._publish_vhdl(_vhdl_out);
      _vhdl_out << " );\n";
    }
  }

  block_declarative_part._publish_vhdl_decl(_vhdl_out);

  _vhdl_out << " begin\n";
  block_statement_part._publish_vhdl(_vhdl_out);
  _vhdl_out << " end block ";

  if (get_label() != NULL) {
    get_label()->_publish_vhdl(_vhdl_out);
  }  
  _vhdl_out << ";\n";
}

void 
IIRScram_BlockStatement::_type_check(){
  // Things we need to check...
  // if we have a guard expression, it must be boolean valued.
  if( get_guard_expression() != NULL ){
    set_guard_expression( _type_check_and_resolve_boolean_condition( get_guard_expression() ) );

    // Now that we've resolved this, we'll set the default value of the implicit signal to
    // the resolved expression.
    _get_implicit_guard_signal()->set_value( get_guard_expression() );
  }
  
#ifdef DEVELOPER_ASSERTIONS
  if( get_guard_expression() == NULL ){
    ASSERT( _get_implicit_guard_signal() == NULL );
  }
  else{
    ASSERT( _get_implicit_guard_signal() != NULL );
  }
#endif

  // We need to type check the port map and generic map if they have stuff in them.

  generic_map_aspect._resolve_and_order( &generic_clause, NULL, this );
  port_map_aspect._resolve_and_order( &port_clause, NULL, this );  

  _get_symbol_table()->open_scope( this );
  _make_interface_visible( _get_symbol_table() );
  block_declarative_part._type_check_disconnection_specifications();
  block_declarative_part._type_check_configuration_specifications( block_statement_part );
  block_declarative_part._type_check_attribute_specifications( block_statement_part );
  _get_symbol_table()->close_scope( this );


}

void 
IIRScram_BlockStatement::_type_check_instantiate_statements(){ 
  _get_symbol_table()->open_scope( this );
  _make_interface_visible( _get_symbol_table() );
  block_statement_part._type_check_instantiate_statements();
  _get_symbol_table()->close_scope( this );
}


set<IIR_Declaration> *
IIRScram_BlockStatement::_find_declarations( IIR_Name *to_find ){
  set<IIR_Declaration> *retval = new set<IIR_Declaration>;
  
  set<IIR_Declaration> *current_set = port_clause._find_declarations(to_find );
  
  if( current_set != NULL ){
    retval->add( current_set );
    delete current_set;
  }

  current_set = generic_clause._find_declarations(to_find );
  if( current_set != NULL ){
    retval->add( current_set );
    delete current_set;
  }

  current_set = block_declarative_part._find_declarations(to_find );
  if( current_set != NULL ){
    retval->add( current_set );
    delete current_set;
  }

  if( retval->num_elements() == 0 ){
    delete retval;
    retval = NULL;
  }

  return retval;
}

void 
IIRScram_BlockStatement::_make_interface_visible( symbol_table *sym_tab ){
  sym_tab->make_visible( &generic_clause );
  sym_tab->make_visible( &port_clause );
  sym_tab->make_visible( &block_declarative_part );
  block_statement_part._make_interface_visible( sym_tab );
}

IIR_List *
IIRScram_BlockStatement::_get_statement_list(){
  return &block_statement_part;
}

void
IIRScram_BlockStatement::_publish_cc_blocks_elaborate( published_file &_cc_out ){
  IIR_ArchitectureStatement *arch_stmt;
  const string temp = _get_current_architecture_name();
  const string old_current_name = _get_current_publish_name();

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_blocks_elaborate" );
  
  _set_current_publish_name( "" );
  
  arch_stmt = block_statement_part.first();
  
  while (arch_stmt != NULL) {
    if (arch_stmt->get_kind() != IIR_COMPONENT_INSTANTIATION_STATEMENT &&
	arch_stmt->get_kind() != IIR_PROCESS_STATEMENT) {
      arch_stmt->_publish_cc_elaborate( _cc_out );
    }
    arch_stmt = block_statement_part.successor(arch_stmt);
  }
  _set_current_architecture_name( temp );
  _set_current_publish_name( old_current_name );
}

void
IIRScram_BlockStatement::_publish_cc( published_file &_cc_out ) {
  IIR *temp = _get_current_publish_node();
  const string old_architecture_name = _get_current_architecture_name();
  const string old_current_name = _get_current_publish_name();
  IIR *old_guard_expr;

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc" );

  PublishedUnit _saved_publishing_unit = _get_currently_publishing_unit();
  _set_currently_publishing_unit(BLOCK);
  _set_current_publish_name( "" );
  _set_current_publish_node( this );

  _publish_cc_declarations( );
  old_guard_expr = guard_expr;
  // Set the static variable so that it can be used in the process using guards
  guard_expr = get_guard_expression();
  block_statement_part._publish_cc( _cc_out );
  guard_expr = old_guard_expr;

  _set_current_publish_name( old_current_name );
  _set_current_architecture_name( old_architecture_name );
  _set_current_publish_node( temp );
  _set_currently_publishing_unit(_saved_publishing_unit);
}

void
IIRScram_BlockStatement::_publish_cc_declarations( ) {
  string block_unit_name;
  ostringstream ent_arch_block;

  ent_arch_block << _get_current_entity_name() << "_"
		 << _get_current_architecture_name();
  _publish_cc_enclosing_stmt_to_architecture_path(ent_arch_block);
  ent_arch_block << "_" << *_get_label();
  block_unit_name = ent_arch_block.str();

  published_header_file header_file( _get_work_library()->_get_path_to_directory(), 
				     block_unit_name,
				     this );
  SCRAM_CC_REF( header_file, "IIRScram_BlockStatement::_publish_cc_declarations" );
  _publish_cc_include_elab( header_file );
  // If this block has an implicit guard signal then add it to the list
  if (_get_implicit_guard_signal() != NULL) {
    block_declarative_part.append(_get_implicit_guard_signal());
  }

  generic_clause._publish_cc_elaborate_as_pointers( header_file, FALSE );
  block_declarative_part._publish_cc( header_file );
  
  published_cc_file cc_file( _get_work_library()->_get_path_to_directory(), 
			     block_unit_name,
			     this );
  SCRAM_CC_REF( cc_file, "IIRScram_BlockStatement::_publish_cc_declarations" );
  cc_file << "#include \"" << block_unit_name << ".hh\""
	  << NL() << NL();

  generic_clause._publish_cc_elaborate_as_pointers( cc_file, TRUE );
  block_declarative_part._publish_cc_decl( cc_file );
}

void
IIRScram_BlockStatement::_publish_cc_elaborate( published_file &_cc_out ){
  IIR* old_publish_node = _get_current_publish_node();

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_elaborate" );

  _set_current_publish_node( this );
  
  _publish_cc_headerfile( );
  _publish_cc_ccfile( );

  _publish_cc_blocks_elaborate( _cc_out );

  _set_current_publish_node( old_publish_node );
}

void
IIRScram_BlockStatement::_publish_cc_headerfile(){
  const string filename = _get_cc_elaboration_class_name();

  published_header_file header_file( _get_work_library()->_get_path_to_directory(), 
				     filename,
				     this );
  SCRAM_CC_REF( header_file, "IIRScram_BlockStatement::_publish_cc_headerfile" );
  
  IIRScram::_publish_cc_include( header_file, "tyvis/_savant_entity_elab.hh" );

  IIRScram::_publish_cc_include( header_file, _get_cc_elaboration_class_name() + "_elab.hh" );

//   header_file << "#include \"";
//   header_file << _get_current_publish_name() << _get_current_entity_name() << "_"
// 	  << _get_current_architecture_name() ;
//   _publish_cc_enclosing_stmt_to_architecture_path( header_file );
//   header_file << "_elab.hh\"\n";
  // Include the declarations file 
//   header_file << "#include \"";
//   header_file << _get_current_entity_name() << "_" << _get_current_architecture_name();
//   _publish_cc_enclosing_stmt_to_architecture_path( header_file );
//   header_file << "_" << *_get_label();
//   header_file << ".hh\"\n";

  IIRScram::_publish_cc_include( header_file, _get_cc_elaboration_class_name() + "_decls.hh" );

  // Request the Hierarchy.hh for our hierarchy stuff
  IIRScram::_publish_cc_include( header_file, "tyvis/Hierarchy.hh" );

  header_file << "\n";
  _publish_cc_class_includes( header_file );
  header_file << "\n";
  _publish_cc_class( header_file );
}

void
IIRScram_BlockStatement::_publish_cc_class_includes( published_file &_cc_out ) {
  const string temp = _get_current_architecture_name();
  const string old_current_name = _get_current_publish_name();

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_class_includes" );
  
  _set_current_publish_name( "" );
  _cc_out << "class " << _get_cc_elaboration_class_name() << ";\n";

  IIRScram::_publish_cc_class_includes( _cc_out, &block_statement_part );

  _set_current_architecture_name( temp );
  _set_current_publish_name( old_current_name );
}

void
IIRScram_BlockStatement::_publish_cc_class( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_class" );
  
  _cc_out << "class " << _get_cc_elaboration_class_name()
	  << " : public _savant_entity_elab {\n"
	  << "public:\n\n"
	  << _get_cc_elaboration_class_name()
	  << " (" << _get_current_publish_name() << _get_current_entity_name() << "_"
	  << _get_current_architecture_name();
  _publish_cc_enclosing_stmt_to_architecture_path( _cc_out.get_stream() );
  _cc_out << "_elab *);\n";

  _cc_out << _get_current_publish_name() << _get_current_entity_name() << "_"
	  << _get_current_architecture_name();
  _publish_cc_enclosing_stmt_to_architecture_path( _cc_out.get_stream() );
  _cc_out  << "_elab *enclosingScope;\n\n";
  
  if (generic_clause.num_elements() > 0) {
    _cc_out << _get_cc_elaboration_class_name()
	    << "(";
    _cc_out << _get_current_publish_name() << _get_current_entity_name() << "_"
	    << _get_current_architecture_name();
    _publish_cc_enclosing_stmt_to_architecture_path( _cc_out.get_stream() );
    _cc_out  << "_elab *enclosingScope, \n";
    generic_clause._publish_generic_parameter_list( _cc_out );
    _cc_out << ");\n";
  }

  _cc_out << "~" << _get_cc_elaboration_class_name() << "();\n";

  _cc_out << "void instantiate(Hierarchy * hier);\n";
  _cc_out << "void createNetInfo();\n";
  _cc_out << "void connect(int, int, ...);\n\n";
  
  block_declarative_part._publish_cc_extern_type_info( _cc_out );
  block_statement_part._publish_cc_extern_type_info( _cc_out );
  
  _publish_cc_signals( _cc_out, &block_declarative_part );
  generic_clause._publish_cc_elaborate( _cc_out );
  port_clause._publish_cc_elaborate( _cc_out );
  
  _publish_cc_object_pointers( _cc_out, &block_statement_part );

  if (generic_clause.first() != NULL) {
    _cc_out << "  void copyGenericsToGlobals();\n";
  }
  
  _cc_out << "};\n";
}

void
IIRScram_BlockStatement::_publish_cc_ccfile( ){
  const string filename = _get_cc_elaboration_class_name();
  published_cc_file cc_file( _get_work_library()->_get_path_to_directory(),
			     filename,
			     this );
  SCRAM_CC_REF( cc_file, "IIRScram_BlockStatement::_publish_cc_ccfile" );

  _publish_cc_include_elab( cc_file );
  _publish_cc_headerfiles_for_cc( cc_file );
  cc_file << "extern VHDLKernel *proc_array[];\n";

  _publish_cc_constructor( cc_file );
  _publish_cc_destructor( cc_file, &block_statement_part );
  if (generic_clause.first() != NULL) {
    _publish_cc_copy_generics_to_globals( cc_file );
  }
  _publish_cc_instantiate( cc_file );
  _publish_cc_createNetInfo( cc_file );
  _publish_cc_connect_call( cc_file );
  //_publish_cc_partition( cc_file );
}

void
IIRScram_BlockStatement::_publish_cc_headerfiles_for_cc( published_file &_cc_out ){
  const string temp = _get_current_architecture_name();
  const string old_current_name = _get_current_publish_name();

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_headerfiles_for_cc" );

  _set_current_publish_name( "" );

  _publish_cc_headerfiles_for_cc_default( _cc_out );

  IIR_Declaration* decl = block_declarative_part.first();
  while(decl != NULL) {
    if(decl->get_kind() == IIR_CONFIGURATION_SPECIFICATION) {
      if (decl->_get_entity_aspect() != NULL) {
	decl->_get_entity_aspect()->_publish_cc_headerfiles_for_cc_default( _cc_out );
      }
    }
    // Nothing for other nodes.
    decl =  block_declarative_part.successor(decl);
  }

  IIR_ArchitectureStatement *arch_stmt = block_statement_part.first();
  while (arch_stmt != NULL) {
    switch( arch_stmt->get_kind() ){
    case IIR_PROCESS_STATEMENT:
    case IIR_BLOCK_STATEMENT:
    case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    case IIR_CONCURRENT_GENERATE_IF_STATEMENT:
      arch_stmt->_publish_cc_headerfiles_for_cc_default( _cc_out );
      break;

    case IIR_COMPONENT_INSTANTIATION_STATEMENT:
      arch_stmt->_publish_cc_headerfiles_for_cc_default( _cc_out );
      if (((IIR_ComponentInstantiationStatement *) arch_stmt)->_get_configuration() != NULL) {
	((IIR_ComponentInstantiationStatement *) arch_stmt)->_get_configuration()->_publish_cc_headers( _cc_out );
      }
      break;
      
    default:
      // do nothing.
      ;
    }

    arch_stmt = block_statement_part.successor( arch_stmt );
  }
  
  _set_current_architecture_name( temp );
  _set_current_publish_name( old_current_name );
}

void
IIRScram_BlockStatement::_publish_cc_constructor( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_constructor" );

  _publish_cc_constructor_with_no_arguments( _cc_out );
  if (generic_clause.num_elements() > 0) {
    _publish_cc_constructor_with_arguments( _cc_out );
  }
}

void
IIRScram_BlockStatement::_publish_cc_constructor_with_no_arguments( published_file &_cc_out ){
  int numGenericClause     = generic_clause.num_elements();
  int numPortClause        = port_clause.num_elements();
  IIR_Boolean firstDeclFlag= FALSE;
  
  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_constructor_with_no_arguments" );
  
  _cc_out << _get_cc_elaboration_class_name () << "::" 
	  << _get_cc_elaboration_class_name() << "("
	  << _get_current_publish_name() << _get_current_entity_name() << "_"
	  << _get_current_architecture_name();
  _publish_cc_enclosing_stmt_to_architecture_path( _cc_out.get_stream() );
  _cc_out << "_elab *outerScope)";
  
  firstDeclFlag = block_declarative_part._publish_cc_type_info( _cc_out, TRUE );
  
  if(numGenericClause > 0) {
    if (firstDeclFlag == TRUE) {
      _cc_out << ":\n";
    }
    else {
      _cc_out << ",\n";
    }
    
    generic_clause._publish_generic_init( _cc_out );
    if(numPortClause > 0) {
      _cc_out << ",\n";
      port_clause._publish_cc_port_init( _cc_out );
    }
  }
  else {
    if(numPortClause > 0) {
      if (firstDeclFlag == TRUE) {
	_cc_out << ":\n";
      }
      else {
	_cc_out << ",\n";
      }
      
      port_clause._publish_cc_port_init( _cc_out );
    }
  }
  
  _publish_cc_signal_objects_init( _cc_out, firstDeclFlag );
  _cc_out << " {\n";
  _cc_out << "    enclosingScope = outerScope;\n";

  // before newing the elab guys down the heirarchy copy generics of this
  // block to the generic pointers in the block_decls file
  if (numGenericClause > 0) {
    _cc_out << "    copyGenericsToGlobals();\n";
  }
  _publish_cc_object_pointers_init( _cc_out,
				    &block_statement_part,
				    &block_declarative_part);
  _cc_out << "}\n";
}

void
IIRScram_BlockStatement::_publish_cc_constructor_with_arguments( published_file &_cc_out ){
  int numGenericClause      = generic_clause.num_elements();
  int numPortClause         = port_clause.num_elements();
  IIR_Boolean firstDeclFlag = TRUE;
  
  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_constructor_with_arguments" );
  _cc_out << _get_cc_elaboration_class_name() << "::" << _get_cc_elaboration_class_name() 
	  << "(\n"
  	  << _get_current_publish_name() << _get_current_entity_name() << "_"
	  << _get_current_architecture_name();
  _publish_cc_enclosing_stmt_to_architecture_path( _cc_out.get_stream() );
  _cc_out << "_elab *outerScope, ";
    
  generic_clause._publish_generic_parameter_list( _cc_out );
  _cc_out << " )";

  firstDeclFlag = block_declarative_part._publish_cc_type_info( _cc_out, TRUE );
  
  if(numGenericClause > 0) {
    if (firstDeclFlag == TRUE) {
      _cc_out << ":\n";
    }
    else {
      _cc_out << ",\n";
    }
    
    generic_clause._publish_generic_init_by_arguments( _cc_out );
    if(numPortClause > 0) {
      _cc_out << ",\n";
      port_clause._publish_cc_port_init( _cc_out );
    }
  }
  else {
    if(numPortClause > 0) {
      if (firstDeclFlag == TRUE) {
	_cc_out << ":\n";
      }
      else {
	_cc_out << ",\n";
      }
      port_clause._publish_cc_port_init( _cc_out );
    }
  }
  
  _publish_cc_signal_objects_init( _cc_out, TRUE );
  _cc_out << " {\n";
  _cc_out << "  enclosingScope = outerScope;\n";
  _publish_cc_object_pointers_init( _cc_out, &block_statement_part, &block_declarative_part );
  _cc_out << "}\n";
}

void
IIRScram_BlockStatement::_publish_cc_signal_objects_init( published_file &_cc_out,
							  IIR_Boolean firstDeclFlag ){
  IIR_Declaration* decl = NULL;
  int first = 0;
  decl = block_declarative_part.first();
  int numGenericClause = generic_clause.num_elements();
  int numPortClause    = port_clause.num_elements();

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_signal_objects_init" );
  
  while (decl != NULL) {
    if(decl->get_kind() == IIR_SIGNAL_DECLARATION) {
      if((first == 0) && (firstDeclFlag == TRUE) && (numGenericClause == 0)
	 && (numPortClause == 0)) {
	_cc_out << ":\n";
	first = 1;
      }
      else {
	_cc_out << ",\n";
      }
      decl->_publish_cc_elaborate( _cc_out );
      _cc_out << "(ObjectBase::SIGNAL_NETINFO";
      
      if (((decl->get_subtype()->_is_array_type() == TRUE) ||
	   (decl->get_subtype()->_is_record_type() ==  TRUE)) &&
	  (decl->get_subtype()->_is_access_type() == FALSE)) {
	decl->get_subtype()->_publish_cc_object_type_info( _cc_out );
	_cc_out << ", ";
	decl->get_subtype()->_publish_cc_resolution_function_id( _cc_out );

	if (decl->get_subtype()->_is_anonymous() == TRUE) {
	  _cc_out << ", ";
	  decl->get_subtype()->_publish_cc_range( _cc_out );
	}
      }
      
      _cc_out << ")";
      //If the Signals has implicit signals
      //They are also needed for elaboration info
      if( decl->_get_implicit_declarations() != NULL && 
	  decl->_get_implicit_declarations()->num_elements() != 0) {
	IIR_Declaration* imp_decl = decl->_get_implicit_declarations()->get_element();
	while(imp_decl != NULL) {
	  if(imp_decl->get_kind() == IIR_SIGNAL_DECLARATION) {
	    _cc_out << ",\n";
	    imp_decl->_publish_cc_elaborate( _cc_out );
	    _cc_out << "(ObjectBase::SIGNAL_NETINFO";
	    
	    if (((imp_decl->get_subtype()->_is_array_type() == TRUE) ||
		 (imp_decl->get_subtype()->_is_record_type() ==  TRUE)) &&
		(imp_decl->get_subtype()->_is_access_type() ==  FALSE)) {
	      imp_decl->get_subtype()->_publish_cc_object_type_info(_cc_out);
	      _cc_out << ", ";
	      imp_decl->get_subtype()->_publish_cc_resolution_function_id( _cc_out );
	      
	      if (imp_decl->get_subtype()->_is_anonymous() == TRUE) {
		_cc_out << ", ";
		imp_decl->get_subtype()->_publish_cc_range( _cc_out );
	      }
	    }
	    _cc_out << ")";
	  }
	  imp_decl = decl->_get_implicit_declarations()->get_next_element();
	}
      }
    }
    else if(decl->get_kind() == IIR_ALIAS_DECLARATION) {
      if(((IIR_AliasDeclaration*)decl)->get_name()->_is_signal()) {
	if((first == 0) && (firstDeclFlag == 1) && (numGenericClause == 0)
	   && (numPortClause == 0)) {
	  _cc_out << ":\n";
	  first = 1;
	}
	else {
	  _cc_out << ",\n";
	}
	((IIR_AliasDeclaration*)decl)->_publish_cc_elaborate_alias_init( _cc_out );
      }
    }
    decl = block_declarative_part.successor(decl);    
  }
}

void
IIRScram_BlockStatement::_publish_cc_instantiate( published_file &_cc_out )
{
  IIR_ArchitectureStatement *arch_stmt;

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_instantiate" );

  _cc_out << "void\n" << _get_cc_elaboration_class_name() 
    	  << "::instantiate(Hierarchy * hier)  {\n";

  arch_stmt = block_statement_part.first();
  while (arch_stmt != NULL) {
    arch_stmt->_publish_cc_instantiate_call( _cc_out );
    arch_stmt = block_statement_part.successor(arch_stmt);
  }
  _cc_out << "createNetInfo();\n";
  _cc_out << "}\n";
}

void
IIRScram_BlockStatement::_publish_cc_createNetInfo( published_file &_cc_out )
{
  IIR_ArchitectureStatement *arch_stmt;
  IIR_Boolean found = false;
  int wanted_instantiation = 1;
  const string tmp  = _get_current_elab_name();
  const string tmp2 = _get_current_publish_name();
  IIR* tmpNode   = _get_current_publish_node();
  PublishedUnit oldunit = _get_currently_publishing_unit();

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_createNetInfo" );
  
  _cc_out << "void\n"
	  << _get_cc_elaboration_class_name() <<"::createNetInfo() {\n";

  _set_currently_publishing_unit(BLOCK);
  _set_current_publish_node( this );
  
  arch_stmt = block_statement_part.first();
  while (arch_stmt!=NULL) {
    _set_current_publish_node( this );
    if (arch_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      ostringstream objectname;
      if(arch_stmt->_get_label() != NULL) {
	objectname << *(arch_stmt->_get_label()->_get_declarator());
      }
      else {
	objectname << "ANON_PROCESS" << arch_stmt;
      }
      objectname << "_elab_obj";
      _set_current_elab_name( objectname.str() );
      arch_stmt->_publish_createNetInfo( _cc_out );
      _set_current_elab_name( "" );
    }
    else if ((arch_stmt->_is_block_statement() == TRUE) ||
	     (arch_stmt->_is_concurrent_generate_statement() == TRUE )){
    }
    else if ( arch_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT ){
      arch_stmt->_publish_createNetInfo( _cc_out );
    }
    else {
      cerr << "ERROR! IIRScram_BlockStatement::"
	   << "_publish_cc_createNetInfo( _cc_out ): unknown conc_statement "
	   << "type |" << arch_stmt->get_kind_text() << "| in arch" << endl;
    }
    found = FALSE;
    wanted_instantiation = 1;
    arch_stmt = block_statement_part.successor(arch_stmt);
  }
  _cc_out << "}\n\n";

  _set_currently_publishing_unit(oldunit);
  
  _set_current_elab_name( tmp );
  _set_current_publish_name( tmp2 );
  _set_current_publish_node( tmpNode );
}

void
IIRScram_BlockStatement::_publish_cc_connect_call( published_file &_cc_out ){
  IIR_AssociationElement* actual_clause = NULL;

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_connect" );

  _cc_out << "void" << NL()
	  << _get_cc_elaboration_class_name()
	  << OS("::connect(int inputsignals, int outputsignals, ...) {")
	  << "int NoofSignals = inputsignals + outputsignals;" << NL()
	  << "va_list ap;" << NL()
	  << "va_start(ap, outputsignals);" << NL()
	  << "for(int i=0; i <NoofSignals; i++) {" << NL()
	  << "  addToFanOut( va_arg(ap, VHDLType*) );" << NL()
	  << "}" << NL()
	  << "va_end(ap);" << NL();
  
  //Pass on the output connection  inforamtion to its output signals
  IIR_InterfaceDeclaration* portelement = NULL;
  IIR_Mode mode;
  int index = 0;

  _cc_out << OS("if(inputsignals > 0){");
  portelement = port_clause.first();
  actual_clause = port_map_aspect.first();
  for(; portelement != NULL; ) {
    mode = portelement->get_mode();
    if( mode == IIR_IN_MODE ) {
      if (!(actual_clause->_get_actual()->_is_static_expression())){
	_cc_out << "setSourceInfo(";
	portelement->_publish_cc_elaborate( _cc_out ); 
	_cc_out << "," << NL() 
		<< "*(fanOutInfo["
		<< index << "]));" << NL();
	index++;
      }
    }
    portelement = port_clause.successor(portelement);
    if (port_map_aspect.successor(actual_clause)!=NULL){
      actual_clause = port_map_aspect.successor(actual_clause);
    }
  }
  _cc_out << CS("}");
  _cc_out << OS("if( outputsignals > 0 ){");
  portelement = port_clause.first();
  for(; portelement != NULL; ) {
    mode = portelement->get_mode();
    if(mode == IIR_INOUT_MODE || mode == IIR_OUT_MODE) {
      if (!(actual_clause->_get_actual()->_is_static_expression())){
	_cc_out << OS("Add(");
	portelement->_publish_cc_elaborate( _cc_out ); 
	_cc_out << "," << NL() 
		<< "*(fanOutInfo["<< index << "])" << CS(");") << NL();
	if (mode != IIR_OUT_MODE){
	  _cc_out << OS("setSourceInfo(");
	  portelement->_publish_cc_elaborate( _cc_out ); 
	  _cc_out << "," << NL()
		  << "*(fanOutInfo[" << index << "])" << CS(");");
	}
	index++;
      }
    }
    portelement = port_clause.successor(portelement);
  }
  _cc_out << CS("}");

  IIR_ArchitectureStatement *arch_stmt;
  arch_stmt = block_statement_part.first();
  while (arch_stmt != NULL) {
    arch_stmt->_publish_cc_connect_call( _cc_out );
    arch_stmt = block_statement_part.successor(arch_stmt);
  }
  _cc_out << CS("}");
}

void
IIRScram_BlockStatement::_publish_connect( published_file &_cc_out ){
  IIR_AssociationElement* actual_clause         = NULL;
  int                     noofinputsignals      = 0;
  int                     noofoutputsignals     = 0;
  IIR_Boolean             first                 = false;
  IIR_FunctionCall        *downTypeConversionFn = NULL;
  IIR_Declaration         *tempDeclaration      = NULL;
  IIR_Label               *label                = _get_label();
  int                     literal_count         = 0;

  _get_label()->_publish_cc_elaborate( _cc_out.get_stream() );

  if (_get_currently_publishing_unit() == GENERATE_FOR) {
    _cc_out << "_elab_obj[i - generateLeft].connect(";
  }
  else {
    _cc_out << "_elab_obj->connect(";
  }

  actual_clause = port_map_aspect.first();
  
  if (actual_clause == NULL) {
    _cc_out << "0, 0);\n";
    return;
  }

  while(actual_clause != NULL) {
    ASSERT(actual_clause->get_formal() != NULL);
    switch(actual_clause->get_formal()->_get_mode()) {
    case IIR_IN_MODE:
      noofinputsignals++;
      first = true;
      break;
    case IIR_OUT_MODE:
    case IIR_INOUT_MODE:
      noofoutputsignals++;
      first = true;
      break;
    default:
      // do nothing
      break;
    }
    if (actual_clause->_get_actual()->_is_static_expression() == TRUE){ 
        literal_count++;
    }
    actual_clause = port_map_aspect.successor(actual_clause);
  }
  
   if (literal_count == noofinputsignals + noofoutputsignals) {
    _cc_out << "0, 0);\n";
    return;
  }

  actual_clause = port_map_aspect.first();

  while(actual_clause != NULL) {
    ASSERT(actual_clause->get_formal() != NULL);
    switch(actual_clause->get_formal()->_get_mode()) {
    case IIR_IN_MODE:
      if (actual_clause->_get_actual()->_is_static_expression()){
	  noofinputsignals--;
      }
      first = true;
      break;
    case IIR_OUT_MODE:
    case IIR_INOUT_MODE:
      if (actual_clause->_get_actual()->_is_static_expression()){
	  noofoutputsignals--;
      }
      first = true;
      break;
    default:
      // do nothing
      break;
    }
    actual_clause = port_map_aspect.successor(actual_clause);
  }
  
  _cc_out << noofinputsignals << ", " << noofoutputsignals;

  if(first == true) {
    actual_clause = port_map_aspect.first();  
    while(actual_clause != NULL) {
      if (actual_clause->_get_actual()->_is_static_expression() == FALSE){
	if( actual_clause->get_formal()->_get_mode() == IIR_IN_MODE ){
	  _cc_out << ", &";
	  if(actual_clause == NULL) {
	    ASSERT(actual_clause != NULL);
	  }
	  ASSERT(actual_clause->_get_actual() != NULL);
	  if (actual_clause->_get_actual()->get_kind() ==IIR_FUNCTION_CALL) {
	    ((IIR_FunctionCall *) actual_clause->_get_actual())->_publish_cc_elaborate_arg( _cc_out );
	  }
	  else {
	    actual_clause->_get_actual()->_publish_cc_elaborate( _cc_out );
	  }
	}
       }
       actual_clause = port_map_aspect.successor(actual_clause);
     }
    
    actual_clause = port_map_aspect.first();  
    for(;actual_clause != NULL; ) {
      switch(actual_clause->get_formal()->_get_mode()) {
      case IIR_OUT_MODE:
      case IIR_INOUT_MODE:
	_cc_out << ", ";
	_cc_out << "&";
	if (actual_clause->_get_actual()->get_kind() == IIR_FUNCTION_CALL) {
	  ((IIR_FunctionCall *) actual_clause->_get_actual())->_publish_cc_elaborate_arg( _cc_out );
	}
	else {
	  if (actual_clause->_get_actual()->_is_iir_declaration() == TRUE) {
	    tempDeclaration = (IIR_Declaration *) actual_clause->_get_actual();
	    ASSERT (tempDeclaration->_is_iir_declaration() == TRUE );
	    _publish_cc_scoping_prefix( _cc_out.get_stream(), 
					tempDeclaration->_get_declarative_region(),
					_get_enclosing_scope() );
	  }
	  else if (actual_clause->_get_actual()->get_kind() == IIR_INDEXED_NAME) {
	    IIR_IndexedName *tempIdxName = (IIR_IndexedName *) actual_clause->_get_actual();
	    ASSERT ( tempIdxName->get_kind() == IIR_INDEXED_NAME );
	    if (tempIdxName->get_prefix()->_is_iir_declaration() == TRUE) {
	      tempDeclaration = (IIR_Declaration *) tempIdxName->get_prefix();

	      ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
	      _publish_cc_scoping_prefix( _cc_out.get_stream(),
					  tempDeclaration->_get_declarative_region(),
					  _get_enclosing_scope());
	    }
	  }
	  
	  actual_clause->_get_actual()->_publish_cc_elaborate( _cc_out );
	}
	break;
      default:
	// do nothing;
	break;
      }
      actual_clause = port_map_aspect.successor(actual_clause);
    }
  }
  _cc_out << ");\n";
  
  actual_clause = port_map_aspect.first();  
  while(actual_clause != NULL) {
    switch(actual_clause->get_formal()->_get_mode()) {
    case IIR_IN_MODE:
    case IIR_INOUT_MODE:
      ASSERT(actual_clause->_get_actual() != NULL);
      if(actual_clause->_get_actual()->get_kind() == IIR_FUNCTION_CALL) {
	downTypeConversionFn = (IIR_FunctionCall*)actual_clause->_get_actual();
	_cc_out << "addDownTypeConversionFn(";
	if(actual_clause->get_formal()->get_kind() == IIR_FUNCTION_CALL) {
	  label->_publish_cc_elaborate( _cc_out.get_stream() );
	  _cc_out << "_elab_obj->";
	  ((IIR_FunctionCall *)actual_clause->get_formal())->_publish_cc_elaborate_arg( _cc_out );
	}
	else {
	  label->_publish_cc_elaborate( _cc_out.get_stream() );
	  
	  if (_get_currently_publishing_unit() == GENERATE_FOR) {
	    _cc_out << "_elab_obj[i - generateLeft].";
	  }
	  else {
	    _cc_out << "_elab_obj->";
	  }
	  actual_clause->get_formal()->_publish_cc_elaborate( _cc_out );
	}
	_cc_out << ", ";
	((IIR_FunctionCall *)actual_clause->_get_actual())->_publish_cc_elaborate_arg( _cc_out );
	_cc_out << ", ";
	downTypeConversionFn->get_implementation()->_publish_cc_type_conversion_function_name( _cc_out );
	_cc_out << ");\n";
      }
      break;
    default:
      // do nothing
      break;
    }
    actual_clause = port_map_aspect.successor(actual_clause);
  }
}
  

void
IIRScram_BlockStatement::_publish_createNetInfo( published_file &_cc_out ){    
  IIR_AssociationElement* actual_clause = NULL;
  IIR* formal = NULL;
  IIR_Label *label = _get_label();

  actual_clause = port_map_aspect.first();
  if(actual_clause == NULL){
    return;
  }
  
  while(actual_clause != NULL) {
    formal = actual_clause->get_formal();
    ASSERT(formal != NULL);
    ASSERT(formal->_is_resolved() == TRUE);
    switch(formal->_get_mode()) {
    case IIR_IN_MODE:
      _publish_cc_elaborate_Add( _cc_out, actual_clause, label);
      break;
    case IIR_OUT_MODE:
      // Code for adding the resolution information tree
      _publish_cc_elaborate_addChild( _cc_out, actual_clause, label );
      break;
    case IIR_INOUT_MODE:
      _publish_cc_elaborate_Add( _cc_out, actual_clause, label );
      // Code for adding the resolution information tree
      _publish_cc_elaborate_addChild( _cc_out, actual_clause, label );
      break;
    default:
      cerr << "Buffer and linkage mode not yet supported in "
	   << "Component instantiation statements\n";
      break;
    }
    actual_clause = port_map_aspect.successor(actual_clause);
  }
}


void 
IIRScram_BlockStatement::_publish_cc_elaborate_Add( published_file &_cc_out,
						    IIR_AssociationElement *actual_clause,
						    IIR_Label *label) {
  IIR* decl = actual_clause->get_formal();
  IIR_FunctionCall *upTypeConversionFn = NULL;
  IIR_FunctionCall *downTypeConversionFn = NULL;
  IIR_Declaration* tempDeclaration = NULL;

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_elaborate_Add" );
    
  if (actual_clause->_get_actual()->get_kind()!=IIR_FUNCTION_CALL &&
      actual_clause->get_formal()->get_kind()!=IIR_FUNCTION_CALL){
    _cc_out << "Add(";
    ASSERT(actual_clause->_get_actual() != NULL);
    if(actual_clause->_get_actual()->get_kind() == IIR_FUNCTION_CALL) {
      downTypeConversionFn = (IIR_FunctionCall *)actual_clause->_get_actual();
      downTypeConversionFn->_publish_cc_elaborate_arg( _cc_out );
    } else {
      if(actual_clause->_get_actual()->_is_iir_declaration() == TRUE && 
	 actual_clause->_get_actual()->_is_static_expression() == FALSE) {
	ASSERT ( actual_clause->_get_actual()->_is_iir_declaration() == TRUE );
	tempDeclaration = (IIR_Declaration *) actual_clause->_get_actual();
	
	ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
	_publish_cc_scoping_prefix( _cc_out.get_stream(), tempDeclaration->_get_declarative_region(),_get_enclosing_scope( ));
      }
      else if(actual_clause->_get_actual()->_is_name() == TRUE) {
	tempDeclaration = (IIR_Declaration *) actual_clause->_get_actual()->_get_prefix_declaration();
	ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
	_publish_cc_scoping_prefix( _cc_out.get_stream(), 
				    tempDeclaration->_get_declarative_region(),
				    _get_enclosing_scope());
      }
      else if (actual_clause->_get_actual()->_is_static_expression() == TRUE) {
	_cc_out << "*(new ";
	actual_clause->_get_actual()->get_subtype()->_publish_cc_type_string( _cc_out );
	_cc_out << "(ObjectBase::SIGNAL_NETINFO, ";
	actual_clause->_get_actual()->_publish_cc_value( _cc_out );
	actual_clause->_get_actual()->get_subtype()->_publish_cc_object_type_info( _cc_out );
	_cc_out << "))";
      }
      else{
	cerr << "Error : IIRScram_BlockStatement::_publish_cc_elaborate_addChild( published_file &_cc_out ) - declaration kind not handled yet.\n";
      }
      
      if (actual_clause->_get_actual()->_is_static_expression() == FALSE){
	actual_clause->_get_actual()->_publish_cc_elaborate( _cc_out );
      }
      
      _cc_out << ", ";
      label->_publish_cc_elaborate( _cc_out.get_stream() );
      
      if (_get_currently_publishing_unit() == GENERATE_FOR) {
	_cc_out << "_elab_obj[i - generateLeft].";
      }
      else {
	_cc_out << "_elab_obj->";
      }
      
      if(decl->get_kind() == IIR_FUNCTION_CALL) {
	upTypeConversionFn = (IIR_FunctionCall *)decl;
	upTypeConversionFn->_publish_cc_elaborate_arg( _cc_out );
      } else {
	decl->_publish_cc_elaborate( _cc_out );
      }    
      _cc_out << ");\n";
      
      if(upTypeConversionFn != NULL) {
	_cc_out << "setUpConversionFunctionId(";
	label->_publish_cc_elaborate( _cc_out.get_stream() );
	
	if (_get_currently_publishing_unit() == GENERATE_FOR) {
	  _cc_out << "_elab_obj[i - generateLeft].";
	}
	else {
	  _cc_out << "_elab_obj->";
	}
	
	upTypeConversionFn->_publish_cc_elaborate_arg( _cc_out );
	_cc_out << ", ";
	upTypeConversionFn->get_implementation()->_publish_cc_type_conversion_function_name( _cc_out );
	_cc_out << ");\n";
      }
    }
  }
}


void 
IIRScram_BlockStatement::_publish_cc_elaborate_addChild( published_file &_cc_out,
							 IIR_AssociationElement *actual_clause,
							 IIR_Label *label) {
  IIR* decl = actual_clause->get_formal();
  IIR_Declaration* tempDeclaration = NULL;

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_elaborate_addChild" );
 
  if (actual_clause->_get_actual()->get_kind() != IIR_FUNCTION_CALL &&
      actual_clause->get_formal()->get_kind() != IIR_FUNCTION_CALL){
     _cc_out << "addChild(";
  if(actual_clause->_get_actual()->get_kind() == IIR_FUNCTION_CALL) {
    //((IIR_FunctionCall *)actual_clause->_get_actual())->_publish_cc_elaborate_arg( _cc_out );
  } else {
    if(actual_clause->_get_actual()->_is_iir_declaration() == TRUE) {
      ASSERT ( actual_clause->_get_actual()->_is_iir_declaration() == TRUE );
      tempDeclaration = (IIR_Declaration *) actual_clause->_get_actual();
      ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
      _publish_cc_scoping_prefix( _cc_out.get_stream(), 
				  tempDeclaration->_get_declarative_region(),
				  _get_enclosing_scope());
    }
    else if(actual_clause->_get_actual()->_is_name() == TRUE) {
      tempDeclaration = (IIR_Declaration *) actual_clause->_get_actual()->_get_prefix_declaration();
      ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
      _publish_cc_scoping_prefix( _cc_out.get_stream(), 
				  tempDeclaration->_get_declarative_region(),
				  _get_enclosing_scope());
    }
    else if (actual_clause->_get_actual()->_is_literal()){
             
      cerr << "Error : IIRScram_BlockStatement::_publish_cc_elaborate_addChid( published_file &_cc_out ) - declaration kind not handled yet.\n";
    }
    actual_clause->_get_actual()->_publish_cc_elaborate( _cc_out );
  }
  
  
  _cc_out << ", ";
  label->_publish_cc_elaborate( _cc_out.get_stream() );

  if (_get_currently_publishing_unit() == GENERATE_FOR) {
    _cc_out << "_elab_obj[i - generateLeft].";
  }
  else {
    _cc_out << "_elab_obj->";
  }
  
  if(decl->get_kind() == IIR_FUNCTION_CALL) {
    //upTypeConversionFn = (IIR_FunctionCall *)decl;
    //upTypeConversionFn->_publish_cc_elaborate_arg( _cc_out );
  } else {
    if ( decl->_is_iir_declaration() == TRUE ) {
      tempDeclaration = (IIR_Declaration *) decl;
      ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
    }
    else if ( decl->_is_name() == TRUE ) {
      tempDeclaration = (IIR_Declaration *) decl->_get_prefix_declaration();
      ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
      _publish_cc_scoping_prefix( _cc_out.get_stream(), 
				  tempDeclaration->_get_declarative_region(),
				  _get_enclosing_scope());
    }
    else {
      cerr << "Error : IIRScram_BlockStatement::_publish_cc_elaborate_addChid( published_file &_cc_out ) - declaration kind not handled yet.\n";
    }
    
    decl->_publish_cc_elaborate( _cc_out );
  }    
  _cc_out << ");\n";
  }
}


IIR_Boolean
IIRScram_BlockStatement::_publish_cc_type_info( published_file &_cc_out, 
						IIR_Boolean firstDeclFlag ){
  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_type_info" );

  firstDeclFlag = block_declarative_part._publish_cc_type_info( _cc_out, firstDeclFlag );
  return block_statement_part._publish_cc_type_info( _cc_out, firstDeclFlag );
}

void
IIRScram_BlockStatement::_publish_cc_extern_type_info( published_file &_cc_out ){

  SCRAM_CC_REF( _cc_out, "IIRScram_BlockStatement::_publish_cc_extern_type_info" );

  block_declarative_part._publish_cc_extern_type_info( _cc_out );
  block_statement_part._publish_cc_extern_type_info( _cc_out );
}

#ifdef PROCESS_GRAPH
void
IIRScram_BlockStatement::_publish_cc_driver_info( published_file &_cc_out ) {
  IIR_InterfaceDeclaration* portelement = NULL;
  IIR_AssociationElement* actual_clause = NULL;
  
  portelement = port_clause.first();
  actual_clause = port_map_aspect.first();
  for(; portelement != NULL; ) {
    _cc_out << "fp << \"";
    portelement->_publish_cc_elaborate( _cc_out );
    _cc_out << "\" << NL();" << NL();
    portelement->_publish_cc_elaborate( _cc_out );
    _cc_out << ".dump_connectivity_info(fp);" << NL();
    portelement = port_clause.successor(portelement);
  }

  IIR_ArchitectureStatement *arch_stmt;
  arch_stmt = block_statement_part.first();
  while (arch_stmt != NULL) {
    if (arch_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
      ((IIRScram_ComponentInstantiationStatement*)arch_stmt)->_publish_cc_driver_info( _cc_out );
    }
    else if (arch_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      ((IIRScram_ProcessStatement*)arch_stmt)->_publish_cc_driver_info( _cc_out );
    }
    else if (arch_stmt->_is_block_statement() == TRUE) {
      ((IIRScram_BlockStatement *) arch_stmt)->_publish_cc_driver_info( _cc_out );
    }
    arch_stmt = block_statement_part.successor(arch_stmt);
  }
}
#endif

visitor_return_type *
IIRScram_BlockStatement::_accept_visitor( node_visitor *visitor, 
					  visitor_argument_type *arg ){
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_BlockStatement(this, arg);
}

void
IIRScram_BlockStatement::_publish_cc_include_elab( published_file &_cc_out ){
  const string filename = _get_cc_elaboration_class_name() + ".hh";
  IIRScram::_publish_cc_include( _cc_out, filename );
}
