
// Copyright (c) 2002-2003 The University of Cincinnati.
// All rights reserved.
 
// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.
 
// Authors: Harish Venkataramani venkath@ececs.uc.edu

#include "IIRScram_BranchQuantityDeclaration.hh"
#include "set.hh"
#include "symbol_table.hh"
#include "IIR_TypeDefinition.hh"
#include "IIR_TerminalDeclaration.hh"
#include "IIR_Declaration.hh"
#include "IIR_Attribute.hh"
#include "IIR_StringLiteral.hh"
#include <string.h>
#include "published_file.hh"
#include "IIR_IndexedName.hh"

extern symbol_table *cgen_sym_tab_ptr;
extern symbol_table *cgen_arch_sym_tab_ptr;

IIRScram_BranchQuantityDeclaration::IIRScram_BranchQuantityDeclaration() {
   _stmt_qty_index = -1;
   default_rtol = 1e-3;
   default_atol = 1e-12;
   _qty_occurence_index = 0;
}

IIRScram_BranchQuantityDeclaration::~IIRScram_BranchQuantityDeclaration() {}

void
IIRScram_BranchQuantityDeclaration::_publish_vhdl_decl(ostream &_vhdl_out) {
  _vhdl_out << " quantity ";
  if ( across_aspect_identifier_list.num_elements() != 0 ){
    get_declarator()->_publish_vhdl(_vhdl_out);
    if ( get_across_aspect_tolerance() != NULL ){
      _vhdl_out << " tolerance ";
      get_across_aspect_tolerance()->_publish_vhdl(_vhdl_out);
    }
    
    if ( get_across_aspect_expression() != NULL ){
      _vhdl_out << " := ";
      get_across_aspect_expression()->_publish_vhdl(_vhdl_out);
    }
    _vhdl_out << "  across ";
  }
  if ( through_aspect_designator_list.num_elements() != 0 ){
    get_declarator()->_publish_vhdl(_vhdl_out);
    if ( get_through_aspect_tolerance() != NULL ){
      _vhdl_out << " tolerance ";
      get_through_aspect_tolerance()->_publish_vhdl(_vhdl_out);
    }
    if ( get_through_aspect_expression() != NULL ){
      _vhdl_out << " := ";
      get_through_aspect_expression()->_publish_vhdl(_vhdl_out);
    }
    _vhdl_out << "  through  ";
  }
  ASSERT( get_plus_terminal_name() != NULL );
  get_plus_terminal_name()->_publish_vhdl(_vhdl_out);
  if ( get_minus_terminal_name() != NULL ){
    _vhdl_out << " to " ;
    get_minus_terminal_name()->_publish_vhdl(_vhdl_out);
  }
  _vhdl_out << " ;\n" ;
}

void
IIRScram_BranchQuantityDeclaration::_publish_cc_lvalue(published_file &_cc_out) {
  ASSERT( get_declarator()->get_kind() == IIR_IDENTIFIER );
  if ( cgen_sym_tab_ptr != NULL &&
      !cgen_sym_tab_ptr->in_scope(this) ){
    cgen_sym_tab_ptr->add_declaration(this);
  }
  
  IIRScram::_publish_cc_prefix_string((published_file&)_cc_out);
  _get_declarator()->_publish_cc_lvalue(_cc_out);
}

IIR_Int32
IIRScram_BranchQuantityDeclaration::get_plus_terminal_id() {
  ASSERT( get_plus_terminal_name() != NULL );
  return ((IIR_TerminalDeclaration *)get_plus_terminal_name())->terminal_id;
}

IIR_Int32
IIRScram_BranchQuantityDeclaration::get_minus_terminal_id() {
  if ( get_minus_terminal_name() != NULL  ){
    return ((IIR_TerminalDeclaration *)get_minus_terminal_name())->terminal_id;
  }
  else {
    return 0;
  }
}

IIR_Int32
IIRScram_BranchQuantityDeclaration::get_num_across_elements() {
  return across_aspect_identifier_list.num_elements();
}

IIR_Int32
IIRScram_BranchQuantityDeclaration::get_num_through_elements() {
  return through_aspect_designator_list.num_elements();
}


void
IIRScram_BranchQuantityDeclaration::_set_stmt_qty_index(IIR_Int32 *index,
                                                        set<IIR_Declaration>
                                                        *quantity_set) {
  if (!cgen_arch_sym_tab_ptr->in_scope(this)) {
    cgen_arch_sym_tab_ptr->add_declaration(this);
  }

  if (cgen_sym_tab_ptr != NULL) {
    if (!cgen_sym_tab_ptr->in_scope(this)) {
      cgen_sym_tab_ptr->add_declaration(this);
    }
  }
    
  if ( _get_stmt_qty_index() == -1 && index != NULL) {
    _stmt_qty_index = *index;
    (*index)++;  
  }
  if (quantity_set != NULL)
    quantity_set->add(this);
}     
  
IIR_Int32
IIRScram_BranchQuantityDeclaration::_get_stmt_qty_index() {
  return _stmt_qty_index;
}

void
IIRScram_BranchQuantityDeclaration::_set_stmt_node_index(IIR_Int32 *index, bool _is_right_child, bool &reducibleFlag) {
  if (_is_right_child == true) {
   _stmt_node_index.push_back(2 * (*index) + 1);
  }
  else {
   _stmt_node_index.push_back(2 * (*index));
  }
}

IIR_Boolean
IIRScram_BranchQuantityDeclaration::_is_across_quantity() {
  if ( get_num_across_elements() != 0 && get_num_through_elements() == 0 ){
    return TRUE;
  }
  else {
    return FALSE;
  }
}
                                                        
void
IIRScram_BranchQuantityDeclaration::_flush_stmt_index() {
  _stmt_qty_index = -1;
}     

void
IIRScram_BranchQuantityDeclaration::_publish_cc_ams_function(published_file &_cc_out) {
  if (_get_currently_publishing_unit() == SIMULTANEOUS_IF ) {
    ASSERT(get_subtype() != NULL);
    _cc_out << get_subtype()->_get_cc_type_name();
    _cc_out << "(ObjectBase::VARIABLE, qty["<<_get_stmt_qty_index() <<"])";
  }
  else {
   _cc_out << "  new equationNode('Q'," << _get_stmt_qty_index()
           << ",node" << (int)(_stmt_node_index[_qty_occurence_index++]/2) <<");" <<NL();
  }
}
  
void
IIRScram_BranchQuantityDeclaration::_publish_cc_decl(published_file &_cc_out) {
  _cc_out << "AMSType ";
  _get_declarator()->_publish_cc_lvalue(_cc_out.get_stream());
  _cc_out << ";" << NL();
}

void
IIRScram_BranchQuantityDeclaration::_publish_cc_tolerances( published_file &_cc_out ){
  _cc_out << RTOL;
  _cc_out << ", ";
  _cc_out << ATOL;
  _cc_out << ", ";
}

void
IIRScram_BranchQuantityDeclaration::_type_check() {
  RTOL = default_rtol ;
  ATOL = default_atol ;
  IIR_ObjectDeclaration::_type_check();
}

IIR_Boolean
IIRScram_BranchQuantityDeclaration::_is_readable() {
  return TRUE;
}

