// Copyright (c) 1996-2000 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//	    Magnus Danielson	cfmd@swipnet.se

//---------------------------------------------------------------------------

#include "IIR_ObjectDeclaration.hh"
#include "IIR_RecordTypeDefinition.hh"
#include "IIR_AccessTypeDefinition.hh"
#include "IIR_Attribute.hh"
#include "IIR_Identifier.hh"
#include "IIR_ArrayTypeDefinition.hh"
#include "set.hh"
#include "resolution_func.hh"
#include "error_func.hh"
#include "published_file.hh"
#include "symbol_table.hh"
#include "sstream-wrap.hh"


extern symbol_table *cgen_sym_tab_ptr;

IIRScram_ObjectDeclaration::~IIRScram_ObjectDeclaration() {}

IIR_Int32 
IIRScram_ObjectDeclaration::_get_num_indexes(){
  return get_subtype()->_get_num_indexes();
}

IIR_TypeDefinition *
IIRScram_ObjectDeclaration::_get_type_of_element( int index ){
  IIR_TypeDefinition *retval;
  retval = get_subtype()->_get_type_of_element(index);

  return retval;
}

set<IIR_Declaration> *
IIRScram_ObjectDeclaration::_find_declarations( IIR_Name *to_find ){
  set<IIR_Declaration> *retval = NULL;

  ASSERT( get_subtype() != NULL );

  if( get_subtype()->_is_record_type() == TRUE ){
    IIR_RecordTypeDefinition *my_subtype;
    my_subtype = (IIR_RecordTypeDefinition *)get_subtype();
    retval = my_subtype->_find_declarations( to_find );
  }
  else if( get_subtype()->_is_access_type() == TRUE ){
    IIR_AccessTypeDefinition *my_subtype;
    my_subtype = (IIR_AccessTypeDefinition *)get_subtype();
    if( my_subtype == NULL ){
      ostringstream err;
      err << "|" << *this << "| cannot refer to element |" << *(IIR *)to_find << "|, as " 
	  << "subtype |" << *get_subtype() << "| has not yet been defined.";
      report_error( (IIR *)to_find, err.str() );
      return NULL;
    }
    else{
      retval = my_subtype->_find_declarations( to_find );
    }
  }

  return retval;
}

void 
IIRScram_ObjectDeclaration::_type_check(){
  if( get_value() != NULL ){
    set<IIR_TypeDefinition> *initializer_types = get_value()->_get_rval_set();
    if( initializer_types == NULL ){
      report_undefined_symbol( get_value() );
      return;
    }

    IIR_Boolean exact_match = FALSE;
    IIR_TypeDefinition *current_type = initializer_types->get_element();
    while( current_type != NULL ){
      if( current_type == get_subtype() ){
	exact_match = TRUE;
	break;
      }
      current_type = initializer_types->get_next_element();
    }
    
    IIR_Boolean one_matched = FALSE;
    if( exact_match == FALSE ){
      current_type = initializer_types->get_element();
      while( current_type != NULL ){
	if( current_type->_is_compatible( get_subtype() ) != NULL ){
	  one_matched = TRUE;
	  break;
	}
	current_type = initializer_types->get_next_element();
      }
    }

    if( exact_match == TRUE ){
      set_value( get_value()->_semantic_transform( current_type ) );
      get_value()->_type_check( current_type );
      set_value( get_value()->_rval_to_decl( current_type ) );      
    }
    else if( one_matched == TRUE ){
      set_value( get_value()->_semantic_transform( get_subtype() ) );
      get_value()->_type_check( get_subtype() );
      set_value( get_value()->_rval_to_decl( get_subtype() ) );
    }
    else{
      ostringstream err;
      err << "Initializer |" << *get_value() << "| is incompatible with type |" << 
	*get_subtype()->_get_declarator() << "|.";
      report_error( this, err.str() );
    }

    delete initializer_types;
  }
  // There's no initializer.
  if( (_is_interface() == FALSE && (_is_variable() == TRUE || _is_signal() == TRUE))
      || _is_element() == TRUE ){
    if( get_subtype()->_is_access_type() == FALSE && 
	get_subtype()->_is_unconstrained_array_type() == TRUE ){
      ostringstream err;
      err << "A signal or variable object with an array type must be constrained.";
      report_error( this, err.str() ); 
    }
  }
  
}

void 
IIRScram_ObjectDeclaration::_publish_cc_first_objectParameter( published_file &_cc_out ){
  _publish_cc_lvalue( _cc_out );
}


IIR_Boolean 
IIRScram_ObjectDeclaration::_is_scalar_type(){
  return get_subtype()->_is_scalar_type();
}

IIR_Boolean 
IIRScram_ObjectDeclaration::_is_array_type(){
  return get_subtype()->_is_array_type();
}

IIR_Boolean
IIRScram_ObjectDeclaration::_is_record_type() {
  return get_subtype()->_is_record_type();
}

IIR_Boolean 
IIRScram_ObjectDeclaration::_is_access_type(){
  return get_subtype()->_is_access_type();
}

void 
IIRScram_ObjectDeclaration::_publish_cc_constructor_args( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ObjectDeclaration::_publish_cc_constructor_args" );
  _cc_out << OS("(");
  if (get_subtype()->_is_access_type() == FALSE) {
    _cc_out << "ObjectBase::" << _get_cc_object_type();
  }  
  
  if (((get_subtype()->_is_array_type() == TRUE) ||
       (get_subtype()->_is_record_type() == TRUE)) &&
      (get_subtype()->_is_access_type() == FALSE)) {
    get_subtype()->_publish_cc_object_type_info( _cc_out );
    _cc_out << ", " << NL();
    get_subtype()->_publish_cc_resolution_function_id( _cc_out );
  }

  if ((get_subtype()->_is_array_type() == TRUE) &&
      (get_subtype()->_is_access_type() == FALSE)) {
    if (get_subtype()->_is_anonymous() == TRUE) {
      _cc_out << ", " << NL();
      get_subtype()->_publish_cc_range( _cc_out );
    } else if(get_subtype()->_is_unconstrained_array_type()){
      // ASSERT(get_value() != NULL);
      if((get_value() != NULL) &&
	 (get_value()->get_kind() != IIR_CONCATENATION_OPERATOR)) {
	_cc_out << ", " << NL();
	get_value()->_publish_cc_range( _cc_out );
      }
    }
  }
  
  if (get_subtype()->_is_access_type() == TRUE)  {
    IIR_TypeDefinition *accessType = ((IIR_AccessTypeDefinition *) get_subtype())->get_designated_type();
    if (accessType->_is_array_type() == TRUE) {
      if ((accessType->_is_anonymous()) ||
	  (accessType->_is_unconstrained_array_type())) {
	if (get_value() != NULL) {
	  if (get_value()->_is_object() == TRUE) {
	    // Pass in a pointer to the array type object being used to initialize
	    // this object.
	    _cc_out << "&";
	  }
	  get_value()->_publish_cc_lvalue( _cc_out );
	  _cc_out << ")";
	  return;
	}
      }
    }
  }
  
  // Well publishing the value of guard signals .ie. the guard expression
  // also drops here. The guard expression could use signal attributes that
  // are not defined at the time of construction. The actual value will be
  // set in initialize() routine in the Kernel. So we don't need to publish
  // the guard expression. Just get out if the signal is a guard.

  if ((_is_signal() == TRUE) && (_is_guard_signal() == TRUE)) {
    _cc_out << ")";
    return;
  }

  // If the value is a parameter in a function/procedure then don't publish it
  if ((get_value() != NULL) &&
      (get_value()->get_kind() == IIR_CONSTANT_INTERFACE_DECLARATION) &&
      (((IIR_Declaration *) get_value())->_get_declarative_region()->_is_subprogram() == TRUE))  {
    if ((_get_currently_publishing_unit() != IIRScram::PROCEDURE) && 
	(_get_currently_publishing_unit() != IIRScram::FUNCTION))   {
      _cc_out << ")";
      return;
    }
  }
  if(get_value() != NULL) {
    if (get_subtype()->_is_access_type() == FALSE) {
      _cc_out << ", " << NL();
    }
    if ((get_subtype()->_is_scalar_type() == TRUE) &&
	(get_subtype()->_is_access_type() == FALSE) &&
	(get_value()->_is_literal() == FALSE) && 
	(get_value()->_is_enumeration_literal() == FALSE) && 
	(get_value()->get_kind() != IIR_PHYSICAL_LITERAL)) {
      _cc_out << "(const ";
      _cc_out << get_value()->get_subtype()->_get_cc_type_name();
      _cc_out << " &) ";
    }
    get_value()->_publish_cc_initialization_value( _cc_out );
  }
  else {
    if ((get_subtype()->_is_iir_scalar_type_definition() == TRUE) &&
	(get_subtype()->_is_anonymous() == TRUE) &&
	(_is_constant() == FALSE)) {
      _cc_out << ", " << NL();
      get_subtype()->_publish_cc_universal_left( _cc_out );
    }
  }
  
  // Check and publish the necessary type_info for scalar types.
  
  if ((get_subtype()->_is_scalar_type() == TRUE) &&
      (get_subtype()->_is_access_type() == FALSE)) {
    get_subtype()->_publish_cc_object_type_info( _cc_out );
  }
  
  _cc_out << CS(")");
}

void
IIRScram_ObjectDeclaration::_publish_cc_universal_type( published_file &_cc_out ) {
  get_subtype()->_publish_cc_universal_type( _cc_out );
}

void
IIRScram_ObjectDeclaration::_publish_vhdl_range( ostream &_vhdl_out ){
  _publish_vhdl( _vhdl_out );
}

void
IIRScram_ObjectDeclaration::_publish_vhdl_subtype_indication(ostream &_vhdl_out){
  if (get_subtype()->_is_anonymous() == TRUE) {
    get_subtype()->_publish_vhdl_decl(_vhdl_out);
  }
  else {
    get_subtype()->_publish_vhdl(_vhdl_out);
  }
}

void
IIRScram_ObjectDeclaration::_publish_vhdl_expression(ostream &_vhdl_out){
  if (get_value() != NULL) {
    _vhdl_out << " := ";
    get_value()->_publish_vhdl(_vhdl_out);
  }
}

void
IIRScram_ObjectDeclaration::_publish_vhdl_subtype_indication_with_expression(ostream &_vhdl_out){
  _publish_vhdl_subtype_indication(_vhdl_out);
  _publish_vhdl_expression(_vhdl_out);
  _vhdl_out << ";\n";
}

void
IIRScram_ObjectDeclaration::_publish_vhdl_signal_kind(ostream &_vhdl_out){
  switch(_get_signal_kind()) {
  case IIR_REGISTER_KIND:
    _vhdl_out << " register ";
    break;
  case IIR_BUS_KIND:
    _vhdl_out << " bus ";
    break;
  case IIR_NO_SIGNAL_KIND:
    break;
  default:
    cerr << " ERROR!!! undefined Signal Kind in node " << get_kind_text();
    abort();
   }
}

void
IIRScram_ObjectDeclaration::_publish_cc_decl_with_args( published_file &_cc_out ){

  SCRAM_CC_REF( _cc_out, "IIRScram_ObjectDeclaration::_publish_cc_decl_with_args" );

  get_subtype()->_publish_cc_lvalue( _cc_out );
  _cc_out << " ";
  _get_declarator()->_publish_cc_lvalue( _cc_out );
  _cc_out << "(";
  _publish_cc_constructor_args( _cc_out );
  _cc_out << ");" << NL();
}

void
IIRScram_ObjectDeclaration::_publish_cc_state_object_init( published_file &_cc_out ){
  SCRAM_CC_REF( _cc_out, "IIRScram_ObjectDeclaration::_publish_cc_state_object_init" );
  _publish_cc_lvalue( _cc_out );
  _publish_cc_constructor_args( _cc_out );
  _cc_out << "," << NL();
}

void 
IIRScram_ObjectDeclaration::_publish_cc_type_and_object_name( published_file &_cc_out ){
  _cc_out << get_subtype()->_get_cc_type_name();
  _cc_out << " ";
  _publish_cc_object_name( _cc_out );
}

void 
IIRScram_ObjectDeclaration::_publish_cc_decl( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ObjectDeclaration::_publish_cc_decl" );
  switch (get_subtype()->get_kind()) {
  case IIR_ACCESS_TYPE_DEFINITION:
  case IIR_ACCESS_SUBTYPE_DEFINITION:
    _cc_out << get_subtype()->_get_cc_type_name();
    _cc_out <<  " ";
    _publish_cc_lvalue( _cc_out );
    _cc_out << ";\n";
    break;
  case IIR_FILE_TYPE_DEFINITION:
    get_subtype()->_publish_cc_lvalue( _cc_out );
    break;
  default:
    get_subtype()->_publish_cc_kernel_type( _cc_out );
    _cc_out << " ";
    _publish_cc_lvalue( _cc_out );
    _cc_out << ";\n";
    break;
  }
}

const string
IIRScram_ObjectDeclaration::_get_cc_type_name(){
  string retval;

  ASSERT(get_subtype() != NULL);
  ASSERT(get_subtype()->_is_iir_type_definition() == TRUE);
  if (get_subtype()->_is_scalar_type() == TRUE)  {
    retval = get_subtype()->_get_cc_kernel_type();
  }
  else {
    retval = get_subtype()->_get_cc_type_name();
  }

  return retval;
}

void
IIRScram_ObjectDeclaration::_publish_cc_range( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ObjectDeclaration::_publish_cc_range" );

  ASSERT(get_subtype() != NULL);
  ASSERT(get_subtype()->_is_iir_type_definition() == TRUE);
  get_subtype()->_publish_cc_range( _cc_out );
}

void
IIRScram_ObjectDeclaration::_publish_cc_bounds( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ObjectDeclaration::_publish_cc_bounds" );

  ASSERT(get_subtype() != NULL);
  ASSERT(get_subtype()->_is_iir_type_definition() == TRUE);
  get_subtype()->_publish_cc_bounds( _cc_out );
}

const IIR_Char*
IIRScram_ObjectDeclaration::_get_mangling_prefix()
{
  return "";
}

IIR_SignalKind
IIRScram_ObjectDeclaration::_get_signal_kind(){
  _report_undefined_scram_fn("_get_signal_kind()");

  return IIR_NO_SIGNAL_KIND;
}
  
void
IIRScram_ObjectDeclaration::_add_declarations_in_initializations() {
  if(get_value() != NULL) {
    get_value()->_add_decl_into_cgen_symbol_table();
  }
  if(get_subtype() != NULL) {
    get_subtype()->_add_decl_into_cgen_symbol_table();
  }
}


void 
IIRScram_ObjectDeclaration::_add_decl_into_cgen_symbol_table(){
  if(get_value() != NULL) {
    get_value()->_add_decl_into_cgen_symbol_table();
  }
  
  if ((cgen_sym_tab_ptr != NULL) && (!cgen_sym_tab_ptr->in_scope(this))) {
    cgen_sym_tab_ptr->add_declaration(this);
  }
}


void
IIRScram_ObjectDeclaration::_clone( IIR *clone ) {
  ASSERT( clone->_is_iir_object_declaration() == TRUE );
  IIR_ObjectDeclaration *as_object = (IIR_ObjectDeclaration *)clone;

  IIRScram_Declaration::_clone(clone);

  as_object->attributes = attributes;
  as_object->set_subtype(get_subtype());
}

IIR_Boolean 
IIRScram_ObjectDeclaration::_is_locally_static(){
  IIR_Boolean retval = TRUE;

  ASSERT( get_subtype() != NULL );
  if( get_subtype()->_is_access_type() == TRUE ){
    retval = FALSE;
  }

  return retval;
}

IIR_Boolean 
IIRScram_ObjectDeclaration::_is_globally_static(){
  IIR_Boolean retval = TRUE;

  ASSERT( get_subtype() != NULL );
  if( get_subtype()->_is_access_type() == TRUE ){
    retval = FALSE;
  }

  return retval;
}

void
IIRScram_ObjectDeclaration::_publish_cc_headers( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ObjectDeclaration::_publish_cc_headers" );
  get_subtype()->_publish_cc_headers( _cc_out );
  if( _get_declarative_region() != 0 ){
    _get_declarative_region()->_publish_cc_include( _cc_out );
  }
}

void
IIRScram_ObjectDeclaration::_publish_cc_lvalue( published_file &_cc_out ) {
  if(_get_currently_publishing_unit() == IIRScram::TYPE) {
    get_value()->_publish_cc_lvalue( _cc_out );
  } 
  else{
    _get_declarator()->_publish_cc_lvalue( _cc_out );
  }
}

void
IIRScram_ObjectDeclaration::_publish_cc_init_signal( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_SignalInterfaceDeclaration::_publish_cc_init" );
  
  //#### yet to handle composite types.  Will be solved eventually
  const string anothername = _get_current_another_name();
  IIR *temporaryNode = _get_current_publish_node();
  IIR_Declaration* tempDecl = NULL;
  // since c_p_n is changed here, it must be reset before returning
  const string procname = _get_current_publish_name();
  ostringstream labelstr;
  ostringstream signalNetinfostr;
  int index = 0;
  IIR_Boolean last_activeAttribRequired = false;
  IIR_Boolean last_eventAttribRequired = false;
  IIR_Boolean last_activeAttribPresent = false;
  IIR_Boolean last_eventAttribPresent = false;
  IIR_Declaration* implied_declarations = NULL;
  
  //Reset the index counter since, a new signal interface declaration
  //is initialized
  _set_index_level( 0 );

  if( _get_implicit_declarations() != NULL ){
    implied_declarations = _get_implicit_declarations()->get_element();
    while(implied_declarations != NULL) {
      if(cgen_sym_tab_ptr->in_scope(implied_declarations)) {
	switch(implied_declarations->get_kind()) {
	case IIR_SIGNAL_DECLARATION:
	case IIR_SIGNAL_INTERFACE_DECLARATION:
	  implied_declarations->_set_declarative_region(_get_declarative_region());
	  implied_declarations->_publish_cc_init_signal( _cc_out ); 
	  break;
	case IIR_FUNCTION_DECLARATION:
	  //	  implied_declarations->_publish_cc_init_function( _cc_out ); 
	  break;
	default:
	  cerr << "Unknown declaration initialization: "
	       << implied_declarations->get_kind_text() << "\n";
	  abort();
	  break;
	}

	// Count the number of attributes of this signal
	index++;

	switch(implied_declarations->_get_attribute_name()->get_kind()) {
	case IIR_QUIET_ATTRIBUTE:
	case IIR_ACTIVE_ATTRIBUTE:
	  last_activeAttribRequired = true;
	  break;
	case IIR_STABLE_ATTRIBUTE:
	case IIR_EVENT_ATTRIBUTE:
	  last_eventAttribRequired = true;
	  break;
	case IIR_LAST_ACTIVE_ATTRIBUTE:
	    last_activeAttribPresent = true;
	    break;
	case IIR_LAST_EVENT_ATTRIBUTE:
	  last_eventAttribPresent = true;

	default:
	  break;
	}
      }
      implied_declarations = _get_implicit_declarations()->get_next_element();
    }
  }

  // Implementation of the QIUET, STABLE, ACTIVE and EVENT attributes
  // require last_active or last_event attribute.  This will be declared
  // transparent to the user.  This is why index is incremented by one here.
  // CAUTION: However, this must not be done if last_active or last_event is
  // already declared in the process.
  if(last_activeAttribRequired == TRUE &&
     last_activeAttribPresent == FALSE) {
    index++;
  }
  if(last_eventAttribRequired == TRUE &&
     last_eventAttribPresent == FALSE) {
    index++;
  }

  labelstr << *_get_declarator();
  _set_current_publish_name( labelstr.str() );
  
  if ( _is_implicit_declaration() == FALSE ) {
    ASSERT ( _get_declarative_region() != NULL );
    if( _get_current_publish_node() != NULL && 
	_get_current_publish_node()->_is_iir_concurrent_statement() == TRUE ){
      _get_current_publish_node()->_publish_cc_scoping_prefix( signalNetinfostr,
							       _get_declarative_region(), 
							       _get_current_publish_node() );
    }
  }
  else {
    ASSERT ( _get_signal_decl()->_is_iir_declaration() == TRUE );
    tempDecl = (IIR_Declaration *) _get_signal_decl();
    if( _get_current_publish_node() != NULL && 
	_get_current_publish_node()->_is_iir_concurrent_statement() == TRUE ){
      _get_current_publish_node()->_publish_cc_scoping_prefix( signalNetinfostr,
							       tempDecl->_get_declarative_region(),
							       this );
    }
  }
 
  _set_current_publish_node( this );
  signalNetinfostr << *_get_declarator() << "_info";
  _set_current_another_name( signalNetinfostr.str() );
  _cc_out << OS("{");//begin dummy block for scoping
  _cc_out << _get_current_publish_name() << OS(".setElaborationInfo(");
  if((_get_declarative_region()->_is_iir_package_declaration() == FALSE) &&
     (_get_declarative_region()->get_kind() != IIR_PACKAGE_BODY_DECLARATION)) {
    _cc_out << "proc->";
  }
  _cc_out << _get_current_another_name() << CS(");");

  // publish attributes for this signal
  if(index > 0) {
    ASSERT( _get_implicit_declarations() != NULL );
    implied_declarations = _get_implicit_declarations()->get_element();

    while(implied_declarations != NULL) {
      if(cgen_sym_tab_ptr->in_scope(implied_declarations)) {
	_cc_out << _get_current_publish_name() << OS(".setAttrib(");
	implied_declarations->_get_attribute_name()->_publish_cc_name_in_caps( _cc_out );
	_cc_out << "," << NL();
	
	if (implied_declarations->_get_attribute_name() != NULL) {
	  implied_declarations->_get_attribute_name()->_publish_cc_declarator( _cc_out );
	}
	else {
	  implied_declarations->_get_declarator()->_publish_cc_lvalue( _cc_out );
	}
	_cc_out << CS(");");
      }
      implied_declarations = _get_implicit_declarations()->get_next_element();
    }

    if(last_activeAttribRequired == TRUE && 
       last_activeAttribPresent == FALSE) {
      _cc_out << _get_current_publish_name() << OS(".setAttrib(");
      _cc_out << "LAST_ACTIVE," << NL();
      ASSERT( _get_current_publish_node()->_is_signal() == true);
      _get_current_publish_node()->_get_declarator()->_publish_cc_lvalue( _cc_out );
      _cc_out << "_last_active" << CS(");");
      
      last_activeAttribPresent = TRUE;
    }
    
    if(last_eventAttribRequired == TRUE && 
       last_eventAttribPresent == FALSE) {
      _cc_out << _get_current_publish_name() << OS(".setAttrib(");
      _cc_out << "LAST_EVENT," << NL();
      
      ASSERT( _get_current_publish_node()->_is_signal() == true);
      
      // This function takes care of publishing last events
      // properly for the corresponding types

      _get_current_publish_node()->_get_declarator()->_publish_cc_lvalue( _cc_out );
      _cc_out << "_last_event" << CS(");");

      last_eventAttribPresent = TRUE;
    }
  }

  if ( _is_implicit_declaration() == FALSE ){
    _cc_out << _get_current_publish_name() << ".resolveAndUpdate(processPointer);" << NL();
  }
  
  _cc_out << CS("}");//end dummy block
  _set_current_publish_name( procname );
  _set_current_publish_node( temporaryNode );
  _set_current_another_name( anothername );
  if(_is_implicit_declaration() == TRUE) {
    _get_attribute_name()->_publish_cc_necessary_decl_init( _cc_out );
  }
}
