
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer
//          Malolan Chetlur    
//          Krishnan Subramani 
//          Radharamanan Radhakrishnan
//          Umesh Kumar V. Rajasekaran
//          Narayanan Thondugulam
//          Swaminathan Subramanian
//          Magnus Danielson    cfmd@swipnet.se

#include "IIR_ProcessStatement.hh"
#include "IIR_AliasDeclaration.hh"
#include "IIR_AboveAttribute.hh"
#include "IIR_BlockStatement.hh"
#include "IIR_BreakElement.hh"
#include "IIR_BreakStatement.hh"
#include "IIR_DesignatorExplicit.hh"
#include "IIR_FileDeclaration.hh"
#include "IIR_Identifier.hh"
#include "IIR_IndexedName.hh"
#include "IIR_Label.hh"
#include "IIR_LibraryDeclaration.hh"
#include "IIR_ProcedureCallStatement.hh"
#include "IIR_SelectedName.hh"
#include "IIR_SelectedNameByAll.hh"
#include "IIR_SimpleName.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_SliceName.hh"
#include "IIR_VariableDeclaration.hh"
#include "IIR_WaitStatement.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "symbol_table.hh"
#include "StandardPackage.hh"
#include "published_header_file.hh"
#include "published_cc_file.hh"
#include "string_utils.hh"
#include "sstream-wrap.hh"
#include "language_processing_control.hh"
#include "set.hh"

extern symbol_table *cgen_sym_tab_ptr;
extern language_processing_control *lang_proc;

IIRScram_ProcessStatement::IIRScram_ProcessStatement(){ }

IIRScram_ProcessStatement::~IIRScram_ProcessStatement() { }

void 
IIRScram_ProcessStatement::_type_check(){
  process_declarative_part._type_check_attribute_specifications( process_statement_part ); 
}

void 
IIRScram_ProcessStatement::_publish_vhdl(ostream &_vhdl_out) {

  _publish_vhdl_stmt_label(_vhdl_out);
    
  if (get_postponed() == true) {
    _vhdl_out << "postponed ";
  }
  _vhdl_out << "process\n";
  process_declarative_part._publish_vhdl_decl(_vhdl_out);
  _vhdl_out << "\nbegin\n";
  process_statement_part._publish_vhdl(_vhdl_out);
  _vhdl_out << "end ";
  if (get_postponed() == true) {
    _vhdl_out << "postponed ";
  }
  _vhdl_out << "process";
  if (get_label() != NULL) {
    _vhdl_out << " ";
    get_label()->_publish_vhdl(_vhdl_out);
  }
  _vhdl_out << ";\n\n";
}

void 
IIRScram_ProcessStatement::_publish_cc_file_open( published_file &_cc_out ) {
  IIR_Declaration *decl;
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    ASSERT(decl->get_kind() == IIR_FILE_DECLARATION);
    ASSERT(decl->_is_read_file() || decl->_is_write_file());

    _cc_out << "static_cast<" << _get_cc_state_class() << "*>(getState())->";
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << OS(".openFile(this, ");
    ((IIR_FileDeclaration*)decl)->get_file_logical_name()->_publish_cc_lvalue( _cc_out );
    if (decl->_is_read_file()) {
      _cc_out << ", _savant_file_open_kind::READ_MODE"; 
    }
    else if (decl->_is_write_file()) {
      _cc_out << ", _savant_file_open_kind::WRITE_MODE";
      if (decl->_get_declarative_region()->get_kind() == IIR_ARCHITECTURE_DECLARATION) {
	_cc_out << ", ios::app";
      }
    }
    _cc_out << CS(");");
    
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
}

void 
IIRScram_ProcessStatement::_publish_cc( published_file & ){
  IIR_Declaration *decl = NULL;
  IIR_WaitStatement *wait = NULL;
  int numInFiles = 0, numOutFiles = 0;

  const string design_unit_name = _get_design_unit_name();
  const string process_decls_file_name = _get_cc_process_class() + "_decls";

  IIR_Boolean _above_attribute_in_process =
    process_statement_part._is_above_attribute_found();
  IIR_Boolean _above_attribute_in_wait_stmts = FALSE;
  IIR_AboveAttribute *current_above_attribute = NULL;
  IIR_DesignatorExplicit *current_designator = NULL;

  symbol_table cgen_sym_tab( 4093, false );
  IIR_Boolean first = TRUE;
  
  cgen_sym_tab_ptr = &cgen_sym_tab;

  _set_current_process_statement((IIR_ProcessStatement*)this);

  // Publish the required stuff in the declarative part of the process.
  published_header_file process_decls_header_file( _get_work_library()->_get_path_to_directory(),
						   process_decls_file_name,
						   this );

  SCRAM_CC_REF( process_decls_header_file, 
		"IIRScram_ProcessStatement::_publish_cc( published_file & )" );
  // Include the declarative region to which this process belongs
  process_decls_header_file << "#include \"" << design_unit_name << ".hh\"\n";

  // Include all the stuff in the declarative part of process in this file
  process_declarative_part._publish_cc( process_decls_header_file );

  SCRAM_CC_REF( process_decls_header_file, 
		"IIRScram_ProcessStatement::_publish_cc( published_file & )" );

  published_cc_file process_decls_cc_file( _get_work_library()->_get_path_to_directory(),
					   process_decls_file_name,
					   this );

  SCRAM_CC_REF( process_decls_cc_file, 
		"IIRScram_ProcessStatement::_publish_cc( published_file & )" );
  process_decls_cc_file << "#include \"" << _get_current_entity_name() << ".hh\"\n\n";
  _get_label()->_publish_cc_include( process_decls_cc_file );
  process_declarative_part._publish_cc_decl( process_decls_cc_file );
  
  _build_wait_list();

  IIR_SequentialStatement *stmt = process_statement_part.first();
  while ( stmt != NULL ) {
    stmt->_build_above_attribute_set(&_above_attribute_set);//VHDL AMS addition - jkm
    stmt = process_statement_part.successor(stmt);
  }
  wait = _wait_stmt_list.first();
  while(wait != NULL) {
    _above_attribute_in_wait_stmts = _above_attribute_in_wait_stmts || wait->_is_above_attribute_found();
    wait = _wait_stmt_list.successor(wait);
  }
  if( _above_attribute_in_wait_stmts == FALSE &&
      _above_attribute_in_process == TRUE) {
    current_above_attribute = _above_attribute_set.get_element();
    while(current_above_attribute != NULL) {
      wait = new IIR_WaitStatement();
      wait->wait_id = 0;
      current_designator = new IIR_DesignatorExplicit();
      current_designator->set_name(((IIR_Attribute*)current_above_attribute)->
				   _get_implicit_declaration());
      wait->sensitivity_list.append(current_designator);
      _wait_stmt_list.append(wait);
      process_statement_part.append(wait);
      current_above_attribute = _above_attribute_set.get_next_element();
    }
  }
  wait = NULL;

  // generate code for the <process>.hh file
  published_header_file process_header_file( _get_work_library()->_get_path_to_directory(),
					     _get_cc_process_class(),
					     this );

  SCRAM_CC_REF( process_header_file, "IIRScram_ProcessStatement::_publish_cc( published_file &process_header_file )" );
  
  IIRScram::_publish_cc_include( process_header_file, "tyvis/VHDLProcess.hh" );
  IIRScram::_publish_cc_include( process_header_file, "tyvis/STDTypes.hh" );
  _publish_cc_include_state( process_header_file );

  //redefine savantnow so that any initializations through savant now is 
  //performed
  _publish_cc_savantnow_no_process_ptr( process_header_file );
  
  process_header_file << "class " << _get_current_publish_name() << _get_current_entity_name();

  // This if check is needed as process stmts in entity decls will not have
  // arch set for them.
  if( _get_current_architecture_name() != "" ) {
    process_header_file << "_" << _get_current_architecture_name();
    _publish_cc_enclosing_stmt_to_architecture_path( process_header_file.get_stream() );
  }
  
  process_header_file << "_elab;\n";
  
  _publish_cc_extern_declarations( process_header_file );
  process_header_file << "class " << _get_cc_process_class();
  process_header_file.open_scope( " : public VHDLProcess {" );	
  process_header_file << "public:" << NL()
		      << _get_cc_process_class() << "( " << _get_current_publish_name()
		      << design_unit_name
		      << "_elab *proc );" << NL();
  process_header_file << "~" << _get_cc_process_class() << "();" << NL();
  process_header_file << "void executeVHDL();" << NL();
  process_header_file << "State *allocateState();" << NL();
  process_header_file << "void initialize();" << NL();
  process_header_file.insert_comment( "type_info structures for types declared within this process" );

  // _publish_cc_extern_type_info( process_header_file );
  process_declarative_part._publish_cc_type_info_and_constant_declarations( process_header_file );
  
  // End the .hh file later after searching for guard signal

  // generate code for the <process>.cc file
  published_cc_file process_cc_file( _get_work_library()->_get_path_to_directory(),
				     _get_cc_process_class(),
				     this );
  SCRAM_CC_REF( process_cc_file, "IIRScram_ProcessStatement::_publish_cc( published_file &process_cc_file )" );
  _publish_cc_include( process_cc_file );
  if (lang_proc->processing_vhdl_ams()) {
    process_cc_file << NL();
    process_cc_file <<"extern int breakflag; "<< NL();
    process_cc_file <<"extern int qsflag; "<< NL();
  }
  PublishedUnit _saved_publishing_unit = _get_currently_publishing_unit();
  _set_currently_publishing_unit(PROCESS);

  // executeVHDL
  process_cc_file << "void " << _get_cc_process_class(); 
  process_cc_file.open_scope( "::executeVHDL() {" );
  process_cc_file << "VHDLProcess *processPtr = this;";
  process_cc_file << "\n#ifdef VHDLDBG\n"
		  << "  cout << getName() << \" executing VHDL at \" << getTimeNow() << endl;\n"
		  << "#endif" << NL();

  process_cc_file << _get_cc_state_class() <<" *currentState = static_cast<" << 
    _get_cc_state_class() <<" *>(getState())";
  process_cc_file.end_statement();

  for (wait = _wait_stmt_list.first(); 
       wait != NULL; 
       wait = _wait_stmt_list.successor(wait))  {
    process_cc_file << "if (currentState->waitLabel == "
	    << wait->wait_id 
	    << ") goto ";
    wait->_publish_cc_wait_label( process_cc_file );
    process_cc_file << ";" << NL();
  }

  IIR_ProcedureCallStatement *proc;
  dl_list<IIR_ProcedureCallStatement> _proc_stmt_list;
  process_statement_part._build_procedure_call_stmt_list(&_proc_stmt_list);
  // "goto"s for procedure call statements in the process.
  process_cc_file << "if(currentState->waitLabel == ";
  process_cc_file.open_scope( "WAITING_IN_PROC) {" );
  process_cc_file << "currentState->stack.setCurrentToTop();" << NL();
  for (proc = _proc_stmt_list.first(); proc != NULL; 
       proc = _proc_stmt_list.successor(proc))  {
    process_cc_file << "if(currentState->stack.getCurrentTop()->waitLabel == "
		    << proc << ") goto ";
    proc->_publish_cc_proc_label( process_cc_file );
    process_cc_file << ";" << NL();
  } // for
  process_cc_file.close_scope( "}" );

  for (IIR_SequentialStatement* seqstmt = process_statement_part.first();
       seqstmt != NULL;
       seqstmt = process_statement_part.successor(seqstmt)) {
    ostringstream breakstr;
    IIR_Identifier *new_identifier;
    if (seqstmt->get_kind() == IIR_BREAK_STATEMENT) {
      ostringstream breakstringstream;
      breakstringstream << _get_cc_process_class() << "_breakSignal" ;
      string breakstring = breakstringstream.str();
      int    strlength   = breakstring.size();
      char  *breaklabel  = new char[strlength];
      strcpy(breaklabel,breakstring.c_str());
      new_identifier = IIR_Identifier::get(breaklabel, strlen(breaklabel));
      seqstmt->_set_process_stmt_label(new IIR_Label);
      seqstmt->_get_process_stmt_label()->set_declarator(new_identifier);
    }
  }
  process_cc_file.open_scope( "while(true){" );
  const string prefix( "currentState->" );
  _set_publish_prefix_string( prefix );
  
  process_statement_part._publish_cc( process_cc_file );
  _set_publish_prefix_string( "" );
  
  process_cc_file.close_scope( "} // end primary while loop" );
  process_cc_file.close_scope( "} // end execute_VHDL" );
  
  // This piece of code is very tricky it has to come only after 
  // the publish_cc is called on the process statement part
  IIR_Declaration *guardSignal = NULL;
  if (IIRScram_BlockStatement::guard_expr != NULL) {
    // This means an implicit guard signal exists in this scope
    // If the signal is used in this process then get it
    guardSignal = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
    while (guardSignal != NULL) {
      if (!IIRScram_TextLiteral::_cmp(guardSignal->get_declarator(), "guard")) {
	break;
      }
      guardSignal = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(guardSignal);
    }
  }

  //  -----** This is done for guard signals
  SCRAM_CC_REF( process_header_file, "IIRScram_ProcessStatement::_publish_cc( published_file &process_header_file )" );
  if (guardSignal != NULL) {
    // This means the process is using the implicit guard signal
    process_header_file << "  VHDLType *getGuard();\n";
    process_header_file << "  VHDLType *getGuardExpression();\n";
  }
  
  process_header_file.close_scope( "};" );
  //  ------**

  // Back to publishing code in the cc file
  SCRAM_CC_REF( process_cc_file, "IIRScram_ProcessStatement::_publish_cc( published_file &process_cc_file )" );
  // <process> destructor
  process_cc_file << _get_cc_process_class() << "::~" << _get_cc_process_class() << "() {}" 
		  << NL();

  // find out the number of things declared & used in this scope.

  // <process> constructor
  process_cc_file << _get_cc_process_class() << "::" << _get_cc_process_class() 
		  << "( " << _get_current_publish_name()
		  << design_unit_name;
  process_cc_file.open_scope( "_elab* ptr)" );
  process_cc_file << ": VHDLProcess(";
  process_cc_file << "\"" << _get_cc_process_class() << "\", ptr )";
  process_cc_file.close_scope("");
  
  first = process_declarative_part._publish_cc_type_info_and_constant_definitions( process_cc_file, FALSE );  
  
  process_cc_file.open_scope("{");
  process_declarative_part._publish_cc_global_constants_assignments( process_cc_file );
  _set_currently_publishing_unit(_saved_publishing_unit);
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    ASSERT(decl->get_kind() == IIR_FILE_DECLARATION);
    ASSERT(decl->_is_read_file() || decl->_is_write_file());

    if (decl->_is_read_file()) {
      numInFiles++;
    }
    else if (decl->_is_write_file()) {
      numOutFiles++;
    }

    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
  // All the files are actually opened in initialize() function.

  process_cc_file.close_scope("}");

  process_cc_file << "State *" << NL()
		  << OS( _get_cc_process_class() + "::allocateState(){" )
		  << "getProc()->copyGenericsToGlobals();" << NL()
		  << "return new " << _get_cc_state_class() << "(this);" << NL()
		  << CS( "}" );
  
  process_cc_file << "void\n" << _get_cc_process_class() << "::initialize(){" << NL();
  _publish_cc_file_open( process_cc_file );
  process_cc_file << "  VHDLProcess::initialize();\n";
  process_cc_file << "}" << NL();
  
  _set_publish_prefix_string( prefix );
  if (guardSignal != NULL) {
    // This means the implicit guard signal is used in this process
    process_cc_file << "VHDLType*\n" << _get_cc_process_class() << "::getGuard() {\n";
		    
    process_cc_file << "  " << _get_cc_state_class() << " *currentState = static_cast<"
		    << _get_cc_state_class() << " *>(getState());\n\n"
		    << "  return &(";
    
    guardSignal->_publish_cc_lvalue( process_cc_file );
    process_cc_file <<");\n}\n";
      
    process_cc_file << "VHDLType*\n"  << _get_cc_process_class()
		    << "::getGuardExpression() {\n";
    process_cc_file << "  " << _get_cc_state_class() << "*currentState = static_cast<"
		    << _get_cc_state_class() << " *>(getState());\n\n";
    
    process_cc_file <<"  return new EnumerationType(ObjectBase::SIGNAL, ";
    IIRScram_BlockStatement::guard_expr->_publish_cc_lvalue( process_cc_file );
    process_cc_file << ", SavantbooleanType_info);\n}";
  }
  _set_publish_prefix_string( "" );
  
  _publish_cc_state();

  cgen_sym_tab_ptr = NULL;
  _set_current_process_statement(NULL);
}

const string
IIRScram_ProcessStatement::_get_design_unit_name(){
  string retval = "";
  // This check is needed as process stmnts in entity decls have no arch
  if( _get_current_architecture_name() == "" ){
    retval = _get_current_entity_name();
  }
  else {
    retval = _get_current_architecture_name() + _get_enclosing_stmt_to_architecture_path();
  }

  return retval;
}

void
IIRScram_ProcessStatement::_publish_cc_state(){
  //Reset the code-generation attributes set during publishing any of the
  //previously encountered process statements
  _clear();

  //All objects to be defined in _sate.hh file should not
  //be printed with prefix "state.current->"
  //So setting the flag so all object and Interface declarations
  //print as per the value of that flag
  IIR_Boolean tmp_value = _get_publish_object_without_prefix();
  _set_publish_object_without_prefix(TRUE);

  published_header_file header_file( _get_work_library()->_get_path_to_directory(),
				     _get_cc_state_class(),
				     this );

  SCRAM_CC_REF( header_file, 
		"IIRScram_ProcessStatement::_publish_cc_state()");

  PublishedUnit _saved_publishing_unit = _get_currently_publishing_unit();
  _set_currently_publishing_unit(PROCESS_STATE);

  //The following function adds the declarations in initializations of
  //visible declarations
  // This routine is not ok -- Probably we can remove it altogether
  _add_declarations_in_initializations();


  _publish_cc_headers( header_file );
  _publish_cc_state_class( header_file );

  published_cc_file cc_file( _get_work_library()->_get_path_to_directory(),
			     _get_cc_state_class(),
			     this );  
  SCRAM_CC_REF( cc_file,
		"IIRScram_ProcessStatement::_publish_cc_state( )" );

  _publish_cc_include_state( cc_file );
  _publish_cc_include( cc_file );
  
  //redefine savantnow so that any initializations through savant now is 
  //performed
  _publish_cc_savantnow_process_ptr( cc_file );
  
  _publish_cc_constructor( cc_file );
  _publish_cc_initstate( cc_file );
  _publish_cc_cleanstate( cc_file );
  _publish_cc_locatesignal( cc_file );
  _publish_cc_operator_equalto( cc_file );

  //Resetting the flag value to the original value
  _set_publish_object_without_prefix(tmp_value);
  _set_currently_publishing_unit(_saved_publishing_unit);
}

void
IIRScram_ProcessStatement::_publish_cc_headers( published_file &_cc_out ) {
  const string design_unit_name = _get_design_unit_name();
  SCRAM_CC_REF( _cc_out, 
		"IIRScram_ProcessStatement::_publish_cc_headers( published_file &_cc_out )" );

  IIRScram::_publish_cc_include( _cc_out, "tyvis/SignalNetinfo.hh" );
  IIRScram::_publish_cc_include( _cc_out, "tyvis/Wait.hh" );
  IIRScram::_publish_cc_include( _cc_out, "tyvis/AccessVariable.hh" );

  _cc_out << "#include \"" << _get_current_publish_name() << _get_design_unit_name()
	  << "_elab.hh\"\n"
	  << "#include \"" << _get_cc_process_class() << "_decls.hh\"\n";
}

void
IIRScram_ProcessStatement::_publish_cc_file_declarations( published_file &_cc_out ) {
  IIR_Declaration *decl;
  int numifiles = 0;
  int numofiles = 0;

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    ASSERT(decl->get_kind() == IIR_FILE_DECLARATION);
    _cc_out << "const int ";
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << "_" << decl << " = ";

    if( decl->_is_read_file()) {
      numifiles++;
      _cc_out << numifiles - 1 << ";\n";
    }
    else if( decl->_is_write_file() ){
      numofiles++;
      _cc_out << numofiles - 1 << ";\n";
    }
    else {
      cerr << "_publish_process_stmt::_publish_cc ERROR!" 
	   << "  File is neither read nor write!\n";
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
}

const string
IIRScram_ProcessStatement::_get_cc_binding_name(){
  ostringstream os;
  if( _get_label() != NULL ) {
    _get_label()->_publish_cc_elaborate( os );
  }
  else {
    os << "ANON_PROCESS" << this;
  }
  return os.str();
}

void 
IIRScram_ProcessStatement::_publish_cc_binding_name( ostream &os ){
  os << _get_cc_binding_name();
}

void
IIRScram_ProcessStatement::_publish_cc_extern_declarations( published_file &_cc_out ) {
  IIR_Declaration *decl;

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while( decl != NULL ){
    ASSERT(decl->get_kind() == IIR_FILE_DECLARATION);
    _cc_out << "extern const int ";
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << "_" << decl;
    _cc_out << ";\n";
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
  _cc_out << "extern const int maxExtraInFilesInProcedure;\n";
  _cc_out << "extern const int maxExtraOutFilesInProcedure;\n";
}

void
IIRScram_ProcessStatement::_publish_cc_read_or_write( published_file &_cc_out,
						      const string &functionName,
						      const string &streamName ){
  IIR_Declaration *decl;
  // Publish copy operator for constants in this state.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    switch (decl->_get_declarative_region()->get_kind()) {
    case IIR_PACKAGE_DECLARATION:
    case IIR_PACKAGE_BODY_DECLARATION:
    case IIR_ARCHITECTURE_DECLARATION:
    case IIR_PROCESS_STATEMENT:
      break;
    default:
      decl->_publish_cc_lvalue( _cc_out );
      _cc_out << "." << functionName << "(" << streamName;
      _cc_out << ");\n";
      break;
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  // Publish copy operator for signals in this state.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    if (decl->_is_implicit_declaration() == FALSE) {
      decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
      _cc_out << "." << functionName << "(" << streamName;
      _cc_out << ");\n";
      decl->_publish_cc_implicit_signal_attributes_read_or_write( _cc_out, 
								  functionName, 
								  streamName);
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << "." << functionName << "(" << streamName;
    _cc_out << ");\n";
    decl->_publish_cc_implicit_signal_attributes_read_or_write( _cc_out, 
								functionName, 
								streamName );
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
  
  // Publish copy operator for variables in this state.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    if (decl->get_subtype()->_is_access_type() == FALSE) {
      decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
      _cc_out << "." << functionName << "(" << streamName;
      _cc_out << ");\n";
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }
  
  // Publish copy operator for interface variables in this state - removed.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].first();
  while (decl != NULL) {
    decl->_publish_cc_lvalue( _cc_out );
    _cc_out << "." << functionName << "(" << streamName << ");\n";
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].successor(decl);
  }
  
  _cc_out << "\n";
}

void
IIRScram_ProcessStatement::_clear() {
  IIR_Declaration *decl;

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    decl->_clear();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    decl->_clear();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
}

const string
IIRScram_ProcessStatement::_get_cc_state_class() const {
  return _get_cc_process_class() + "_state";
}

void
IIRScram_ProcessStatement::_publish_cc_state_class( published_file &_cc_out ) {
  IIR_Declaration *decl;

  SCRAM_CC_REF( _cc_out,
		"IIRScram_ProcessStatement::_publish_cc_state_class( published_file & )" );

  IIRScram::_publish_cc_include( _cc_out, "tyvis/VHDLKernel_state.hh" );
  _cc_out << "class " << _get_cc_process_class() << ";\n\n"
	  << "class " << _get_cc_state_class() <<" : public VHDLKernel_state {\n"
	  << "public:\n"
	  << "  " << _get_cc_state_class() << "(" << _get_cc_process_class() << "*);\n"
	  << "  ~" << _get_cc_state_class() << "(){}\n"
	  << "  void initState(_savant_entity_elab *, VHDLKernel * );\n"
	  << "  void cleanState();\n"
	  << "  " << _get_cc_state_class() << "& operator=(const " << _get_cc_state_class() << "&);\n"
	  << "  void copyState(const State *rhs){\n"
	  << "    *this = *(static_cast<const " << _get_cc_state_class() << " *>(rhs));\n"
	  << "  }\n\n"
	  << "  SignalBase* locateSig(int);\n\n";
  
  //Constants are made to publish first because,
  //other types of objects can be initialized using these constants
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    switch (decl->_get_declarative_region()->get_kind()) {
    case IIR_PACKAGE_DECLARATION:
    case IIR_PACKAGE_BODY_DECLARATION:
    case IIR_ARCHITECTURE_DECLARATION:
    case IIR_PROCESS_STATEMENT:
      break;
    default:
      decl->_publish_cc_decl( _cc_out );
      break;
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    if (decl->_is_implicit_declaration()) {
      IIR_Declaration* decl_prefix;
      decl_prefix = decl->_get_signal_prefix();
      if (!cgen_sym_tab_ptr->in_scope(decl_prefix)) {
	cgen_sym_tab_ptr->add_declaration(decl_prefix);
      }
    }
    else {
      _cc_out << "  ";
      decl->_publish_cc_decl( _cc_out );
      //This function publishes all the declarations that are in scope.
      decl->_publish_cc_implicit_signal_attributes( _cc_out );
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    _cc_out << "  ";
    decl->_publish_cc_decl( _cc_out );
    //This function publishes all the declarations that are in scope.
    decl->_publish_cc_implicit_signal_attributes( _cc_out );
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    _cc_out << "  ";
    decl->_publish_cc_decl( _cc_out );
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }
  
  // interface variables are no longer handled here - removed code
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].first();
  while (decl != NULL) {
    _cc_out << "  ";
    decl->_publish_cc_decl( _cc_out );
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].first();
  while (decl != NULL) {
    _cc_out << "  ";
    decl->_publish_cc_decl( _cc_out );
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    _cc_out << "  ";
    decl->_publish_cc_decl( _cc_out );
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::QUANTITY].first();
  IIR_Declaration *implied_declaration = NULL;
  while (decl != NULL) {
    if(decl->_get_implicit_declarations() != NULL &&
       decl->_get_implicit_declarations()->num_elements() != 0) {
      for(implied_declaration = decl->_get_implicit_declarations()->get_element();
          implied_declaration != NULL;
          implied_declaration = decl->_get_implicit_declarations()->get_next_element()) {
        if(implied_declaration->_is_signal() == TRUE) {
          implied_declaration->_publish_cc_decl(_cc_out);
	  implied_declaration->_publish_cc_implicit_signal_attributes(_cc_out);
        }
      }
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::QUANTITY].successor(decl);
  }
  
  for (IIR_SequentialStatement* seqstmt = process_statement_part.first();
       seqstmt != NULL;
       seqstmt = process_statement_part.successor(seqstmt)) {
    if (seqstmt->get_kind() == IIR_BREAK_STATEMENT) {
      ASSERT(seqstmt->_get_process_stmt_label() != NULL);
      string signalname = seqstmt->_get_process_stmt_label()->_to_string();
      _cc_out << "EnumerationType " << signalname << ";" << NL();
    }
  }
  _cc_out << "\n"
	  << "  const int numWaits;\n"
	  << "};\n\n";
  
}

void
IIRScram_ProcessStatement::_publish_cc_constructor( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out,
		"IIRScram_ProcessStatement::_publish_cc_constructor( )" );
  // generate <process>_state constructor
  _cc_out << _get_cc_state_class() << "::" << _get_cc_state_class() << "("
	  << _get_cc_process_class() << " *processPtr )" << OS(":");
  set<IIR_Declaration> busSignals;
  _publish_cc_state_objects_init( _cc_out, &busSignals );
  // signal names could go here for human readability--not required.
  _cc_out << "numWaits(" << _wait_stmt_list.num_elements() << ")" << CS("") << OS("{");
  // If the signal is of kind bus then set it 
  IIR_Declaration *decl = busSignals.get_element();
  while( decl != 0 ){
    _cc_out << "  ";
    decl->_publish_cc_lvalue( _cc_out );
    _cc_out <<".setBusKind();" << NL();
    decl = busSignals.get_next_element();
  }

  _cc_out << "  wait = new Wait[" << _wait_stmt_list.num_elements()
	  << "];" << NL();
  
  //For a composite Resolved signals, certain data members namely
  // compositeResolvedSignal, and parentCompositeType of the kernel Signal
  //has to be initialized. The following function publishes them.
  _publish_cc_composite_resolved_signal_init( _cc_out );

  _cc_out << CS("}");
}

void
IIRScram_ProcessStatement::_publish_cc_composite_resolved_signal_init( published_file &_cc_out ) {
  IIR_Declaration *decl = NULL;
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    if(decl->_is_composite_resolved_signal() == TRUE) {
      decl->_publish_cc_lvalue( _cc_out );
      _cc_out << ".setParentCompositeType(&";
      decl->_publish_cc_lvalue( _cc_out );
      _cc_out << ");" << NL();
      decl->_publish_cc_lvalue( _cc_out );
      _cc_out << ".setCompositeResolvedSignal(true);" << NL();
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    if(decl->_is_composite_resolved_signal() == TRUE) {
      decl->_publish_cc_lvalue( _cc_out );
      _cc_out << ".setParentCompositeType(&";
      decl->_publish_cc_lvalue( _cc_out );
      _cc_out << ");" << NL();
      decl->_publish_cc_lvalue( _cc_out );
      _cc_out << ".setCompositeResolvedSignal(true);" << NL();
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
}

void
IIRScram_ProcessStatement::_publish_cc_initstate( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out,
		"IIRScram_ProcessStatement::_publish_cc_initstate()" );
  const string old_current_publish_name = _get_current_publish_name();
  IIR *oldNode = _get_current_publish_node();
  // generate_state::initState
  _cc_out << "void" << NL() 
	  << _get_cc_state_class() << "::initState(_savant_entity_elab *ptr," << NL()
	  << OS("VHDLKernel *processPointer ){");
  // We need to check if _current_architecture_name is NULL or not since
  // process statements in entity decl will not have an arch. In other
  // places i.e. in arch decls, blocks etc, this will be set.
  
  const string elabString = _get_current_publish_name() + _get_design_unit_name() + "_elab";

  _cc_out << elabString << "* proc = (" << elabString << " *) ptr;" << NL()
	  << _get_cc_process_class() << " *processPtr = (" << _get_cc_process_class()
	  << " *) processPointer;" << NL()
	  << "VHDLKernel_state::initState(ptr, processPointer);" << NL();
  for (IIR_SequentialStatement* seqstmt = process_statement_part.first();
       seqstmt != NULL;
       seqstmt = process_statement_part.successor(seqstmt)) {
    ostringstream breakstr;
    const string design_unit_name = _get_design_unit_name();
    if (seqstmt->get_kind() == IIR_BREAK_STATEMENT) {
      ASSERT(seqstmt->_get_process_stmt_label() != NULL);
      breakstr << _get_cc_process_class() << "_breakSignal" ;
      string breakstring = breakstr.str();
      _cc_out << "{" << NL();
      _cc_out << "\t Signal<UniversalBoolean> *signal = " << NL();
      _cc_out << "\t (Signal<UniversalBoolean>*) (*(ScalarType*)&this->" ;
      _cc_out << breakstring << ").object;" << NL();
      _cc_out << "\t SignalNetinfo *signal_info = " << NL();
      _cc_out << "\t (SignalNetinfo*) (*(ScalarType*)&proc->"
              << breakstring << "_info).object;" << NL();
      _cc_out << "\t signal->fanDest = signal_info->obj_ids;" << NL();
      _cc_out << "\t signal->fanout = signal_info->fanout;" << NL();
      _cc_out << "\t signal->id = signal_info->id;" << NL();
      _cc_out << "\t signal->type = EXPLICIT;" << NL();
      _cc_out << "\t signal->sensitive = true;" << NL();
      _cc_out << "\t signal->numAttributes = 0;" << NL();
      _cc_out << "\t signal->attributeList = NULL;" << NL();
      _cc_out << "\t signal->source = signal_info->source;" << NL();
      _cc_out << "\t signal->name =\"" << breakstring << "\";" << NL();
      _cc_out << "}" << NL();
    }
  }
  _set_current_publish_name( _get_cc_process_class() );
  SCRAM_CC_REF( _cc_out,
		"IIRScram_ProcessStatement::_publish_cc_initstate()" );
  // Set the file handles for the files in this process state
  int inputFileHandleCount = 0, outputFileHandleCount = 0;
  IIR_Declaration *decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    ASSERT ( decl->get_kind() == IIR_FILE_DECLARATION );
    ASSERT ( decl->_is_read_file() == TRUE ||
	     decl->_is_write_file() == TRUE );
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << ".set_fileHandle(";
    if (decl->_is_read_file() == TRUE) {
      _cc_out << inputFileHandleCount;
      inputFileHandleCount++;
    }
    else {
      ASSERT ( decl->_is_write_file() == TRUE );
      _cc_out << outputFileHandleCount;
      outputFileHandleCount++;
    }
    _cc_out << ");" << NL();
    
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
  // Publish initialization for Constants and Constant Interfaces
  _cc_out  << NL();
  SCRAM_CC_REF( _cc_out,
		"IIRScram_ProcessStatement::_publish_cc_initstate()" );
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    if ( decl->_get_declarative_region()->get_kind() != IIR_PACKAGE_DECLARATION && decl->_get_declarative_region()->get_kind() != IIR_PACKAGE_BODY_DECLARATION ){
      
      decl->_publish_cc_init( _cc_out );
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  // Publish intialization for Signals and Signal Interfaces
  // Need this stuff for scoping of signals.
  SCRAM_CC_REF( _cc_out,
		"IIRScram_ProcessStatement::_publish_cc_initstate()" );
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    _set_current_publish_node( _get_enclosing_scope() );
    if ( decl->_is_implicit_declaration() == FALSE &&
	 decl->_get_declarative_region() != NULL &&
	 decl->_get_declarative_region()->_is_subprogram() == FALSE ){
      decl->_publish_cc_init( _cc_out );
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    _set_current_publish_node( _get_enclosing_scope() );
    if( decl->_is_implicit_declaration() == FALSE &&
	decl->_get_declarative_region() != NULL &&
	decl->_get_declarative_region()->_is_subprogram() == FALSE ){
      decl->_publish_cc_init( _cc_out );
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
  _set_current_publish_node( oldNode );
  
  // Publish initialization for Variables and Variable Interfaces
  _cc_out  << NL();
  SCRAM_CC_REF( _cc_out,
		"IIRScram_ProcessStatement::_publish_cc_initstate()" );
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    decl->_publish_cc_init( _cc_out );
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }
  // interface variables no longer handled here - code removed 
  _cc_out << NL();
  IIR_Declaration *implied_decl = NULL;
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::QUANTITY].first();
  while (decl != NULL) {
    if(decl->_get_implicit_declarations() != NULL &&
       decl->_get_implicit_declarations()->num_elements() != 0) {
      for(implied_decl = decl->_get_implicit_declarations()->get_element();
          implied_decl != NULL;
          implied_decl = decl->_get_implicit_declarations()->get_next_element()) {

        if(implied_decl->_is_signal() == TRUE) {
	  PublishedUnit _temp_publishing_unit = _get_currently_publishing_unit();
	  _set_currently_publishing_unit(ABOVE_ATTRIBUTE);
          implied_decl->_publish_cc_init(_cc_out);
	  _set_currently_publishing_unit(_temp_publishing_unit);
	}
      }
    }
    decl =
      cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::QUANTITY].successor(decl);
  }

  _cc_out << NL();
  _publish_cc_wait_init( _cc_out );
  _cc_out << CS("}");

  _set_current_publish_name( old_current_publish_name );
}

void
IIRScram_ProcessStatement::_publish_cc_cleanstate( published_file &_cc_out ) {
  // generate <process>_state::cleanState
  _cc_out << "void" << NL();
  _cc_out << _get_cc_state_class() << "::cleanState() {" << NL();
  _cc_out << "//" << " Hey!!! cleanup is done in the signal's destructor" << NL();
  for(int i=0; i < _wait_stmt_list.num_elements(); i++){ 
    _cc_out << "wait[" << i <<"].destructSensitivityList();" << NL();
  }

  _cc_out << "}" << NL();
}

void
IIRScram_ProcessStatement::_publish_cc_locatesignal( published_file &_cc_out ) {
  IIR_Declaration *decl;

  // generate <process>_state::locateSig
  _cc_out << "SignalBase*" << NL()
	  << _get_cc_state_class() << "::locateSig(int sigId) {" << NL()
	  << "SignalBase* ptr = NULL;" << NL();
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    decl->_publish_cc_locatesignal( _cc_out );
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    decl->_publish_cc_locatesignal( _cc_out );
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
  for (IIR_SequentialStatement* seqstmt = process_statement_part.first();
       seqstmt != NULL;
       seqstmt = process_statement_part.successor(seqstmt)) {
    ostringstream breakstr;
    const string design_unit_name = _get_design_unit_name();
    if (seqstmt->get_kind() == IIR_BREAK_STATEMENT) {
      ASSERT(seqstmt->_get_process_stmt_label() != NULL);
      breakstr << _get_cc_process_class() << "_breakSignal" ;
      string breakstring = breakstr.str();
      _cc_out << "if(sigId == ((SignalBase*)(*(ScalarType*)&" << breakstring
              << ").object)->id)" << NL();
      _cc_out << " return ((SignalBase*)(*(ScalarType*)&" << breakstring
              << ").object);" << NL();
    }
  }
  _cc_out  << NL()
	   << "    cerr << \""
	   << _get_cc_state_class() << "::locateSig ERROR: signal \" << sigId\n"
	   << "         << \" not in state!\";\n"
	   << "  abort();\n \n  return ptr; \n}\n" << NL();
}

void
IIRScram_ProcessStatement::_publish_cc_print( published_file &_cc_out ) {
  // generate <process>_state::print
  _cc_out << "void" << NL();
  _cc_out << _get_cc_state_class() << "::print(ostream& os) {" << NL();
  _cc_out << "  os << *this;\n}\n\n" << NL();
}

void
IIRScram_ProcessStatement::_publish_cc_operator_equalto( published_file &_cc_out ) {
  IIR_Declaration *decl;
  // generate <process>_state::operator=
  _cc_out << _get_cc_state_class() + " &" << NL();
  _cc_out << OS(_get_cc_state_class() + "::operator=(const " + _get_cc_process_class() + "_state& s) {");
  _cc_out << "(VHDLKernel_state&)*this = (const VHDLKernel_state&)s;" << NL();

  // Publish copy operator for constants in this state.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    switch (decl->_get_declarative_region()->get_kind()) {
    case IIR_PACKAGE_DECLARATION:
    case IIR_PACKAGE_BODY_DECLARATION:
    case IIR_ARCHITECTURE_DECLARATION:
    case IIR_PROCESS_STATEMENT:
      break;
    default:
      _cc_out << "this->";
      decl->_publish_cc_lvalue( _cc_out );
      _cc_out << " = s.";
      decl->_publish_cc_lvalue( _cc_out );
      _cc_out << ";" << NL();
      break;
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  _cc_out  << NL();
  
  // Publish copy operator for signals in this state.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while( decl != NULL ){
    if( decl->_is_implicit_declaration() == FALSE ){
      _cc_out << "this->";
      decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
      _cc_out << " = s.";
      decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
      _cc_out << ";" << NL();
      decl->_publish_cc_implicit_signal_attributes_copying( _cc_out );
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while( decl != NULL ){
    _cc_out << "this->";
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << " = s.";
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << ";" << NL();
    decl->_publish_cc_implicit_signal_attributes_copying( _cc_out );
    
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
  
  // Publish copy operator for variables in this state.
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    _cc_out << "this->";
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << " = s.";
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << ";" << NL();
    
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }

  // Publish copy operator for interface variables in this state - removed.

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].first();
  while (decl != NULL) {
    _cc_out << "this->";
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << " = s.";
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << ";" << NL();
    
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].first();
  while (decl != NULL) {
    _cc_out << "this->";
    decl->_publish_cc_lvalue( _cc_out );
    _cc_out << " = s.";
    decl->_publish_cc_lvalue( _cc_out );
    _cc_out << ";" << NL();
    
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    _cc_out << "this->";
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << " = s.";
    decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
    _cc_out << ";" << NL();
    
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
  
  // Publish copy operator for interface constants in this state.
  // This is not necessary since we directly replace the constant with the
  // value of the constant.

  _cc_out << "for (register int i = 0; i < numWaits; i++) wait[i] = s.wait[i];" << NL();
  _cc_out << "return *this;" << NL()
	  << CS("}\n");
}

void
IIRScram_ProcessStatement::_publish_createNetInfo( published_file &_cc_out ){
  set<IIR> input_signal_list;
  string objectname;
  _get_list_of_input_signals(&input_signal_list);

  objectname = ", " + _get_cc_process_class();
  if (_is_currently_publishing_generate_for()) {
    objectname += "_elab_obj[i - generateLeft]);";
  }
  else {
    objectname += "_elab_obj);";
  }
  
  IIR *signals = input_signal_list.get_element();
  ASSERT ( _get_enclosing_scope() != NULL );
  while( signals != NULL ){
    SCRAM_CC_REF( _cc_out, "IIRScram_ProcessStatement::_publish_createNetInfo( published_file &_cc_out )" );
    _cc_out << OS("Add(");
    IIR *declarative_region = NULL;
    if( signals->_is_iir_declaration() ){
      IIR_Declaration *as_declaration = (IIR_Declaration *)signals;
      if( as_declaration->_is_implicit_declaration() == TRUE ){
	declarative_region = as_declaration->_get_signal_decl()->_get_declarative_region();
      }
      else{
	declarative_region = as_declaration->_get_declarative_region();
      }
    }
    else if( signals->_is_iir_name() ){
      declarative_region = signals->_get_prefix_declaration()->_get_declarative_region();
    }
    else if( signals->get_kind() == IIR_DESIGNATOR_EXPLICIT ){
      IIR_DesignatorExplicit *as_designator = (IIR_DesignatorExplicit *)signals;
      ASSERT( as_designator->get_name()->_is_iir_declaration() );
      IIR_Declaration *as_declaration = (IIR_Declaration *)as_designator->get_name();
      declarative_region = as_declaration->_get_declarative_region();
    }
    _get_enclosing_scope()->_publish_cc_scoping_prefix( _cc_out.get_stream(),
							declarative_region,
							_get_enclosing_scope() );
    signals->_publish_cc_elaborate( _cc_out );

    _cc_out << objectname << CS("");  // The objectname has a ", <whatever> )" in it...
    signals = input_signal_list.get_next_element();
  }
  for (IIR_SequentialStatement* seqstmt = process_statement_part.first();
       seqstmt != NULL;
       seqstmt = process_statement_part.successor(seqstmt)) {
    ostringstream breakstr;
    if (seqstmt->get_kind() == IIR_BREAK_STATEMENT) {
      ASSERT(seqstmt->_get_process_stmt_label() != NULL);
      breakstr << _get_cc_process_class() << "_breakSignal" ;
      string breakstring = breakstr.str();
      _cc_out << "{" << NL();
      _cc_out << "VHDLType* ptr = new EnumerationType(ObjectBase::SIGNAL, "
	      <<"SavantbooleanType_info);" << NL();
      _cc_out << "addChild(" << breakstring << "_info, *(ptr)" << objectname ;
      _cc_out << "};" << NL();
    }
  }
  const string _saved_current_publish_name = _get_current_publish_name();
  _set_current_publish_name( objectname );
  IIR *_saved_current_publish_node      = _get_current_publish_node();
  
  set<IIR> signal_source_set;
  _get_signal_source_info(&signal_source_set);
  
  //  IIR_Declaration* sig_decl = signal_source_set.get_element();
  IIR* sig_decl = signal_source_set.get_element();
  while( sig_decl != NULL ){
    SCRAM_CC_REF( _cc_out, "IIRScram_ProcessStatement::_publish_createNetInfo( published_file &_cc_out )" );
    if (sig_decl->_is_iir_declaration() == TRUE || sig_decl->_is_name() == TRUE ) {
      sig_decl->_publish_cc_addChild( _cc_out );
    }
    sig_decl = signal_source_set.get_next_element();
  }
  _set_current_publish_name( _saved_current_publish_name );
  _set_current_publish_node( _saved_current_publish_node );
}

IIR_Boolean
IIRScram_ProcessStatement::_publish_cc_type_info( published_file &_cc_out,
						  IIR_Boolean firstDeclFlag) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ProcessStatement::_publish_cc_type_info" );
  return process_declarative_part._publish_cc_type_info( _cc_out, firstDeclFlag );
}

void
IIRScram_ProcessStatement::_publish_cc_extern_type_info( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ProcessStatement::_publish_cc_extern_type_info" );
  process_declarative_part._publish_cc_extern_type_info( _cc_out );
}

void
IIRScram_ProcessStatement::_get_signal_source_info( set<IIR> *siginfo ){
  process_statement_part._get_signal_source_info(siginfo);
}

void
IIRScram_ProcessStatement::_get_list_of_input_signals( set<IIR> *list ){
  process_statement_part._get_list_of_input_signals(list);
}

void
IIRScram_ProcessStatement::_publish_cc_implicit_declarations( published_file &_cc_out,
							      IIR_Declaration *decl) {
  ASSERT(decl->_is_implicit_declaration() == FALSE);
  if( decl->_get_implicit_declarations() != NULL ){
    IIR_Declaration* implied_declarations = decl->_get_implicit_declarations()->get_element();
    while (implied_declarations != NULL) {
      if (cgen_sym_tab_ptr->in_scope(implied_declarations)) {
	implied_declarations->_publish_cc_decl( _cc_out );
	implied_declarations->_publish_cc_implicit_signal_attributes( _cc_out );
      }
      implied_declarations = decl->_get_implicit_declarations()->get_next_element();
    }
  }
}

void
IIRScram_ProcessStatement::_publish_cc_state_objects_init( published_file &_cc_out,
							   set<IIR_Declaration> *busSigs ){

  SCRAM_CC_REF( _cc_out,
		"IIRScram_ProcessStatement::_publish_cc_state_objects_init()" );
  // Publish inititalization for constant
  IIR_Declaration *decl = 
    cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    switch(decl->_get_declarative_region()->get_kind()) {
    case IIR_PACKAGE_DECLARATION:
    case IIR_PACKAGE_BODY_DECLARATION:
    case IIR_ARCHITECTURE_DECLARATION:
      break;
    default:
      if ( decl->_get_declarative_region()->get_kind() != IIR_PROCESS_STATEMENT &&
	   decl->_get_declarative_region()->get_kind() != IIR_ARCHITECTURE_DECLARATION ){
	decl->_publish_cc_state_object_init( _cc_out );
	_cc_out  << NL();
      }
      break;
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  // Publish inititalization for signals
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    if(decl->_is_implicit_declaration() == FALSE) {
      decl->_publish_cc_state_object_init( _cc_out );
      _cc_out  << NL();
      decl->_publish_cc_implicit_state_objects_init( _cc_out );
      if (decl->get_kind() == IIR_SIGNAL_DECLARATION) {
	if (((IIR_SignalDeclaration*)decl)->get_signal_kind() == IIR_BUS_KIND) {
	  busSigs->add(decl);
	}
      }
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }
  // Publish inititalization for interface signals
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    if(decl->_is_implicit_declaration() == FALSE) {
      decl->_publish_cc_state_object_init( _cc_out );
      _cc_out  << NL();
      decl->_publish_cc_implicit_state_objects_init( _cc_out );
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
  // Publish inititalization for variable
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    decl->_publish_cc_state_object_init( _cc_out );
    _cc_out  << NL();      
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }
  // Publish inititalization for alias declarations
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].first();
  while (decl != NULL) {
    decl->_publish_cc_state_object_init( _cc_out );
    _cc_out  << NL();      
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].successor(decl);
  }
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].first();
  while (decl != NULL) {
    decl->_publish_cc_state_object_init( _cc_out );
    _cc_out  << NL();      
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].successor(decl);
  }
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::QUANTITY].first();
  IIR_Declaration *implied_declaration = NULL;
  while (decl != NULL) {
    _cc_out << NL();
    if(decl->_get_implicit_declarations() != NULL &&
       decl->_get_implicit_declarations()->num_elements() != 0) {
      for(implied_declaration = decl->_get_implicit_declarations()->get_element();
          implied_declaration != NULL;
          implied_declaration = decl->_get_implicit_declarations()->get_next_element()) {
        if(implied_declaration->_is_signal() == TRUE) {
          PublishedUnit _temp_publishing_unit = _get_currently_publishing_unit();
          _set_currently_publishing_unit(ABOVE_ATTRIBUTE);
          implied_declaration->_publish_cc_state_object_init(_cc_out);
          _cc_out << NL();
          implied_declaration->_publish_cc_implicit_state_objects_init(_cc_out);
  	  _set_currently_publishing_unit(_temp_publishing_unit);
        }
      }
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::QUANTITY].successor(decl);
  }
  for (IIR_SequentialStatement* seqstmt = process_statement_part.first();
       seqstmt != NULL;
       seqstmt = process_statement_part.successor(seqstmt)) {
    ostringstream breakstr;
    if (seqstmt->get_kind() == IIR_BREAK_STATEMENT) {
      ASSERT(seqstmt->_get_process_stmt_label() != NULL);
      breakstr << _get_cc_process_class() << "_breakSignal" ;
      string breakstring = breakstr.str();
      _cc_out << breakstring << "(ObjectBase::SIGNAL)," << NL();
    }
  }
}

void
IIRScram_ProcessStatement::_publish_cc_wait_init( published_file &_cc_out ) {

  IIR_WaitStatement *wait;
  IIR_Boolean sens_list_newed = FALSE; // Flag used to indicate that the
  // new'ed memory should be deleted.
  for (wait = _wait_stmt_list.first();
       wait != NULL;
       wait = _wait_stmt_list.successor(wait)) {
    if (wait->sensitivity_list.num_elements() == 0) { 
      if (wait->get_condition_clause() != NULL) {
	wait->get_condition_clause()->_build_sensitivity_list(&wait->sensitivity_list);
	sens_list_newed = TRUE;
      }
    }
    if (wait->sensitivity_list.num_elements() != 0) {
      int sig = 0;
      IIR *sens_sig;
      IIR_Designator *current_signal_designator;
      current_signal_designator = wait->sensitivity_list.first();
      while (current_signal_designator != NULL) {
	ASSERT(current_signal_designator->get_kind()==IIR_DESIGNATOR_EXPLICIT);
	sens_sig = ((IIR_DesignatorExplicit *)current_signal_designator)->get_name();
	
	_cc_out <<"wait[" << wait->wait_id << "].addSensitivityTo( "
		<< "(VHDLType*)&(";
	sens_sig->_publish_cc_declarator( _cc_out );
	_cc_out << "));" << NL();
	current_signal_designator = wait->sensitivity_list.successor( current_signal_designator );
	sig++;
      }	// while
      if (sens_list_newed == TRUE) {
	current_signal_designator = wait->sensitivity_list.first();
	while (current_signal_designator != NULL) {
	  IIR_Designator *next =  wait->sensitivity_list.successor(current_signal_designator);
	  wait->sensitivity_list.remove( current_signal_designator );
	  delete (IIR_DesignatorExplicit *)current_signal_designator;
	  current_signal_designator = next;
	} // while
      }	// if(sens_list_newed == TRUE)
    }
  } // for
  _cc_out  << NL();
}

set<IIR_Declaration> *
IIRScram_ProcessStatement::_find_declarations( IIR_Name *name ){
  return process_declarative_part._find_declarations( name );
}

IIR *
IIRScram_ProcessStatement::_clone() {
  IIR_Declaration *decl;
  IIR_SequentialStatement *clone, *stmt = process_statement_part.first();
  IIR_ProcessStatement *newproc = new IIR_ProcessStatement;
  IIRScram::_clone( newproc );
  IIR_Label *label = get_label();

  if (label != NULL) {
    label = (IIR_Label*)label->_clone();
  }
  newproc->set_label(label);

#ifdef DEBUG_ELAB
  cout << "cloning process |";
  if (label == NULL || label->get_declarator() == NULL) {
    cout << "NONAME|" << NL();
  }
  else {
    cout << *label->get_declarator() << "|" << NL();;
  }
#endif

  decl = process_declarative_part.first();
  while ( decl != NULL ) {
      // This forces a distinct clone of these node types to be created
      // for this process.
    switch (decl->get_kind()) {
    case IIR_SIGNAL_DECLARATION:
      ((IIRScram_SignalDeclaration*)decl)->_clear_clone();
      break;
    case IIR_FILE_DECLARATION:
      ((IIRScram_FileDeclaration*)decl)->_clear_clone();
      break;
    case IIR_VARIABLE_DECLARATION:
      ((IIR_VariableDeclaration*)decl)->_clear_clone();
      break;
    case IIR_ALIAS_DECLARATION:
      ((IIR_AliasDeclaration*)decl)->_clear_clone();
      break;
    default:
      break;
    }
    newproc->process_declarative_part.append((IIR_Declaration*)decl->_clone());
    decl = process_declarative_part.successor( decl );
  }

  newproc->set_postponed(get_postponed());
  while (stmt != NULL) {
    clone = (IIR_SequentialStatement*)stmt->_clone();
    newproc->process_statement_part.append(clone);
    stmt = process_statement_part.successor(stmt);
  }
  return newproc;
}

#ifdef PROCESS_COMBINATION
void 
IIRScram_ProcessStatement::_static_elaborate(IIR_ArchitectureDeclaration *arch,
					     IIR_DeclarationList *cfglist,
					     char *hier_location) {
  ostringstream newname;
  IIR_Label *label;
  IIR_Char *text = NULL;
  IIR_Declaration *decl, *declclone, *next;
  char *hier_ptr;
  int hier_length = strlen( hier_location );

  label = get_label();
  if (label != NULL) {
    newname << hier_location << *label->get_declarator() << "_";
  }
  else {
    label = new IIR_Label;
    copy_location( this, label );
    newname << hier_location << this;
  }
  text = newname.str();

  // remove the trailing underscore from the process name
  *strrchr(text, '_') = '\0';
#ifdef DEBUG_ELAB
  cout << "elaborated process |" << text << "|" << NL();
#endif
  label->set_declarator(IIR_Identifier::get(text, strlen(text)));

  set_label(label);
  
  // replace the trailing underscore for furhter hierarchy building
  text[ strlen(text) ] = '_';

  strcpy( hier_location, text );

  decl = process_declarative_part.first();
  while ( decl != NULL ) {
    next = process_declarative_part.successor(decl);
    decl->_static_elaborate_decl(hier_location);
    decl = next;
  }

  hier_location[hier_length] = '\0';
  newname.rdbuf()->freeze(0);
}

bool
IIRScram_ProcessStatement::_convert_to_TWF() {
  // A process might have waits inside procedure calls it contains.
  // Therefore, I consider processes with procedure calls un-combinable.
  // This prevents processes with e.g. writes or reads inside them from
  // being combined.  This is fine, because the generated code will
  // contain duplicate labels otherwise, as each statement gets generated
  // twice: once in the initialization part, and once in the execution
  // part.
  dl_list<IIR_ProcedureCallStatement> proc_stmt_list;
  process_statement_part._build_procedure_call_stmt_list(&proc_stmt_list);
  if ( 0 < proc_stmt_list.num_elements() ) {
    return false;
  }

  if ( IIR_WAIT_STATEMENT != process_statement_part.last()->get_kind() ) {
    cout << "Process ";
    if ( NULL != get_label() ) {
      cout << *get_label();
    }
    else {
      cout << "UNNAMED";
    }
    cout << " needs to be TWF-converted." << NL();
    return false;
  }
  return true;
}
#endif

void
IIRScram_ProcessStatement::_add_declarations_in_initializations() {
  IIR_Declaration *decl;
  //Constants are made to publish first because,
  //other types of objects can be initialized using these constants
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    if ( decl->_get_declarative_region()->get_kind() != IIR_PACKAGE_DECLARATION && decl->_get_declarative_region()->get_kind() != IIR_PACKAGE_BODY_DECLARATION ){
      
      decl->_add_declarations_in_initializations();
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    if(decl->_is_implicit_declaration() == FALSE) {
      decl->_add_declarations_in_initializations();
    }
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    decl->_add_declarations_in_initializations();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    decl->_add_declarations_in_initializations();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }

  //   decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_VARIABLE].first();
  //   while (decl != NULL) {
  //     decl->_add_declarations_in_initializations();
  //     decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::INTERFACE_VARIABLE].successor(decl);
  //   }

  decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].first();
  while (decl != NULL) {
    decl->_add_declarations_in_initializations();
    decl = cgen_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::ALIAS].successor(decl);
  }
}

void
IIRScram_ProcessStatement::_build_wait_list() {
  // This core of this method will only execute if the wait statement list
  // has no elements.  It will have no elements if 1) the method hasn't
  // been called yet, or 2) it has been called, but the process has no
  // wait statements (which is bad VHDL, but should cause no problems
  // here).  This check is to make the method idempotent.
  if ( 0 == _wait_stmt_list.num_elements() ) {
    IIR_SequentialStatement *stmt;
 
    stmt = process_statement_part.first();
    while ( stmt != NULL ) {
      stmt->_build_wait_list((dl_list<IIRScram_WaitStatement>*)&_wait_stmt_list);
      stmt = process_statement_part.successor(stmt);
    }
  }
}

void
IIRScram_ProcessStatement::_publish_cc_concurrent_stmt_init( published_file &_cc_out,
							     IIR_DeclarationList * ){
  _cc_out << _get_cc_process_class() << "_elab_obj = NULL;" << NL();
}

const string
IIRScram_ProcessStatement::_get_cc_process_class() const {
  string retval;
  if( _get_label() != NULL ){
    retval += _get_label()->_to_string();
  }
  else {
    retval += "ANON_PROCESS" + ptr_to_string( this );
  }

  return retval;
}

void
IIRScram_ProcessStatement::_publish_cc_include( published_file &_cc_out,
						const string &to_insert ){
  ostringstream file_name_stream;
  file_name_stream << *_get_declarator() << to_insert << ".hh";
  IIRScram::_publish_cc_include( _cc_out, file_name_stream.str() );
}

void
IIRScram_ProcessStatement::_publish_cc_include_state( published_file &_cc_out ){
  _publish_cc_include( _cc_out, "_state" );
}

void
IIRScram_ProcessStatement::_publish_cc_instantiate_call( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ProcessStatement::_publish_cc_instantiate_call" );
  _cc_out << _get_cc_binding_name() << "_elab_obj = new " << _get_cc_binding_name() << "(this);" << NL();
  _set_number_of_processes( _get_number_of_processes() + 1 );
}

visitor_return_type *
IIRScram_ProcessStatement::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_ProcessStatement(this, arg);
}

void
IIRScram_ProcessStatement::_publish_cc_ams_break_function_include(published_file &_cc_out) {
  _cc_out << "#include \"" << _get_cc_process_class() << ".hh\"\n";
}

void
IIRScram_ProcessStatement::_publish_cc_ams_breakExpression_function_body(published_file &_cc_out) {
  
  const string procname = _get_cc_process_class();
  const string old_current_publish_name = _get_current_publish_name();
  IIR *oldNode = _get_current_publish_node();
  IIR_BreakElement *element;
  
  _set_currently_publishing_unit(IIRScram::PROCESS);
  ASSERT(_get_label() != NULL);
  
  for (IIR_SequentialStatement* seqstmt = process_statement_part.first();
       seqstmt != NULL;
       seqstmt = process_statement_part.successor(seqstmt)) {
    if (seqstmt->get_kind() == IIR_BREAK_STATEMENT) {
      element = ((IIR_BreakStatement *)seqstmt)->break_list.first();
      while (element != NULL) {
        _cc_out << "RealType *\n";
        _cc_out << "break_expr_func_" << element << "(VHDLKernelBase "
                << "*digitalProcess) {\n";
	_cc_out << procname << " *process =\n";
        _cc_out << "((" << procname << "*)digitalProcess);\n";
        if ((element->get_expression()->_is_iir_declaration() == TRUE)||
	    (element->get_expression()->_is_monadic_operator())) {
          _cc_out << "RealType *ptr = new RealType(ObjectBase::VARIABLE," ;
          element->get_expression()->_publish_cc_lvalue(_cc_out);
          _cc_out << ");" << NL();
          _cc_out << "return ptr;" << NL();
        }
        else if (element->get_expression()->_is_floating_literal() == TRUE) {
          _cc_out << "return (new ";
          element->get_expression()->_publish_cc_lvalue(_cc_out);
          _cc_out << ");" << NL();
        }
        else {
          _cc_out << "return (";
          element->get_expression()->_publish_cc_lvalue(_cc_out);
          _cc_out << ");" << NL();
        }
        _cc_out << "}" << NL() << NL();
        element = ((IIR_BreakStatement *)seqstmt)->break_list.successor(element);
      }
    }
  }
  _set_current_publish_node( oldNode );
  _set_current_publish_name( old_current_publish_name );
}

void
IIRScram_ProcessStatement::_publish_cc_ams_breakExpression_function_prototype(published_file &_cc_out) {
  
  IIR_BreakElement *element;
  for (IIR_SequentialStatement* seqstmt = process_statement_part.first();
       seqstmt != NULL;
       seqstmt = process_statement_part.successor(seqstmt)) {
    if (seqstmt->get_kind() == IIR_BREAK_STATEMENT) {
      element = ((IIR_BreakStatement *)seqstmt)->break_list.first();
      while (element != NULL) {
        _cc_out << "RealType *\n";
        _cc_out << "break_expr_func_" << element << "(VHDLKernelBase *);\n";
        _cc_out << NL();
        element = ((IIR_BreakStatement *)seqstmt)->break_list.successor(element);
      }
    }
  }
}

void
IIRScram_ProcessStatement::_publish_cc_ams_tick_above_info(published_file &_cc_out) {

  const string procname = _get_cc_process_class();
  IIR_AboveAttribute *current_above_attribute = _above_attribute_set.get_element();
  while(current_above_attribute != NULL) {
    _cc_out << "  tickAbove = new aboveAttributeInfo("; 
    _cc_out << procname << "_elab_obj, ";
    _cc_out << "above_attr_func_" << current_above_attribute
            << ", &(";
    ((IIR_Attribute*)current_above_attribute)->
      _get_implicit_declaration()->_publish_cc_elaborate(_cc_out);
    _cc_out << "), &(";
    current_above_attribute->get_prefix()->_publish_cc_lvalue(_cc_out);
    _cc_out << "));\n";
    _cc_out << "  globalAboveAttributeList.push_back(tickAbove);\n";
    current_above_attribute = _above_attribute_set.get_next_element();
  }
}

void
IIRScram_ProcessStatement::_publish_cc_ams_tick_above_function_prototype(published_file
									 &_cc_out) {
  IIR_AboveAttribute *current_above_attribute = _above_attribute_set.get_element();
  while(current_above_attribute != NULL) {
    _cc_out << "RealType * \n";
    _cc_out << "above_attr_func_" << current_above_attribute << "(VHDLKernelBase *);\n";
    _cc_out << NL();
    current_above_attribute = _above_attribute_set.get_next_element();
  }
}

void
IIRScram_ProcessStatement::_publish_cc_ams_tick_above_function_body(published_file &_cc_out) {

  const string procname = _get_cc_process_class();
  IIR_AboveAttribute *current_above_attribute = _above_attribute_set.get_element();
  _cc_out << "#include \"" << procname << ".hh\"\n";
  while(current_above_attribute != NULL) {
    _cc_out << "RealType *\n";
    _cc_out << "above_attr_func_" << current_above_attribute << "(VHDLKernelBase "
            << "*digitalProcess) {\n";
    _cc_out << procname << " *process =\n";
    _cc_out << "((" << procname << "*)digitalProcess);\n";
    if(current_above_attribute->get_suffix()->_is_iir_declaration() == TRUE ) {
      _cc_out << "RealType *ptr = new RealType(ObjectBase::VARIABLE,";
      current_above_attribute->get_suffix()->_publish_cc_lvalue(_cc_out);
      _cc_out << ");\n";
      _cc_out << "return ptr;" << NL();
    }
    else if (current_above_attribute->get_suffix()->_is_floating_literal() == TRUE) {
      _cc_out << "RealType *ptr = new RealType(ObjectBase::VARIABLE,";
      current_above_attribute->get_suffix()->_publish_cc_lvalue(_cc_out);
      _cc_out << ");\n";
      _cc_out << "return ptr;" << NL();
    }
    else {
      _cc_out << "RealType *ptr = new RealType(ObjectBase::VARIABLE,";
      current_above_attribute->get_suffix()->_publish_cc_lvalue(_cc_out);
      _cc_out << ");" << NL() ;
      _cc_out << "return ptr;" << NL() ;
    }
    _cc_out << "}" << NL() << NL ();
    current_above_attribute = _above_attribute_set.get_next_element();
  }
}

