/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "ttyaccess.h"
#include "common.h"
#include "helper.h"
#include "atcommand.h"
#include "config.h" //needed for TTYSPEED

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <strings.h>
#include <termios.h>
#include <unistd.h>

#ifndef OS2
int tty_portfd;
#else
#  define  INCL_OS2
#  define  INCL_DOSDEVICES
#  define  INCL_DOSDEVIOCTL
#  include <os2.h>
HFILE tty_portfd;
#endif

speed_t tty_speed(char *newttyspeed){
  switch (atoi(newttyspeed)) {
  case 9600: return B9600;
  case 19200: return B19200;
  case 38400: return B38400;
#ifdef B57600
  case 57600: return B57600;
#endif
#ifdef B115200
  case 115200: return B115200;
#endif
  default:
    myprintf(0,"%s","No valid baudrate defined. Using compiled in value...\n");
    return tty_speed(TTYSPEED);
  }
}

#include <termios.h>

#include "config.h"
#ifdef NO_CFMAKERAW
# undef cfmakeraw
int cfmakeraw(struct termios *termios_p) {
  termios_p->c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|IGNCR|ICRNL|IXON);
  termios_p->c_oflag &= ~OPOST;
  termios_p->c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
  termios_p->c_cflag &= ~(CSIZE|PARENB);
  termios_p->c_cflag |= CS8;
  return 1;
}
#endif

#define NONBLOCK_OPEN 1
void tty_open(struct port_args_t* args) {
#ifndef OS2
  int flags = O_RDWR | O_NOCTTY;
  struct termios newtio;
#else //OS2
  LINECONTROL lc;
#endif
  char port[1024];

  //finding real device name
  memset(port,0,sizeof(port));
  strncpy(port,args->device,sizeof(port));
  //opening the device
  if (strcmp(args->device,port) == 0) {
    myprintf(0,"Accessing device %s...",port);
  } else {
    myprintf(0,"Accessing device %s -> %s...",args->device,port);    
  }
#ifndef OS2  
  /* FHS (http://www.pathname.com/fhs/pub/fhs-2.3.html#VARLOCKLOCKFILES)
   * specifies lock files for devices but the Serial HowTo (13.3)
   * also mentions the problem when it comes to DevFS (Linux).
   * Since this makes it an unreliable test, forget about it.
   */
#if NONBLOCK_OPEN
  flags |= O_NONBLOCK;
#endif
  if ((tty_portfd=open(port,flags)) == -1) {
    myprintf(0,"Cannot open %s: %s\n",port,strerror(errno));
    exit(EXIT_FAILURE);
  }
#if NONBLOCK_OPEN
  if (fcntl(tty_portfd,F_SETFL,flags&(~O_NONBLOCK)) == -1) {
    fprintf(stderr,"Error in setting port attributes: %s\n",strerror(errno));
    exit(EXIT_FAILURE);
  }
#endif //NONBLOCK_OPEN
  if (!args->ignorebits) {
    //getting port parameters
    if (tcgetattr(tty_portfd,&newtio) < 0){
      fprintf(stderr,"Error in getting device attributes: %s\n",strerror(errno));
      exit(EXIT_FAILURE);
    }
  } else {
    memset(&newtio,0,sizeof(newtio));
  }
  cfsetispeed(&newtio, tty_speed(args->baud)); //input at baudrate
  cfsetospeed(&newtio, tty_speed(args->baud)); //ouput at baudrate
  cfmakeraw(&newtio); //make raw, 8N1
  /* CLOCAL: ignore modem control lines
   * CREAD: enable receiver
   * ~CSTOPB: do not send two stop bits (but one)
   */
  newtio.c_cflag |= (CLOCAL | CREAD);
  newtio.c_cflag &= ~CSTOPB;
  if (args->timeout == 0) {
    newtio.c_cc[VMIN]=1; //wait for at least one character (never times out)
  } else {
    newtio.c_cc[VMIN]=0; //return after timeout, even with nothing
  }
  newtio.c_cc[VTIME]=args->timeout; //try reading for this amount of time (in deciseconds)
  tcflush(tty_portfd, TCIOFLUSH); //clear serial tty
  if(tcsetattr(tty_portfd,TCSANOW,&newtio) < 0){ //set now
    fprintf(stderr,"Error in setting device attributes: %s\n",strerror(errno));
    exit(EXIT_FAILURE);
  }
#else //OS2
  long action,size;
  int bps,rc,timeout;
  DCBINFO devinfo;


  rc = DosOpen(port,&tty_portfd, &action, 0, FILE_NORMAL, FILE_OPEN,
	       OPEN_ACCESS_READWRITE|OPEN_SHARE_DENYNONE,
	       (PEAOP2)NULL);
  
  if (rc) {
    myprintf(0,"Cannot open %s: rc=%d\n",port,rc);
    exit(EXIT_FAILURE);
  }
  // setting baud rate
  bps = tty_speed(args->baud);
  rc = DosDevIOCtl(tty_portfd, IOCTL_ASYNC, ASYNC_SETBAUDRATE,
		   &bps, sizeof(ULONG), &size, NULL, 0, NULL);
  if (rc) {
    fprintf(stderr,"Error in setting device speed:rc= %d\n",rc);
    exit(EXIT_FAILURE);
  }
  // set 8,N,1
  lc.bDataBits = 8;
  lc.bParity = 0;
  lc.bStopBits = 0;
  lc.fTransBreak = 0;
  rc = DosDevIOCtl(tty_portfd,
		   IOCTL_ASYNC, ASYNC_SETLINECTRL,
		   &lc, sizeof(LINECONTROL), &size,
		   NULL, 0, NULL);
  if (rc) {
    fprintf(stderr,"Error in setting device attributes:rc= %d\n",rc);
    exit(EXIT_FAILURE);
  }

   size=sizeof(DCBINFO);
   rc=DosDevIOCtl(tty_portfd, /* file decsriptor */
		 IOCTL_ASYNC, /*asyncronous change */
		 ASYNC_GETDCBINFO, /* get device control block info */
		 NULL, /*  */
		 0, /* length of the previous parameter */
		 NULL, /* max length of data ret */
		 &devinfo, /* data to be recieved */
		 size, /* length of data */
		 (unsigned long *)&size); /* length of data returned */
   
  if (args->timeout == 0) { // infinity waiting
	timeout=65535;
  } else {
	timeout=args->timeout*10;
	if (timeout>65535)
	    timeout=65535;
  } 	  
  devinfo.fbTimeout = 2;
  devinfo.fbFlowReplace = 0; // diable rts/cts control
  devinfo.fbCtlHndShake = 1; // Truning on DTR.
  devinfo.usReadTimeout=timeout;
  devinfo.usWriteTimeout=timeout;

   

   size=sizeof(DCBINFO);
   rc=DosDevIOCtl(tty_portfd, 
		 IOCTL_ASYNC, 
		 ASYNC_SETDCBINFO,
		 &devinfo, /* parameters to set  */
		 size,
		 (unsigned long *)&size, 
		 NULL, 
		 0,
		 NULL); 
 // sleep(1); /* i dont know why but without this delay it is unusable :( */
#endif //OS2
  myprintf(0,"%s","done\n");
  /* The following seems to be needed for C45
   * which seems to react too slow
   */
  if (args->startdelay) {
    myprintf(0,"Waiting for %d seconds as requested...\n",args->startdelay);
    sleep(args->startdelay);
  }
}

int tty_write (const char* data, size_t count) {
#ifndef OS2
  ssize_t status = 0;
  ssize_t instatus;

  do {
    instatus = write(tty_portfd,&data[status],count-status);
    if (instatus != -1) {
      status += instatus;
    } else {
      if (errno == EAGAIN) usleep(1);
      else return 0;
    }
  } while (count-status > 0);
#else
  int rc;
  ULONG status = 0;
  ULONG instatus;

  do { 
    rc = DosWrite(tty_portfd,&data[status],count-status,&instatus);
    if(!rc) status += instatus;
    else {
	 fprintf(stderr,"Error in DosWrite: rc= %d\n",rc);
     return 0;
    }
  } while (count-status > 0);
#endif
  return 1;
}


/* This function returns a char* that must be freed.
 * If it returns NULL, there was an error. Look at errno for the reason.
 * If the strlen(char*)==0, reading from the device timed out. Retry or abort.
 * (the timeout might only happen when the device was not opened with O_NONBLOCK,
 *  the timeout time refers to the value that was set as c_cc[VTIME])
 */
char* tty_read_line (int (*stop_condition)(const char*,const size_t))
{	
  char* retval;
  int retvalsize = 0;
#ifndef OS2
  int status;
#else
  ULONG status;
  int rc;
#endif
  char buffer; //buffer the read character for checks
  int counter = 0; //count the read-and-used characters
  int repeat = 0; //tell the inner loop to loop

  retval = NULL;
  do {
#ifndef OS2	  
    do {
      status = read(tty_portfd,&buffer,1);
      switch (status) {
      case -1:
	if (errno == EAGAIN) {
	  usleep(1);
	  repeat = 1;
	} else {
	  mem_realloc(retval,0);
	  return NULL;
	}
      case 0:
	mem_realloc(retval,0);
	return str_dup("");
      default:
	repeat = 0;
      }
    } while (repeat);
#else //OS2
    do {
      rc = DosRead(tty_portfd,&buffer,1,&status); /* expect a single byte */
      if (rc) {
	fprintf(stderr, "Serial RX: timeout\n");
	mem_realloc(retval,0);
	return NULL;
      }
      usleep(1);
      if (status==0) {
	mem_realloc(retval,0);
	return str_dup("");
      }
    } while (repeat);
#endif //OS2

    // allocate space on stack
    if (counter >= retvalsize) {
      retvalsize += BUFSIZ;
      retval = mem_realloc(retval,retvalsize+1);
      memset(retval+counter,0,retvalsize-counter+1);
    }

    /* fix the '@'=0x00 (GSM character set) problem here :-(
     * we simply set the MSB to 1, so when processing:
     * only look at the last 7 bits (char&0x7F)
     */
    retval[counter++] = (buffer == 0) ? 128 : buffer;
  } while (!stop_condition(retval,counter));
  /* There are two types of possible answers:
   * "> " (2 characters) for data input requests
   * "....<CR><LF>" for all other things (even empty)
   */
  return mem_realloc(retval,strlen(retval)+1);
}

void tty_close(){
#ifndef OS2
  close(tty_portfd);
#else
  DosClose(tty_portfd); 
#endif
}
