/*
 * Smart Gnome Control - database interface file
 * Copyright (C) 2004 by Mark J. Fine <fineware@users.sourceforge.net>
 *
 * This program provides a graphical user interface to multiple
 * communications receivers using the Hamlib library.
 *
 * $Id: database.c,v 1.14 2005/01/01 00:33:17 fineware Exp $
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 *      Free Software Foundation, Inc.
 *      59 Temple Place - Suite 330
 *      Boston, MA  02111-1307
 *      USA
 *
 */

#include "support.h"
#include "extra.h"
#include "database.h"
#include "logfile.h"

unsigned int convert_bytes_to_int(unsigned char* b);
long convert_bytes_to_long(unsigned char* b);
float convert_bytes_to_float(unsigned char* b);
//void print_float_chars(float f);

gint	q_mode;

void
get_database_file()
{
	file_selector = gtk_file_selection_new("Select Database File");
	gtk_file_selection_set_filename(GTK_FILE_SELECTION(file_selector),basedbpath);
	gtk_file_selection_complete(GTK_FILE_SELECTION(file_selector),"SWBCGIDE.*");
	gtk_window_set_position(GTK_WINDOW(file_selector), GTK_WIN_POS_CENTER);
	gtk_window_set_modal(GTK_WINDOW(file_selector), TRUE);
	gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(file_selector)->ok_button),
						"clicked", GTK_SIGNAL_FUNC(store_filename), NULL);
	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(file_selector)->ok_button),
						"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
						(gpointer) file_selector);
	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(file_selector)->cancel_button),
						"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
						(gpointer) file_selector);
	gtk_widget_show(file_selector);
}


void
store_filename						   (GtkFileSelection *selector,
                                        gpointer         user_data)
{
	gchar* temp;
	
	temp = gtk_file_selection_get_filename(GTK_FILE_SELECTION(file_selector));
	db_name = g_strdup(temp);
	open_database();
}


unsigned int
convert_bytes_to_int(unsigned char* b)
{
	ShortsInt		si;

	si.s[0] = b[0];
	si.s[1] = b[1];
	return si.i;
}


long
convert_bytes_to_long(unsigned char* b)
{
	ShortsFloat		sf;

	sf.c[0] = b[0];
	sf.c[1] = b[1];
	sf.c[2] = b[2];
	sf.c[3] = b[3];
	return sf.l;
}


float
convert_bytes_to_float(unsigned char* b)
{
	ShortsFloat		sf;

	sf.c[0] = b[1];
	sf.c[1] = b[4];
	if ((b[0] & 0x01) == 0x01)
		sf.c[2] = (b[5] | 0x80);
	else
		sf.c[2] = b[5];
	sf.c[3] = ((b[0] >> 1) - 0x01);
	return sf.f;
}


gchar*
days_to_str                            (char in_days)
{
	int		i;
	char	days[7] = "       ";
	char	sDow[7] = "smtwhfa";
	static gchar*	result;
	
	if (result)
		g_free(result);
	if (in_days == 0x00)
		result = g_strdup(" daily ");
	else {
		for (i = 0; i < 7; i++) {
    		if ((in_days & (1 << i)) != 0x00)
      			days[i] = sDow[i];
		}
	  	result = g_strndup(days,7);
	}
	return result;
}


gchar*
time_rec_to_str                        (db_record_t *out_rec_p)
{
	static gchar*	result;
	
	if (result)
		g_free(result);
	result = g_strdup_printf("%c%c%c%c",
							out_rec_p->pOf1[0],
							out_rec_p->pOf1[1],
							out_rec_p->pOf2[0],
							out_rec_p->pOf2[1]);
	return result;
}


gchar*
target_rec_to_str                      (db_record_t *out_rec_p)
{
	static gchar*	result;
	
	if (result)
		g_free(result);
	result = g_strdup_printf("%c%c",
							out_rec_p->pTar[0],
							out_rec_p->pTar[1]);
	return result;
}


gchar*
get_station                            (int in_stat)
{
	if (in_stat > 0)
		return g_strstrip((gchar*)g_list_nth_data(statDB,in_stat));
	else
		return "";
}


gchar*
strip_station                          (gchar* in_stat)
{
	char*	p_pos;
	guint	len;
	static gchar*	out_stat;
	
	if (out_stat)
		g_free(out_stat);
	
	if (in_stat) {
		if (db_ver == 3)
			p_pos = strrchr(in_stat,'-');
		else
			p_pos = NULL;
		if (p_pos) {
			len = p_pos - in_stat;
			out_stat = g_strndup(in_stat,len);
		}
		else
			out_stat = g_strdup(in_stat);
	}
	else
		out_stat = NULL;
	
	return out_stat;
}


gchar*
get_country                            (int in_coun)
{
	if (in_coun > 0)
		return g_strstrip((gchar*)g_list_nth_data(counDB,in_coun));
	else
		return "";
}


gchar*
get_language                           (int in_lang)
{
	if (in_lang > 0)
		return g_strstrip((gchar*)g_list_nth_data(langDB,in_lang));
	else
		return "";
}


/*void
print_float_chars(float f)
{
	ShortsFloat		sf;
	sf.f = f;
	rig_debug(RIG_DEBUG_ERR," %2x %2x %2x %2x \n", sf.c[0],sf.c[1],sf.c[2],sf.c[3]);
}*/


void
open_database()
{
	GtkWidget*		gen_item;
	int				i, rec_size, add_on, len, db_size;
	FILE			*fp;
	char			in_rec[26];
	db_record_t		out_rec;
	db_record_t		*out_rec_p;
	gchar*			db_title_str;
	gchar*			db_path_name = NULL;
	gchar*			db_extension = NULL;
	gchar*			db_stat_name;
	gchar*			db_coun_name;
	gchar*			db_lang_name;
	gchar*			ptr;
	gchar*			temp;
	struct stat		st;

	num_freqs = 0;
	g_list_free(freqDB);
	freqDB = NULL;
	num_stats = 0;
	g_list_free(statDB);
	statDB = NULL;
	num_couns = 0;
	g_list_free(counDB);
	counDB = NULL;
	num_langs = 0;
	g_list_free(langDB);
	langDB = NULL;
		
	if (db_ext_name)
		g_free(db_ext_name);
	
	ptr = strrchr(db_name,'/');
	if (ptr) {
		len = ptr - (gchar*)db_name + 1;
		db_path_name = g_strndup(db_name,len);
	}
	ptr = strrchr(db_name,'.');
	if (ptr)
		db_extension = g_strdup(ptr);
	
	if (db_extension) {
		db_ext_name = g_strdup(db_extension+1);
		g_strup(db_ext_name);
		if (strcmp(db_ext_name,"DAT") == 0) {
			g_free(db_ext_name);
			db_ext_name = g_strdup("SRC");
		}
	}
	
	set_app_title();
	db_title_str = g_strdup_printf("Tuning Database - [%s]",db_ext_name);
	gtk_window_set_title(GTK_WINDOW(tun1),db_title_str);
	g_free(db_title_str);
	
	db_stat_name = g_strdup_printf("%s%s%s",db_path_name,"STATIONS",db_extension);
	db_coun_name = g_strdup_printf("%s%s%s",db_path_name,"COUNTRYS",db_extension);
	db_lang_name = g_strdup_printf("%s%s%s",db_path_name,"LANGUAGE",db_extension);
	
	db_ver = 2;
	db_size = 0;
	rec_size = 16;
	add_on = 0;
	i = 0;
	
	fp = fopen(db_name,"rb");
	if (fp) {
		fread(&in_rec,16,1,fp);
		fclose(fp);
		if ((in_rec[15] != 'T') && (in_rec[15] != 'F')) {
			db_ver = 3;
			rec_size = 26;
			add_on = 10;
		}
	}
	else {
		temp = g_strdup_printf("%s not found",db_name);
		show_message("Open Database",temp,"warning");
		g_free(temp);
	}
	
	gen_item = lookup_widget(app1,"appbar1");
	stat(db_name,&st);
	if (st.st_size > 0)
		db_size = st.st_size / rec_size;
	gnome_appbar_set_status(GNOME_APPBAR(gen_item),"Reading frequencies...");
	update_immediate();
	fp = fopen(db_name,"rb");
	if (fp) {
		for (i = 0; i < db_size; i++) {
			fread(&in_rec,rec_size,1,fp);
			out_rec.pOn = convert_bytes_to_int(&in_rec[0]);
			out_rec.pOff = convert_bytes_to_int(&in_rec[2]);
			out_rec.pDays = in_rec[4];
			if (db_ver == 3) {
				out_rec.pSInx = convert_bytes_to_long(&in_rec[5]);
				out_rec.pCInx = convert_bytes_to_long(&in_rec[9]);
				out_rec.pLInx = convert_bytes_to_long(&in_rec[13]);
			}
			else {
				out_rec.pSInx = convert_bytes_to_int(&in_rec[5]);
				out_rec.pCInx = 0;
				out_rec.pLInx = 0;
			}
			out_rec.pFreq = convert_bytes_to_float(&in_rec[7+add_on]);
			out_rec.pTar[0] = in_rec[13+add_on];
			out_rec.pTar[1] = in_rec[14+add_on];
			out_rec.pHear = in_rec[15+add_on];
			out_rec_p = &out_rec;
			ptr = g_memdup(out_rec_p, sizeof(out_rec));
			freqDB = g_list_append(freqDB,ptr);
			if ((i % 100) == 0) {
				gnome_appbar_set_progress(GNOME_APPBAR(gen_item),((gfloat)i+1)/db_size);
				update_immediate();
			}
		}
		fclose(fp);
		num_freqs = db_size;
	}
	else {
		temp = g_strdup_printf("%s not found",db_name);
		show_message("Open Database",temp,"warning");
		g_free(temp);
	}

	stat(db_stat_name,&st);
	if (st.st_size > 0)
		db_size = st.st_size / 25;
	gnome_appbar_set_status(GNOME_APPBAR(gen_item),"Reading stations...");
	update_immediate();
	ptr = g_strdup(" ");
	statDB = g_list_append(statDB,ptr);
	fp = fopen(db_stat_name,"rb");
	if (fp) {
		for (i = 0; i < db_size; i++) {
			fread(&in_rec,25,1,fp);
			ptr = g_strndup(in_rec,25);
			statDB = g_list_append(statDB,ptr);
			gnome_appbar_set_progress(GNOME_APPBAR(gen_item),((gfloat)i+1)/db_size);
			update_immediate();
		}
		fclose(fp);
		num_stats = db_size;
	}
	else {
		temp = g_strdup_printf("%s not found",db_stat_name);
		show_message("Open Database",temp,"warning");
		g_free(temp);
	}

	num_couns = 1;	
    ptr = g_strdup(" ");
	counDB = g_list_append(counDB,ptr);
	
	num_langs = 1;	
	ptr = g_strdup(" ");
	langDB = g_list_append(langDB,ptr);
	
	if (db_ver == 3) {
		stat(db_coun_name,&st);
		if (st.st_size > 0)
			db_size = st.st_size / 25;
		gnome_appbar_set_status(GNOME_APPBAR(gen_item),"Reading countries...");
		update_immediate();
		fp = fopen(db_coun_name,"rb");
		if (fp) {
			for (i = 0; i < db_size; i++) {
				fread(&in_rec,25,1,fp);
				ptr = g_strndup(in_rec,25);
				counDB = g_list_append(counDB,ptr);
				gnome_appbar_set_progress(GNOME_APPBAR(gen_item),((gfloat)i+1)/db_size);
				update_immediate();
			}
			fclose(fp);
			num_couns = db_size;
		}
		else {
			temp = g_strdup_printf("%s not found",db_coun_name);
			show_message("Open Database",temp,"warning");
			g_free(temp);
		}

		stat(db_lang_name,&st);
		if (st.st_size > 0)
			db_size = st.st_size / 25;
		gnome_appbar_set_status(GNOME_APPBAR(gen_item),"Reading languages...");
		update_immediate();
		fp = fopen(db_lang_name,"rb");
		if (fp) {
			for (i = 0; i < db_size; i++) {
				fread(&in_rec,25,1,fp);
				ptr = g_strndup(in_rec,25);
				langDB = g_list_append(langDB,ptr);
				gnome_appbar_set_progress(GNOME_APPBAR(gen_item),((gfloat)i+1)/db_size);
				update_immediate();
			}
			fclose(fp);
			num_langs = db_size;
		}
		else {
			temp = g_strdup_printf("%s not found",db_lang_name);
			show_message("Open Database",temp,"warning");
			g_free(temp);
		}
	}
	
	gnome_appbar_set_status(GNOME_APPBAR(gen_item),"");
	gnome_appbar_set_progress(GNOME_APPBAR(gen_item),0);
	update_immediate();
	
	g_free(db_lang_name);
	g_free(db_stat_name);
	g_free(db_extension);
	g_free(db_path_name);

	set_enabled(tun1,"LangBox",(db_ver == 3));
	set_enabled(tun1,"LangBtn",(db_ver == 3));
	set_enabled(tun1,"language_filter1",(db_ver == 3));
	set_enabled(tun1,"FilterBtn",(db_ver == 3));
	if (db_ver == 3) {
		gen_item = lookup_widget(tun1,"LangBox");
		if (gen_item)
			gtk_combo_set_popdown_strings(GTK_COMBO(gen_item),langDB);
		if (db_filt)
			set_text(tun1,"LangBox",db_filt);
	}
	gconf_client_set_string(confclient,"/apps/sgcontrol/last_db",db_name,NULL);
	gen_item = lookup_widget(query1,"StationEntry");
	if (gen_item)
		gtk_combo_set_popdown_strings(GTK_COMBO(gen_item),statDB);
	index_db();
	for (i = 0; i < 5; i++)
		db_changed[i] = TRUE;
	update_freq_db_display();
	if (get_visible(tun1,"TuningDlg"))
		set_time_filter();
}


void
index_db()
{
	GtkWidget*	gen_item;
	int			on1, i, j;
	db_record_t	*out_rec_p;

	gen_item = lookup_widget(app1,"appbar1");
	gnome_appbar_set_status(GNOME_APPBAR(gen_item),"Indexing...");
	update_immediate();
	j = 0;
	for (i = 0; i < 26; i++)
		db_index[i] = 0;
	for (i = 0; i < num_freqs; i++) {
		out_rec_p = g_list_nth_data(freqDB,i);
		if (out_rec_p->pOn > 2400)
			on1 = 25;
		else
			on1 = out_rec_p->pOn / 100;
		if (j == on1) {
			db_index[j++] = i;
		}
		if ((i % 100) == 0) {
			gnome_appbar_set_progress(GNOME_APPBAR(gen_item),((gfloat)i+1)/num_freqs);
			update_immediate();
		}
	}
	if (db_index[25] == 0)
		db_index[25] = num_freqs;

	gnome_appbar_set_status(GNOME_APPBAR(gen_item),"");
	gnome_appbar_set_progress(GNOME_APPBAR(gen_item),0);
	update_immediate();
}


void
get_freq_records                       (GtkCList* clist,
                                        gdouble   inFreq)
{
	int			i, j, inx, color_toggle;
	db_record_t	*out_rec_p;
	long		gmt_long;
	gchar 		*buf, **text;
	guint16		last_on, last_off;
	guint32		last_stat;

	gmt_long = get_gmt_time();
	inx = gmt_long / 100;

	j = 0;
	last_on = 9999;
	last_off = 9999;
	last_stat = 0;
	color_toggle = -1;
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	/* get normal timed frequencies */
	for (i = db_index[inx]; i < db_index[inx+1]; i++) {
		out_rec_p = g_list_nth_data(freqDB,i);
		if ((inFreq >= (out_rec_p->pFreq-0.2)) && (inFreq <= (out_rec_p->pFreq+0.2)) &&
			((gmt_long >= out_rec_p->pOn) && (gmt_long < out_rec_p->pOff))) {
			buf = g_strdup_printf("%04d-%04d;%s;%s;%s;%s;%s",
									out_rec_p->pOn, out_rec_p->pOff,
									days_to_str(out_rec_p->pDays),
									strip_station(get_station(out_rec_p->pSInx)),
									get_country(out_rec_p->pCInx),
									get_language(out_rec_p->pLInx),
									target_rec_to_str(out_rec_p));
			text = g_strsplit(buf, ";", 6);
			gtk_clist_append(clist,text);
			if (strcmp(target_rec_to_str(out_rec_p),"in") == 0)
				set_clist_row_color(app1,"clist1",inact_color_str,j);
			else	
				set_clist_row_color(app1,"clist1",reg_color_str,j);
			if ((out_rec_p->pOn != last_on) || 
				(out_rec_p->pOff != last_off) || 
				(out_rec_p->pSInx != last_stat)) {
					if (color_toggle < 0)
						color_toggle +=2;
					else
						color_toggle -=2;
			}
			if (color_toggle > 0)
				set_clist_row_background(app1,"clist1",bg_color_str,j);
			else
				set_clist_row_background(app1,"clist1",bg_color_str2,j);
			gtk_clist_set_row_data(clist, j++, GUINT_TO_POINTER(i));
			last_on = out_rec_p->pOn;
			last_off = out_rec_p->pOff;
			last_stat = out_rec_p->pSInx;
			g_free(buf);
			g_strfreev(text);
		}
	}
	/* get 24hr and Utility frequencies */
	for (i = db_index[24]; i < num_freqs; i++) {
		out_rec_p = g_list_nth_data(freqDB,i);
		if ((inFreq >= (out_rec_p->pFreq-0.2)) && (inFreq <= (out_rec_p->pFreq+0.2))) {
			if (out_rec_p->pOn == 2400)
				buf = g_strdup_printf("24hr;%s;%s;%s;%s;%s",
										days_to_str(out_rec_p->pDays),
										strip_station(get_station(out_rec_p->pSInx)),
										get_country(out_rec_p->pCInx),
										get_language(out_rec_p->pLInx),
										target_rec_to_str(out_rec_p));
			else
				buf = g_strdup_printf("Util-%s;%s;%s;%s;%s;%s",
										time_rec_to_str(out_rec_p),
										days_to_str(out_rec_p->pDays),
										strip_station(get_station(out_rec_p->pSInx)),
										get_country(out_rec_p->pCInx),
										get_language(out_rec_p->pLInx),
										target_rec_to_str(out_rec_p));
			text = g_strsplit(buf, ";", 6);
			gtk_clist_append(clist,text);
			if (strcmp(target_rec_to_str(out_rec_p),"in") == 0)
				set_clist_row_color(app1,"clist1",inact_color_str,j);
			else	
				set_clist_row_color(app1,"clist1",reg_color_str,j);
			if ((out_rec_p->pOn != last_on) || 
				(out_rec_p->pOff != last_off) || 
				(out_rec_p->pSInx != last_stat)) {
					if (color_toggle < 0)
						color_toggle +=2;
					else
						color_toggle -=2;
			}
			if (color_toggle > 0)
				set_clist_row_background(app1,"clist1",bg_color_str,j);
			else
				set_clist_row_background(app1,"clist1",bg_color_str2,j);
			gtk_clist_set_row_data(clist, j++, GUINT_TO_POINTER(i));			
			last_on = out_rec_p->pOn;
			last_off = out_rec_p->pOff;
			last_stat = out_rec_p->pSInx;
			g_free(buf);
			g_strfreev(text);
		}
	}
	gtk_clist_thaw(clist);
}


void
get_sked_records                       (gchar* in_stat,
                                        long   in_time)
{
	int			i, j, t_start, t_stop, all_stats, all_time, inx, color_toggle;
	db_record_t	*out_rec_p;
	gchar 		*buf, **text, *hdr_label, *in_lang;
	GtkCList*	clist;
	gboolean	is_lang;
	guint16		last_on, last_off;
	guint32		last_stat;

	all_stats = (strcmp(in_stat,"*")==0);
	all_time = (in_time < 0);
	
	if (all_time) {
		t_start = 0;
		t_stop = db_index[24];
	}
	else {
		inx = in_time / 100;
		t_start = db_index[inx];
		t_stop = db_index[inx+1];
	}

	in_lang = g_strndup(db_filt,3);
	is_lang = (get_down(tun1,"LangBtn") && get_enabled(tun1,"LangBtn") && (strcmp(in_lang," ") != 0));
	buf = NULL;
	
	j = 0;
	last_on = 9999;
	last_off = 9999;
	last_stat = 0;
	color_toggle = -1;
	clist = GTK_CLIST(lookup_widget(tun1,"SkedList"));
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	if (all_stats)
		hdr_label = g_strdup(" Full Frequency Schedule");
	else
		hdr_label = g_strdup_printf(" Frequency Schedule for %s",in_stat);
	set_text(tun1,"SkedLbl",hdr_label);
	g_free(hdr_label);
	/* get normal timed frequencies */
	for (i = t_start; i < t_stop; i++) {
		out_rec_p = g_list_nth_data(freqDB,i);
		if ((all_time) || ((in_time >= out_rec_p->pOn) && (in_time < out_rec_p->pOff))) {
			if ((all_stats) || (strstr(get_station(out_rec_p->pSInx),g_strstrip(in_stat)))) {
				if ((!is_lang) || (strstr(get_language(out_rec_p->pLInx),in_lang))) {
					buf = g_strdup_printf("%04d-%04d;%s;%s;%s;%s;%11.3f;%s",
											out_rec_p->pOn, out_rec_p->pOff,
											days_to_str(out_rec_p->pDays),
											strip_station(get_station(out_rec_p->pSInx)),
											get_country(out_rec_p->pCInx),
											get_language(out_rec_p->pLInx),
											out_rec_p->pFreq,
											target_rec_to_str(out_rec_p));
					text = g_strsplit(buf, ";", 7);
					gtk_clist_append(clist,text);
					if (strcmp(target_rec_to_str(out_rec_p),"in") == 0)
						set_clist_row_color(tun1,"SkedList",inact_color_str,j);
					else	
						set_clist_row_color(tun1,"SkedList",reg_color_str,j);
					if ((out_rec_p->pOn != last_on) || 
						(out_rec_p->pOff != last_off) || 
						(out_rec_p->pSInx != last_stat)) {
							if (color_toggle < 0)
								color_toggle +=2;
							else
								color_toggle -=2;
					}
					if (color_toggle > 0)
						set_clist_row_background(tun1,"SkedList",bg_color_str,j);
					else
						set_clist_row_background(tun1,"SkedList",bg_color_str2,j);
					gtk_clist_set_row_data(clist, j++, GUINT_TO_POINTER(i));
					last_on = out_rec_p->pOn;
					last_off = out_rec_p->pOff;
					last_stat = out_rec_p->pSInx;
					g_free(buf);
					g_strfreev(text);
				}
			}
		}
	}
	/* get 24hr and Utility frequencies */
	for (i = db_index[24]; i < num_freqs; i++) {
		out_rec_p = g_list_nth_data(freqDB,i);
		if ((all_stats) || (strstr(get_station(out_rec_p->pSInx),g_strstrip(in_stat)))) {
			if (out_rec_p->pOn == 2400) {
				if ((!is_lang) || (strstr(get_language(out_rec_p->pLInx),in_lang)))
					buf = g_strdup_printf("24hr;%s;%s;%s;%s;%11.3f;%s",
											days_to_str(out_rec_p->pDays),
											strip_station(get_station(out_rec_p->pSInx)),
											get_country(out_rec_p->pCInx),
											get_language(out_rec_p->pLInx),
											out_rec_p->pFreq,
											target_rec_to_str(out_rec_p));
				else
					buf = NULL;
			}
			else
				buf = g_strdup_printf("Util-%s;%s;%s;%s;%s;%11.3f;%s",
										time_rec_to_str(out_rec_p),
										days_to_str(out_rec_p->pDays),
										strip_station(get_station(out_rec_p->pSInx)),
										get_country(out_rec_p->pCInx),
										get_language(out_rec_p->pLInx),
										out_rec_p->pFreq,
										target_rec_to_str(out_rec_p));
			if (buf) {
				text = g_strsplit(buf, ";", 7);
				gtk_clist_append(clist,text);
				if (strcmp(target_rec_to_str(out_rec_p),"in") == 0)
					set_clist_row_color(tun1,"SkedList",inact_color_str,j);
				else	
					set_clist_row_color(tun1,"SkedList",reg_color_str,j);
				if ((out_rec_p->pOn != last_on) || 
					(out_rec_p->pOff != last_off) || 
					(out_rec_p->pSInx != last_stat)) {
						if (color_toggle < 0)
							color_toggle +=2;
						else
							color_toggle -=2;
				}
				if (color_toggle > 0)
					set_clist_row_background(tun1,"SkedList",bg_color_str,j);
				else
					set_clist_row_background(tun1,"SkedList",bg_color_str2,j);
				gtk_clist_set_row_data(clist, j++, GUINT_TO_POINTER(i));			
				last_on = out_rec_p->pOn;
				last_off = out_rec_p->pOff;
				last_stat = out_rec_p->pSInx;
				g_free(buf);
				g_strfreev(text);
			}
		}
	}
	g_free(in_lang);
	db_changed[0] = FALSE;
	gtk_clist_thaw(clist);
}


gboolean
check_stat_record                      (GtkCList* clist,
                                        int       in_row)
{
	int inx;
	inx = GPOINTER_TO_INT(gtk_clist_get_row_data(clist, in_row));
	if (inx <= -1) {
		gtk_clist_remove(clist,in_row);
		return TRUE;
	}
	else
		return FALSE;
}

void
get_stat_records                       (int       type,
                                        long      in_time)
{
	int			i, j, k, inx, old_on, old_off, old_stat;
	double		old_freq;
	db_record_t	*out_rec_p;
	gchar 		*buf, *hdr_label, *in_lang;
	gboolean	is_lang;
	GtkCList*	list = NULL;
	
	inx = in_time / 100;
	
	old_on = 0;
	old_off = 0;
	old_stat = 0;
	old_freq = 0;
	j = 0;
	k = -1;
	
	in_lang = g_strndup(db_filt,3);
	is_lang = (get_down(tun1,"LangBtn") && get_enabled(tun1,"LangBtn") && (strcmp(in_lang," ") != 0));
	
	switch (type) {
		case 0 : /* get normal timed stations */
			list = GTK_CLIST(lookup_widget(tun1,"BCList"));
			gtk_clist_freeze(list);
			gtk_clist_clear(list);
			hdr_label = g_strdup_printf(" Available Shortwave Stations - [as of %04d]",(int)in_time);
			set_text(tun1,"StationsLbl",hdr_label);
			g_free(hdr_label);
			for (i = db_index[inx]; i < db_index[inx+1]; i++) {
				out_rec_p = g_list_nth_data(freqDB,i);
				if  ((in_time >= out_rec_p->pOn) && (in_time < out_rec_p->pOff)) {
					if ((out_rec_p->pOn != old_on) || (out_rec_p->pOff != old_off)) {
						if (check_stat_record(list,j-1))
							j--;
						buf = g_strdup_printf("=========[%04d-%04d]=========", out_rec_p->pOn, out_rec_p->pOff);
						gtk_clist_append(list,&buf);
						set_clist_row_color(tun1,"BCList",high_color_str,j);
						gtk_clist_set_row_data(list, j++, GINT_TO_POINTER(k--));
						g_free(buf);
						old_on = out_rec_p->pOn;
						old_off = out_rec_p->pOff;
						old_stat = 0;
					}
					if (old_stat != out_rec_p->pSInx) {
						if ((!is_lang) || (strstr(get_language(out_rec_p->pLInx),in_lang))) {
							buf = g_strdup_printf("%s", get_station(out_rec_p->pSInx));
							gtk_clist_append(list,&buf);
							gtk_clist_set_row_data(list, j++, GINT_TO_POINTER(i));
							g_free(buf);
							old_stat = out_rec_p->pSInx;
						}
					}
				}
			}
			for (i = db_index[24]; i < db_index[25]; i++) {
				out_rec_p = g_list_nth_data(freqDB,i);
				if ((out_rec_p->pOn != old_on) || (out_rec_p->pOff != old_off)) {
					buf = g_strdup("===========[24 hr]===========");
					if (check_stat_record(list,j-1))
						j--;
					gtk_clist_append(list,&buf);
					set_clist_row_color(tun1,"BCList",high_color_str,j);
					gtk_clist_set_row_data(list, j++, GINT_TO_POINTER(k--));
					g_free(buf);
					old_on = out_rec_p->pOn;
					old_off = out_rec_p->pOff;
					old_stat = 0;
				}
				if (old_stat != out_rec_p->pSInx) {
					if ((!is_lang) || (strstr(get_language(out_rec_p->pLInx),in_lang))) {
						buf = g_strdup_printf("%s", get_station(out_rec_p->pSInx));
						gtk_clist_append(list,&buf);
						gtk_clist_set_row_data(list, j++, GINT_TO_POINTER(i));
						g_free(buf);
						old_stat = out_rec_p->pSInx;
					}
				}
			}
			db_changed[1] = FALSE;
			gtk_clist_thaw(list);
			break;
		case 1 : /* get utility stations */
			list = GTK_CLIST(lookup_widget(tun1,"UteList"));
			gtk_clist_freeze(list);
			gtk_clist_clear(list);
			set_text(tun1,"StationsLbl"," Available Utility Stations");
			for (i = db_index[25]; i < num_freqs; i++) {
				out_rec_p = g_list_nth_data(freqDB,i);
				if ((strcmp(time_rec_to_str(out_rec_p),"BCB ") != 0) &&
					(strcmp(time_rec_to_str(out_rec_p),"CB  ") != 0)) {
					if ((out_rec_p->pOn != old_on) || (out_rec_p->pOff != old_off)) {
						buf = g_strdup_printf("=========[Util-%s]=========",
										time_rec_to_str(out_rec_p));
						gtk_clist_append(list,&buf);
						set_clist_row_color(tun1,"UteList",high_color_str,j);
						gtk_clist_set_row_data(list, j++, GINT_TO_POINTER(k--));
						g_free(buf);
						old_on = out_rec_p->pOn;
						old_off = out_rec_p->pOff;
						old_stat = 0;
					}
					if (old_stat != out_rec_p->pSInx) {
						buf = g_strdup_printf("%s", get_station(out_rec_p->pSInx));
						gtk_clist_append(list,&buf);
						gtk_clist_set_row_data(list, j++, GINT_TO_POINTER(i));
						g_free(buf);
						old_stat = out_rec_p->pSInx;
					}
				}
			}
			db_changed[2] = FALSE;
			gtk_clist_thaw(list);
			break;
		case 2 : /* get bcb/cb stations */
			list = GTK_CLIST(lookup_widget(tun1,"BCBList"));
			gtk_clist_freeze(list);
			gtk_clist_clear(list);
			set_text(tun1,"StationsLbl"," Available BCB/CB Stations");
			for (i = db_index[25]; i < num_freqs; i++) {
				out_rec_p = g_list_nth_data(freqDB,i);
				if ((strcmp(time_rec_to_str(out_rec_p),"BCB ") == 0) &&
					(out_rec_p->pFreq < 1800)) {
					if (out_rec_p->pFreq != old_freq) {
						buf = g_strdup_printf("=========[%4.0f]=========", out_rec_p->pFreq);
						gtk_clist_append(list,&buf);
						set_clist_row_color(tun1,"BCBList",high_color_str,j);
						gtk_clist_set_row_data(list, j++, GINT_TO_POINTER(k--));
						g_free(buf);
						old_freq = out_rec_p->pFreq;
					}
					buf = g_strdup_printf("%s", get_station(out_rec_p->pSInx));
					gtk_clist_append(list,&buf);
					gtk_clist_set_row_data(list, j++, GINT_TO_POINTER(i));
					g_free(buf);
				}
			}
			buf = g_strdup("=========[Feeds]=========");
			gtk_clist_append(list,&buf);
			set_clist_row_color(tun1,"BCBList",high_color_str,j);
			gtk_clist_set_row_data(list, j++, GINT_TO_POINTER(k--));
			g_free(buf);
			old_stat = 0;
			for (i = db_index[25]; i < num_freqs; i++) {
				out_rec_p = g_list_nth_data(freqDB,i);
				if ((strcmp(time_rec_to_str(out_rec_p),"BCB ") == 0) &&
					(out_rec_p->pFreq > 1800)) {
					if (old_stat != out_rec_p->pSInx) {
						buf = g_strdup_printf("%s", get_station(out_rec_p->pSInx));
						gtk_clist_append(list,&buf);
						gtk_clist_set_row_data(list, j++, GINT_TO_POINTER(i));
						g_free(buf);
						old_stat = out_rec_p->pSInx;
					}
				}
			}
			buf = g_strdup("=========[ CB ]==========");
			gtk_clist_append(list,&buf);
			set_clist_row_color(tun1,"BCBList",high_color_str,j);
			gtk_clist_set_row_data(list, j++, GINT_TO_POINTER(k--));
			g_free(buf);
			for (i = db_index[25]; i < num_freqs; i++) {
				out_rec_p = g_list_nth_data(freqDB,i);
				if (strcmp(time_rec_to_str(out_rec_p),"CB  ") == 0) {
					buf = g_strdup_printf("%s", get_station(out_rec_p->pSInx));
					gtk_clist_append(list,&buf);
					gtk_clist_set_row_data(list, j++, GINT_TO_POINTER(i));
					g_free(buf);
				}
			}
			db_changed[3] = FALSE;
			gtk_clist_thaw(list);
			break;
	}
	g_free(in_lang);
}


void
get_who_records                        (gdouble in_freq,
                                        long    in_time)
{
	int			i, j, t_start, t_stop, all_time, inx, color_toggle;
	db_record_t	*out_rec_p;
	gchar 		*buf, **text, *hdr_label, *in_lang;
	GtkCList*	clist;
	gboolean	is_lang;
	guint32		last_stat;

	all_time = (in_time < 0);
	
	if (all_time) {
		t_start = 0;
		t_stop = db_index[24];
	}
	else {
		inx = in_time / 100;
		t_start = db_index[inx];
		t_stop = db_index[inx+1];
	}
	
	in_lang = g_strndup(db_filt,3);
	is_lang = (get_down(tun1,"LangBtn") && get_enabled(tun1,"LangBtn") && (strcmp(in_lang," ") != 0));
	buf = NULL;

	j = 0;
	last_stat = 0;
	color_toggle = -1;
	clist = GTK_CLIST(lookup_widget(tun1,"WhoList"));
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	hdr_label = g_strdup_printf(" Other Stations on %11.3f",in_freq);
	set_text(tun1,"WhoLbl",hdr_label);
	g_free(hdr_label);
	/* get normal timed frequencies */
	for (i = t_start; i < t_stop; i++) {
		out_rec_p = g_list_nth_data(freqDB,i);
		if ((all_time) || ((in_time >= out_rec_p->pOn) && (in_time < out_rec_p->pOff))) {
			if ((in_freq >= (out_rec_p->pFreq-0.2)) && (in_freq <= (out_rec_p->pFreq+0.2))) {
				if ((!is_lang) || (strstr(get_language(out_rec_p->pLInx),in_lang))) {
					buf = g_strdup_printf("%04d-%04d;%s;%s;%s;%s;%11.3f;%s",
											out_rec_p->pOn, out_rec_p->pOff,
											days_to_str(out_rec_p->pDays),
											strip_station(get_station(out_rec_p->pSInx)),
											get_country(out_rec_p->pCInx),
											get_language(out_rec_p->pLInx),
											out_rec_p->pFreq,
											target_rec_to_str(out_rec_p));
					text = g_strsplit(buf, ";", 7);
					gtk_clist_append(clist,text);
					if (strcmp(target_rec_to_str(out_rec_p),"in") == 0)
						set_clist_row_color(tun1,"WhoList",inact_color_str,j);
					else	
						set_clist_row_color(tun1,"WhoList",reg_color_str,j);			
					if (out_rec_p->pSInx != last_stat) {
							if (color_toggle < 0)
								color_toggle +=2;
							else
								color_toggle -=2;
					}
					if (color_toggle > 0)
						set_clist_row_background(tun1,"WhoList",bg_color_str,j);
					else
						set_clist_row_background(tun1,"WhoList",bg_color_str2,j);
					gtk_clist_set_row_data(clist, j++, GUINT_TO_POINTER(i));			
					last_stat = out_rec_p->pSInx;
					g_free(buf);
					g_strfreev(text);
				}
			}
		}
	}
	/* get 24hr and Utility frequencies */
	for (i = db_index[24]; i < num_freqs; i++) {
		out_rec_p = g_list_nth_data(freqDB,i);
		if ((in_freq >= (out_rec_p->pFreq-0.2)) && (in_freq <= (out_rec_p->pFreq+0.2))) {
			if (out_rec_p->pOn == 2400) {
				if ((!is_lang) || (strstr(get_language(out_rec_p->pLInx),in_lang)))
					buf = g_strdup_printf("24hr;%s;%s;%s;%s;%11.3f;%s",
											days_to_str(out_rec_p->pDays),
											strip_station(get_station(out_rec_p->pSInx)),
											get_country(out_rec_p->pCInx),
											get_language(out_rec_p->pLInx),
											out_rec_p->pFreq,
											target_rec_to_str(out_rec_p));
				else
					buf = NULL;
			}	
			else
				buf = g_strdup_printf("Util-%s;%s;%s;%s;%s;%11.3f;%s", 
										time_rec_to_str(out_rec_p),
										days_to_str(out_rec_p->pDays),
										strip_station(get_station(out_rec_p->pSInx)),
										get_country(out_rec_p->pCInx),
										get_language(out_rec_p->pLInx),
										out_rec_p->pFreq,
										target_rec_to_str(out_rec_p));
			if (buf) {
				text = g_strsplit(buf, ";", 7);
				gtk_clist_append(clist,text);
				if (strcmp(target_rec_to_str(out_rec_p),"in") == 0)
					set_clist_row_color(tun1,"WhoList",inact_color_str,j);
				else	
					set_clist_row_color(tun1,"WhoList",reg_color_str,j);
				if (out_rec_p->pSInx != last_stat) {
						if (color_toggle < 0)
							color_toggle +=2;
						else
							color_toggle -=2;
				}
				if (color_toggle > 0)
					set_clist_row_background(tun1,"WhoList",bg_color_str,j);
				else
					set_clist_row_background(tun1,"WhoList",bg_color_str2,j);
				gtk_clist_set_row_data(clist, j++, GUINT_TO_POINTER(i));			
				last_stat = out_rec_p->pSInx;
				g_free(buf);
				g_strfreev(text);
			}
		}
	}
	g_free(in_lang);
	db_changed[4] = FALSE;
	gtk_clist_thaw(clist);
}


void
get_freq_data                          (gchar* in_stat,
                                        long   in_time)
{
	int			i, j, inx, color_toggle;
	db_record_t	*out_rec_p;
	gchar 		*stat1, *buf, **text, *in_lang;
	GtkCList*	clist;
	gboolean	is_lang;
	guint16		last_on, last_off;

	inx = in_time / 100;

	in_lang = g_strndup(db_filt,3);
	is_lang = (get_down(tun1,"LangBtn") && get_enabled(tun1,"LangBtn") && (strcmp(in_lang," ") != 0));
	buf = NULL;

	if (get_down(tun1,"FilterBtn"))
		stat1 = g_strdup(in_stat);
	else
		stat1 = g_strdup(strip_station(in_stat));
	
	j = 0;
	last_on = 9999;
	last_off = 9999;
	color_toggle = -1;
	clist = GTK_CLIST(lookup_widget(tun1,"FreqList"));
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	set_text(tun1,"FreqLbl",stat1);
	/* get normal timed frequencies */
	for (i = db_index[inx]; i < db_index[inx+1]; i++) {
		out_rec_p = g_list_nth_data(freqDB,i);
		if ((in_time >= out_rec_p->pOn) && (in_time < out_rec_p->pOff)) {
			if (strstr(get_station(out_rec_p->pSInx),g_strstrip(stat1))) {
				if ((!is_lang) || (strstr(get_language(out_rec_p->pLInx),in_lang))) {
					buf = g_strdup_printf("%04d-%04d;%s;%s;%11.3f;%s",
											out_rec_p->pOn, out_rec_p->pOff,
											days_to_str(out_rec_p->pDays),
											get_language(out_rec_p->pLInx),
											out_rec_p->pFreq,
											target_rec_to_str(out_rec_p));
					text = g_strsplit(buf, ";", 5);
					gtk_clist_append(clist,text);
					if (strcmp(target_rec_to_str(out_rec_p),"in") == 0)
						set_clist_row_color(tun1,"FreqList",inact_color_str,j);
					else	
						set_clist_row_color(tun1,"FreqList",reg_color_str,j);
					if ((out_rec_p->pOn != last_on) || 
						(out_rec_p->pOff != last_off)) {
							if (color_toggle < 0)
								color_toggle +=2;
							else
								color_toggle -=2;
					}
					if (color_toggle > 0)
						set_clist_row_background(tun1,"FreqList",bg_color_str,j);
					else
						set_clist_row_background(tun1,"FreqList",bg_color_str2,j);
					gtk_clist_set_row_data(clist, j++, GUINT_TO_POINTER(i));			
					last_on = out_rec_p->pOn;
					last_off = out_rec_p->pOff;
					g_free(buf);
					g_strfreev(text);
				}
			}
		}
	}
	/* get 24hr and Utility frequencies */
	for (i = db_index[24]; i < num_freqs; i++) {
		out_rec_p = g_list_nth_data(freqDB,i);
		if (strstr(get_station(out_rec_p->pSInx),g_strstrip(stat1))) {
			if (out_rec_p->pOn == 2400) {
				if ((!is_lang) || (strstr(get_language(out_rec_p->pLInx),in_lang)))
					buf = g_strdup_printf("24hr;%s;%s;%11.3f;%s",
											days_to_str(out_rec_p->pDays),
											get_language(out_rec_p->pLInx),
											out_rec_p->pFreq,
											target_rec_to_str(out_rec_p));
				else
					buf = NULL;
			}
			else
				buf = g_strdup_printf("Util-%s;%s;%s;%11.3f;%s",
										time_rec_to_str(out_rec_p),
										days_to_str(out_rec_p->pDays),
										get_language(out_rec_p->pLInx),
										out_rec_p->pFreq,
										target_rec_to_str(out_rec_p));
			if (buf) {
				text = g_strsplit(buf, ";", 5);
				gtk_clist_append(clist,text);
				if (strcmp(target_rec_to_str(out_rec_p),"in") == 0)
					set_clist_row_color(tun1,"FreqList",inact_color_str,j);
				else	
					set_clist_row_color(tun1,"FreqList",reg_color_str,j);
				if ((out_rec_p->pOn != last_on) || 
					(out_rec_p->pOff != last_off)) {
						if (color_toggle < 0)
							color_toggle +=2;
						else
							color_toggle -=2;
				}
				if (color_toggle > 0)
					set_clist_row_background(tun1,"FreqList",bg_color_str,j);
				else
					set_clist_row_background(tun1,"FreqList",bg_color_str2,j);
				gtk_clist_set_row_data(clist, j++, GUINT_TO_POINTER(i));			
				last_on = out_rec_p->pOn;
				last_off = out_rec_p->pOff;
				g_free(buf);
				g_strfreev(text);
			}
		}
	}
	g_free(stat1);
	g_free(in_lang);
	gtk_clist_thaw(clist);
}


void
get_stat_list_items_ex                 (GtkWidget* in_widget,
                                        gchar*     list_name,
                                        gint       in_row)
{
	GtkWidget* gen_item = lookup_widget(in_widget,list_name);
	if (gen_item)
		get_stat_list_items(gen_item,in_row);
}


void
get_stat_list_items                    (GtkWidget* in_widget,
                                        gint in_row)
{
	int		rec_num;
	long	in_time;
	gchar*	in_stat;
	db_record_t	*out_rec_p;
	
	rec_num = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST(in_widget), in_row));
	if (rec_num > -1) {
		in_time = get_time_filter();
		out_rec_p = g_list_nth_data(freqDB,rec_num);
		in_stat = g_strdup_printf(" %s",get_station(out_rec_p->pSInx));
		get_freq_data(in_stat,abs(in_time));
	}
}


void
show_curr_freq                         (gchar* in_stat)
{
	gint	in_time;
	
	in_time = abs(get_time_filter());
	get_freq_data(in_stat,in_time);
	show_freq_sidebar();
}


void
tune_from_freq_list_ex                 (GtkWidget* in_widget,
                                        gchar*     list_name,
                                        gint       in_row)
{
	GtkWidget* gen_item = lookup_widget(in_widget,list_name);
	tune_from_freq_list(gen_item,in_row);
}


void
tune_from_freq_list                    (GtkWidget* in_widget,
                                        gint in_row)
{
	int			rec_num;
	db_record_t	*out_rec_p;
	
	if (in_row > -1) {
		rec_num = GPOINTER_TO_INT(gtk_clist_get_row_data(GTK_CLIST(in_widget), in_row));
		out_rec_p = g_list_nth_data(freqDB,rec_num);
		frequency = out_rec_p->pFreq;
		set_freq(TRUE);
	}
}


long
get_gmt_time()
{
	gchar		*gmt_time;
	time_t    	s_time;
	struct tm 	g_time;
	long		out_time;
		
	time(&s_time);
	gmtime_r(&s_time,&g_time);
	gmt_time = g_strdup_printf("%02d%02d",g_time.tm_hour,g_time.tm_min);
	out_time = strtol(gmt_time,NULL,10);
	g_free(gmt_time);
	
	return out_time;	
}


long
get_time_filter()
{
	gchar		*time_str;
	long		out_time;
	
	if (get_down(tun1,"TimeFiltBtn")) {
		get_text(tun1,"TimeFilt",&time_str);
		if (strcmp(g_strstrip(time_str),"")!=0)
			out_time = strtol(time_str,NULL,10);
		else {
			out_time = get_gmt_time();
		}
	}
	else
		out_time = -get_gmt_time();
	
	return out_time;
}


void
set_time_filter()
{
	GtkWidget* gen_item;
	int		page_num;
	
	if (db_ver == 3) {
		get_text(tun1,"LangBox",&db_filt);
		if (get_down(tun1,"LangBtn") && get_enabled(tun1,"LangBtn"))
			gconf_client_set_string(confclient,"/apps/sgcontrol/last_filter",db_filt,NULL);
	}
	gen_item = lookup_widget(tun1,"DBNotebook");
	if (gen_item) {
		page_num = gtk_notebook_get_current_page(GTK_NOTEBOOK(gen_item));
		switch (page_num) {
			case 0 : refresh_sked_list(NULL); break;
			case 1 : refresh_stat_list(-1); break;
			case 2 : refresh_who_list(NULL); break;
		}
	}
}


void
show_skeds_list                        (gchar* in_stat)
{
	refresh_sked_list(in_stat);
	set_visible(tun1,"FrequenciesBox",FALSE);
	GtkWidget* gen_item = lookup_widget(tun1,"DBNotebook");
	if (gen_item)
		gtk_notebook_set_page(GTK_NOTEBOOK(gen_item),0);
	gtk_widget_show(tun1);
}


void
clear_sked_list()
{
	GtkWidget* gen_item = lookup_widget(tun1,"SkedList");
	if (gen_item) {
		gtk_clist_clear(GTK_CLIST(gen_item));
	}
}


void
refresh_sked_list                      (gchar* in_stat)
{
	long	in_time;
	
	in_time = get_time_filter();
	if (in_stat == NULL)
		get_sked_records("*",in_time);
	else
		get_sked_records(in_stat,in_time);
}


void
show_stat_list                         (int in_what)
{
	refresh_stat_list(in_what);
	GtkWidget* gen_item = lookup_widget(tun1,"DBNotebook");
	if (gen_item)
		gtk_notebook_set_page(GTK_NOTEBOOK(gen_item),1);
	gen_item = lookup_widget(tun1,"StatNotebook");
	if (gen_item)
		gtk_notebook_set_page(GTK_NOTEBOOK(gen_item),in_what);
	gtk_widget_show(tun1);
}


void
refresh_stat_list                      (int whichwhat)
{
	long	in_time;
	
	in_time = get_time_filter();
	GtkWidget* gen_item = lookup_widget(tun1,"StatNotebook");
	if (gen_item) {
		if (whichwhat == -1)
			whichwhat = gtk_notebook_get_current_page(GTK_NOTEBOOK(gen_item));
		get_stat_records(whichwhat,abs(in_time));
	}
}


void
show_freq_list                         (gchar* in_freq)
{
	refresh_who_list(in_freq);
	GtkWidget* gen_item = lookup_widget(tun1,"DBNotebook");
	if (gen_item)
		gtk_notebook_set_page(GTK_NOTEBOOK(gen_item),2);
	gtk_widget_show(tun1);
}


void
refresh_who_list                       (gchar* in_freq)
{
	long	in_time;
	double	d_freq;
	
	in_time = get_time_filter();
	if (in_freq == NULL)
		get_who_records(frequency,in_time);
	else {
		d_freq = strtod(in_freq,NULL);
		get_who_records(d_freq,in_time);
	}
}


void
show_freq_sidebar()
{
	GtkWidget* gen_item = lookup_widget(tun1,"DBNotebook");
	if (gen_item)
		gtk_notebook_set_page(GTK_NOTEBOOK(gen_item),1);
	gtk_widget_show(tun1);
}


void
update_tuning_buttons                  (GtkCList* clist,
                                        gint      row)
{
	gint		num_rows, page_num, last_row1, last_row2, stat_row;
	gchar*		clist_name;
	gchar*      stat_list = NULL;
	gboolean	freq_ok1,freq_ok2;

	clist_name = gtk_widget_get_name(GTK_WIDGET(clist));
	if (get_focus(tun1,clist_name)) {
		cur_clist = clist_name;
		cur_row = row;
		freq_ok1 = (!strcmp(clist_name,"FreqList"));
		freq_ok2 = (!strcmp(clist_name,"FreqList"));
		if (strcmp(clist_name,"FreqList") == 0) {
			page_num = get_notebook_page(tun1,"StatNotebook");
			switch (page_num) {
				case 0 : stat_list = g_strdup("BCList"); break;
				case 1 : stat_list = g_strdup("UteList"); break;
				case 2 : stat_list = g_strdup("BCBList"); break;
			}
			last_row1 = get_list_num_rows(tun1,stat_list)-1;
			stat_row = get_list_selected_row(tun1,stat_list);
			last_row2 = get_list_num_rows(tun1,"FreqList")-1;
			g_free(stat_list);
			freq_ok2 = ((stat_row != last_row1) || (row != last_row2));
		}
		num_rows = clist->rows;
		set_enabled(tun1,"LeftBtn",((num_rows > 0) && (row > 0))||freq_ok1);
		set_enabled(tun1,"RightBtn",((num_rows > 0) && (row < (num_rows-1)))||freq_ok2);
	}
}


void
select_previous()
{
	gint		row_num, row_num2, page_num;
	gchar*		stat_list = NULL;

	if (cur_clist) {
		row_num = cur_row;
		row_num--;
		if ((row_num > -1) && (row_num < get_list_num_rows(tun1,cur_clist)))
			set_list_selected_row(tun1,cur_clist,row_num);
		if (strcmp(cur_clist,"SkedList") == 0)
			tune_from_freq_list_ex(tun1,cur_clist,row_num);
		else if (strcmp(cur_clist,"FreqList") == 0) {
			if ((row_num > -1) && (row_num < get_list_num_rows(tun1,"FreqList")))
				tune_from_freq_list_ex(tun1,cur_clist,row_num);
			else {
				page_num = get_notebook_page(tun1,"StatNotebook");
				switch (page_num) {
					case 0 : stat_list = g_strdup("BCList"); break;
					case 1 : stat_list = g_strdup("UteList"); break;
					case 2 : stat_list = g_strdup("BCBList"); break;
				}
				if (get_list_selected_row(tun1,stat_list) > 0) {
					cur_clist = g_strdup(stat_list);
					set_focus(tun1,cur_clist);
					row_num = get_list_selected_row(tun1,stat_list)-1;
					set_list_selected_row(tun1,cur_clist,row_num);
				}
				g_free(stat_list);
			}
		}
		if ((strcmp(cur_clist,"BCList") == 0) || 
		     (strcmp(cur_clist,"UteList") == 0) ||
			 (strcmp(cur_clist,"BCBList") == 0)) {
			if (GPOINTER_TO_INT(get_list_row_data(tun1,cur_clist,row_num)) > -1) {
				page_num = get_notebook_page(tun1,"StatNotebook");
				get_stat_list_items_ex(tun1,cur_clist,row_num);
				set_focus(tun1,"FreqList");
				row_num2 = get_list_num_rows(tun1,"FreqList")-1;
				set_list_selected_row(tun1,"FreqList",row_num2);
				tune_from_freq_list_ex(tun1,"FreqList",row_num2);					
			}
		}
	}
}


void
select_next()
{
	gint		row_num, page_num, num_rows;
	gchar*		stat_list = NULL;

	if (cur_clist) {
		row_num = cur_row;
		row_num++;
		if ((row_num > -1) && (row_num < get_list_num_rows(tun1,cur_clist)))
			set_list_selected_row(tun1,cur_clist,row_num);
		if (strcmp(cur_clist,"SkedList") == 0)
			tune_from_freq_list_ex(tun1,cur_clist,row_num);
		else if (strcmp(cur_clist,"FreqList") == 0) {
			if ((row_num > -1) && (row_num < get_list_num_rows(tun1,"FreqList")))
				tune_from_freq_list_ex(tun1,cur_clist,row_num);
			else {
				page_num = get_notebook_page(tun1,"StatNotebook");
				switch (page_num) {
					case 0 : stat_list = g_strdup("BCList"); break;
					case 1 : stat_list = g_strdup("UteList"); break;
					case 2 : stat_list = g_strdup("BCBList"); break;
				}
				num_rows = get_list_num_rows(tun1,stat_list);
				if (get_list_selected_row(tun1,stat_list) < (num_rows-1)) {
					cur_clist = g_strdup(stat_list);
					set_focus(tun1,cur_clist);
					row_num = get_list_selected_row(tun1,stat_list)+1;
					set_list_selected_row(tun1,cur_clist,row_num);
				}
				g_free(stat_list);
			}
		}
		if ((strcmp(cur_clist,"BCList") == 0) || 
			(strcmp(cur_clist,"UteList") == 0) ||
			(strcmp(cur_clist,"BCBList") == 0)) {
			if (GPOINTER_TO_INT(get_list_row_data(tun1,cur_clist,row_num)) > -1) {
				page_num = get_notebook_page(tun1,"StatNotebook");
				get_stat_list_items_ex(tun1,cur_clist,row_num);
				set_focus(tun1,"FreqList");
				set_list_selected_row(tun1,"FreqList",0);
				tune_from_freq_list_ex(tun1,"FreqList",0);					
			}
			
		}
	}
}

void
set_query                              (gint in_mode)
{ 
	gchar* buf = NULL;

	q_mode = in_mode;
	if (q_mode == 4)
		gtk_window_set_title(GTK_WINDOW(query1),"Frequency Query");
	else
		gtk_window_set_title(GTK_WINDOW(query1),"Station Query");
	
	set_visible(query1,"StationBox",(q_mode != 4));
	set_visible(query1,"FreqBox",(q_mode == 4));
	if ((ispower) && (q_mode == 4)) {
		buf = g_strdup_printf("%11.3f",frequency);
		set_text(query1,"FreqEntry",buf);
		g_free(buf);
	}
	gtk_widget_show(GTK_WIDGET(query1));
}

void
do_query()
{
	gchar*	buf = NULL;
	
	if (q_mode == 4)
		get_text(query1,"FreqEntry",&buf);
	else
	    get_text(query1,"StationEntry",&buf);
		
	switch (q_mode) {
		case 1 : show_skeds_list(buf); break;
		case 2 : show_curr_freq(buf); break;
		case 3 : show_freq_sidebar(); break;
		case 4 : show_freq_list(buf); break;
	}
}
