/* $Id: instimage.c,v 1.4 2003/05/30 00:42:37 cgd Exp $ */

/*
 * Copyright 2001, 2003
 * Broadcom Corporation. All rights reserved.
 *
 * This software is furnished under license and may be used and copied only
 * in accordance with the following terms and conditions.  Subject to these
 * conditions, you may download, copy, install, use, modify and distribute
 * modified or unmodified copies of this software in source and/or binary
 * form. No title or ownership is transferred hereby.
 *
 * 1) Any source code used, modified or distributed must reproduce and
 *    retain this copyright notice and list of conditions as they appear in
 *    the source file.
 *
 * 2) No right is granted to use any trade name, trademark, or logo of
 *    Broadcom Corporation.  The "Broadcom Corporation" name may not be
 *    used to endorse or promote products derived from this software
 *    without the prior written permission of Broadcom Corporation.
 *
 * 3) THIS SOFTWARE IS PROVIDED "AS-IS" AND ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING BUT NOT LIMITED TO, ANY IMPLIED WARRANTIES OF
 *    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 *    NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL BROADCOM BE LIABLE
 *    FOR ANY DAMAGES WHATSOEVER, AND IN PARTICULAR, BROADCOM SHALL NOT BE
 *    LIABLE FOR DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 *    OR OTHERWISE), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <linux/unistd.h>

#define le_to_cpu32(val) (val)

typedef unsigned int u32;

typedef struct {
	unsigned char active;
	unsigned char start_chs[3];
	unsigned char id;
	unsigned char end_chs[3];
	u32 start_sector;
	u32 size;
} part_entry_t;

#define VERSION_MAJOR 0
#define VERSION_MINOR 1

#define BYTES_PER_SECTOR 512

#define EXIT_ERROR    (-1)
#define EXIT_SUCCESS    0

#define COPY_BLOCK_SIZE (1024 * 32)

static void usage(void)
{
	fprintf(stderr, "Usage: instimage [-p <partition number>] <ext2fs image> <disk image>\n");
}

static int verbose = 0;
static long partition_num = 0;
static char *fs_image = 0;
static char *disk_image = 0;

static void parse_args(int argc, char *argv[]) 
{
	int i;
	char *tmp;
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
			case 'p':
				if (i == (argc-1)) {
					usage();
					exit(EXIT_ERROR);
				}
				i++;
				partition_num = strtol(argv[i], &tmp, 0);
				if (*tmp) {
					fprintf(stderr, "Couldn't parse %s as a partition number\n",
						argv[i]);
					usage();
					exit(EXIT_ERROR);
				}
				break;
			case 'V':
				fprintf(stderr, "instimage v%i.%i; (C) 2001 Broadcom\n",
					VERSION_MAJOR,
					VERSION_MINOR);
				break;
			case 'v':
				verbose = 1;
				break;
			default:
				usage();
				exit(EXIT_ERROR);
			}
		} else if (!fs_image) {
			fs_image = argv[i];
		} else if (!disk_image) {
			disk_image = argv[i];
		}
	}
	if (!(fs_image 
	      && disk_image)) {
		usage();
		exit(EXIT_ERROR);
	}
}

static int read_table(int fd, int *entry_num, unsigned long ext_part_sector, unsigned long sector_offset, 
	       unsigned long *start_sector, unsigned long *size)
{
	int i;
	part_entry_t entry;
	long offset = (512 * (ext_part_sector + sector_offset)) + 0x1be;
	for (i = 0; i < 4; i++) {
		if (lseek(fd, offset, SEEK_SET) < 0) {
			perror("llseek failed!");
			exit(1);
		}
		read(fd, &entry, sizeof(part_entry_t));
		if (entry.size) {
			if (entry.id == 0x83) {
				if (*entry_num) {
					(*entry_num)--;
				} else {
					*start_sector = (le_to_cpu32(entry.start_sector) +
							 ext_part_sector +
							 sector_offset) * BYTES_PER_SECTOR;
					*size = entry.size * BYTES_PER_SECTOR;
					return 1;
				}
/*				printf("Partition ID:    0x%x\n"
				       "Starting sector: %u\n"
				       "Size in sectors: %u\n"
				       "Real start sec:  %lu\n\n",
				       entry.id,
				       le_to_cpu32(entry.start_sector),
				       le_to_cpu32(entry.size),
				       le_to_cpu32(entry.start_sector) + ext_part_sector + sector_offset);*/
			} else if (entry.id == 0x5) {
				if (!ext_part_sector) {
					ext_part_sector = le_to_cpu32(entry.start_sector);
					entry.start_sector = 0;
				}
				if (read_table(fd, entry_num, ext_part_sector, le_to_cpu32(entry.start_sector),
					       start_sector, size)) {
					return 1;
				}
			}
		}
		offset += 16;
	}
	return 0;
}


int main(int argc, char *argv[])
{
	int fs_fd, di_fd;
	unsigned long part_ofs, part_size ;
	struct stat fs_stat;
	int tmp_int;
	unsigned long image_size;
	char *buf;
	
	parse_args(argc, argv);
	fs_fd = open(fs_image, O_RDONLY);
	
	if (fs_fd < 0) {
		perror("Open failed");
		exit(EXIT_ERROR);
	}
	if (fstat(fs_fd, &fs_stat)) {
		perror("Stat failed");
		exit(EXIT_ERROR);
	}
	image_size = fs_stat.st_size;

	if (verbose) {
		printf("Found disk image of %li bytes\n", image_size);
	}

	di_fd = open(disk_image, O_RDWR);
	
	if (di_fd < 0) {
		perror("Open failed");
		exit(EXIT_ERROR);
	}
	
	tmp_int = partition_num;

	if(!read_table(di_fd, &tmp_int, 0, 0, &part_ofs, &part_size)) {
		fprintf(stderr, "Didn't find partition table entry %li\n", 
			partition_num);
		exit(EXIT_ERROR);
	}
	
	if (verbose) {
		printf("Found linux partition %li:\n"
		       "  Starting sector:  %li\n"
		       "  Size, in sectors: %li\n",
		       partition_num,
		       part_ofs/BYTES_PER_SECTOR,
		       part_size/BYTES_PER_SECTOR);
	}
	if (image_size > part_size) {
		fprintf(stderr, "Filesystem is bigger than the selected partition\n");
		exit(EXIT_ERROR);
	}
	if (lseek(di_fd, part_ofs, SEEK_SET) < 0) {
		perror("lseek failed");
		exit(EXIT_ERROR);
	}

	buf = malloc(COPY_BLOCK_SIZE);
	if (!buf) {
		fprintf(stderr, "Out of memory\n");
		exit(EXIT_ERROR);
	}

	if (verbose) {
		printf("Grafting in filesystem image");
	}

	while (image_size > COPY_BLOCK_SIZE) {
		read(fs_fd, buf, COPY_BLOCK_SIZE);
		write(di_fd, buf, COPY_BLOCK_SIZE);
		if (verbose) {
			printf(".");
		}
		image_size -= COPY_BLOCK_SIZE;
	}
	read(fs_fd, buf, image_size);
	write(di_fd, buf, image_size);
	if (verbose) {
		printf("\n");
	}
	free(buf);
	printf("Image successfully grafted\n");
	return EXIT_SUCCESS;
}
