/*
    This file is part of the sly ident daemon (slidentd).  slidentd 
    was written by Sean Hunter <sean@uncarved.com> as a minimal 
    RFC1413 (ident) daemon.

    slidentd is copyright (c) 2001 Uncarved Systems Ltd.

    slidentd is free software; you can redistribute it and/or modify
    it under the terms of iversion 2 of the GNU General Public License 
    as published by the Free Software Foundation.

    slidentd is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    in the file COPYING along with slidentd; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "slid_chroot.h"
#include "slid_log.h"
#include "slid_die.h"
#include <sys/stat.h>
#include <sys/types.h>

#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <errno.h>

#include <byte.h>
#include <buffer.h>
#include <str.h>

static int slid_unique_chroot();

/* change directory to a safe path, and chroot if possible */
void
slid_chroot(void)
{
	int retval;
	buffer *log_buffer = slid_get_log_buffer();

	/* We only attempt a chroot if we are running as root */
	if (geteuid() == 0) {
		stralloc path = { 0 };

		stralloc_copys(&path, SLIDENTD_CHROOT_PATH);
		stralloc_0(&path);
		retval = chdir(path.s);
		if (retval != 0) {
			if (!slid_unique_chroot()) {
				slid_put_datestamp(log_buffer);
				buffer_puts(log_buffer, "Unable to chdir to ");
				buffer_puts(log_buffer, path.s);
				buffer_putsflush(log_buffer,
						 " or to create unique chroot dir.\n");
				slid_dies();
			}
		} else {
			errno = 0;
			retval = chroot(path.s);
			if (retval != 0) {
				slid_put_datestamp(log_buffer);
				buffer_puts(log_buffer, "Unable to chroot to ");
				buffer_puts(log_buffer, path.s);
				buffer_puts(log_buffer, ": ");
				buffer_puts(log_buffer, strerror(errno));
				buffer_putsflush(log_buffer, "\n");
				slid_dies();
			}
		}
		stralloc_free(&path);
	} else {
		/* we're not root.  Just chdir to / */
		retval = chdir("/");
		if (retval != 0) {
			slid_die("Unable to chdir to /\n");
		}
	}
}

/* create a unique per-instance chroot jail and chdir and chroot into it.
   Algorithm from an idea by Chris Evans.  All bugs are my own */
int
slid_unique_chroot()
{
	int pid;
	buffer *log_buffer = slid_get_log_buffer();
	stralloc path = { 0 };

	assert(geteuid() == 0);

	pid = getpid();
	stralloc_copys(&path, SLIDENTD_CHROOT_PREFIX);
	stralloc_cats(&path, ".");
	stralloc_catint(&path, pid);
	stralloc_0(&path);

	errno = 0;
	if (mkdir(path.s, 0500) != 0) {
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "Unable to make directory ");
		buffer_puts(log_buffer, path.s);
		buffer_puts(log_buffer, ": ");
		buffer_puts(log_buffer, strerror(errno));
		buffer_putsflush(log_buffer, "\n");
		slid_dies();
	}

	if (chdir(path.s) != 0) {
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "Unable to change to directory ");
		buffer_puts(log_buffer, path.s);
		buffer_puts(log_buffer, ": ");
		buffer_puts(log_buffer, strerror(errno));
		buffer_putsflush(log_buffer, "\n");
		slid_dies();
	}

	if (rmdir(path.s) != 0) {
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "Unable to remove directory ");
		buffer_puts(log_buffer, path.s);
		buffer_puts(log_buffer, ": ");
		buffer_puts(log_buffer, strerror(errno));
		buffer_putsflush(log_buffer,
				 "\nPlease remove directory manually.\n");
		slid_dies();
	}

	if (chroot(".") != 0) {
		slid_put_datestamp(log_buffer);
		buffer_puts(log_buffer, "Unable to chroot to directory ");
		buffer_puts(log_buffer, path.s);
		buffer_puts(log_buffer, ": ");
		buffer_putsflush(log_buffer, strerror(errno));
		buffer_putsflush(log_buffer, "\n");
		slid_dies();
	}
	stralloc_free(&path);

	return 1;
}
