/*
 * raw2webcam.cc -- Connects a Raw DV stream to an ftp Webcam Service
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <unistd.h>

#include <iostream>
#include <string>
#include <cmath>
using std::string;
using std::cerr;
using std::cin;
using std::endl;

#include <preferences.h>
#include <Diagnostics.h>
#include <RawDVFileInput.h>
#include <Threader.h>
#include <frame.h>
#include <time.h>
#include <pthread.h>

class Raw2Webcam : public Threader
{
	private:
		string server;
		string user;
		string password;
		string temp_file;
		string real_file;
		string temp_local;
		double fps;
		int width;
		int height;
		bool deinterlace;
		int quality;
		FILE *ftppipe;
		uint8_t *image;

	public:
		Raw2Webcam( ) : 
			temp_file( "kino.tmp" ), 
			real_file( "kino.jpeg" ), 
			temp_local( "/tmp/kino.jpeg" ),
			fps( 1 ), 
			width( 188 ), 
			height( 144 ), 
			deinterlace( true ), 
			quality( 80 )
		{
			image = new uint8_t[ 720 * 576 * 3 ];
		}

		virtual ~Raw2Webcam( )
		{
			delete image;
		}

		virtual string LogId( )
		{
			return "Raw2Webcam";
		}

		void SetServer( string _server )
		{
			server = _server;
		}

		void SetUser( string _user )
		{
			user = _user;
		}

		void SetPassword( string _password )
		{
			password = _password;
		}

		void SetRealFile( string _real_file )
		{
			real_file = _real_file;
		}

		void SetTempFile( string _temp_file )
		{
			temp_file = _temp_file;
		}

		void SetLocalTempFile( string _temp_local )
		{
			temp_local = _temp_local;
		}

		void SetFPS( double _fps )
		{
			fps = _fps;
		}

		void SetWidth( int _width )
		{
			width = _width;
		}

		void SetHeight( int _height )
		{
			height = _height;
		}

		void SetDeinterlace( bool _deinterlace )
		{
			deinterlace = _deinterlace;
		}

		void SetQuality( int _quality )
		{
			quality = _quality;
		}

		bool Connect( )
		{
			bool ret = false;
			string command = "ftp -n " + server;

			ftppipe = popen( command.c_str( ), "w" );

			if ( ftppipe != NULL )
			{
				fprintf( ftppipe, "user %s %s\n", user.c_str( ), password.c_str( ) );
				fflush( ftppipe );
				ret = true;
			}

			return ret;
		}

		void Close( )
		{
			if ( ftppipe != NULL )
			{
				fprintf( ftppipe, "quit\n" );
				fflush( ftppipe );
				pclose( ftppipe );
				ftppipe = NULL;
			}
		}

	protected:
		bool Send( Frame &frame )
		{
			bool ret = true;
			char command[ 256 ];
			sprintf( command, "raw2image %s -s %dx%d -q %d -i %d", temp_local.c_str( ), width, height, quality, deinterlace );

			FILE *pipe = popen( command, "w" );
			if ( pipe != NULL )
			{
				if ( fwrite( frame.data, frame.GetFrameSize( ), 1, pipe ) != 1 )
					ret = false;
				pclose( pipe );

				if ( ret )
				{
					fprintf( ftppipe, "put %s %s\n", temp_local.c_str( ), temp_file.c_str( ) );
					fflush( ftppipe );
					fprintf( ftppipe, "rename %s %s\n", temp_file.c_str( ), real_file.c_str( ) );
					fflush( ftppipe );
				}
			}
			else
			{
				ret = false;
			}

			return ret;
		}

		void Thread( )
		{
			bool running = true;

			RawDVFileInput input;
			input.SetPumpSize( 25 );
			input.SetFile( stdin );
			input.SetAutoFlush( true );
			input.ThreadStart( );

			while( running && ThreadIsRunning( ) )
			{
				struct timespec t = { 0, 0 };
				if ( input.GetOutputAvailable( ) > 0 )
				{
					Frame &frame = input.GetOutputFrame( );
					if ( Send( frame ) )
					{
						input.QueueInputFrame( );
						if ( fps > 0 )
						{
							t.tv_sec = (time_t)floor( 1 / fps );
							t.tv_nsec = (long)( ( 1 / fps ) - t.tv_sec ) * 1000000000;
							nanosleep( &t, NULL );
						}
					}
				}
				else
				{
					running = input.ThreadIsRunning( ) && input.PumpIsNotCleared( );
				}
			}
			input.ThreadStop( );
		}
};

static void Usage( )
{
	cerr << "Usage: raw2webcam [ options ]" << endl;
	cerr << "Where: options are" << endl;
	cerr << "       -s server    - ftp server to connect to" << endl;
	cerr << "       -u user      - ftp user to connect with" << endl;
	cerr << "       -p password  - ftp password to connect with" << endl;
	cerr << "       -r realfile  - file to create on server (default: kino.jpeg)" << endl;
	cerr << "       -t tempfile  - temp file to use on server (default: kino.tmp)" << endl;
	cerr << "       -l localtemp - temp file to use on local system (default: /tmp/kino.jpeg)" << endl;
	cerr << "       -f fps       - frames per second (default: 1)" << endl;
	cerr << "       -w width     - width of image (default: 160)" << endl;
	cerr << "       -h height    - height of image (default: 144)" << endl;
	cerr << "       -q quality   - quality of jpeg image (default: 80)" << endl;
	cerr << "       -d           - turn deinterlacing off" << endl;
	exit( 0 );
}

int main( int argc, char **argv )
{
	Raw2Webcam ftp;

	Diagnostics::SetApp( "raw2webcam" );

	if ( isatty( fileno( stdin ) ) )
	{
		cerr << "Input must must be obtained from a pipe or a file." << endl;
		Usage( );
	}

	try
	{
		for ( int i = 1; i < argc; i ++ )
		{
			if ( !strcmp( argv[ i ], "--help" ) )
				Usage( );
			else if ( !strcmp( argv[ i ], "-s" ) )
				ftp.SetServer( string( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-u" ) )
				ftp.SetUser( string( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-p" ) )
				ftp.SetPassword( string( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-r" ) )
				ftp.SetRealFile( string( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-t" ) )
				ftp.SetTempFile( string( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-l" ) )
				ftp.SetLocalTempFile( string( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-f" ) )
				ftp.SetFPS( atof( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-w" ) )
				ftp.SetWidth( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-h" ) )
				ftp.SetHeight( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-d" ) )
				ftp.SetDeinterlace( false );
			else if ( !strcmp( argv[ i ], "-q" ) )
				ftp.SetQuality( atoi( argv[ ++ i ] ) );
			else if ( !strcmp( argv[ i ], "-v" ) )
				Diagnostics::SetLevel( atoi( argv[ ++ i ] ) );
			else
				Usage( );
		}

		if ( ftp.Connect( ) == false )
			Usage( );
		ftp.ThreadExecute( );
		ftp.Close( );
	}
	catch ( string exc )
	{
		cerr << "Exception thrown: " << exc << endl;
	}

	return 0;
}
