import cPickle, os, traceback
from urlparse import urlparse
from error import log, logtb, logparam
import pygtk
pygtk.require('2.0')
import gconf
import Event
import LookupManager

GCONF_STRAW_ROOT = "/apps/straw"

OPTION_LAST_POLL = "/general/last_poll"
OPTION_ITEMS_STORED = "/general/number_of_items_stored"
OPTION_ITEM_ORDER = "/general/item_order_newest"
OPTION_WINDOW_SIZE_W = "/ui/window_width"
OPTION_WINDOW_SIZE_H = "/ui/window_height"
OPTION_MAIN_PANE_POS = "/ui/main_pane_position"
OPTION_SUB_PANE_POS = "/ui/sub_pane_position"
OPTION_OFFLINE = "/general/offline"
OPTION_POLL_FREQUENCY = "/general/poll_frequency"
OPTION_FEED_ID_SEQ = "feed_id_seq"
OPTION_FEEDS = "feeds"
OPTION_FEED_INFO_VISIBILITY = "/general/feed_info_visibility"
OPTION_CATEGORIES = "categories"

def ensure_directory(strawdir):
    if os.path.exists(strawdir):
        if not os.path.isdir(strawdir):
            raise Exception
        return 0
    os.mkdir(strawdir)
    return 1

#############
# Config persistence classes
class ConfigPicklePersistence:
    def __init__(self, filename):
        self._config_file = filename
        self._dict = None

    def save_option(self, option, value):
        if self._dict is None:
            self._initialize_dict()
        temp_config_file = self._config_file + ".working"
        pickle_file = open(temp_config_file, "w")
        self._dict[option] = value
        cPickle.dump(self._dict, pickle_file, True)
        pickle_file.close()
        os.rename(temp_config_file, self._config_file)
        return

    def load_option(self, option):
        if self._dict is None:
            self._initialize_dict()
        return self._dict.get(option, None)

    def _initialize_dict(self):
        if os.path.exists(self._config_file):
            pickle_file = open(self._config_file, "r")
            self._dict = cPickle.load(pickle_file)
            pickle_file.close()
        else:
            self._dict = {}

class ConfigGConfPersistence:
    SAVERS = {OPTION_LAST_POLL: 'int',
              OPTION_ITEMS_STORED: 'int',
              OPTION_ITEM_ORDER: 'bool',
              OPTION_WINDOW_SIZE_W: 'int',
              OPTION_WINDOW_SIZE_H: 'int',
              OPTION_MAIN_PANE_POS: 'int',
              OPTION_SUB_PANE_POS: 'int',
              OPTION_OFFLINE: 'bool',
              OPTION_POLL_FREQUENCY: 'int',
              OPTION_FEED_INFO_VISIBILITY: 'bool'}

    def __init__(self, client):
        self.client = client

    def save_option(self, option, value):
        getattr(self.client, 'set_' + self.SAVERS[option])(
            GCONF_STRAW_ROOT + option, value)

    def load_option(self, option):
        return getattr(self.client, 'get_' + self.SAVERS[option])(
            GCONF_STRAW_ROOT + option)

class ConfigPersistence:
    def __init__(self, *backends):
        self.backends = backends

    def save_option(self, option, value):
        for b in self.backends:
            if option in b[1]:
                b[0].save_option(option, value)

    def load_option(self, option):
        for b in self.backends:
            if option in b[1]:
                return b[0].load_option(option)

################
# Proxy config
class BaseProxyConfig(object):
    def __init__(self):
        self._config_working = False
        
    def get_config_working(self):
        return self._config_working

    config_working = property(get_config_working, None, None, "")

    def get_host(self):
        return self._host

    def set_host(self, host):
        self._host = host
        self._ip = None

    def lookup_host(self):
        if self._host and not self._ip:
            LookupManager.get_instance().lookup(
                self._host, self._ip_lookup_callback)

    host = property(get_host, set_host, None, "")

    def get_ip(self):
        return self._ip

    def set_ip(self, ip):
        assert self._ip is None, "Overwriting proxy IP not allowed!"
        if ip is None:
            self.use = 0
        self._ip = ip

    ip = property(get_ip, set_ip, None, "")

    def _ip_lookup_callback(self, host, ip, data):
        self.ip = ip

    def _is_waiting(self):
        return self.use and not self._ip

    is_waiting = property(_is_waiting, None, None, "")

class GconfProxyConfig(BaseProxyConfig):
    """Encapsulate proxy use and location information (host, port, ip),
    gconf reading and name lookup logic"""

    GCONF_PROXY_ROOT = "/system/proxy"
    GCONF_PROXY_MODE = GCONF_PROXY_ROOT + "/mode"
    GCONF_HTTP_PROXY_ROOT = "/system/http_proxy"
    GCONF_HTTP_PROXY_HOST = GCONF_HTTP_PROXY_ROOT + "/host"
    GCONF_HTTP_PROXY_PORT = GCONF_HTTP_PROXY_ROOT + "/port"
    GCONF_HTTP_PROXY_USE_AUTHENTICATION = GCONF_HTTP_PROXY_ROOT + "/use_authentication"
    GCONF_HTTP_PROXY_USER = GCONF_HTTP_PROXY_ROOT + "/authentication_user"
    GCONF_HTTP_PROXY_PASSWORD = GCONF_HTTP_PROXY_ROOT + "/authentication_password"

    def __init__(self):
        BaseProxyConfig.__init__(self)
        client = gconf.client_get_default()
        self._config_working = client.dir_exists(self.GCONF_PROXY_ROOT) and client.dir_exists(self.GCONF_HTTP_PROXY_ROOT)
        if not self._config_working:
            return
        client.add_dir(self.GCONF_PROXY_ROOT, gconf.CLIENT_PRELOAD_RECURSIVE)
        client.add_dir(self.GCONF_HTTP_PROXY_ROOT, gconf.CLIENT_PRELOAD_RECURSIVE)
        client.notify_add(self.GCONF_PROXY_MODE, self.proxy_mode_changed)
        client.notify_add(self.GCONF_HTTP_PROXY_HOST, self.proxy_host_changed)
        client.notify_add(self.GCONF_HTTP_PROXY_PORT, self.proxy_port_changed)
        client.notify_add(self.GCONF_HTTP_PROXY_USE_AUTHENTICATION,
                          self.proxy_auth_changed)
        client.notify_add(self.GCONF_HTTP_PROXY_USER, self.proxy_auth_changed)
        client.notify_add(self.GCONF_HTTP_PROXY_PASSWORD, self.proxy_auth_changed)

        self._ip = None
        pm = client.get_string(self.GCONF_PROXY_MODE)
        self.use = pm is not None and pm != "none"
        self.host = client.get_string(self.GCONF_HTTP_PROXY_HOST)
        self.port = client.get_int(self.GCONF_HTTP_PROXY_PORT)
        self.use_authentication = client.get_bool(
            self.GCONF_HTTP_PROXY_USE_AUTHENTICATION)
        self.user = client.get_string(self.GCONF_HTTP_PROXY_USER)
        self.password = client.get_string(self.GCONF_HTTP_PROXY_PASSWORD)

    # here be gconf logic
    def proxy_mode_changed(self, client, notify_id, entry, *args):
        value = entry.value.get_string()
        self.use = value is not None and value != "none"

    def proxy_host_changed(self, client, notify_id, entry, *args):
        value = entry.value.get_string()
        if value is not None and len(value):
            self.host = value
        else:
            self.use = 0

    def proxy_port_changed(self, client, notify_id, entry, *args):
        value = entry.value.get_int()
        if value is not None and value > 0:
            self.port = value
        else:
            self.use = 0

    def proxy_auth_changed(self, client, notify_id, entry, *args):
        value = entry.value
        if entry.key == self.GCONF_HTTP_PROXY_USE_AUTHENTICATION:
            if not value:
                self.use_authentication = 0
            else:
                self.use_authentication = 1
        elif entry.key == self.GCONF_HTTP_PROXY_USER:
            self.user = value
        elif entry.key == self.GCONF_HTTP_PROXY_PASSWORD:
            self.password = value

class EnvironmentProxyConfig(BaseProxyConfig):
    """Encapsulate proxy use and location information, environment reading"""

    def __init__(self):
        BaseProxyConfig.__init__(self)
        try:
            result = self._read_env()
        except:
            result = 0
        self._config_working = result
        self.use = self._config_working

    def _read_env(self):
        proxy = os.getenv("http_proxy")
        if not proxy:
            return 0
        pparts = urlparse(proxy)
        proto = pparts[0]
        host = pparts[1]
        if proto != "http" or not len(host):
            return 0
        hparts = host.split('@')
        auth = self.user = self.password = self.use_authentication = None
        if len(hparts) > 2:
            return 0
        if len(hparts) == 2:
            auth, host = hparts
        if auth:
            aparts = auth.split(':')
            if len(aparts) != 2:
                return 0
            self.user, self.password = aparts
            self.use_authentication = 1
        self.host = host
        self.port = 80
        hparts = host.split(':')
        if len(hparts) > 2:
            return 0
        if len(hparts) == 2:
            self.host = hparts[0]
            self.port = int(hparts[1])
        return 1

class NullProxyConfig(BaseProxyConfig):
    def __init__(self):
        BaseProxyConfig.__init__(self)
        self.use = 0
        self._config_working = 0
        self._host = None
        self._ip = None
        self.port = None
        self.user = None
        self.password = None
        self.use_authentication = None

###################
# The main man
class Config(object, Event.SignalEmitter):
    _straw_dir = os.path.join(os.getenv("HOME"), ".straw")
    _straw_config_file = os.path.join(_straw_dir, "config")

    def __init__(self, persistence):
        Event.SignalEmitter.__init__(self)
        self.persistence = persistence
        self.initialize_slots(Event.ItemOrderChangedSignal,
                              Event.OfflineModeChangedSignal,
                              Event.PollFrequencyChangedSignal)
        pcchoices = (GconfProxyConfig, EnvironmentProxyConfig, NullProxyConfig)
        for p in pcchoices:
            self._proxy_config = p()
            if self._proxy_config.config_working:
                break

        self.first_time = ensure_directory(self._straw_dir)

        if os.path.exists(self.straw_config_file):
            self._feed_id_seq = self.persistence.load_option(OPTION_FEED_ID_SEQ)
        else:
            self._feed_id_seq = 0

        self._poll_freq = self.persistence.load_option(OPTION_POLL_FREQUENCY)
        self._last_poll = self.persistence.load_option(OPTION_LAST_POLL)
        self._items_stored = self.persistence.load_option(OPTION_ITEMS_STORED)
        self._item_order = self.persistence.load_option(OPTION_ITEM_ORDER)
        self._main_window_size = (
            self.persistence.load_option(OPTION_WINDOW_SIZE_W),
            self.persistence.load_option(OPTION_WINDOW_SIZE_H))
        self._main_pane_position = self.persistence.load_option(
            OPTION_MAIN_PANE_POS)
        self._sub_pane_position = self.persistence.load_option(
            OPTION_SUB_PANE_POS)
        self._offline = self.persistence.load_option(OPTION_OFFLINE)
        self._feed_info_visibility = self.persistence.load_option(OPTION_FEED_INFO_VISIBILITY)
        self._use_threads = True
        self._reload_css = False

    def get_straw_dir(self):
        return self._straw_dir

    straw_dir = property(get_straw_dir, None, None, "")

    def get_straw_config_file(self):
        return self._straw_config_file

    straw_config_file = property(get_straw_config_file, None, None, "")
    
    def get_proxy_config(self):
        return self._proxy_config

    proxy_config = property(get_proxy_config, None, None, "")

    # and the rest of the stuff
    def set_feeds(self, feeddata):
        self.persistence.save_option(
            OPTION_FEEDS, feeddata)
        return

    def get_feeds(self):
        return self.persistence.load_option(OPTION_FEEDS)

    feeds = property(get_feeds, set_feeds, None, "Marshalled feed data")

    def set_categories(self, categorydata):
        self.persistence.save_option(
            OPTION_CATEGORIES, categorydata)
        return

    def get_categories(self):
        return self.persistence.load_option(OPTION_CATEGORIES)

    categories = property(get_categories, set_categories)

    def set_poll_frequency(self, poll_frequency):
        if self._poll_freq != poll_frequency:
            self._poll_freq = poll_frequency
            self.persistence.save_option(OPTION_POLL_FREQUENCY, poll_frequency)
            self.emit_signal(Event.PollFrequencyChangedSignal(self, poll_frequency))
        return

    def get_poll_frequency(self):
        return self._poll_freq

    poll_frequency = property(get_poll_frequency, set_poll_frequency, doc="Polling frequency")

    def set_last_poll(self, last_poll):
        if self._last_poll != last_poll:
            self._last_poll = last_poll
            self.persistence.save_option(OPTION_LAST_POLL, last_poll)
        return

    def get_last_poll(self):
        return self._last_poll

    last_poll = property(get_last_poll, set_last_poll, doc="Last polled")

    def get_number_of_items_stored(self):
        return self._items_stored

    def set_number_of_items_stored(self, num=30):
        if self._items_stored != num:
            self._items_stored = num
            self.persistence.save_option(OPTION_ITEMS_STORED, num)
            self.emit_signal(Event.NumberOfItemsStoredChangedSignal(self))
        return

    number_of_items_stored = property(get_number_of_items_stored,
                                      set_number_of_items_stored,
                                      doc="Number of items to store per feed")

    def set_item_order(self, order):
        if self._item_order != order:
            self._item_order = order
            self.persistence.save_option(OPTION_ITEM_ORDER, order)
            self.emit_signal(Event.ItemOrderChangedSignal(self))
        return

    def get_item_order(self):
        return self._item_order

    item_order = property(get_item_order, set_item_order, doc="Ordering of items")


    def set_feed_info_visibility(self, visibility):
        if self._feed_info_visibility != visibility:
            self._feed_info_visibility = visibility
            self.persistence.save_option(OPTION_FEED_INFO_VISIBILITY, visibility)
        return

    def get_feed_info_visibility(self):
        return self._feed_info_visibility

    feed_info_visibility = property(get_feed_info_visibility,
                                 set_feed_info_visibility, doc="Sets the visibility of feed info widget")

    def get_feed_id_seq(self):
        return self._feed_id_seq

    def set_feed_id_seq(self, id):
        self._feed_id_seq = id
        self.persistence.save_option(OPTION_FEED_ID_SEQ, id)
        return

    def next_feed_id_seq(self):
        self.feed_id_seq += 1
        return self._feed_id_seq

    feed_id_seq = property(get_feed_id_seq, set_feed_id_seq, None, "")

    def set_main_window_size(self, size):
        if self._main_window_size != size:
            self._main_window_size = size
            self.persistence.save_option(OPTION_WINDOW_SIZE_W, size[0])
            self.persistence.save_option(OPTION_WINDOW_SIZE_H, size[1])
            self.emit_signal(Event.MainWindowSizeChangedSignal(self))
        return

    def get_main_window_size(self):
        return self._main_window_size

    main_window_size = property(get_main_window_size, set_main_window_size, "")

    def set_offline(self, mode):
        if self._offline != mode:
            self._offline = mode
            self.persistence.save_option(OPTION_OFFLINE, mode)
            self.emit_signal(Event.OfflineModeChangedSignal(self))
        return

    def get_offline(self):
        return self._offline

    offline = property(get_offline, set_offline, None, "")

    def set_main_pane_position(self, position):
        if self._main_pane_position != position:
            self._main_pane_position = position
            self.persistence.save_option(OPTION_MAIN_PANE_POS, position)
            self.emit_signal(Event.MainPanePositionChangedSignal(self))

    def get_main_pane_position(self):
        return self._main_pane_position

    main_pane_position = property(get_main_pane_position, set_main_pane_position, None, "")

    def set_sub_pane_position(self, position):
        if self._sub_pane_position != position:
            self._sub_pane_position = position
            self.persistence.save_option(OPTION_SUB_PANE_POS, position)
            self.emit_signal(Event.SubPanePositionChangedSignal(self))

    def get_sub_pane_position(self):
        return self._sub_pane_position

    sub_pane_position = property(get_sub_pane_position, set_sub_pane_position, None, "")

    def set_use_threads(self, threads):
        self._use_threads = threads

    def get_use_threads(self):
        return self._use_threads

    use_threads = property(get_use_threads, set_use_threads)

    def set_reload_css(self, reload): self._reload_css = reload
    def get_reload_css(self): return self._reload_css
    reload_css = property(get_reload_css, set_reload_css)

def convert_if_necessary(config):
    if not os.path.exists(config.straw_config_file):
        return

    try:
        f = open(config.straw_config_file, "r")
        cf = cPickle.load(f)

        feeds = cf.get('feeds', None)
        if feeds is not None and feeds[0].get('_n_items_unread', None) is None:
            import ItemStore

            def _convert_feeds(feeds, parent):
                itemstore = ItemStore.get_instance(config.straw_dir)

                for feed in feeds:
                    if isinstance(feed, list):
                        _convert_feeds(feed[1:], parent)
                    else:
                        stored = feed['_items_stored']
                        if stored > -1:
                            cutoff = stored
                        else:
                            cutoff = config.number_of_items_stored
                        feed['_n_items_unread'] = itemstore.get_number_of_unread(feed['_id'], cutoff)
                return feeds

            config.feeds = _convert_feeds(feeds, None)

        if cf.has_key('poll_frequency'):
            config.poll_frequency = cf.get('poll_frequency')
            config.number_of_items_stored = cf.get('number_of_items_stored')
            config.item_order = cf.get('item_order')
            config.main_window_size = cf.get('main_window_size')
            config.main_pane_position = cf.get('main_pane_position')
            config.sub_pane_position = cf.get('sub_pane_position')
            config.offline = cf.get('offline')

            del cf['poll_frequency']
            del cf['number_of_items_stored']
            del cf['item_order']
            del cf['main_window_size']
            del cf['main_pane_position']
            del cf['sub_pane_position']
            del cf['offline']

            f.close()
            f = open(config.straw_config_file, "w")
            f.seek(0)
            f.truncate()
            cPickle.dump(cf, f, True)
    finally:
        f.close()
    return

def create_gconf_persistence():
    client = gconf.client_get_default()
    client.add_dir(GCONF_STRAW_ROOT, gconf.CLIENT_PRELOAD_ONELEVEL)
    return ConfigGConfPersistence(client)

def create_pickle_persistence():
    return ConfigPicklePersistence(Config._straw_config_file)

def create_instance():
    # Add your GConf key here as well!
    cp = ConfigPersistence(
        (create_gconf_persistence(),
         (OPTION_LAST_POLL, OPTION_ITEMS_STORED, OPTION_ITEM_ORDER,
          OPTION_WINDOW_SIZE_W, OPTION_WINDOW_SIZE_H,
          OPTION_MAIN_PANE_POS, OPTION_SUB_PANE_POS, OPTION_OFFLINE,
          OPTION_POLL_FREQUENCY, OPTION_FEED_INFO_VISIBILITY)),
        (create_pickle_persistence(),
         (OPTION_FEED_ID_SEQ, OPTION_FEEDS, OPTION_CATEGORIES)))
    config = Config(cp)
    convert_if_necessary(config)
    return config

config_instance = None
def get_instance():
    global config_instance
    if config_instance is None:
        config_instance = create_instance()
    return config_instance
