/*
 *	tardy - a tar post-processor
 *	Copyright (C) 1998, 1999, 2002, 2004 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: interface definition for common/tar/input.cc
 */

#ifndef COMMON_TAR_INPUT_H
#define COMMON_TAR_INPUT_H

#include <tar/header.h>

/**
  * The tar_input class represents an abstract archive input source.
  * It was originally tar(5) formatted archives, but has since been
  * generalized to cpio(1) archives and even list of file names.
  */
class tar_input
{
public:
    /**
      * The destructor.
      */
    virtual ~tar_input();

    /**
      * The default constructor.
      */
    tar_input();

    /**
      * The read_data method is used to read a block of data from
      * the input.  The `buffer_length' argument specifies the
      * maximum number of bytes in the `buffer'.  Returns the number
      * of bytes read, or 0 at end of file.
      */
    virtual int read_data(void *buffer, int buffer_length) = 0;

    /**
      * The read_data_padding is used to read any extra data between
      * files.  By default, this method does nothing.
      */
    virtual void read_data_padding();

    /**
      * The read_header method is used to read file information from
      * the input.  Returns 0 at end of input.
      */
    virtual int read_header(tar_header &) = 0;

    /**
      * The read_header_padding is used to read any extra data between
      * headers.  By default, this method does nothing.
      */
    virtual void read_header_padding();

    /**
      * The `filename' method is used to obtain the name of the
      * input file (not the current archiove member file name).
      */
    virtual const char *filename() const = 0;

    /**
      * The fatal method is used to print a fatal error message.
      * This method does not return.
      */
    void fatal(char *, ...) const ATTR_PRINTF(2, 3);

    /**
      * The warning method is used to print a warning message.
      */
    void warning(char *, ...) const ATTR_PRINTF(2, 3);

private:
    /**
      * The copy constructor.  Do not use.
      */
    tar_input(const tar_input &);

    /**
      * The assignment operator.  Do not use.
      */
    tar_input &operator=(const tar_input &);
};

#endif /* COMMON_TAR_INPUT_H */
