/*
 * window.c - a module for X-Window interface
 * by Hirotsugu Kakugawa
 *
 *  4 Jun 1997  Changed cache mechanism.
 *  6 Jun 1997  Implemented 'color' feature. Added smart cache size choice.
 * 16 Jun 1997  Implemented color figure feature for Direct/TrueColor.
 * 17 Jun 1997  Implemented color figure feature for Pseudo/StaticColor.
 */
/*
 * Copyright (C) 1996-1997 Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <ctype.h>

#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#include "dvi-2_6.h"
#include "defs.h"
#include "cf-xldvi.h"
#include "resource.h"
#include "window.h"
#include "../X11/x11-defs.h"

Public Display         *x_display;
Public Window          x_window;
Public int             x_offset_x, x_offset_y;

Private Visual         *x_visual;
Private int            x_depth;
Private Colormap       x_cmap;
Private GC             x_gc_copy, x_gc_clear;
Private XImage         x_image;
Private unsigned long  x_pix_char_default, x_pix_char;
Private unsigned long  x_pix_paper_default, x_pix_paper;
Private unsigned long  x_pix_back;
Private unsigned long  x_pix_frame;
Private unsigned int   x_page_width, x_page_height;
Private int            x_looking_at_x, x_looking_at_y;
Private Cursor         x_cursor_current;
Private Cursor         x_cursor_reading, x_cursor_ready; 
Private Cursor         x_cursor_drawing, x_cursor_ps;
Private int            x_rgb_shift_r, x_rgb_shift_g, x_rgb_shift_b;
Private unsigned int   x_rgb_mask_r,  x_rgb_mask_g,  x_rgb_mask_b;

Private int            x_pc_cache_size   = 0;
Private WIN_CACHE      x_pc_cache        = NULL;
Private WIN_CACHE      *x_pc_cache_table = NULL;
Private unsigned int    x_pc_w, x_pc_h;
Private GC             x_pc_gc_image, x_pc_gc_image_mask, x_pc_gc_image_rgb;
Private GC             x_pc_gc_rect;
Private GC             x_pc_gc_clear, x_pc_gc_frame;


Private void    x_rgb_init(void);
Private int     first_bit(int);
Private void    x_rectangle(long,long,long,long,GC);
Private void    x_clear_window2(int);
Private void    x_cache_clear_page(int);
Private void    x_window_resize(double); 
Private void    put_pixmap(int,int,DVI_PIXMAP_RGB,long,long);
Private void    put_graymap(int,DVI_GRAYMAP,long,long);



Public void
x_open_display(char *disp)
{
  if ((x_display = XOpenDisplay(disp)) == NULL){
    fprintf(stderr, "Can't open X display...\n");
    exit(1);
  }
  x_visual = DefaultVisual(x_display, 0);
  x_depth  = DefaultDepth(x_display, 0);
  x_cmap   = DefaultColormap(x_display, 0);

#if 0
  printf("Visual Class: %d\n", x_visual->class);
  printf("  red mask:   0x%08x\n", x_visual->red_mask);
  printf("  green mask: 0x%08x\n", x_visual->green_mask);
  printf("  blue mask:  0x%08x\n", x_visual->blue_mask);
  printf("  bits per RGB: %d\n", x_visual->bits_per_rgb);
  printf("Depth:  %d\n", x_depth);
  printf("Planes: %d\n", XDisplayPlanes(x_display, 0));
#endif

#if !defined(HAVE_OPENWIN)
  XInitImage(&x_image);
#endif
  x_image.format                      = XYBitmap;
  x_image.xoffset                     = 0;
  x_image.byte_order                  = MSBFirst;
  x_image.bitmap_unit                 = 8;
  x_image.bitmap_bit_order            = MSBFirst;
  x_image.bitmap_pad                  = 8;
  x_image.depth                       = 1;

  x_rgb_init();
}

Private void
x_rgb_init(void)
{
  x_rgb_shift_r = first_bit(x_visual->red_mask);
  x_rgb_shift_g = first_bit(x_visual->green_mask);
  x_rgb_shift_b = first_bit(x_visual->blue_mask);
  x_rgb_mask_r  = x_visual->red_mask   >> x_rgb_shift_r;
  x_rgb_mask_g  = x_visual->green_mask >> x_rgb_shift_g;
  x_rgb_mask_b  = x_visual->blue_mask  >> x_rgb_shift_b;
}

Private int
first_bit(int n)
{
  int  bit, i;

  bit = 1;
  for (i = 0; i < 32; i++){
    if ((n & bit) == bit)
      break;
    bit = bit << 1;
  }
  return i;
}

Public void
x_init(void)
{
  char          name[64];
  XColor        xc0, xc1;
  Window        root;
  int           x, y;
  unsigned int  b, d;

  x_looking_at_x = 0;
  x_looking_at_y = 0;

  XAllocNamedColor(x_display, x_cmap, Resource.color_char, &xc1, &xc0);
  x_pix_char_default = x_pix_char = xc1.pixel;
  XAllocNamedColor(x_display, x_cmap, Resource.color_paper, &xc1, &xc0);
  x_pix_paper_default = x_pix_paper = xc1.pixel;
  XAllocNamedColor(x_display, x_cmap, Resource.color_back, &xc1, &xc0);
  x_pix_back = xc1.pixel;
  XAllocNamedColor(x_display, x_cmap, Resource.color_frame, &xc1, &xc0);
  x_pix_frame = xc1.pixel;

  paper_size(Resource.paper, &x_page_width, &x_page_height);
  x_window = XCreateSimpleWindow(x_display, RootWindow(x_display, 0),
				 0, 0, x_page_width, x_page_height, 
				 2, x_pix_frame, x_pix_paper);
  XSetWindowBackground(x_display, x_window, x_pix_back);
  sprintf(name, "%s %s", PROG_NAME, VERSION);
  XStoreName(x_display, x_window, name);
  XSelectInput(x_display, x_window, EVENT_MASK);

  x_gc_clear = XCreateGC(x_display, x_window, 0, NULL);
  x_gc_copy  = XCreateGC(x_display, x_window, 0, NULL);

  XSetGraphicsExposures(x_display, x_gc_copy, False);

  x_cursor_reading = XCreateFontCursor(x_display, XC_shuttle);
  x_cursor_drawing = XCreateFontCursor(x_display, XC_watch);
  x_cursor_ready   = XCreateFontCursor(x_display, XC_circle);
  x_cursor_ps      = XCreateFontCursor(x_display, XC_coffee_mug);
  x_cursor_current = x_cursor_ready;

  XMapWindow(x_display, x_window);
  XFlush(x_display);
  XGetGeometry(x_display, x_window, &root,
	       &x, &y, &x_page_width, &x_page_height, &b, &d);

  x_cache_reset();
  x_cache_get();

  x_clear_window2(1);
  x_cache_clear_page(1);
}

Public int
x_visual_color_direct(void)
{
  switch (x_visual->class){
  case StaticGray:
  case GrayScale:
  case StaticColor:
  case PseudoColor:
    return 0;
  case TrueColor:
  case DirectColor:
    return 1;
  }
  return 0;
}

Public int
x_visual_color(void)
{
  switch (x_visual->class){
  case StaticGray:
  case GrayScale:
    return 0;
  case StaticColor:
  case PseudoColor:
  case TrueColor:
  case DirectColor:
    return 1;
  }
  return 0;
}



Public void  
x_resized_window(unsigned int w, unsigned int h)
{
  x_page_width  = w;
  x_page_height = h;
}

Public void
x_window_enlarge(void)
{
  x_window_resize(WINDOW_ENLARGE_RATIO);
}

Public void
x_window_shrink(void)
{
  x_window_resize(1.0/WINDOW_ENLARGE_RATIO);
}

Private void
x_window_resize(double ratio)
{
  Window        root;
  int           x, y;
  unsigned int  b, d;

  XGetGeometry(x_display, x_window, &root,
	       &x, &y, &x_page_width, &x_page_height, &b, &d);
  if (ratio < 0)
    return;
  x_page_width  *= ratio;
  x_page_height *= ratio;
  if (x_page_width < 10)
    x_page_width = 10;
  if (x_page_height < 10)
    x_page_height = 10;
  XResizeWindow(x_display, x_window, x_page_width, x_page_height);
}

Public int
x_shift_view_x(int shift, int check_only)
{
  int  s;

  if (shift >= WIN_MOVE_X_MAX){
    s = x_pc_w - x_looking_at_x - x_page_width;
  } else if (shift <= -WIN_MOVE_X_MAX){
    s = -x_looking_at_x;
  } else {
    s = ((double)x_page_width * (double)shift) / 100.0;
  }
  if ((x_looking_at_x + s) < 0)
    return -1;
  if (x_pc_w - (x_looking_at_x + s) < x_page_width)
    return -1;
  if (check_only == 1)
    return 0;

  x_looking_at_x += s;
  return 0;
}

Public int
x_shift_view_y(int shift, int check_only)
{
  int  s;

  s = ((double)x_page_height * (double)shift) / 100.0;
  if ((x_looking_at_y + s) < 0)
    return -1;
  if (x_pc_h < (x_looking_at_y + s + x_page_height))
    return -1;
  if (check_only == 1)
    return 0;

  x_looking_at_y += s;
  return 0;
}

Public void
x_correct_view(void)
{
  if (x_pc_w < (x_looking_at_x + x_page_width))
    x_looking_at_x = x_pc_w - x_page_width;
  if (x_looking_at_x < 0)
    x_looking_at_x = 0;

  if (x_pc_h < (x_looking_at_y + x_page_height))
    x_looking_at_y = x_pc_h - x_page_height;
  if (x_looking_at_y < 0)
    x_looking_at_y = 0;
}



/*
 * Color Change 
 */

Public void  
x_change_color_fg(int page_resume_flag, double r, double g, double b)
{
  XColor        xc;

  if ((r < 0) || (g < 0) || (b < 0)){
    x_pix_char = x_pix_char_default;
  } else {
    xc.red   = (unsigned int)(0xffff * r) & 0xffff;
    xc.green = (unsigned int)(0xffff * g) & 0xffff;
    xc.blue  = (unsigned int)(0xffff * b) & 0xffff;
    XAllocColor(x_display, x_cmap, &xc);
    x_pix_char = xc.pixel;
  }
  XSetForeground(x_display, x_pc_gc_image, x_pix_char);
  XSetForeground(x_display, x_pc_gc_rect,  x_pix_char);
}

Public void  
x_change_color_bg(int page_resume_flag, double r, double g, double b)
{
  XColor        xc;

  if ((r < 0) || (g < 0) || (b < 0)){
    x_pix_paper = x_pix_paper_default;
  } else {
    xc.red   = (unsigned int)(0xffff * r) & 0xffff;
    xc.green = (unsigned int)(0xffff * g) & 0xffff;
    xc.blue  = (unsigned int)(0xffff * b) & 0xffff;
    XAllocColor(x_display, x_cmap, &xc);
    x_pix_paper = xc.pixel;
  }
  if (page_resume_flag == 0){
    x_clear_window2(1);
    x_cache_clear_page(1);
  } else {
    x_clear_window2(0);
    x_cache_clear_page(0);
  }
}



/*
 * Put Bitmap & Put Rectangle
 */

Public void
x_put_bitmap(DVI_BITMAP bitmap, long x, long y)
{
  if ((bitmap->width == 0) || (bitmap->height == 0)) 
    return;

  x_image.width          = bitmap->width;
  x_image.height         = bitmap->height;
  x_image.bytes_per_line = bitmap->raster;
  x_image.data           = (char*)bitmap->bitmap;

  x += x_offset_x;
  y += x_offset_y;
  XPutImage(x_display, x_pc_cache->win, x_pc_gc_image_mask, 
	    &x_image, 0, 0, x, y, bitmap->width, bitmap->height);
  XPutImage(x_display, x_pc_cache->win, x_pc_gc_image, 
	    &x_image, 0, 0, x, y, bitmap->width, bitmap->height);
  XCopyArea(x_display, x_pc_cache->win, x_window, x_gc_copy,
	    x, y, bitmap->width, bitmap->height, 
	    x - x_looking_at_x, y - x_looking_at_y);
}

Public void
x_put_rectangle(long x, long y, long w, long h)
{
  x_rectangle(x, y, w, h, x_pc_gc_rect);
}


Public void
x_put_pixmap_rgb(DVI_PIXMAP_RGB pixmap_rgb, long x, long y)
{
  switch (x_visual->class){
  default:
  case StaticGray:
  case GrayScale:
    break;
  case StaticColor:
  case PseudoColor:
    put_pixmap(256, 16, pixmap_rgb, x, y);
    break;
  case TrueColor:
  case DirectColor:
    put_pixmap(16*16*16, 18, pixmap_rgb, x, y);
    break;
  }
}

Private void
put_pixmap(int ncolors, int nlevels, DVI_PIXMAP_RGB pixmap, long x, long y)
{
  static unsigned long   *pix_table   = NULL;
  static char            *pix_alloced = NULL;
  static int             left_ncolors = -1;
  static int             last_ncolors = -1;
  static int             last_nlevels = -1;
  unsigned long   pix;
  XImage          *x_image_rgb;
  unsigned char   *buffer, *p;
  XColor          col;
  int             r, g, b, xxx, yyy, i, n;
  int 	          cr, cg, cb, cd, r0, b0, g0, d0;

  if (pixmap == NULL)
    return;
  if ((pixmap->width == 0) || (pixmap->height == 0)) 
    return;
  
  XFlush(x_display);

  if ((last_ncolors != ncolors) || (last_nlevels != nlevels)){
    if (pix_table != NULL){
      free(pix_table);
      pix_table = NULL;
    }
    if (pix_alloced != NULL){
      free(pix_alloced);
      pix_alloced = NULL;
    }
  }
  if (pix_table == NULL){
    n = nlevels*nlevels*nlevels;
    pix_table   = (unsigned long*)calloc(n, sizeof(unsigned long));
    pix_alloced = (char*)malloc(n);
    if ((pix_table == NULL) || (pix_alloced == NULL)){
      if (pix_table != NULL)
	free(pix_table);
      if (pix_alloced != NULL)
	free(pix_alloced);
      pix_table   = NULL;
      pix_alloced = NULL;
      return;
    }
    for (i = 0; i < n; i++)
      pix_alloced[i] = 0;
    left_ncolors = ncolors;
    last_ncolors = ncolors;
    last_nlevels = nlevels;
  }

  buffer = (unsigned char*)malloc(x_depth * pixmap->height * pixmap->width);
  if (buffer == NULL)
    return;

  x_image_rgb
    = XCreateImage(x_display, x_visual, x_depth, XYPixmap,
		   0, (char*)buffer,
		   pixmap->width, pixmap->height,
		   8, (pixmap->width+7)/8);

  col.flags = DoRed|DoGreen|DoBlue;
  for (yyy = 0; yyy < pixmap->height; yyy++){
    p = &pixmap->bitmap[yyy * pixmap->raster];
    for (xxx = 0; xxx < pixmap->width; xxx++, p += 3){
      r = ((nlevels-1) * p[0]) / pixmap->max;
      g = ((nlevels-1) * p[1]) / pixmap->max;
      b = ((nlevels-1) * p[2]) / pixmap->max;
      pix  = r | (g * nlevels) | (b * nlevels * nlevels);
      if (pix_alloced[pix] == 1){
	col.pixel = pix_table[pix];
      } else {
	if (left_ncolors == 0){
	  cr = cg = cb = nlevels - 1;
	  cd = (cr - r)*(cr - r) + (cg - g)*(cg - g) + (cb - b)*(cb - b);
	  col.pixel = 0;
	  for (i = 0; i < nlevels*nlevels*nlevels; i++){
	    if (pix_alloced[i] == 1){
	      r0 = i % nlevels;
	      g0 = (i/nlevels) % nlevels;
	      b0 = i/(nlevels*nlevels);
	      if (cd > (d0 = (r0-r)*(r0-r) + (g0-g)*(g0-g) + (b0-b)*(b0-b))){
		cr = r0;
		cg = g0; 
		cb = b0;
		col.pixel = pix_table[i];
		if (d0 == 0)
		  break;
	      }
	    }
	  }
	} else {
	  col.red   = (65535L * p[0]) / pixmap->max;
	  col.green = (65535L * p[1]) / pixmap->max;
	  col.blue  = (65535L * p[2]) / pixmap->max;
	  if (XAllocColor(x_display, x_cmap, &col) == 0){
	    if (x_cmap == DefaultColormap(x_display, 
					  DefaultScreen(x_display))){
	      x_cmap = XCopyColormapAndFree(x_display, x_cmap);
	    }
	    XAllocColor(x_display, x_cmap, &col);
	  }
	  left_ncolors--;
	  pix_alloced[pix] = 1;
	}
	pix_table[pix] = col.pixel;
      }
      XPutPixel(x_image_rgb, xxx, yyy, col.pixel);
    }
  }

  x += x_offset_x;
  y += x_offset_y;
  XPutImage(x_display, x_pc_cache->win, x_pc_gc_image_rgb, 
	    x_image_rgb, 0, 0, x, y, 
	    pixmap->width, pixmap->height);
  XCopyArea(x_display, x_pc_cache->win, x_window, x_gc_copy,
	    x, y, pixmap->width, pixmap->height,
	    x - x_looking_at_x, y - x_looking_at_y);
  XDestroyImage(x_image_rgb);
  XFlush(x_display);
}

Public void
x_put_graymap(DVI_GRAYMAP graymap, long x, long y)
{
  switch (x_visual->class){
  default:
  case StaticGray:
  case GrayScale:
    put_graymap(16, graymap, x, y);
    break;
  case StaticColor:
  case PseudoColor:
    put_graymap(64, graymap, x, y);
    break;
  case TrueColor:
  case DirectColor:
    put_graymap(256, graymap, x, y);
    break;
  }
}

Private void
put_graymap(int nlevels, DVI_GRAYMAP graymap, long x, long y)
{
  static unsigned long   *pix_table   = NULL;
  static int             last_nlevels = -1;
  unsigned long   pix;
  XImage          *x_image_rgb;
  unsigned char   *buffer, *p;
  XColor          col;
  int             xxx, yyy, i;
  
  if (graymap == NULL)
    return;
  if ((graymap->width == 0) || (graymap->height == 0)) 
    return;
  
  buffer = (unsigned char*)malloc(x_depth * graymap->height * graymap->raster);
  if (buffer == NULL)
    return;

  x_image_rgb
    = XCreateImage(x_display, x_visual, x_depth, XYPixmap, 0, (char*)buffer, 
		   graymap->width, graymap->height, 8, (graymap->width+7)/8);

  if (last_nlevels != nlevels){
    if (pix_table != NULL){
      free(pix_table);
      pix_table = NULL;
    }
  }
  if (pix_table == NULL){
    pix_table = (unsigned long*)calloc(nlevels, sizeof(unsigned long));
    if (pix_table == NULL){
      XDestroyImage(x_image_rgb);
      return;
    }
  }
  last_nlevels = nlevels;

  for (i = 0; i < nlevels; i++)
    pix_table[i]  = ~0L;
  col.flags = DoRed|DoGreen|DoBlue;
  for (yyy = 0; yyy < graymap->height; yyy++){
    p = &graymap->bitmap[yyy * graymap->raster];
    for (xxx = 0; xxx < graymap->width; xxx++, p++){
      pix = (*p * (nlevels-1)) / graymap->max;
      if ((col.pixel = pix_table[pix]) == ~0L){
	col.red = col.green = col.blue = (65535L * (*p)) / graymap->max;
	if (XAllocColor(x_display, x_cmap, &col) == 0){
	  if (x_cmap == DefaultColormap(x_display, DefaultScreen(x_display)))
	    x_cmap = XCopyColormapAndFree(x_display, x_cmap);
	  XAllocColor(x_display, x_cmap, &col);
	}
	pix_table[pix] = col.pixel;
      }
      XPutPixel(x_image_rgb, xxx, yyy, col.pixel);
    }
  }

  x += x_offset_x;
  y += x_offset_y;
  XPutImage(x_display, x_pc_cache->win, x_pc_gc_image_rgb, 
	    x_image_rgb, 0, 0, x, y, 
	    graymap->width, graymap->height);
  XCopyArea(x_display, x_pc_cache->win, x_window, x_gc_copy,
	    x, y, graymap->width, graymap->height,
	    x - x_looking_at_x, y - x_looking_at_y);
  XSync(x_display, 0);
  XDestroyImage(x_image_rgb);
}



Private void
x_rectangle(long x, long y, long w, long h, GC gc)
{
  long   xw, yh;

  x += x_offset_x;
  y += x_offset_y;

  if ((xw = w) <= 0)
    xw =1;
  if (x+xw > x_pc_w)
    xw = x_pc_w - x;

  if ((yh = h) <= 0)
    yh = 1;
  if (y+yh > x_pc_h)
    yh = x_pc_h - y;

  XFillRectangle(x_display, x_pc_cache->win, x_pc_gc_image_mask, x, y, xw, yh);
  XFillRectangle(x_display, x_pc_cache->win, gc,                 x, y, xw, yh);
  XCopyArea(x_display, x_pc_cache->win, x_window, x_gc_copy,
	    x, y, xw, yh, x - x_looking_at_x, y - x_looking_at_y);
}


/* 
 * Page Clear & Page Restore
 */
Public void 
x_clear_window(void)
{
  x_clear_window2(1);
  XSync(x_display, 0);
}

Private void 
x_clear_window2(int fill)
{
  XSetForeground(x_display, x_gc_clear, x_pix_back);
  if (fill == 1)
    XFillRectangle(x_display, x_window, x_gc_clear, 0, 0, 
		   x_page_width, x_page_height);
  XSetForeground(x_display, x_gc_clear, x_pix_paper);
  if (fill == 1)
    XFillRectangle(x_display, x_window, x_gc_clear, 0, 0,
		   x_pc_w, x_pc_h);
}

Public void
x_restore_current_page(void)
{
  XCopyArea(x_display, x_pc_cache->win, x_window, x_gc_copy, 
	    x_looking_at_x, x_looking_at_y, x_page_width, x_page_height, 
	    0, 0);
}

Public void 
x_put_border(void)
{
  x_rectangle(0-x_offset_x, 0-x_offset_y, x_pc_w, 1, x_pc_gc_frame);
  x_rectangle(0-x_offset_x, 0-x_offset_y, 1, x_pc_h, x_pc_gc_frame);
  x_rectangle(0-x_offset_x, x_pc_h-1-x_offset_y, x_pc_w, 1, x_pc_gc_frame);
  x_rectangle(x_pc_w-1-x_offset_x, 0-x_offset_y, 1, x_pc_h, x_pc_gc_frame);
}



/* 
 * Cursor
 */
Public void
x_change_cursor(int new)
{
  Cursor  new_cursor;

  switch (new){
  default:
  case CURSOR_READING:
    new_cursor = x_cursor_reading;
    break;
  case CURSOR_READY:
    new_cursor = x_cursor_ready;
    break;
  case CURSOR_DRAWING:
    new_cursor = x_cursor_drawing;
    break;
  case CURSOR_DRAWING_PS:
    new_cursor = x_cursor_ps;
    break;
  }
  XDefineCursor(x_display, x_window, new_cursor);
  XFlush(x_display);
  x_cursor_current = new_cursor;
}

Public int
x_currenct_cursor(void)
{
  return x_cursor_current;
}



/*
 * Cache
 */

Public void
x_cache_reset(void)
{
  WIN_CACHE  wc;
  Window     wcdrw;
  XGCValues  gcval;
  int        p;
  double     sr;

  if (x_pc_cache_table != NULL){
    for (p = 0; p < x_pc_cache_size; p++){
      if ((wc = x_pc_cache_table[p]) != NULL){
	XFreePixmap(x_display, wc->win);
	if (wc->context != NULL)
	  DVI_FREE_CONTEXT(dviobj, dev, wc->context);
	free(x_pc_cache_table[p]);
      }
    }
    free(x_pc_cache_table);
    x_pc_cache_table = NULL;
    XFreeGC(x_display, x_pc_gc_image_mask);
    XFreeGC(x_display, x_pc_gc_image);
    XFreeGC(x_display, x_pc_gc_rect);
    XFreeGC(x_display, x_pc_gc_frame);
    XFreeGC(x_display, x_pc_gc_clear);
  }

  sr = (shrink/Resource.shrink);
  x_pc_cache_size = Resource.page_cache * (sr * sr);
  if (x_pc_cache_size < 1)
    x_pc_cache_size = 1;
  if (x_pc_cache_size > 2*Resource.page_cache)
    x_pc_cache_size  = 2*Resource.page_cache;
  
  x_pc_cache = NULL;
  x_pc_cache_table 
    = (WIN_CACHE*)calloc(x_pc_cache_size, sizeof(WIN_CACHE));
  if (x_pc_cache_table == NULL){
    fprintf(stderr, "No memory...\n");
    exit(1);
  }
  paper_size(Resource.paper, &x_pc_w, &x_pc_h);

  wcdrw = x_window;
  gcval.function   = GXandInverted;
  x_pc_gc_image_mask = XCreateGC(x_display, wcdrw, GCFunction, &gcval);
  XSetForeground(x_display, x_pc_gc_image_mask, 0xffffffff);
  XSetBackground(x_display, x_pc_gc_image_mask, 0x00000000);
  gcval.function   = GXor;
  x_pc_gc_image = XCreateGC(x_display, wcdrw, GCFunction, &gcval);
  XSetForeground(x_display, x_pc_gc_image, x_pix_char);
  XSetBackground(x_display, x_pc_gc_image, 0x00000000);
  x_pc_gc_image_rgb = XCreateGC(x_display, wcdrw, 0, NULL);
  XSetForeground(x_display, x_pc_gc_image, x_pix_char);
  gcval.function = GXcopy;
  x_pc_gc_rect = XCreateGC(x_display, wcdrw, GCFunction, &gcval);
  XSetForeground(x_display, x_pc_gc_rect, x_pix_char);
  x_pc_gc_frame = XCreateGC(x_display, x_window, 0, NULL);
  XSetForeground(x_display, x_pc_gc_frame, x_pix_frame);
  XSetBackground(x_display, x_pc_gc_image, 0x00000000);
  x_pc_gc_clear = XCreateGC(x_display, wcdrw, 0, NULL);

  /*(void)x_cache_get();*/
}

Public Pixmap
x_cache_get_window(void)
{
  WIN_CACHE   wc;

  wc = x_cache_get();
  return wc->win;
}

Public WIN_CACHE
x_cache_get(void)
{
  int   p;

  p = current_page % x_pc_cache_size;
  if ((x_pc_cache = x_pc_cache_table[p]) == NULL){
    x_pc_cache_table[p] = (WIN_CACHE)calloc(1, sizeof(struct s_win_cache));
    if ((x_pc_cache = x_pc_cache_table[p]) == NULL){
      fprintf(stderr, "No memory...\n");
      exit(1);
    }
    x_pc_cache->page    = current_page;
    x_pc_cache->flag    = WIN_CACHE_EMPTY;
    x_pc_cache->context = NULL;
    x_pc_cache->win
      = XCreatePixmap(x_display, DefaultRootWindow(x_display), 
		      x_pc_w, x_pc_h, x_depth);
    x_cache_clear_page(1);
  }
  if (x_pc_cache->page != current_page){
    x_pc_cache->page = current_page;
    x_pc_cache->flag = WIN_CACHE_EMPTY;
    if (x_pc_cache->context != NULL)
      DVI_FREE_CONTEXT(dviobj, dev, x_pc_cache->context);
    x_pc_cache->context = NULL;
    x_cache_clear_page(1);
  }

  return x_pc_cache;
}

Private void
x_cache_clear_page(int fill)
{
  XSetForeground(x_display, x_pc_gc_clear, x_pix_paper);
  if (fill == 1)
    XFillRectangle(x_display, x_pc_cache->win, x_pc_gc_clear,
		   0, 0, x_pc_w, x_pc_h);
}

/*EOF*/
