/* 
 * test.c - a test program for DVIlib (for checking memory leaks.)
 * by Hirotsugu Kakugawa
 *
 *
 */
/*
 * Copyright (C) 1996-2001 Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#ifdef __FreeBSD__
#include <sys/types.h> 
#include <sys/time.h> 
#include <sys/resource.h> 
#define RUSAGE_SELF 0
#endif

#include "dvi-2_6.h"

#define DPI             "300"
#define KPATHSEA_MODE   "cx"
#define VFLIBCAP        "vflibcap-tex"

char     *dpi;
char     *kpathsea_mode;
char     *vflibcap;

int      opt_font_delay = 0;
int      opt_async_gs   = 0;
int      opt_draw_page  = 0;

void        usage(void);
DVI_DEVICE  device_create(void);



int
main(int argc, char** argv)
{
  DVI            dvi;
  DVI_DEVICE     dev;
  DVI_PROPERTY   prop;
  double      shrink;
  char        *dvi_file;
  char        vars[1024];
  int         i;
#ifdef __FreeBSD__
  struct rusage  ru;
#endif

  shrink = 1.0;
  dvi_file = NULL;
  vflibcap      = VFLIBCAP;
  dpi           = DPI;
  kpathsea_mode = KPATHSEA_MODE;

  argc--;
  argv = &argv[1];
  while (argc >= 1){
    if (strcmp(argv[0], "-s") == 0){
      shrink = atof(argv[1]);
      argc -= 2;
      argv = &argv[2];
    } else if ((strcmp(argv[0], "-h") == 0)
	       || (strcmp(argv[0], "--h") == 0)
	       || (strcmp(argv[0], "-help") == 0)
	       || (strcmp(argv[0], "--help") == 0)){
      usage();
      exit(0);
    } else if (strcmp(argv[0], "-v") == 0){
      vflibcap = argv[1];
      argc -= 2;
      argv = &argv[2];
    } else if (strcmp(argv[0], "-dpi") == 0){
      dpi = argv[1];
      argc -= 2;
      argv = &argv[2];
    } else if (strcmp(argv[0], "-mode") == 0){
      kpathsea_mode = argv[1];
      argc -= 2;
      argv = &argv[2];
    } else if (strcmp(argv[0], "-df") == 0){
      opt_font_delay = 1;
      argc--; argv++;
    } else if (strcmp(argv[0], "-ag") == 0){
      opt_async_gs = 1;
      argc--; argv++;
    } else if (strcmp(argv[0], "-dp") == 0){
      opt_draw_page = 1;
      argc--; argv++;
    } else if (argv[0][0] == '-'){
      fprintf(stderr, "Unknown option: %s\n", argv[0]);
      exit(1);
    } else {
      dvi_file = argv[0];
      break;
    }
  }


  if (dvi_file == NULL){
    fprintf(stderr, "Error: No DVI file.\n");
    usage();
    exit(1);
  }

  sprintf(vars, "TeX_DPI=%s, TeX_KPATHSEA_MODE=%s",
	  dpi, kpathsea_mode);
  if (DVI_INIT(vflibcap, vars) < 0){
    fprintf(stderr, "Error: Cannot initialize dvilib.\n");
    exit(1);
  }
  if ((dev = device_create()) == NULL){
    fprintf(stderr, "Error: Cannot initialize device.\n");
    exit(1);
  }
  prop = DVI_PROPERTY_ALLOC();
  if (opt_font_delay == 1)
    DVI_PROPERTY_SET(prop, DVI_PROP_DELAYED_FONT_OPEN);
  if (opt_async_gs == 1)
    DVI_PROPERTY_SET(prop, DVI_PROP_ASYNC_GS_INVOCATION);

  dvi = NULL;
  i = 0;
  for (;;){
    i++;
    if ((i % 10) == 1){
#ifdef __FreeBSD__
      printf("\n");
      if (getrusage(RUSAGE_SELF, &ru) >= 0){
	printf(" maxrss=%ldK", ru.ru_maxrss);
      }
#endif
      printf("\n% 4d  ", i);
    }
    printf("*");fflush(stdout);
    if ((dvi = DVI_CREATE(dev, dvi_file, prop)) == NULL){
      printf("Error: Cannot open DVI file.\n");
      exit(1);
    }
    DVI_OPEN_FONT(dvi, dev);

    if (opt_draw_page == 1){
      DVI_DRAW_PAGE(dvi, dev, 1, shrink);
    }

    DVI_DISPOSE(dvi, dev);
    dvi = NULL;
  }

  return 0;
}

void usage(void)
{
  printf("Usage: test [options] dvi_file\n");
  printf("Options: \n");
  printf("         -v VFLIBCAP\n");
  printf("         -dpi DPI\n");
  printf("         -mode MODE  kpathsea device mode name (e.g., \"cx\")\n");
  printf("         -dp         draw page\n");
  printf("         -df         delay font open\n");
  printf("         -ag         invoke ghostscript asynchronously\n");
}



void
dvi_dump_bitmap(DVI_BITMAP bm)
{
  static unsigned char bit_table[] = {
    0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01};
  unsigned char *p, *p0;
  int           x, y;
  
  if ((p = bm->bitmap) == NULL)
    return;
  for (x = 0; x < bm->width+2; x++)
    putchar('-');
  putchar('\n');
  for (y = 0; y < bm->height; y++){
    p0 = p;
    --p;
    putchar('|');
    for (x = 0; x < bm->width; x++){
      if (x%8 == 0)
	p++;
      if ((*p & bit_table[x%8]) != 0)
	putchar('@');
      else
	putchar('.');
    }
    p = p0 + bm->raster;
    putchar('|');
    putchar('\n');
  }
  for (x = 0; x < bm->width+2; x++)
    putchar('-');
  putchar('\n');
}



int 
dvidev_poll(DVI_DEVICE dev, DVI dvi, int poll_type)
{
  return 0; /* no interruption */
}

void 
dvidev_message(DVI_DEVICE dev, DVI dvi, char *fmt, ...)
{
  va_list ap;

  va_start(ap, fmt);
  vprintf(fmt, ap);
  va_end(ap);
  printf("\n");
}

void 
dvidev_put_rectangle(DVI_DEVICE dev, DVI dvi, 
		     long x, long y, long w, long h)
{
  printf("Put Rectangle (%ld,%ld)-(%ld,%ld) W=%ld,H=%ld\n", 
	 x, y, x+w, y+h, w, h);
}
void 
dvidev_put_bitmap(DVI_DEVICE dev, DVI dvi, DVI_BITMAP bm, 
		  int font_id, long id2, long code_point,
		  long x, long y)
{
  if (bm != NULL){
    printf("Put Bitmap (%ld,%ld)  W=%ld, H=%ld\n",
	   x, y, bm->width, bm->height);
#if 0
    dvi_dump_bitmap(bm);
#endif
  }
}

char*
dvidev_gs_path(DVI_DEVICE dev, DVI dvi)
{
  return "/usr/local/bin/gs";
}

DVI_DEVICE
device_create(void)
{
  DVI_DEVICE  dev;

  if ((dev = DVI_DEVICE_ALLOC()) == NULL)
    return NULL;
  dev->h_dpi = atoi(dpi);
  dev->v_dpi = atoi(dpi);
  dev->put_rectangle        = dvidev_put_rectangle;
  dev->put_bitmap           = dvidev_put_bitmap;
  dev->gs_program_path      = dvidev_gs_path;
  dev->message_advice       = dvidev_message;
  dev->message_warning      = dvidev_message;
  dev->message_error        = dvidev_message;
  dev->message_fatal        = dvidev_message;
  dev->device_polling       = dvidev_poll;
  return dev;
}

/*EOF*/
