/******************************************************************************
*   TinTin++                                                                  *
*   Copyright (C) 2004 (See CREDITS file)                                     *
*                                                                             *
*   This program is protected under the GNU GPL (See COPYING)                 *
*                                                                             *
*   This program is free software; you can redistribute it and/or modify      *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with this program; if not, write to the Free Software               *
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
*******************************************************************************/

/*********************************************************************/
/* file: llist.c - linked-list datastructure                         */
/*                             TINTIN III                            */
/*          (T)he K(I)cki(N) (T)ickin D(I)kumud Clie(N)t             */
/*                     coded by peter unold 1992                     */
/*********************************************************************/

#include "tintin.h"

/*
	init list - return: ptr to listhead
*/

struct listroot *init_list()
{
	struct listroot *listhead;

	if ((listhead = calloc(1, sizeof(struct listroot))) == NULL)
	{
		fprintf(stderr, "couldn't alloc listhead\n");
		exit(1);
	}
	listhead->flags = LIST_FLAG_DEFAULT;
	return listhead;
}

/*
	kill list - run throught list and free nodes
*/

void kill_list(struct listroot *listhead)
{
	struct listnode *node;

	while ((node = listhead->f_node))
	{
		UNLINK(node, listhead->f_node, listhead->l_node, next, prev);

		free(node->left);
		free(node->right);
		free(node->pr);
		free(node);
	}

	listhead->count = 0;
}

/*
	This function will clear all lists associated with a session
*/

DO_COMMAND(do_killall)
{
	int cnt;

	push_call("kill_all(%p,%p)",ses,arg);

	for (cnt = 0 ; cnt < LIST_MAX ; cnt++)
	{
		kill_list(ses->list[cnt]);

		if (arg == NULL)
		{
			free(ses->list[cnt]);
		}
	}

	if (arg != NULL)
	{
		tintin_puts2("#KILLALL: LISTS CLEARED.", ses);
	}
	pop_call();
	return ses;
}

/*
	make a copy of a list - return: ptr to copy
*/

struct listroot *copy_list(struct listroot *sourcelist, int mode)
{
	struct listroot *resultlist;
	struct listnode *result;

	resultlist = init_list();

	for (result = sourcelist->f_node ; result ; result = result->next)
	{
		insertnode_list(resultlist, result->left, result->right, result->pr, mode);
	}
	resultlist->flags = sourcelist->flags;
	return resultlist;
}

/*
	create a node containing the ltext, rtext fields and stuff it
	into the list - in lexicographical order, or by numerical
	priority (dependent on mode) - Mods by Joann Ellsworth 2/2/94
	- More mods by Scandum
*/

void insertnode_list(struct listroot *listhead, const char *ltext, const char *rtext, const char *prtext, int mode)
{
	struct listnode *nptr, *newnode;

	if ((newnode = calloc(1, sizeof(struct listnode))) == NULL)
	{
		fprintf(stderr, "couldn't calloc listhead");
		exit(1);
	}

	newnode->left  = strdup(ltext);
	newnode->right = strdup(rtext);
	newnode->pr    = strdup(prtext);

	nptr = listhead->f_node;

	listhead->count++;

	switch (mode)
	{
		case PRIORITY:
			for (nptr = listhead->f_node ; nptr ; nptr = nptr->next)
			{
				if (atoi(prtext) < atoi(nptr->pr))
				{
					INSERT_LEFT(newnode, nptr, listhead->f_node, next, prev);
					return;
				}
			}
			LINK(newnode, listhead->f_node, listhead->l_node, next, prev);
			break;

		case ALPHA:
			for (nptr = listhead->f_node ; nptr ; nptr = nptr->next)
			{
				if (strcmp(ltext, nptr->left) < 0)
				{
					INSERT_LEFT(newnode, nptr, listhead->f_node, next, prev);
					return;
				}
			}
			LINK(newnode, listhead->f_node, listhead->l_node, next, prev);
			break;

		case APPEND:
			LINK(newnode, listhead->f_node, listhead->l_node, next, prev);
			break;

		default:
			tintin_printf2(NULL, "insertnode_list: bad mode: %d", mode);
			break;
	}
}


void updatenode_list(struct listroot *listhead, const char *ltext, const char *rtext, const char *prtext, int mode)
{
	struct listnode *node;

	for (node = listhead->f_node ; node ; node = node->next)
	{
		if (strcmp(node->left, ltext) == 0)
		{
			if (mode == PRIORITY)
			{
				if (strcmp(node->pr, prtext) != 0)
				{
					deletenode_list(listhead, node);
					break;
				}
			}
			else
			{
				if (strcmp(node->pr, prtext) != 0)
				{
					free(node->pr);
					node->pr = strdup(prtext);
				}
			}

			if (strcmp(node->right, rtext) != 0)
			{
				free(node->right);
				node->right = strdup(rtext);
			}
			return;
		}
	}
	insertnode_list(listhead, ltext, rtext, prtext, mode);
}

void deletenode_list(struct listroot *listhead, struct listnode *node)
{
	push_call("deletenode_list(%p,%p)",listhead,node);

	if ((node->next == NULL && node != listhead->l_node)
	||  (node->prev == NULL && node != listhead->f_node))
	{
		tintin_puts2("#ERROR: delete_nodelist: unlink error.", NULL);		
	}
	else
	{
		if (node == listhead->update)
		{
			listhead->update = node->next;
		}

		UNLINK(node, listhead->f_node, listhead->l_node, next, prev);

		free(node->left);
		free(node->right);
		free(node->pr);
		free(node);

		listhead->count--;
	}
	pop_call();
	return;
}

/*
	search for a node containing the ltext in left-field
*/

struct listnode *searchnode_list(struct listroot *listhead, const char *cptr)
{
	struct listnode *node;

	for (node = listhead->f_node ; node ; node = node->next)
	{
		if (!strcmp(node->left, cptr))
		{
			return node;
		}
	}
	return NULL;
}

/*
	search for a node that has cptr as a beginning
*/

struct listnode *searchnode_list_begin(struct listroot *listhead, const char *cptr, int mode)
{
	struct listnode *node;
	int len;

	len = strlen(cptr);

	for (node = listhead->f_node ; node ; node = node->next)
	{
		if (strncmp(node->left, cptr, len) != 0)
		{
			continue;
		}

		if (node->left[len] == ' ' || node->left[len] == '\0')
		{
			return node;
		}
	}
	return NULL;
}

/*
	show contens of a node on screen
*/

void shownode_list(struct session *ses, struct listnode *nptr, int mode)
{
	char buf[BUFFER_SIZE * BUFFER_SAFE], out[BUFFER_SIZE * BUFFER_SAFE], *pti, *pto;
	int level = 0;

	switch (mode)
	{
		case PRIORITY:
			sprintf(buf, "{%s}<168>={%s} <168>@ {%s}", nptr->left, nptr->right, nptr->pr);
			break;
		default:
			sprintf(buf, "{%s}<168>={%s}", nptr->left, nptr->right);
			break;
	}

	pto = out;
	pti = buf;

	while (*pti)
	{
		switch (*pti)
		{
			case DEFAULT_OPEN:
				sprintf(pto, "<088><1%d8>%c<088>", level % 5 + 1, DEFAULT_OPEN);
				level++;
				pti++;
				pto += strlen(pto);
				break;

			case DEFAULT_CLOSE:
				level--;
				sprintf(pto, "<088><1%d8>%c<088>", level % 5 + 1, DEFAULT_CLOSE);
				pti++;
				pto += strlen(pto);
				break;

			case '\\':
				*pto++ = *pti++;
				if (*pti)
				{
					*pto++ = *pti++;
				}
				break;

			case ';':
				sprintf(pto, "<088><168>;<088>");
				pti++;
				pto += strlen(pto);

			default:
				*pto++ = *pti++;
				break;
		}
	}
	substitute(ses, out, buf, SUB_COL);

	tintin_puts2(buf, ses);
}

/*
	list contens of a list on screen
*/

void show_list(struct session *ses, struct listroot *listhead, int mode)
{
	struct listnode *node;

	for (node = listhead->f_node ; node ; node = node->next)
	{
		shownode_list(ses, node, mode);
	}
}


int show_node_with_wild(struct session *ses, struct listroot *listhead, const char *cptr, int mode)
{
	struct listnode *node;
	int flag = FALSE;

	for (node = listhead->f_node ; node ; node = node->next)
	{
		if (regexp(cptr, node->left))
		{
			shownode_list(ses, node, mode);
			flag = TRUE;
		}
	}
	return flag;
}

struct listnode *search_node_with_wild(struct listroot *listhead, const char *cptr)
{
	struct listnode *node;

	for (node = listhead->f_node ; node ; node = node->next)
	{
		if (regexp(cptr, node->left))
		{
			return node;
		}
	}
	return NULL;
}


/*
	create a node containing the ltext, rtext fields and place at the
	end of a list - as insertnode_list(), but not alphabetical
*/

void addnode_list(struct listroot *listhead, const char *ltext, const char *rtext, const char *prtext)
{
	struct listnode *newnode;

	if ((newnode = calloc(1, sizeof(struct listnode))) == NULL)
	{
		fprintf(stderr, "couldn't calloc listhead");
		exit(1);
	}
	newnode->left  = calloc(1, strlen(ltext)  + 1);
	newnode->right = calloc(1, strlen(rtext)  + 1);
	newnode->pr    = calloc(1, strlen(prtext) + 1);

	sprintf(newnode->left,  "%s", ltext);
	sprintf(newnode->right, "%s", rtext);
	sprintf(newnode->pr,    "%s", prtext);

	LINK(newnode, listhead->f_node, listhead->l_node, next, prev);

	listhead->count++;
}

int count_list(struct listroot *listhead)
{
	struct listnode *node;
	int cnt = 0;
	for (node = listhead->f_node ; node ; node = node->next)
	{
		cnt++;
	}
	return cnt;
}
