namespace eval rmdir {
    variable rmdir

    # Define the rmdir array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set rmdir {
	list.reset ""
	list.clear ""
	entry.dir ""
	dialog ""
	dir ""
	rm_parent ""
    }
}

# rmdir::create --
#
#   Method to create the dialog box for the rmdir command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc rmdir::create { } {
    global tkWorld
    variable rmdir

    # Put the focus on the rmdir dialog if it is already open.
    if [winfo exists $rmdir(dialog)] {
	switch -- [wm state $rmdir(dialog)] {
	    normal {
		raise $rmdir(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $rmdir(dialog)
	    }
	}
	focus $rmdir(dialog)
	return
    } else {
	set rmdir(dialog) [dialog::create .rmdir Rmdir]
    }

    # The first tab has the dir information, along with the make
    # parent command line option.
    set tab1 [tabnotebook::page [dialog::interior \
	    $rmdir(dialog)] "Options"]

    # Use a frame to encapsulate the dir options so that
    # the frame can be centered accross the grid columns.
    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]
    label $f1.dir_label \
	    -text "Directory" \
	    -width 10
    set rmdir(entry.dir) [entry $f1.entry_dir \
	    -width 35 \
	    -textvariable rmdir::rmdir(dir)]
    grid $f1.dir_label $f1.entry_dir \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    # Now build the individual checkbutton options available to
    # the user.
    set f3 [frame $tab1.f3 \
	    -class TabnotebookFrame]
    checkbutton $f3.rm_parent \
	    -text "Remove Parent (Be very careful with this!)" \
	    -variable rmdir::rmdir(rm_parent) \
	    -onvalue "p" \
	    -offvalue ""
    grid $f3.rm_parent \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    # Build the first tab.
    pack $f1 $f3 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Duhhhhhh......
    focus $rmdir(entry.dir)

    # Define the lists for the reset and clear methods
    set rmdir(list.reset) "rm_parent"
    set rmdir(list.clear) "dir"
}

# rmdir::ok --
#
#   Method to insert the command the user has created into the CC.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc rmdir::ok { } {
    global tkWorld
    variable rmdir

    # Insert the working directory in each of the users
    # typed in directory names.
    set dlist ""
    foreach d $rmdir(dir) {
	if [regexp $tkWorld(working_dir) $d] {
	    append dlist "$d "
	} else {
	    append dlist "$tkWorld(working_dir)/$d "
	}
    }

    # Check to remove the parent directory
    set cmd_arg ""
    if [string length $rmdir(rm_parent)] {
	set cmd_arg "-$rmdir(rm_parent)"
    }

    # Insert the Tcl command list in the Command Center with the
    # proper formatting of a space between each argument on the
    # command line.  If there are no options given by the user,
    # then don't display it in the CC.

    if [string length $cmd_arg] {
	set cmd_arg [string trimright $cmd_arg " "]
	$tkWorld(cmd_center) insert insert \
		"rmdir $cmd_arg $dlist"
    } else {
	$tkWorld(cmd_center) insert insert \
		"rmdir $dlist"
    }

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled
}

# rmdir::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc rmdir::reset { } {
    variable rmdir

    # It is easier to list the elements to exclude rather than list
    # all of the values to reset.
    foreach x $rmdir(list.reset) {
	set rmdir($x) ""
    }
}

# rmdir::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc rmdir::clear { } {
    variable rmdir

    # Reset the data structure elements and bg/fg.
    foreach x $rmdir(list.clear) {
	set rmdir($x) ""
	$rmdir(entry.$x) configure -bg #ffffff -fg #000000
    }

    focus $rmdir(entry.dir)
}

# rmdir::help --
#
#   Method to invoke the Rmdir Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc rmdir::help { } {
    global tkWorld

    help::create "help/rmdir.html" "Rmdir Command Help"
}

# rmdir::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc rmdir::close { } {
    variable rmdir
    
    balloonhelp::cancel
    destroy $rmdir(dialog)
}
