# ----------------------------------------------------------------------
#  EXAMPLE: simple panedwindow facility
# ----------------------------------------------------------------------
#  Effective Tcl/Tk Programming
#    Mark Harrison, DSC Communications Corp.
#    Michael McLennan, Bell Labs Innovations for Lucent Technologies
#    Addison-Wesley Professional Computing Series
# ======================================================================
#  Copyright (c) 1996-1997  Lucent Technologies Inc. and Mark Harrison
# ======================================================================
#
# Modified to use the Tk 8.0 namespace facility by Wes Bailey.
#
# Note: This paned window manager only manages the y direction.

namespace eval panedwindow {
    variable pwInfo

    option add *Panedwindow.grip.cursor sb_v_double_arrow widgetDefault
    option add *Panedwindow.divider.cursor sb_v_double_arrow widgetDefault
}

# panedwindow::create  --
#
#   Method to create the parent level frame containing to panes.
#
# Args
#
#   win    - The parent level frame.
#   width  - The width in screen units of the parent frame.
#   height - The height of the parent frame.
#
# Returns
#
#   win - Returns the parent level frame object name.

proc panedwindow::create { win {width 700} {height 400} {pane1_pct 0.25} } {
    variable pwInfo

    # Create the parent frame which contains the 2 panes.
    if {$win == "."} {
	error "Cannot use panedwindow::create with . parent"
    } else {
	frame $win \
		-class Panedwindow \
		-width $width \
		-height $height
    }

    # Define the first pane.
    frame $win.pane1
    place $win.pane1 \
	    -relx 0.5 \
	    -rely 0 \
	    -anchor n \
	    -relwidth 1.0 \
	    -relheight $pane1_pct

    # Define the second pane.
    frame $win.pane2
    place $win.pane2 \
	    -relx 0.5 \
	    -rely 1.0 \
	    -anchor s \
	    -relwidth 1.0 \
	    -relheight [expr 1.0 - $pane1_pct]

    # Build the divider between the two paned frames.
    frame $win.divider \
	    -height 2 \
	    -borderwidth 1 \
	    -relief sunken
    place $win.divider \
	    -relx 0.5 \
	    -rely $pane1_pct \
	    -relwidth 1.0 \
	    -anchor c

    # Build the grip on the divider.
    frame $win.grip \
	    -width 10 \
	    -height 10 \
	    -borderwidth 1 \
	    -relief raised
    place $win.grip \
	    -relx 0.9810 \
	    -rely $pane1_pct \
	    -anchor c

    bind $win.grip <ButtonPress-1> "panedwindow::grab $win"
    bind $win.grip <B1-Motion> "panedwindow::drag $win %Y"
    bind $win.grip <ButtonRelease-1> "panedwindow::drop $win %Y"

    return $win
}

# panedwindow::grab  --
#
#   Method to make the grip look pressed
#
# Args
#
#   win - The object name of the parent window.
#
# Returns
#
#   None.

proc panedwindow::grab { win } {
    # No private variables.

    $win.grip configure \
	    -relief sunken
}

# panedwindow::drag  --
#
#   Method to move the divider up/down and adjust the corresponding
#   pane window frames.
#
# Args
#
#   win - The object name of the parent window.
#   y   - The y coordinate of the mouse when dropped.
#
# Returns
#
#   frac - The fraction of the change in pane displacements.

proc panedwindow::drag { win y } {
    # No private variables.

    # Catch the coordinates of the mouse placement.
    set realY [expr $y - [winfo rooty $win]]
    set Ymax [winfo height $win]
    set frac [expr double($realY)/$Ymax]

    # Define the minimum displacements that are allowed.
    if {$frac < 0.15} {
        set frac 0.15
    }
    if {$frac > 0.85} {
        set frac 0.85
    }

    # Move the divider and grip.
    place $win.divider \
	    -rely $frac
    place $win.grip \
	    -rely $frac

    return $frac
}

# panedwindow::rrop  --
#
#   Method to drop the grip and move the divider.
#
# Args
#
#   win - The object name of the parent frame.
#   y   - The y coordinate of the dropped mouse position.
#
# Returns
#
#   None.

proc panedwindow::drop { win y } {
    # No private variables.

    # Determine the fractional amount to move the panes and divider.
    set frac [panedwindow::drag $win $y]

    # Actually move the objects around.
    panedwindow::divide $win $frac

    # Make the grip look done.
    $win.grip configure \
	    -relief raised
}

# panedwindow::divide  --
#
#   Method to divide the parent frame.
#
# Args
#
#   win - The object name of the parent frame.
#   frac - The amount to displace the panes.
#
# Returns
#
#   None.

proc panedwindow::divide { win frac } {
    # No private variables.

    # Move the divider and grip.
    place $win.divider \
	    -rely $frac
    place $win.grip \
	    -rely $frac

    # Displace the 2 panes.
    place $win.pane1 \
	    -relheight $frac
    place $win.pane2 \
	    -relheight [expr 1-$frac]
}
