// ti990.c - TI-990 Minicomputer Emulator
//
// Copyright (c) 2002, Timothy M. Stark
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// TIMOTHY M STARK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Timothy M Stark shall not
// be used in advertising or otherwise to promote the sale, use or other 
// dealings in this Software without prior written authorization from
// Timothy M Stark.

#include "ti990/ti990.h"

// TI-990/10 Minicomputer Memory Map Area
//
// 000000-0FFFFF  Up to 1 MB Memory Area
// 1FF800-1FFBFF  TPCS
// 1FFC00-1FFFFF  ROM

#define IN_RAM(addr)  ((addr) < ti99cpu->SizeRAM)
#define IN_TPCS(addr) ((addr) < TI990_ROM_BASE)
#define iN_ROM(addr)  ((addr) < TI990_ROM_END)

#define BMEM(addr) ((uint8 *)ti99cpu->RAM)[addr]
#define WMEM(addr) ((uint16 *)ti99cpu->RAM)[addr]

#define BROM(addr) ((uint8 *)ti99cpu->ROM)[addr]
#define WROM(addr) ((uint16 *)ti99cpu->ROM)[addr]

// Aligned Word Read Access
uint16 ti990_ReadW(register TI990_CPU *ti99cpu, uint32 pAddr)
{
	if (IN_RAM(pAddr))        // Main Memory
		return WMEM(pAddr >> 1);
	else if (IN_TPCS(pAddr))  // TPCS - I/O Devices
		return ti990_ReadIO(ti99cpu, pAddr, OP_WORD);
	else if (IN_ROM(pAddr))   // Boot Loader ROM
		return WROM((pAddr & ROM_MASK) >> 1);
	return 0;
}

// Aligned Word Write Access
void ti990_WriteW(register TI990_CPU *ti99cpu, uint32 pAddr, uint16 data)
{
	if (IN_RAM(pAddr))        // Main Memory
		WMEM(pAddr >> 1) = data;
	else if (IN_TPCS(pAddr))  // TPCS - I/O Devices
		ti990_WriteIO(ti99cpu, pAddr, data, OP_WORD);
//	else if (IN_ROM(pAddr))   // Boot Loader ROM
//		WROM((pAddr & ROM_MASK) >> 1) = data;
}

// Aligned Byte Read Access
uint8 ti990_ReadB(register TI990_CPU *ti99cpu, uint32 pAddr)
{
	if (IN_RAM(pAddr))        // Main Memory
		return BMEM(pAddr);
	else if (IN_TPCS(pAddr))  // TPCS - I/O Devices
		return ti990_ReadIO(ti99cpu, pAddr, OP_BYTE);
	else if (IN_ROM(pAddr))   // Boot Loader ROM
		return BROM(pAddr & ROM_MASK);
	return 0;
}

// Aligned Byte Write Access
void ti990_WriteB(register TI990_CPU *ti99cpu, uint32 pAddr, uint16 data)
{
	if (IN_RAM(pAddr))        // Main Memory
		BMEM(pAddr) = data;
	else if (IN_TPCS(pAddr))  // TPCS - I/O Devices
		ti990_WriteIO(ti99cpu, pAddr, data, OP_BYTE);
//	else if (IN_ROM(pAddr))   // Boot Loader ROM
//		BROM(pAddr & ROM_MASK) = data;
}

uint16 ti990_ReadCRU(register TI990_CPU *ti99cpu, uint16 cruAddr)
{
	return 0;
}

void ti990_WriteCRU(register TI990_CPU *ti99cpu, uint16 cruAddr, uint16 data)
{
}

// ********************************************************

void ti990_SystemTimer(void *dptr)
{
	register TI990_CPU *ti99cpu = (TI990_CPU *)dptr;

	if (ti99cpu->TickClock++ >= TI99_SECOND) {
		ti99cpu->TickClock = 0;

#ifdef DEBUG
		if (dbg_Check(DBG_IPS))
			dbg_Printf("%s: %d ips\n", ti99cpu->cpu.devName, IPS);
#else
		printf("%s: %d ips\n", ti99cpu->cpu.devName, IPS);
#endif /* DEBUG */

		IPS = 0;
	}
}

void ti990_EnableTimer(register TI990_CPU *ti99cpu)
{
	ti99cpu->TickClock          = 0;
	ti99cpu->SystemTimer.Flags |= CLK_ENABLE;
}

void ti990_DisableTimer(register TI990_CPU *ti99cpu)
{
	ti99cpu->SystemTimer.Flags &= ~CLK_ENABLE;
}

// ********************************************************

void *ti990_Create(MAP_DEVICE *newMap, int argc, char **argv)
{
	TI99_SYSTEM *ti99sys;
	TI990_CPU   *ti99cpu;
	CLK_QUEUE   *timer;

	if (ti99cpu = (TI990_CPU *)calloc(1, sizeof(TI990_CPU))) {
		// Set up function calls
		ReadW    = ti990_ReadW;
		WriteW   = ti990_WriteW;
		ReadB    = ti990_ReadB;
		WriteB   = ti990_WriteB;
		ReadCRU  = ti990_ReadCRU;
		WriteCRU = ti990_WriteCRU;

		EnableTimer  = ti990_EnableTimer;
		DisableTimer = ti990_DisableTimer;

		// Set up a link between CPU and System device.
		ti99sys = (TI99_SYSTEM *)newMap->devParent->Device;
		ti99cpu->System    = ti99sys;
		ti99sys->Processor = (TI99_CPU *)ti99cpu;

		// Set up a system timer
		timer           = &ti99cpu->SystemTimer;
		timer->Next     = NULL;
		timer->Flags    = CLK_REACTIVE;
		timer->outTimer = TI99_TICK;
		timer->nxtTimer = TI99_TICK;
		timer->Device   = ti99cpu;
		timer->Execute  = ti990_SystemTimer;

		// Now enable system timer.
		ts10_SetRealTimer(timer);

		// Build opcode table for execution.
		ti99_BuildCPU((TI99_CPU *)ti99cpu, 0);

		// Finally, set up device identification and return.
		ti99cpu->cpu.devName    = newMap->devName;
		ti99cpu->cpu.keyName    = newMap->dtName;
		ti99cpu->cpu.emuName    = newMap->emuName;
		ti99cpu->cpu.emuVersion = newMap->emuVersion;
		newMap->Device          = ti99cpu;
	}
	return ti99cpu;
}

int ti990_Boot(MAP_DEVICE *map, int argc, char **argv)
{
	register TI990_CPU *ti99cpu = (TI990_CPU *)map->Device;

	// Power-On State - Load WP and PC register.
	WP = ReadW(ti99cpu, 0xFFFC);
	PC = ReadW(ti99cpu, 0xFFFE);

	printf("%s: Load WP = >%04X, PC = >%04X\n",
		ti99cpu->cpu.devName, WP, PC);

	// Enable TMS9900 to run.
	ti99cpu->cpu.State = TI99_RUN;
	emu_State          = TI99_RUN;

	return TI99_OK;
}

extern COMMAND ti990_Commands[];
//extern COMMAND ti990_SetCommands[];
//extern COMMAND ti990_ShowCommands[];

DEVICE ti990_System =
{
	TI990_KEY,
	TI990_NAME,
	TI990_VERSION,
	NULL,
	DF_USE,
	DT_PROCESSOR,

	ti990_Commands, NULL, NULL,

	// Function Calls
	ti990_Create,    // Create Routine
	NULL,            // Configure Routine
	NULL,            // Delete Routine
	NULL,            // Reset Routine
	NULL,            // Attach Routine
	NULL,            // Detach Routine
	NULL,            // Info Routine
	ti990_Boot,      // Boot Routine
	ti99_Execute,    // Execute Routine
#ifdef DEBUG
	NULL,            // Debug Routine
#endif /* DEBUG */
};
