// cpu_queue.c - VAX Queue Instructions
//
// Written by
//  Timothy Stark <sword7@speakeasy.org>
//
// This file is part of the TS10 Emulator.
// See ReadMe for copyright notice.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "vax/defs.h"

// Due to lack of documentation about inner workings of queue 
// instructions so that I had quoted them from Bob's simh VAX
// emulator to pass all HCORE tests.  When I find complete
// specs about queue instructions, I will write my own code
// here.  That's why VAX Architecture Reference Handbook (First
// Edition) did not give much information.   I apologize for that.

INSDEF(vax, INSQHI)
{
	int32 entry  = OP0;
	int32 header = OP1;
	int32 a, t;
	int32 at;

	// Header must be quadword aligned.
	// header and Entry must not be equal.
	// If so, go to reserved operand fault.
	if ((entry == header) || ((entry | header) & 0x07))
		RSVD_OPND_FAULT;

	vax_Read(entry, OP_BYTE, WA);
	a = vax_Read(header, OP_LONG, WA);

	if (a & 06)
		RSVD_OPND_FAULT;
	if (a & 01) {
		CC = CC_C;
	} else {
		vax_Write(header, a | 1, OP_LONG, WA);
		a += header;
		if (vax_Test(a, WA, &t) < 0)
			vax_Write(header, a - header, OP_LONG, WA);
		vax_Write(a + 4, entry - a, OP_LONG, WA);
		vax_Write(entry, a - entry, OP_LONG, WA);
		vax_Write(entry + 4, header - entry, OP_LONG, WA);
		vax_Write(header, entry - header, OP_LONG, WA);

		CC = (a == header) ? CC_Z : 0;
	}
}

INSDEF(vax, INSQTI)
{
	int32 entry  = OP0;
	int32 header = OP1;
	int32 a, c, t;

	// Header must be quadword aligned.
	// header and Entry must not be equal.
	// If so, go to reserved operand fault.
	if ((header == entry) || ((header | entry) & 07))
		RSVD_OPND_FAULT;

	vax_Read(entry, OP_BYTE, WA);
	a = vax_Read(header, OP_LONG, WA);
	if (a == 0) {
		// Treat as INSQHI instruction.
		INSCALL(vax, INSQHI);
		return;
	}
	if (a & 06)
		RSVD_OPND_FAULT;
	if (a & 01) {
		// Busy, set CC as 0001 (---C)
		CC = CC_C;
	} else {
		vax_Write(header, a | 1, OP_LONG, WA);
		c = vax_Read(header + 4, OP_LONG, RA) + header;
		if (c & 07) {
			vax_Write(header, a, OP_LONG, WA);
			RSVD_OPND_FAULT;
		}
		if (vax_Test(c, WA, &t) < 0)
			vax_Write(header, a, OP_LONG, WA);
		vax_Write(c, entry - c, OP_LONG, WA);
		vax_Write(entry, header - entry, OP_LONG, WA);
		vax_Write(entry + 4, c - entry, OP_LONG, WA);
		vax_Write(header + 4, entry - header, OP_LONG, WA);
		vax_Write(header, a, OP_LONG, WA);

		// Clear all condition codes
		CC = 0;
	}
}

// INSQUE - Insert Entry in Queue
//
// Format:
//   opcode entry.ab, pred.ab
//
// Operation:
//   if {all memory accesses can be completed} then
//     begin
//       (entry) <- (pred);       ! forward link of entry
//       (entry + 4) <- pred;     ! backward link of entry
//       ((pred) + 4) <- entry;   ! backward link of successor
//       (pred) <- entry;         ! forward link of predecessor
//     end

INSDEF(vax, INSQUE)
{
	register int32 entry = OP0;
	register int32 pred  = OP1;
	register int32 succ;

	// Check memory accesses first
	succ = vax_Read(pred, OP_LONG, WA);
	vax_Read(succ + 4, OP_LONG, WA);
	vax_Read(entry + 4, OP_LONG, WA);
	vax_Write(entry, succ, OP_LONG, WA);
	vax_Write(entry + 4, pred, OP_LONG, WA);
	vax_Write(succ + 4, entry, OP_LONG, WA);
	vax_Write(pred, entry, OP_LONG, WA);

	// Update condition codes
	CC_CMP_L(succ, pred);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("INSQUE: Enqueue %08X to %08X with next %08X\n",
			entry, pred, succ);
#endif /* DEBUG */
}

INSDEF(vax, REMQHI)
{
	int32 header = OP0;
	int32 ar, a, b, t;

	// Header must be quadword aligned.
	// Also check address for write access and
	// must not be equal to header.
	if (header & 07)
		RSVD_OPND_FAULT;
	if (OP1 < 0) {
		if (OP2 == header)
			RSVD_OPND_FAULT;
		vax_Read(OP2, OP_LONG, WA);
	}

	ar = vax_Read(header, OP_LONG, WA);
	if (ar & 06)
		RSVD_OPND_FAULT;
	if (ar & 01) {
		CC = CC_V|CC_C;
	} else {
		a = ar + header;
		if (ar) {
			vax_Write(header, ar | 1, OP_LONG, WA);
			if (vax_Test(a, RA, &t) < 0)
				vax_Write(header, ar, OP_LONG, WA);
			b = vax_Read(a, OP_LONG, RA) + a;
			if (b & 07) {
				vax_Write(header, ar, OP_LONG, WA);
				RSVD_OPND_FAULT;
			}
			if (vax_Test(b, RA, &t) < 0)
				vax_Write(header, ar, OP_LONG, WA);
			vax_Write(b + OP_LONG, header - b, OP_LONG, WA);
			vax_Write(header, b - header, OP_LONG, WA);
		}

		LSTORE(OP1, OP2, a);

		// Update condition codes
		if (ar == 0)
			CC = CC_Z|CC_V;
		else if (b == header)
			CC = CC_Z;
		else
			CC = 0;
	}
}

INSDEF(vax, REMQTI)
{
	int32 header = OP0;
	int32 ar, b, c, t;

	// Header must be quadword aligned.
	// Also check address for write access and
	// must not be equal to header.
	if (header & 07)
		RSVD_OPND_FAULT;
	if (OP1 < 0) {
		if (header == OP2)
			RSVD_OPND_FAULT;
		vax_Read(OP2, OP_LONG, WA);
	}

	ar = vax_Read(header, OP_LONG, WA);
	if (ar & 06)
		RSVD_OPND_FAULT;
	if (ar & 01) {
		// Busy, set CC as 0011 (--VC)
		CC = CC_V|CC_C;
	} else {
		if (ar) {
			vax_Write(header, ar | 1, OP_LONG, WA);
			c = vax_Read(header + 4, OP_LONG, RA);
			if (ar == c) {
				vax_Write(header, ar, OP_LONG, WA);
				INSCALL(vax, REMQHI);
				return;
			}
			if (c & 07) {
				vax_Write(header, ar, OP_LONG, WA);
				RSVD_OPND_FAULT;
			}
			c += header;
			if (vax_Test(c + 4, RA, &t) < 0)
				vax_Write(header, ar, OP_LONG, WA);
			b = vax_Read(c + 4, OP_LONG, RA) + c;
			if (b & 07) {
				vax_Write(header, ar, OP_LONG, WA);
				RSVD_OPND_FAULT;
			}
			if (vax_Test(b, WA, &t) < 0)
				vax_Write(header, ar, OP_LONG, WA);
			vax_Write(b, header - b, OP_LONG, WA);
			vax_Write(header + 4, b - header, OP_LONG, WA);
			vax_Write(header, ar, OP_LONG, WA);
		} else
			c = header;

		LSTORE(OP1, OP2, c);

		CC = (ar == 0) ? (CC_Z|CC_V) : 0;
	}
}

INSDEF(vax, REMQUE)
{
	int32 entry = OP0;
	int32 succ, pred;

	succ = vax_Read(entry, OP_LONG, RA);
	pred = vax_Read(entry + 4, OP_LONG, RA);

	CC_CMP_L(succ, pred);
	if (entry != pred) {
		vax_Read(succ + 4, OP_LONG, WA);
		if (OP1 < 0)
			vax_Read(OP2, OP_LONG, WA);
		vax_Write(pred, succ, OP_LONG, WA);
		vax_Write(succ + 4, pred, OP_LONG, WA);
	} else
		CC |= CC_V;

	LSTORE(OP1, OP2, entry);

#ifdef DEBUG
	if (dbg_Check(DBG_TRACE|DBG_DATA))
		dbg_Printf("REMQUE: Dequeue %08X from previous %08X and next %08X\n",
			entry, pred, succ);
#endif /* DEBUG */
}
