#include "ArrayType.hh"
#include "Types.hh"
#include "ElementAssociation.hh"
#include "VHDLKernel.hh"
#include "UniversalInteger.hh"
#include "std_standardPkg.hh"

using std::ends;

ArrayInfo* ArrayType::rangeInfo = NULL;

ArrayType::ArrayType() : VHDLType( false ),
			 isCompositeResolvedType( false ),
			 resolutionFunctionId( -1 ){}

ArrayType::ArrayType( bool alias ) : VHDLType(alias),
				     isCompositeResolvedType( false ),
				     resolutionFunctionId( -1 ){}

ArrayType::ArrayType( ObjectBase::ObjectType objType,
		      const ArrayTypeInfo &typeInfo,
		      ResolutionFnId_t resolveFnId )
  : VHDLType(false) {
  
  arrayInfo = typeInfo;
  resolutionFunctionId = resolveFnId;
  
  if (typeInfo.get_dimensions() > 1) {
    // This is a n-d array...so, setup subArrayInfo, also...
    vector<TypeInfo *> subRanges( typeInfo.getRanges().begin() + 1,
				  typeInfo.getRanges().end() );
    subArrayInfo = ArrayTypeInfo( typeInfo.getElementTypeInfo(), 
				  typeInfo.isUnconstrainedArrayType(), 
				  typeInfo.isCompositeResolvedType(), 
				  typeInfo.getResolutionFunctionId(), 
				  subRanges );
  }
  for( int i = 0; 
       i < typeInfo.get_dimensions(); 
       i++ ){
    bounds.push_back( typeInfo.getBounds(i) );
  }
  
  buildArray( objType );
  
  if ((isCompositeResolvedType = typeInfo.isCompositeResolvedType()) == true) {
    resolutionFunctionId    = typeInfo.getResolutionFunctionId();
  }
  else {
    if (resolveFnId != -1) {
      // This will be an anonymous subtype with a resolution function...
      resolutionFunctionId = resolveFnId;
    }
  }
  
  if (resolveFnId != -1) {
    isCompositeResolvedType = true;
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }
}

ArrayType::ArrayType( ObjectBase::ObjectType objType, 
		      const ArrayTypeInfo &typeInfo,
		      ResolutionFnId_t resolveFnId, 
		      int left, 
		      ArrayInfo::ArrayDirn_t direction, 
		      int right, 
		      const char *value ){
  ASSERT ( typeInfo.get_dimensions() == 1 );

  arrayInfo = typeInfo;
  bounds.push_back( ArrayInfo(left, direction, right) );
  buildArray( objType );

  resolutionFunctionId = resolveFnId;
  
  if ((isCompositeResolvedType = typeInfo.isCompositeResolvedType()) == true) {
    resolutionFunctionId    = typeInfo.getResolutionFunctionId();
  }
  else {
    if (resolveFnId != -1) {
      // This will be an anonymous subtype with a resolution function
      resolutionFunctionId = resolveFnId;
    }
  }
  
  if (resolveFnId != -1) {
    isCompositeResolvedType = true;
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }
  
  int elementCount = bounds[0].length();
  int counter;
  int adjustValue = arrayInfo.getElementTypeInfo()->getCharacterOffset();

  for(counter = 0; (counter < elementCount); counter++) {
    object[counter]->getObject()->updateVal(UniversalInteger((int) (value[counter] - adjustValue)));
  }
}

ArrayType::ArrayType( ObjectBase::ObjectType objType, 
		      const ArrayTypeInfo& typeInfo,
		      ResolutionFnId_t resolveFnId, 
		      const char *value ){
  ASSERT ( typeInfo.get_dimensions() == 1 );
  int    adjustValue = typeInfo.getElementTypeInfo()->getCharacterOffset();
  
  arrayInfo = typeInfo;
  ArrayInfo tempBounds = typeInfo.getBounds(0);
  resolutionFunctionId = resolveFnId;
  
  if (tempBounds.dirn() == ArrayInfo::to) {
    bounds.push_back( ArrayInfo( tempBounds.left(), 
				 tempBounds.dirn(), 
				 tempBounds.left() + strlen(value) - 1) );
  }
  else {
    bounds.push_back( ArrayInfo( tempBounds.left(), 
				 tempBounds.dirn(), 
				 tempBounds.left() - strlen(value) + 1) );
  }
  
  buildArray( objType );
  
  if ((isCompositeResolvedType = typeInfo.isCompositeResolvedType()) == true) {
    resolutionFunctionId    = typeInfo.getResolutionFunctionId();
  }
  else {
    if (resolveFnId != -1) {
      // This will be an anonymous subtype with a resolution function
      resolutionFunctionId = resolveFnId;
    }
  }
  
  if (resolveFnId != -1) {
    isCompositeResolvedType = true;
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }
  
  int elementCount = bounds[0].length();
  int counter;

  for(counter = 0; (counter < elementCount); counter++) {
    object[counter]->getObject()->updateVal(UniversalInteger((int) (value[counter] - adjustValue)));
  }
}


ArrayType::ArrayType( bool alias,
		      ObjectBase::ObjectType,
		      int left, 
		      ArrayInfo::ArrayDirn_t dirn,
		      int right,
		      const ArrayType &actual,
		      const ArrayInfo &boundsOfActual) :
  VHDLType(alias) {
  arrayInfo = actual.arrayInfo;
  
  ArrayInfo actualBounds = boundsOfActual;
  
  if( boundsOfActual.operator==(nullInfo) ){
    actualBounds = actual.get_bounds(0);
  }

  bounds.push_back( ArrayInfo(left, dirn, right) );

  int offset = abs(actualBounds.left() - actual.left(0));
  object.insert( object.begin(),
		 actual.get_array().begin() + offset,
		 actual.get_array().end() );
}

ArrayType::ArrayType( bool alias, 
		      ObjectBase::ObjectType,
		      const ArrayType &actual ) :
  VHDLType(alias) {
  
  arrayInfo = actual.arrayInfo;
  for( int i = 0; i < arrayInfo.get_dimensions(); i++ ){
    bounds.push_back( actual.get_bounds(i) );
  }

  object = actual.get_array();
}

ArrayType::ArrayType(ObjectBase::ObjectType objType, const ArrayTypeInfo& typeInfo,
		     ResolutionFnId_t resolveFnId, int left,
		     ArrayInfo::ArrayDirn_t direction, int right,
		     int noofElmntAssns, ...) {
  
  arrayInfo = typeInfo;

  bounds.push_back( ArrayInfo(left, direction, right) );
  resolutionFunctionId = resolveFnId;

  if (arrayInfo.get_dimensions() > 1) {
    subArrayInfo = ArrayTypeInfo( 1,
				  typeInfo.getElementTypeInfo(),
				  typeInfo.isUnconstrainedArrayType(),
				  typeInfo.isCompositeResolvedType(),
				  typeInfo.getResolutionFunctionId(),
				  arrayInfo.getRangeInfo(1),
				  NULL );
  }
  
  buildArray( objType );
  
  if ((isCompositeResolvedType = typeInfo.isCompositeResolvedType()) == true) {
    resolutionFunctionId    = typeInfo.getResolutionFunctionId();
  }
  else {
    if (resolveFnId != -1) {
      // This will be an anonymous subtype with a resolution function...
      resolutionFunctionId = resolveFnId;
    }
  }
  
  if (resolveFnId != -1) {
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }

  if (noofElmntAssns <= 0) {
    return;
  }

  // Okay...we need to handle one or more element associations here...

  va_list ap;
  ElementAssociation* elmtptr           = NULL;
  ElementAssociation* othersAssociation = NULL;
  ElementAssociation** elmtAssocArray   = NULL;
  char* charptr                         = NULL;
  int i = 0,      j = 0;
  int correct_index = 0;
  int temp          = 0;
  int numElems      = bounds[0].length();
  
  bool positional_association = false;
  
  charptr        = new char[bounds[0].length()];
  elmtAssocArray = (ElementAssociation* *) new char[noofElmntAssns *
						   sizeof(ElementAssociation *)];
  for (i = 0; (i < numElems); i++) {
    charptr[i] = 'U';
  }

  va_start(ap, noofElmntAssns);
  
  for(i = 0; (i < noofElmntAssns); i++) {
    elmtAssocArray[i] = va_arg(ap, ElementAssociation*);
  }
  va_end(ap);

  // Determine if this is a positional association or not
  elmtptr = elmtAssocArray[0];
  if(noofElmntAssns == get_bounds(0).length()) {
    if(elmtptr->choice.left() == elmtptr->choice.right()) {
      if(elmtptr->choice.left() == 0) {
	if(get_bounds(0).contains(0)) {
	  temp = elmtAssocArray[noofElmntAssns -1]->choice.left();
	  if(get_bounds(0).contains(temp)) {
	    positional_association = false;
	  } else {
	    positional_association = true;
	  }
	} else {
	  positional_association = true;
	}
      } else {
	positional_association = false;
      }
    } else {
      positional_association = false;
    }
  } else {
    positional_association = false;
  }
  
  for(i = 0; (i < noofElmntAssns); i++) {
    elmtptr = elmtAssocArray[i];
    if( elmtptr->choice.operator==(Others) ){
      othersAssociation = elmtptr;
      break;
    }
    else {
      if(elmtptr->choice.left() == elmtptr->choice.right()) {
	if(positional_association == true) {
	  correct_index = get_bounds(0).actualIndex(elmtptr->choice.left());
	}
	else {
	  correct_index = elmtptr->choice.left();
	}
	object[storageIndex(0, correct_index)]->assignVal(*elmtptr->value);
	charptr[storageIndex(0, correct_index)] = 'I';
      }
      else {
	if(elmtptr->choice.dirn() == ArrayInfo::to) {
	  for(j = elmtptr->choice.left(); (j <= elmtptr->choice.right()); j++) {
	    if(positional_association == true) {
	      correct_index = get_bounds(0).actualIndex(j);
	    }
	    else {
	      correct_index = j;
	    }
	    object[storageIndex(0, correct_index)]->assignVal(*(elmtptr->value));
	    charptr[storageIndex(0, correct_index)] = 'I';
	  }
	}
	else {
	  for(j = elmtptr->choice.left(); (j >= elmtptr->choice.right()); j--) {
	    if(positional_association == true) {
	      correct_index = get_bounds(0).actualIndex(j);
	    }
	    else {
	      correct_index = j;
	    }
	    object[storageIndex(0, correct_index)]->assignVal(*(elmtptr->value));
	    charptr[storageIndex(0, correct_index)] = 'I';
	  }
	}
      }
    }
  }
  
  if (othersAssociation != NULL) {
    for(i = 0; (i < numElems); i++) {
      if(charptr[i] == 'U') {
	object[i]->operator=(*(othersAssociation->value));
      }
    }
  }

  // Free memory allocated/used by intermediate data structures...
  for(i = 0; (i < noofElmntAssns); i++) {
    delete  elmtAssocArray[i]->value;
    delete  elmtAssocArray[i];
  }
  if(elmtAssocArray != NULL) {
    delete [] elmtAssocArray;
  }
  delete [] charptr;  
}


ArrayType::ArrayType(ObjectBase::ObjectType objType, const ArrayTypeInfo& typeInfo,
		     ResolutionFnId_t resolveFnId, int noofElmntAssns, ...) {
  ASSERT ( typeInfo.isUnconstrainedArrayType() == false );
  
  arrayInfo = typeInfo;
  resolutionFunctionId = resolveFnId;
  
  if (typeInfo.get_dimensions() > 1) {
    // This is a n-d array...so, setup subArrayInfo, also...
    vector<TypeInfo *> subRanges( typeInfo.getRanges().begin() + 1,
				  typeInfo.getRanges().end() );
    subArrayInfo = ArrayTypeInfo( typeInfo.getElementTypeInfo(), 
				  typeInfo.isUnconstrainedArrayType(), 
				  typeInfo.isCompositeResolvedType(), 
				  typeInfo.getResolutionFunctionId(), 
				  subRanges );
  }
  
  for(int i = 0; i < arrayInfo.get_dimensions(); i++) {
    bounds.push_back( typeInfo.getBounds(i) );
  }
  
  buildArray( objType );
  
  if ((isCompositeResolvedType = typeInfo.isCompositeResolvedType()) == true) {
    resolutionFunctionId    = typeInfo.getResolutionFunctionId();
  }
  else {
    if (resolveFnId != -1) {
      // This will be an anonymous subtype with a resolution function...
      resolutionFunctionId = resolveFnId;
    }
  }
  
  if (resolutionFunctionId != -1) {
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }

  if (noofElmntAssns <= 0) {
    return;
  }

  // Okay...we need to handle one or more element associations here...

  va_list ap;
  ElementAssociation* elmtptr           = NULL;
  ElementAssociation* othersAssociation = NULL;
  ElementAssociation** elmtAssocArray   = NULL;
  char* charptr                         = NULL;
  int i = 0,      j = 0;
  int correct_index = 0;
  int temp          = 0;
  int numElems      = bounds[0].length();
  
  bool positional_association = false;
  
  charptr        = new char[bounds[0].length()];
  elmtAssocArray = (ElementAssociation* *) new char[noofElmntAssns *
						   sizeof(ElementAssociation *)];
  for (i = 0; (i < numElems); i++) {
    charptr[i] = 'U';
  }

  va_start(ap, noofElmntAssns);
  
  for(i = 0; (i < noofElmntAssns); i++) {
    elmtAssocArray[i] = va_arg(ap, ElementAssociation*);
  }
  va_end(ap);

  // Determine if this is a positional association or not
  elmtptr = elmtAssocArray[0];
  if(noofElmntAssns == get_bounds(0).length()) {
    if(elmtptr->choice.left() == elmtptr->choice.right()) {
      if(elmtptr->choice.left() == 0) {
	if(get_bounds(0).contains(0)) {
	  temp = elmtAssocArray[noofElmntAssns -1]->choice.left();
	  if(get_bounds(0).contains(temp)) {
	    positional_association = false;
	  } else {
	    positional_association = true;
	  }
	} else {
	  positional_association = true;
	}
      } else {
	positional_association = false;
      }
    } else {
      positional_association = false;
    }
  } else {
    positional_association = false;
  }
  
  for(i = 0; (i < noofElmntAssns); i++) {
    elmtptr = elmtAssocArray[i];
    if( elmtptr->choice.operator==(Others)) {
      othersAssociation = elmtptr;
      break;
    }
    else {
      if(elmtptr->choice.left() == elmtptr->choice.right()) {
	if(positional_association == true) {
	  correct_index = get_bounds(0).actualIndex(elmtptr->choice.left());
	}
	else {
	  correct_index = elmtptr->choice.left();
	}
	object[storageIndex(0, correct_index)]->assignVal(*elmtptr->value);
	charptr[storageIndex(0, correct_index)] = 'I';
      }
      else {
	if(elmtptr->choice.dirn() == ArrayInfo::to) {
	  for(j = elmtptr->choice.left(); (j <= elmtptr->choice.right()); j++) {
	    if(positional_association == true) {
	      correct_index = get_bounds(0).actualIndex(j);
	    }
	    else {
	      correct_index = j;
	    }
	    object[storageIndex(0, correct_index)]->assignVal(*(elmtptr->value));
	    charptr[storageIndex(0, correct_index)] = 'I';
	  }
	}
	else {
	  for(j = elmtptr->choice.left(); (j >= elmtptr->choice.right()); j--) {
	    if(positional_association == true) {
	      correct_index = get_bounds(0).actualIndex(j);
	    }
	    else {
	      correct_index = j;
	    }
	    object[storageIndex(0, correct_index)]->assignVal(*(elmtptr->value));
	    charptr[storageIndex(0, correct_index)] = 'I';
	  }
	}
      }
    }
  }
  
  if (othersAssociation != NULL) {
    for(i = 0; (i < numElems); i++) {
      if(charptr[i] == 'U') {
	object[i]->operator=(*(othersAssociation->value));
      }
    }
  }

  // Free memory allocated/used by intermediate data structures...
  for(i = 0; (i < noofElmntAssns); i++) {
    delete  elmtAssocArray[i]->value;
    delete  elmtAssocArray[i];
  }
  if(elmtAssocArray != NULL) {
    delete [] elmtAssocArray;
  }
  delete [] charptr;    
}

ArrayType::ArrayType( ObjectBase::ObjectType, 
		      const ArrayTypeInfo &typeInfo,
		      ResolutionFnId_t resolveFnId, 
		      int left1,
		      ArrayInfo::ArrayDirn_t direction1, 
		      int right1, 
		      int left2,
		      ArrayInfo::ArrayDirn_t direction2, 
		      int right2, ...) {
  ASSERT ( typeInfo.get_dimensions() > 1 );
  arrayInfo = typeInfo;
  resolutionFunctionId = resolveFnId;
  
  
  bounds.push_back( ArrayInfo(left1, direction1, right1) );
  bounds.push_back( ArrayInfo(left2, direction2, right2) );

  if (typeInfo.get_dimensions() > 1) {
    // This is a n-d array...so, setup subArrayInfo, also...
    vector<TypeInfo *> subRanges( typeInfo.getRanges().begin() + 1,
				  typeInfo.getRanges().end() );
    subArrayInfo = ArrayTypeInfo( typeInfo.getElementTypeInfo(), 
				  typeInfo.isUnconstrainedArrayType(), 
				  typeInfo.isCompositeResolvedType(), 
				  typeInfo.getResolutionFunctionId(), 
				  subRanges );
  }
  
  isCompositeResolvedType = typeInfo.isCompositeResolvedType();
  if( isCompositeResolvedType == true ){
    resolutionFunctionId = typeInfo.getResolutionFunctionId();
  }
  else {
    if (resolveFnId != -1) {
      // This will be an anonymous subtype with a resolution function...
      resolutionFunctionId = resolveFnId;
    }
  }
  
  if (resolveFnId != -1) {
    isCompositeResolvedType = true;
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }

  cerr << "Hit unfinished and broken code" << endl;
  abort();
}

ArrayType::ArrayType( ObjectBase::ObjectType objType,
		      const ArrayTypeInfo& typeInfo,
		      ResolutionFnId_t resolveFnId,
		      const ArrayType &source ){
  arrayInfo    = typeInfo;
  subArrayInfo = source.getSubArrayTypeInfo();

  if( !arrayInfo.isUnconstrainedArrayType() ){
    for(int i = 0; (i < arrayInfo.get_dimensions()); i++) {
      bounds.push_back( arrayInfo.getBounds(i) );
    }
  }
  else {
    // Build bounds based on the source...
    for(int i = 0; (i < arrayInfo.get_dimensions()); i++) {
      ArrayInfo newBounds = arrayInfo.getBounds(i);
      ArrayInfo srcBounds = source.get_bounds(i);
      int size = srcBounds.length() - 1;
      int right = newBounds.left() + ((newBounds.dirn() == ArrayInfo::to) ? size : -size);
      bounds.push_back( ArrayInfo(newBounds.left(), newBounds.dirn(), right) );
    }
  }
  
  resolutionFunctionId = resolveFnId;
  
  if (resolutionFunctionId == -1) {
    resolutionFunctionId = source.getResolutionFunctionId();
  }
  
  buildArray( objType );
  operator=(source);

  if (source._is_composite_resolved_type() == true) {
    setParentCompositeType(this);
    setCompositeResolvedSignal(true);
  }  
}

ArrayType::ArrayType(ObjectBase::ObjectType objType,
		     const ArrayType &source) {
  arrayInfo      = source.getArrayTypeInfo();
  subArrayInfo   = source.getSubArrayTypeInfo();

  for(int i = 0; i < arrayInfo.get_dimensions(); i++ ){
    bounds.push_back( source.get_bounds(i) );
  }

  resolutionFunctionId = source.getResolutionFunctionId();
  
  buildArray( objType );
  operator=(source);

  if (source._is_composite_resolved_type() == true) {
    setParentCompositeType(this);
    setCompositeResolvedSignal(true);
  }  
}

ArrayType::ArrayType(ObjectBase::ObjectType objType, const ArrayTypeInfo &typeInfo,
		     ResolutionFnId_t resolveFnId, int left1,
		     ArrayInfo::ArrayDirn_t direction1, int right1,
		     const ArrayType &source) {
    ASSERT ( typeInfo.get_dimensions() == 1 );

  arrayInfo = typeInfo;
  resolutionFunctionId = resolveFnId;  

  bounds.push_back( ArrayInfo(left1, direction1, right1) );
  buildArray( objType );

  resolutionFunctionId = -1;

  if ((isCompositeResolvedType = typeInfo.isCompositeResolvedType()) == true) {
    resolutionFunctionId    = typeInfo.getResolutionFunctionId();
  }
  else {
    if (resolveFnId != -1) {
      // This will be an anonymous subtype with a resolution function
      resolutionFunctionId = resolveFnId;
    }
  }
  
  if (resolveFnId != -1) {
    isCompositeResolvedType = true;
    setCompositeResolvedSignal(true);
    setParentCompositeType(this);
  }

  operator=(source);
}

ArrayType::ArrayType(const ArrayType &source) :
  VHDLType(),
  bounds( source.bounds ),
  isCompositeResolvedType( source._is_composite_resolved_type() ),
  resolutionFunctionId( source.getResolutionFunctionId() ), 
  arrayInfo( source.getArrayTypeInfo() ), 
  subArrayInfo( source.getSubArrayTypeInfo() ){

  buildArray( source.getKind() );
  operator=(source);
  
  if (source._is_composite_resolved_type() == true) {
    setParentCompositeType(this);
    setCompositeResolvedSignal(true);
  }
}

ArrayType::~ArrayType() { 
  if( !is_alias ){
    // More work is needed here...  Assuming single dimension arrays for
    // the moment.
    if( !bounds.empty() ){
      for( int i = 0; i < bounds[0].length(); i++ ){
	delete object[i];
      }
    }
  }
}

Type 
ArrayType::get_kind() const {
  return ARRAY_TYPE;
}

void 
ArrayType::printDimension( ostream &os, int dimension ) const {
  ArrayInfo currentBounds = get_bounds( dimension );
  os << "(";
  if( currentBounds.isAscending() ){
    for( int i = currentBounds.left(); i < currentBounds.right(); i++ ){
      object[storageIndex(dimension, i)]->print( os );
    }
  }
  else{
    for( int i = currentBounds.left(); i >= currentBounds.right(); i-- ){
      object[storageIndex(dimension, i)]->print( os );
    }
  }
  os << ")";
}

void 
ArrayType::print( ostream &os ) const {
  for( unsigned int i = 0; i < getNumberOfDimensions(); i++ ){
    printDimension( os, i );
    if( i + 1 < getNumberOfDimensions() ){
      os << "\n";
    }
  }
}

VHDLType& 
ArrayType::operator[](const int pos) const {
  if (bounds[0].contains(pos)) {
    return *(object[storageIndex(0, pos)]);
  }
  
  cerr << "Position: " << pos << " is out of bounds. (Range = " << bounds[0]
       << ").\n";
  abort();

  return IntegerType::INVALID_TYPE_OBJECT;
}

VHDLType& 
ArrayType::operator[](const ScalarType &pos) const {
  return operator[](pos.getIntValue());
}

VHDLType& 
ArrayType::operator[](const VHDLType& pos) const {
  ASSERT ( pos.is_scalar_type() == true );
  return operator[](pos.getIntValue());
}

VHDLType& 
ArrayType::operator=(const VHDLType& val) {
  ASSERT ( val.get_kind() == ARRAY_TYPE );

  return operator=( static_cast<const ArrayType &>(val) );
}

VHDLType& 
ArrayType::assignVal(const VHDLType& val) {
  int numElements = bounds[0].length();

  ASSERT ( val.get_kind() == ARRAY_TYPE );
  
  for( int i = 0; i < numElements; i++) {
    object[i]->assignVal(val.get_element(i));
  }
  
  return (*this);
}


ArrayType& 
ArrayType::operator=( const ArrayType &val ){
  if( object.size() == 0 ){
    arrayInfo    = val.getArrayTypeInfo();
    subArrayInfo = val.getSubArrayTypeInfo();
    
    for( int i = 0; i < arrayInfo.get_dimensions(); i++) {
      bounds.push_back( val.get_bounds(i) );
    }
    resolutionFunctionId = val.getResolutionFunctionId();

    buildArray( val.getKind() );
  }
  
  if (val._is_composite_resolved_type() == true) {
    resolutionFunctionId = val.getResolutionFunctionId();
    setParentCompositeType(this);
    setCompositeResolvedSignal(true);
  }
  
  int elementCount = bounds[0].length();
  
  for(int i = 0; i < elementCount; i++ ){
    object[i]->operator=(val.get_element(i));
  }
  
  is_alias = val.is_alias;
  
  return *this;
}

bool
ArrayType::operator==( const RValue &val ) const {
  bool retval = true;

  const ArrayType &rhs = dynamic_cast<const ArrayType &>(val);
  
  int lhs_length = get_bounds(0).length();
  int rhs_length = rhs.get_bounds(0).length();
  
  if(lhs_length != rhs_length) {
    retval = false;
  }
  else{
    for(int i = 0; i < lhs_length; i++) {
      if(!(get_element(i).operator==(rhs.get_element(i)))) {
	retval = false;
	break;
      }
    }
  }

  return retval;
}

bool
ArrayType::operator!=( const RValue &rhs ) const {
  return !operator==( rhs );
}

bool
ArrayType::operator<( const RValue &rhs ) const {
  const ArrayType &compareTo = dynamic_cast<const ArrayType &>(rhs);

  bool is_lhs_null = get_bounds(0).is_null_range();
  bool is_rhs_null = compareTo.get_bounds(0).is_null_range();

  if( is_lhs_null  && is_rhs_null ){ 
    return false;
  } 
  else  { 
    if( is_lhs_null && !is_rhs_null ){ 
      return true;
    }
    else if( !is_lhs_null && is_rhs_null ){ 
      return false;
    }
  }
  
  int lhs_i   =  get_bounds(0).left();
  int rhs_i   =  compareTo.get_bounds(0).left();
  int lhs_increment = (get_bounds(0).dirn() == ArrayInfo::to) ? 1 : -1;
  int rhs_increment = (compareTo.get_bounds(0).dirn() == ArrayInfo::to) ? 1 : -1;
  int max_i   = get_number_of_elements();
  
  if (max_i > compareTo.get_number_of_elements())  {
    max_i = compareTo.get_number_of_elements();
  }
  
  for( int i = 0; i < max_i; i++)  { 
    if( (*this)[lhs_i] < compareTo[rhs_i] ){
      return true;
    }
    else if( (*this)[lhs_i] > compareTo[rhs_i] ){
      return false;
    }
    lhs_i += lhs_increment;
    rhs_i += rhs_increment;
  }
  
  if( max_i == get_number_of_elements() ) {
    return true;
  }
  else {
    return false;
  }
}

bool
ArrayType::operator>=( const RValue &rhs ) const {
  return (*this > rhs || *this == rhs );
}

bool
ArrayType::operator>( const RValue &rhs ) const {
  return !(*this < rhs || *this == rhs );
}

bool
ArrayType::operator<=( const RValue &rhs ) const {
  return (*this < rhs || *this == rhs );
}

VHDLType& 
ArrayType::get_element(const int index) const {
  ASSERT( index >= 0 );
  ASSERT( (unsigned)index <= object.size() - 1 );
  return *(object[index]);
}

const vector<VHDLType*> &
ArrayType::get_array() const {
  return object;
}

int 
ArrayType::get_number_of_elements(int dimension) const {
  return bounds[dimension].length();
}

const ArrayInfo& 
ArrayType::get_bounds(int dimension) const {
  return bounds[dimension];
}

char *
ArrayType::getString() const { 
  char *retval = 0;

  if (arrayInfo.get_dimensions() != 1) {
    cerr << "ArrayType::getString() const called on an array with "
	 << arrayInfo.get_dimensions() << " dimensions!" << endl;
    abort();
  }
  else{
    int elementCount = bounds[0].length();
    string retString;
    for(int i = 0; i < elementCount; i++) {
      retString += object[i]->getString();
    }
    retval = cppStrDup( retString.c_str() );
  }
  return retval;
}

ArrayType &
ArrayType::getSlice( const ArrayInfo& newBounds ) const {
  ArrayType *retval = new ArrayType(true);

  retval->arrayInfo            = arrayInfo;
  retval->subArrayInfo         = subArrayInfo;
  retval->resolutionFunctionId = resolutionFunctionId;

  int dimensions = arrayInfo.get_dimensions();

  for( int i = 0; i < dimensions; i++ ){
    retval->bounds.push_back( get_bounds(i) );
  }
  
  retval->bounds[0] = newBounds;
  retval->object.reserve(retval->length(0));
  int sourceOffset = abs(newBounds.left() - get_bounds(0).left());
  for( int i = 0; i < newBounds.length(); i++) {
    retval->object[i] = get_array()[sourceOffset + i];
  }

  if (arrayInfo.isCompositeResolvedType() == true) {
    retval->setResolutionFunctionId(arrayInfo.getResolutionFunctionId());
    retval->setCompositeResolvedSignal(true);
    retval->setParentCompositeType(retval);
  }
  
  return *retval;
}
		       
VHDLType*
ArrayType::clone() const {
  VHDLType *retval = new ArrayType(*this);
  
  return retval;
}

ObjectBase::ObjectType
ArrayType::getKind() const {
  return object[0]->getKind();
}

int 
ArrayType::left(const int dimension) const {
  return bounds[dimension].left();
}

VHDLType&
ArrayType::left(const ScalarType& dimension) const {
  int dim              = dimension.getIntValue();
  int leftValue        = left(dim);
  
  // Convert it to appropriate type...
  ScalarType *retVal = (ScalarType *) arrayInfo.getRangeInfo(dim - 1)->createObject(ObjectBase::VARIABLE);
  retVal->getObject()->updateVal(UniversalInteger(leftValue));
  
  return *retVal;
}

int 
ArrayType::right(const int dimension) const {
  return bounds[dimension].right();
}

VHDLType&
ArrayType::right(const ScalarType& dimension) const {
  int dim              = dimension.getIntValue();
  int rightValue       = right(dim);
  
  // Convert it to appropriate type...
  ScalarType *retVal = (ScalarType *) arrayInfo.getRangeInfo(dim - 1)->createObject(ObjectBase::VARIABLE);
  retVal->getObject()->updateVal(UniversalInteger(rightValue));
  
  return *retVal;
}

ArrayInfo::ArrayDirn_t 
ArrayType::dirn(const int dimension) const {
  return bounds[dimension].dirn();
}

int 
ArrayType::length(const int dimension) const { 
  return bounds[dimension].length();
}

const ArrayTypeInfo&
ArrayType::getArrayTypeInfo() const {
  return arrayInfo;
}

const ArrayTypeInfo&
ArrayType::getSubArrayTypeInfo() const {
  return subArrayInfo;
}

VHDLType*
ArrayType::resolve(VHDLKernel* processPtr) {
  int elementCount = bounds[0].length();
  VHDLType *retval = NULL, *element = NULL;
  ASSERT( getKind() == ObjectBase::SIGNAL );
  if(_is_composite_resolved_type() == true) {
    if( object.size() > 0 ){
      return object[0]->resolve(processPtr);
    }
  }
  else {
    for( int i = 0; i < elementCount; i++ ){
      // if the subelements are not composite resolved types. then the value
      // is update in their down most scalar types itself. So we dont' need
      // to udpate the value physically.  Otherwise, we need to update it
      // physically.  that is why this assignment.

      element = object[i];
      retval  = element->resolve(processPtr);
      
      if (element->_is_composite_resolved_type()) {
	*element = *retval;
      } 
    }
  }
  return retval;
}

void
ArrayType::updateEffVal(const VHDLType* ptr) {
  ASSERT(getKind() == ObjectBase::SIGNAL);
  ASSERT(ptr->get_kind() == ARRAY_TYPE);

  ArrayType *rhs   = (ArrayType *) ptr;
  int elementCount = length(0);
  int i;

  for(i = 0; (i < elementCount); i++) {
    object[i]->updateEffVal(&rhs->get_element(i));
  }
}

void
ArrayType::setResolutionFunctionId(ResolutionFnId_t resolutionFnId) {
  int elementCount = length(0);
  int i;

  for(i = 0; (i < elementCount); i++) {
    object[i]->setResolutionFunctionId(resolutionFnId);
  }
}

void
ArrayType::setTypeConversionFunctionId(TypeConversionFnId_t typeConversionFnId) {
  int elementCount = length(0);
  for( int i = 0; i < elementCount; i++ ){
    object[i]->setTypeConversionFunctionId(typeConversionFnId);
  }  
}


void
ArrayType::setParentCompositeType(VHDLType* ptr) {
  if (getKind() != ObjectBase::SIGNAL) {
    return;
  }
  int elementCount = bounds[0].length();
  for( int i = 0; i < elementCount; i++ ){
    object[i]->setParentCompositeType(ptr);
  }
  
  isCompositeResolvedType = true;
}
 
void
ArrayType::setCompositeResolvedSignal(bool val) {
  if( getKind() != ObjectBase::SIGNAL ){
    return;
  }
  
  int elementCount = bounds[0].length();
  for( int i = 0; i < elementCount; i++ ){
    object[i]->setCompositeResolvedSignal(val);
  }
  
  isCompositeResolvedType = val;
}

void
ArrayType::setElaborationInfo(const VHDLType &obj_info) {
  ASSERT( _is_signal() == true );
  ASSERT( obj_info.get_kind() == ARRAY_TYPE );
  const ArrayType &rhs = (const ArrayType &) obj_info;
  int elementCount = bounds[0].length();
  for( int i = 0; i < elementCount; i++ ){
    get_element(i).setElaborationInfo(rhs.get_element(i));
  }
}

void
ArrayType::setAttrib(AttribType typ, VHDLType& attr) {
  ASSERT (_is_signal() == true);
  ASSERT (attr.get_kind() == ARRAY_TYPE);
  ArrayType &rhs = (ArrayType &) attr;
  int index;
  
  for (index = 0; (index < length(0)); index++) {
    get_element(index).setAttrib(typ, rhs.get_element(index));
  }
}

SignalBase *
ArrayType::locateSig( int sigId ){
  SignalBase *retval = 0;
  int elementCount = length(0);
  for( int i = 0; i < elementCount; i++ ){
    retval = get_element(i).locateSig(sigId);
    if( retval != NULL) {
      break;
    }
  }
  
  return retval;
}

SignalBase *
ArrayType::findSigInBlock(int sigId, VHDLKernel* srcId){
  SignalBase *driver_ptr = NULL;
  int elementCount = length(0);
  
  for(int i = 0; i < elementCount; i++ ){
    driver_ptr = get_element(i).findSigInBlock(sigId, srcId);
    if( driver_ptr != NULL) {
      break;
    }
  }
  
  return driver_ptr;
}

void
ArrayType::dump_connectivity_info(ofstream& fileStream) {
  get_element( left(0) ).dump_connectivity_info( fileStream );
}
 
bool
ArrayType::_is_signal() const {
  return object[0]->_is_signal();
}

int
ArrayType::savantwrite(AccessVariable &line) const {
  for( int i = 0; i < length(0); i++ ){
    get_element(i).savantwrite(line);
  }

  //  XXX We should probably be checking the return of our subelement writes!
  return NORMAL_RETURN;
}

int
ArrayType::savantwrite(AccessType &line) const {
  int elementCount = length(0);
  for( int i = 0; i < elementCount; i++ ){
    get_element(i).savantwrite(line);
  }

  //  XXX We should probably be checking the return of our subelement writes!
  
  return NORMAL_RETURN;
}  

int
ArrayType::savantread(AccessVariable &line)  {
  int elementCount = length(0);
  
  for( int i = 0; i < elementCount; i++ ){
    get_element(i).savantread(line);
  }

  //  XXX We should probably be checking the return of our subelement reads!
  return NORMAL_RETURN;
}

int
ArrayType::savantread( AccessType &line ){
  int elementCount = length(0);
  for( int i = 0; i < elementCount; i++ ){
    get_element(i).savantread( line );
  }

  //  XXX We should probably be checking the return of our subelement reads!
  return NORMAL_RETURN;
}

IntegerType
ArrayType::LEFT_O(const IntegerType& dimension) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim < arrayInfo.get_dimensions()) {
    ASSERT ( arrayInfo.getRangeInfo(dim) != NULL );
    ASSERT ( arrayInfo.getRangeInfo(dim)->getKind() == RANGE_INFO );
    return IntegerType(ObjectBase::VARIABLE, UniversalInteger(bounds[dim].left()), *arrayInfo.getRangeInfo(dim));
  }
  else {
    cerr << "The parameter of `LEFT_O attribute not within range\n";
    abort();
    return IntegerType::INVALID_TYPE_OBJECT;
  }
}

EnumerationType
ArrayType::LEFT_O(const IntegerType& dimension, const TypeInfo&) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim < arrayInfo.get_dimensions()) {
    ASSERT ( arrayInfo.getRangeInfo(dim) != NULL );
    ASSERT ( arrayInfo.getRangeInfo(dim)->getKind() == ENUM_INFO );
    return EnumerationType(ObjectBase::VARIABLE, UniversalInteger(bounds[dim].left()), *arrayInfo.getRangeInfo(dim));
  }
  else {
    cerr << "The parameter of `LEFT_O attribute not within range\n";
    abort();
    return EnumerationType();
  }
}

IntegerType
ArrayType::RIGHT_O(const IntegerType& dimension) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim <= arrayInfo.get_dimensions()) {
    ASSERT ( arrayInfo.getRangeInfo(dim) != NULL );
    ASSERT ( arrayInfo.getRangeInfo(dim)->getKind() == RANGE_INFO );
    return IntegerType(ObjectBase::VARIABLE, UniversalInteger(bounds[dim].right()), *arrayInfo.getRangeInfo(dim));
  }
  else {
    cerr << "The parameter of `RIGHT_O attribute not within range\n";
    abort();
    return IntegerType::INVALID_TYPE_OBJECT;
  }
}

EnumerationType
ArrayType::RIGHT_O(const IntegerType& dimension, const TypeInfo&) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim <= arrayInfo.get_dimensions()) {
    ASSERT ( arrayInfo.getRangeInfo(dim) != NULL );
    ASSERT ( arrayInfo.getRangeInfo(dim)->getKind() == ENUM_INFO );
    return EnumerationType(ObjectBase::VARIABLE, UniversalInteger(bounds[dim].right()), *arrayInfo.getRangeInfo(dim));
  }
  else {
    cerr << "The parameter of `RIGHT_O attribute not within range\n";
    abort();
    return EnumerationType();
  }
}

IntegerType
ArrayType::HIGH_O(const IntegerType& dimension) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim <= arrayInfo.get_dimensions()) {
    ASSERT ( arrayInfo.getRangeInfo(dim) != NULL );
    ASSERT ( arrayInfo.getRangeInfo(dim)->getKind() == RANGE_INFO );
    int highValue = ((bounds[dim].dirn() == ArrayInfo::to) ? bounds[dim].right() : bounds[dim].left());
    return IntegerType(ObjectBase::VARIABLE, UniversalInteger(highValue), *arrayInfo.getRangeInfo(dim));
  }
  else {
    cerr << "The parameter of `HIGH_O attribute not within range\n";
    abort();
    return IntegerType::INVALID_TYPE_OBJECT;
  }
}

EnumerationType
ArrayType::HIGH_O(const IntegerType& dimension, const TypeInfo&) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim <= arrayInfo.get_dimensions()) {
    ASSERT ( arrayInfo.getRangeInfo(dim) != NULL );
    ASSERT ( arrayInfo.getRangeInfo(dim)->getKind() == ENUM_INFO );
        int highValue = ((bounds[dim].dirn() == ArrayInfo::to) ? bounds[dim].right() : bounds[dim].left());
    return EnumerationType(ObjectBase::VARIABLE, UniversalInteger(highValue), *arrayInfo.getRangeInfo(dim));
  }
  else {
    cerr << "The parameter of `HIGH_O attribute not within range\n";
    abort();
    return EnumerationType();
  }
}

IntegerType
ArrayType::LOW_O(const IntegerType& dimension) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim < arrayInfo.get_dimensions()) {
    ASSERT ( arrayInfo.getRangeInfo(dim) != NULL );
    ASSERT ( arrayInfo.getRangeInfo(dim)->getKind() == RANGE_INFO );
    int lowValue = ((bounds[dim].dirn() == ArrayInfo::to) ? bounds[dim].left() : bounds[dim].right());
    return IntegerType(ObjectBase::VARIABLE, UniversalInteger(lowValue), *arrayInfo.getRangeInfo(dim));
  }
  else {
    cerr << "The parameter of `LOW_O attribute not within range\n";
    abort();
    return IntegerType::INVALID_TYPE_OBJECT;
  }
}

EnumerationType
ArrayType::LOW_O(const IntegerType& dimension, const TypeInfo&) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim < arrayInfo.get_dimensions()) {
    ASSERT ( arrayInfo.getRangeInfo(dim) != NULL );
    ASSERT ( arrayInfo.getRangeInfo(dim)->getKind() == ENUM_INFO );
    int lowValue = ((bounds[dim].dirn() == ArrayInfo::to) ? bounds[dim].left() : bounds[dim].right());
    return EnumerationType(ObjectBase::VARIABLE, UniversalInteger(lowValue), *arrayInfo.getRangeInfo(dim));
  }
  else {
    cerr << "The parameter of `LOW_O attribute not within range\n";
    abort();
    return EnumerationType();
  }
}

IntegerType
ArrayType::LENGTH_O(const IntegerType& dimension) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim < arrayInfo.get_dimensions()) {
    return IntegerType(ObjectBase::VARIABLE, UniversalInteger(bounds[dim].length()), SavantintegerType_info);
  }

  cerr << "The parameter of `LENGTH_O attribute not within range\n";
  abort();
  return IntegerType::INVALID_TYPE_OBJECT;
}

EnumerationType
ArrayType::ASCENDING_O(const IntegerType& dimension) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim < arrayInfo.get_dimensions()) {
    return EnumerationType::toBoolean( bounds[dim].dirn() == ArrayInfo::to );
  }

  cerr << "The parameter of `ASCENDING_O attribute not within range\n";
  abort();
  return SAVANT_BOOLEAN_FALSE;
}


IntegerType
ArrayType::RANGE_O(const IntegerType& dimension) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim < arrayInfo.get_dimensions()) {
    ASSERT ( arrayInfo.getRangeInfo(dim) != NULL );
    ASSERT ( arrayInfo.getRangeInfo(dim)->getKind() == RANGE_INFO );

    return IntegerType(ObjectBase::VARIABLE, UniversalInteger(bounds[dim].left()), IntegerTypeInfo(bounds[dim].left(), bounds[dim].dirn(), bounds[dim].right()));
  }

  cerr << "The parameter of `RANGE_O attribute not within range\n";
  abort();
  return IntegerType::INVALID_TYPE_OBJECT;
}

EnumerationType
ArrayType::RANGE_O(const IntegerType& dimension, const TypeInfo&) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim < arrayInfo.get_dimensions()) {
    ASSERT ( arrayInfo.getRangeInfo(dim) != NULL );
    ASSERT ( arrayInfo.getRangeInfo(dim)->getKind() == ENUM_INFO );
    
    char **imgMap = (((EnumerationTypeInfo *) arrayInfo.getRangeInfo(dim))->get_imageMap() + bounds[dim]. left());
    
    return EnumerationType(ObjectBase::VARIABLE,
			   UniversalInteger(bounds[dim].left()),
			   EnumerationTypeInfo(bounds[dim].length(),
					       imgMap,
					       bounds[dim].left(),
					       bounds[dim].dirn()));
  }
  
  cerr << "The parameter of `RANGE_O attribute not within range\n";
  abort();
  return EnumerationType();
}


IntegerType
ArrayType::REVERSE_RANGE_O(const IntegerType& dimension) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim < arrayInfo.get_dimensions()) {
    ASSERT ( arrayInfo.getRangeInfo(dim) != NULL );
    ASSERT ( arrayInfo.getRangeInfo(dim)->getKind() == RANGE_INFO );
    
    if (bounds[dim].dirn() == ArrayInfo::to) {
      return IntegerType(ObjectBase::VARIABLE, UniversalInteger(bounds[dim].right()), IntegerTypeInfo(bounds[dim].right(), ArrayInfo::downto, bounds[dim].left()));
    }
    else {
      return IntegerType(ObjectBase::VARIABLE, UniversalInteger(bounds[dim].left()), IntegerTypeInfo(bounds[dim].left(), ArrayInfo::to, bounds[dim].right()));
    }
  }
  
  cerr << "The parameter of `RANGE_O attribute not within range\n";
  abort();
  return IntegerType::INVALID_TYPE_OBJECT;
}

EnumerationType
ArrayType::REVERSE_RANGE_O(const IntegerType& dimension, const TypeInfo&) const {
  int dim = dimension.getIntValue() - 1;
  
  if (dim < arrayInfo.get_dimensions()) {
    ASSERT ( arrayInfo.getRangeInfo(dim) != NULL );
    ASSERT ( arrayInfo.getRangeInfo(dim)->getKind() == ENUM_INFO );
    
    char **imgMap = (((EnumerationTypeInfo *) arrayInfo.getRangeInfo(dim))->get_imageMap() + bounds[dim]. left());

    if (bounds[dim].dirn() == ArrayInfo::to) {
      return EnumerationType(ObjectBase::VARIABLE,
			     UniversalInteger(bounds[dim].right()),
			     EnumerationTypeInfo(bounds[dim].length(),
						 imgMap,
						 bounds[dim].right(),
						 ArrayInfo::downto));
    }
    else {
      return EnumerationType(ObjectBase::VARIABLE,
			     UniversalInteger(bounds[dim].left()),
			     EnumerationTypeInfo(bounds[dim].length(),
						 imgMap,
						 bounds[dim].left(),
						 ArrayInfo::to));
    }
  }
  
  cerr << "The parameter of `RANGE_O attribute not within range\n";
  abort();
  return EnumerationType();
}


IntegerType 
ArrayType::LEFT(const IntegerType& dimen, const TypeInfo& tInfo) {
  ASSERT ( tInfo.getKind() == ARRAY_TYPE_INFO );
  const ArrayTypeInfo &aInfo = (const ArrayTypeInfo &) tInfo;
  
  int dimension = dimen.getIntValue() - 1;
  if (dimension < (aInfo.get_dimensions() + 1)) {
    ArrayInfo tempInfo = aInfo.getBounds(dimension);
    return IntegerType(ObjectBase::VARIABLE, tempInfo.left());
  }
  else {
    cerr << "The parameter of `LEFT attribute not within range\n";
    abort();
    return IntegerType::INVALID_TYPE_OBJECT;
  }
}

EnumerationType 
ArrayType::LEFT(const IntegerType& dimen, const TypeInfo& tInfo, const TypeInfo&) {
  ASSERT ( tInfo.getKind() == ARRAY_TYPE_INFO );
  const ArrayTypeInfo &aInfo = (const ArrayTypeInfo &) tInfo;
  
  int dimension = dimen.getIntValue() - 1;
  if (dimension < (aInfo.get_dimensions() + 1)) {
    ArrayInfo tempInfo = aInfo.getBounds(dimension);
    return EnumerationType(ObjectBase::VARIABLE, tempInfo.left());
  }
  else {
    cerr << "The parameter of `LEFT attribute not within range\n";
    abort();
    return EnumerationType();
  }
}

IntegerType 
ArrayType::RIGHT(const IntegerType& dimen, const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == ARRAY_TYPE_INFO );
  const ArrayTypeInfo &aInfo = (const ArrayTypeInfo &) tInfo;
  
  int dimension = dimen.getIntValue() - 1;
  if (dimension < (aInfo.get_dimensions() + 1))  {
    ArrayInfo tempBounds = aInfo.getBounds(dimension);
    return IntegerType(ObjectBase::VARIABLE, tempBounds.right());
  }
  else {
    cerr << "The parameter of `RIGHT attribute not within range\n";
    abort();
    return IntegerType::INVALID_TYPE_OBJECT;
  }
}

EnumerationType 
ArrayType::RIGHT(const IntegerType& dimen, const TypeInfo& tInfo, const TypeInfo&){
  ASSERT ( tInfo.getKind() == ARRAY_TYPE_INFO );
  const ArrayTypeInfo &aInfo = (const ArrayTypeInfo &) tInfo;
  
  int dimension = dimen.getIntValue() - 1;
  if (dimension < (aInfo.get_dimensions() + 1))  {
    ArrayInfo tempBounds = aInfo.getBounds(dimension);
    return EnumerationType(ObjectBase::VARIABLE, tempBounds.right());
  }
  else {
    cerr << "The parameter of `RIGHT attribute not within range\n";
    abort();
    return EnumerationType();
  }
}

IntegerType 
ArrayType::HIGH(const IntegerType& dimen, const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == ARRAY_TYPE_INFO );
  const ArrayTypeInfo &aInfo = (const ArrayTypeInfo &) tInfo;
  
  int dimension = dimen.getIntValue() - 1;
  if (dimension < (aInfo.get_dimensions() + 1)) {

    ArrayInfo tempBounds = aInfo.getBounds(dimension);
    int right = tempBounds.right();
    int left  = tempBounds.left();
    
    if (right > left) {
      return IntegerType(ObjectBase::VARIABLE, right);
    }
    else {
      return IntegerType(ObjectBase::VARIABLE, left);
    }
  }
  else {
    cerr << "The parameter of `HIGH attribute not within range\n";
    abort();
    return IntegerType::INVALID_TYPE_OBJECT;
  }
}

EnumerationType 
ArrayType::HIGH(const IntegerType& dimen, const TypeInfo& tInfo, const TypeInfo&) {
  ASSERT ( tInfo.getKind() == ARRAY_TYPE_INFO );
  const ArrayTypeInfo &aInfo = (const ArrayTypeInfo &) tInfo;
  
  int dimension = dimen.getIntValue() - 1;
  if (dimension < (aInfo.get_dimensions() + 1)) {

    ArrayInfo tempBounds = aInfo.getBounds(dimension);
    int right = tempBounds.right();
    int left  = tempBounds.left();
    
    if (right > left) {
      return EnumerationType(ObjectBase::VARIABLE, right);
    }
    else {
      return EnumerationType(ObjectBase::VARIABLE, left);
    }
  }
  else {
    cerr << "The parameter of `HIGH attribute not within range\n";
    abort();
    return EnumerationType();
  }
}

IntegerType 
ArrayType::LOW(const IntegerType& dimen, const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == ARRAY_TYPE_INFO );
  const ArrayTypeInfo &aInfo = (const ArrayTypeInfo &) tInfo;
  
  int dimension = dimen.getIntValue();
  if (dimension <= (aInfo.get_dimensions() + 1)) {
    ArrayInfo tempBounds = aInfo.getBounds(dimension - 1);
    int right = tempBounds.right();
    int left  = tempBounds.left();
    
    if (right > left) {
      return IntegerType(ObjectBase::VARIABLE, left);
    }
    else {
      return IntegerType(ObjectBase::VARIABLE, right);
    }
  }
  else {
    cerr << "The parameter of `LOW attribute not within range\n";
    abort();
    return IntegerType::INVALID_TYPE_OBJECT;
  }
}

EnumerationType 
ArrayType::LOW(const IntegerType& dimen, const TypeInfo& tInfo, const TypeInfo&) {
  ASSERT ( tInfo.getKind() == ARRAY_TYPE_INFO );
  const ArrayTypeInfo &aInfo = (const ArrayTypeInfo &) tInfo;
  
  int dimension = dimen.getIntValue();
  if (dimension <= (aInfo.get_dimensions() + 1)) {

    ArrayInfo tempBounds = aInfo.getBounds(dimension - 1);
    int right = tempBounds.right();
    int left  = tempBounds.left();
    
    if (right > left) {
      return EnumerationType(ObjectBase::VARIABLE, left);
    }
    else {
      return EnumerationType(ObjectBase::VARIABLE, right);
    }
  }
  else {
    cerr << "The parameter of `LOW attribute not within range\n";
    abort();
    return EnumerationType();
  }
}

IntegerType 
ArrayType::LENGTH(const IntegerType& dimen, const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == ARRAY_TYPE_INFO );
  const ArrayTypeInfo &aInfo = (const ArrayTypeInfo &) tInfo;
  
  int dimension = dimen.getIntValue();
  
  if (dimension <= (aInfo.get_dimensions() + 1))  {
    int length = aInfo.get_number_of_elements(dimension - 1);
    return IntegerType(ObjectBase::VARIABLE, length);
  }
  else {
    cerr << "The parameter of `LENGTH attribute not within range\n";
    abort();
    return IntegerType::INVALID_TYPE_OBJECT;
  }
}

EnumerationType
ArrayType::ASCENDING(const IntegerType& dimen, const TypeInfo& tInfo){
  ASSERT ( tInfo.getKind() == ARRAY_TYPE_INFO );
  const ArrayTypeInfo &aInfo = (const ArrayTypeInfo &) tInfo;
  
  int dimension = dimen.getIntValue();
  
  if (dimension <= (aInfo.get_dimensions() + 1))  {
    ArrayInfo tempBounds = aInfo.getBounds(dimension - 1);
    return EnumerationType::toBoolean( tempBounds.dirn() == ArrayInfo::to );
  }

  cerr << "The parameter of `ASCENDING attribute not within range\n";
  abort();
  return SAVANT_BOOLEAN_FALSE;
}

ArrayType
ArrayType::bitOperation( const ArrayType &rhsArray,
			 const EnumerationType &(EnumerationType::*operation)(const EnumerationType& ) const )const {
  int lhs_length = get_bounds(0).length();
  int rhs_length = rhsArray.get_bounds(0).length();
  
  if(lhs_length != rhs_length) {
    cerr << "Arrays of different lengths are ANDed. Bailing out" << endl;
    abort();
  }

  ArrayType retval( *this );
  for( int i = 0; i < lhs_length; i++ ){
    const EnumerationType &lhs = dynamic_cast<const EnumerationType &>(get_element(i));
    const EnumerationType &rhs = dynamic_cast<const EnumerationType &>(rhsArray.get_element(i));
    retval.get_element(i) = (&(lhs)->*operation)(rhs);
  }
  
  return retval;
}
//ASSUMES the Array is a single Dimensional Array of EnumerationType Type
ArrayType
ArrayType::vhdlAnd( const ArrayType &rhs ) const {
  return bitOperation( rhs, &EnumerationType::vhdlAnd );
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType
ArrayType::vhdlNand( const ArrayType &rhs ) const {
  return bitOperation( rhs, &EnumerationType::vhdlNand );
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType
ArrayType::vhdlOr( const ArrayType &rhs ) const {
  return bitOperation( rhs, &EnumerationType::vhdlOr );
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType
ArrayType::vhdlNor( const ArrayType &rhs ) const {
  return bitOperation( rhs, &EnumerationType::vhdlNor );
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType
ArrayType::vhdlXnor( const ArrayType &rhs ) const {
  return bitOperation( rhs, &EnumerationType::vhdlXnor );
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType
ArrayType::vhdlXor( const ArrayType &rhs ) const {
  return bitOperation( rhs, &EnumerationType::vhdlXor );
}

ArrayType
ArrayType::vhdlNot() const {
  int length = get_bounds(0).length();
  ArrayType retval( *this );
  for(int i =0; i < length; i++) {
    const EnumerationType &lhs = dynamic_cast<const EnumerationType&>(get_element(i));
    retval.get_element(i) = lhs.vhdlNot();
  }
  
  return retval;
}

void
ArrayType::shiftLogical( int shiftAmount ){
  int length = get_bounds(0).length();
  if( length > 0 ){
    // We'll shift left if shiftAmount > 0, and right if it's < 0.
    enum { LEFT, RIGHT } direction = RIGHT;
    if( shiftAmount < 0 ){
      direction = LEFT;
      shiftAmount = -shiftAmount;
    }
    
    if( shiftAmount > length ){
      shiftAmount = length;
    }
    
    if( shiftAmount > 0 ){
      if( direction == LEFT ){
	for( int i = length - 1; i >= 0; i-- ){
	  int sourceIndex = i - shiftAmount;
	  EnumerationType newVal( ObjectBase::VARIABLE );
	  if( sourceIndex >= 0 ){
	    newVal = dynamic_cast<EnumerationType &>(get_element(sourceIndex));
	  }
	  else{
	    newVal = dynamic_cast<EnumerationType &>(get_element(i));
	    newVal = newVal.LEFT( newVal.getTypeInfo() );
	  }
	  get_element(i) = newVal;
	}
      }
      else{
	for( int i = 0; i < length; i++ ){
	  int sourceIndex = i + shiftAmount;
	  EnumerationType newVal( ObjectBase::VARIABLE );
	  if( sourceIndex < length ){
	    newVal = dynamic_cast<EnumerationType &>(get_element(sourceIndex));
	  }
	  else{
	    newVal = dynamic_cast<EnumerationType &>(get_element(i));
	    newVal = newVal.LEFT( newVal.getTypeInfo() );
	  }
	  get_element(i) = newVal;
	}
      }
    }
  }
}

ArrayType
ArrayType::vhdlSll( const RValue &rhs ) const {
  ArrayType retval( *this );
  retval.shiftLogical( rhs.getIntValue() );
  return retval;
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType
ArrayType::vhdlSrl( const RValue &rhs ) const {
  ArrayType retval( *this );
  retval.shiftLogical( -rhs.getIntValue() );
  return retval;
}

ArrayType
ArrayType::vhdlSla( const RValue &rhs ) const {
  int lhsLength = get_bounds(0).length();
  int numberOfShifts = rhs.getIntValue();
  int i = 0;
  
  if (numberOfShifts < 0) {
    return vhdlSll( UniversalInteger(numberOfShifts).vhdlAbs() );
  }
  
  ArrayType retval( *this );
  if ((numberOfShifts == 0) || (lhsLength == 0)) {
    return retval;
  }
  
  EnumerationType last_element = (EnumerationType &) get_element(lhsLength - 1);
  
  for(i = numberOfShifts; i < lhsLength; i++) {
    EnumerationType& enumerationPtr = (EnumerationType&)retval.get_element(i - numberOfShifts);
    EnumerationType& enumerationLhs = (EnumerationType&) get_element(i);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }
  
  for(i = lhsLength - numberOfShifts; i < lhsLength; i++) {
    retval.get_element(i) = last_element;
  }
      
  return retval;
}

//ASSUMES the Array is a single Dimensional Array of Boolean or Bit Type
ArrayType
ArrayType::vhdlSra( const RValue &rhs ) const {
  int lhsLength = get_bounds(0).length();
  int numberOfShifts = rhs.getIntValue();
  int i = 0;
  
  if (numberOfShifts < 0) {
    return vhdlSla( UniversalInteger(numberOfShifts).vhdlAbs() );
  }
  
  ArrayType retval( *this );
  if ((numberOfShifts == 0) || (lhsLength == 0)) {
    return retval;
  }
  
  EnumerationType first_value = (EnumerationType&) get_element(0);
  
  if (numberOfShifts > lhsLength) {
    return retval;
  }
  
  for(i = lhsLength; i >= numberOfShifts; i--) {
    EnumerationType& enumerationPtr = (EnumerationType &) retval.get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType &) get_element(i - numberOfShifts);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }

  for(i = 0; i < lhsLength; i++) {
    retval.get_element(i) = first_value;
  }
  
  return retval;
}

ArrayType
ArrayType::vhdlRor( const RValue &rhs ) const {
  int lhsLength = get_bounds(0).length();
  int numberOfShifts = rhs.getIntValue();
  int i = 0, src_pos = 0;
  
  if (numberOfShifts < 0) {
    return vhdlSra( UniversalInteger(numberOfShifts).vhdlAbs() );
  }

  ArrayType retval(*this);
  numberOfShifts = numberOfShifts % lhsLength;
  
  if ((numberOfShifts == 0) || (lhsLength == 0)) {
    return retval;
  }

  // Copy tail of lhs to head of return value
  for(i = 0, src_pos = lhsLength - numberOfShifts; (src_pos < lhsLength); src_pos++, src_pos++) {
    EnumerationType& enumerationPtr = (EnumerationType &) retval.get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType &) get_element(src_pos);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }

  // Copy head of lhs to tail of return value
  for(src_pos = 0, i = numberOfShifts; i < lhsLength; src_pos++, i++)  {
    EnumerationType& enumerationPtr = (EnumerationType &) retval.get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType &) get_element(src_pos);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }
  
  return retval;
}

ArrayType
ArrayType::vhdlRol( const RValue &rhs ) const {
  int lhsLength = get_bounds(0).length();
  int numberOfShifts = rhs.getIntValue();
  int i = 0, src_pos = 0;
  
  if (numberOfShifts < 0) {
    return vhdlSla( UniversalInteger(numberOfShifts).vhdlAbs() );
  }
  
  ArrayType retval( *this );
  numberOfShifts   = numberOfShifts % lhsLength;
  
  if ((numberOfShifts == 0) || (lhsLength == 0)) {
    return retval;
  }
  
  // Copy head of lhs to tail of return value
  for(src_pos = 0, i = lhsLength - numberOfShifts; (src_pos < numberOfShifts); i++, src_pos++) {
    EnumerationType& enumerationPtr = (EnumerationType &) retval.get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType &) get_element(src_pos);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }
  
  // Copy tail of lhs to head of return value
  for(i = 0, src_pos = lhsLength - numberOfShifts; src_pos < lhsLength; src_pos++, i++)  {
    EnumerationType& enumerationPtr = (EnumerationType &) retval.get_element(i);
    EnumerationType& enumerationLhs = (EnumerationType &) get_element(src_pos);
    
    ASSERT(enumerationPtr.get_kind() == ENUMERATION_TYPE);
    ASSERT(enumerationLhs.get_kind() == ENUMERATION_TYPE);
    
    enumerationPtr = enumerationLhs;
  }
  
  return retval;
}

void
ArrayType::buildArray( ObjectBase::ObjectType objType ){
  ASSERT ( !bounds.empty() );
  ASSERT ( arrayInfo.getElementTypeInfo() != NULL );
  
  if (arrayInfo.get_dimensions() == 1) {
    // A single dimension array...
    int elementCount = bounds[0].length();
    object.clear();
    object.reserve(elementCount);
    for( int i = 0; i < elementCount; i++ ){
      object.push_back( arrayInfo.getElementTypeInfo()->createObject(objType) );
    }
  }
  else{
    // It is a 2-Dimensional array...in this case we build the object as a array of
    // of ArrayTypes
    int firstDimension = bounds[0].length();
    object.clear();
    object.reserve(firstDimension);
    for( int i = 0; i < firstDimension; i++ ){
      object.push_back( new ArrayType(objType, subArrayInfo, -1) );
    }
  }
}

void 
ArrayType::Add( const VHDLType &rhs ){
  const ArrayType &source = dynamic_cast<const ArrayType &>( rhs );
  ASSERT( length(0) >= source.length(0) );
  for( int i = 0; i < source.length(0); i++ ){
    get_element(i).Add( source.get_element(i));
  }
}

void 
ArrayType::Add( const SignalNetinfo *ptr ){
  int elementCount  = length(0);
  for( int i = 0; i < elementCount; i++ ){
    get_element(i).Add( ptr );
  }
}

void 
ArrayType::Add( VHDLKernel *processPtr ){
  ASSERT( processPtr != 0 );
  for(int i = 0; i < length(0); i++) {
    get_element(i).Add( processPtr );
  }
}

void 
ArrayType::Add( VHDLKernel *processPtr, int sigId ){
  ASSERT( processPtr != 0 );
  for(int i = 0; i < length(0); i++) {
    get_element(i).Add( processPtr, sigId );
  }
}

void 
ArrayType::copyId( const VHDLType &src ){
  const ArrayType &as_array_type = dynamic_cast<const ArrayType &>( src );
  if( get_array().size() == 0 ){
    *this = as_array_type;
  }

  ASSERT( length(0) >= as_array_type.length(0) );

  for(int i = 0; i < as_array_type.length(0); i++) {
    get_element(i).copyId( src.get_element(i) );
  }
}

void
ArrayType::resolveAndUpdate( VHDLKernel *processPointer ){
  if ( _is_composite_resolved_type() == false ){
    for( int i = 0; i < get_number_of_elements(); i++ ){
      get_element(i).resolveAndUpdate( processPointer );
    }
  }
  else {
    processPointer->updateSignal( this );
  }
}

bool
ArrayType::is_driver_already_set() const {
  return get_element(0).is_driver_already_set();
}

bool
ArrayType::is_resolved_signal() const {
  return get_element(0).is_resolved_signal();
}

void
ArrayType::set_sourcebase_delete_flag( bool flag ) const {
  int elementCount = length(0);
  for( int i = 0; i < elementCount; i++) {
    get_element(i).set_sourcebase_delete_flag(flag);
  }
}

void
ArrayType::assignSignal( VHDLKernel *destProcess, 
			 VHDLKernel *srcId, 
			 const VHDLType& src,
			 const PhysicalType& delay, 
			 const PhysicalType& rejTime, 
			 const ArrayInfo& dinfo,
			 const ArrayInfo& sinfo ){

  ASSERT( destProcess != 0 );
  ASSERT( srcId != 0 );
  switch(src.get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:{
    const ArrayInfo* src_ainfo  = (ArrayInfo *) &sinfo;
    const ArrayInfo* dest_ainfo = (ArrayInfo *) &dinfo;
    if( sinfo.operator==(nullInfo) ){
      src_ainfo = &src.get_bounds(0);
    }
    if(dinfo.operator==(nullInfo)) {
      dest_ainfo  = &get_bounds(0);
    }
    destProcess->assignSignal( (*this)[get_bounds(0).left()], 
			       srcId,
			       src, 
			       delay, 
			       rejTime, 
			       *dest_ainfo, 
			       *src_ainfo);
    break;
  }
  case ARRAY_TYPE:{
    int dest_i, src_i, dest_lbound, dest_rbound, src_lbound, src_rbound, length;
    ArrayInfo::ArrayDirn_t src_dir, dest_dir;
    if ((sinfo != nullInfo) && (sinfo != defaultInfo)) {
      src_lbound  = sinfo.left();
      src_rbound  = sinfo.right();
      src_dir     = sinfo.dirn();
      length      = sinfo.length();
    } else {
      src_lbound = src.left(0);
      src_rbound = src.right(0);
      src_dir    = src.dirn(0);
      length     = src.length(0);
    }
    if ((dinfo != nullInfo) && (dinfo != defaultInfo)) { 
      dest_lbound = dinfo.left();
      dest_rbound = dinfo.right();
      dest_dir    = dinfo.dirn();
    } else {
      dest_lbound = left(0);
      dest_rbound = right(0);
      dest_dir    = dirn(0);
    }
    dest_i = dest_lbound;
    src_i  = src_lbound;
    for( int i = length ; (i > 0); i--) {
      destProcess->assignSignal( (*this)[dest_i], 
				 srcId, 
				 src[src_i], 
				 delay, 
				 rejTime, 
				 defaultInfo, 
				 defaultInfo );
      dest_i += dest_dir;
      src_i  += src_dir;
    }      
    break;
  }
  default:
    abort();
  } 
}

int 
ArrayType::storageIndex( int dimension, int elementIndex ) const {
  return get_bounds( dimension ).storageIndex( elementIndex );
}

ArrayType
savantConcatenate(const ArrayType &lhs, const ArrayType &rhs) {
  ArrayType retval;

  bool lhs_is_null_range = lhs.get_bounds(0).is_null_range();
  bool rhs_is_null_range = rhs.get_bounds(0).is_null_range();
  
  if( lhs_is_null_range  && rhs_is_null_range ){
    retval = rhs;
  }
  else if ( lhs_is_null_range ){
    retval = rhs;
  }
  else if ( rhs_is_null_range ){
    retval = lhs;
  } 
  else{
    int new_length = lhs.get_bounds(0).length() + rhs.get_bounds(0).length();
    int new_left_bounds  = 0;
    int new_right_bounds = new_left_bounds + new_length - 1;
    retval = ArrayType( ObjectBase::VARIABLE,
			lhs.getArrayTypeInfo(),
			lhs.getResolutionFunctionId(),
			new_left_bounds,
			ArrayInfo::to,
			new_right_bounds,
			0,
			NULL );

    int j = new_left_bounds;
    if (lhs.get_bounds(0).dirn() == ArrayInfo::to)  {
      for( int i = lhs.get_bounds(0).left();
	  i <= lhs.get_bounds(0).right();
	  i++, j++ ){
	retval[j] = lhs[i];
      }
    }
    else {
      for( int i = lhs.get_bounds(0).left();
	   i >= lhs.get_bounds(0).right();
	   i--, j++ ){
	retval[j] = lhs[i];
      }
    }
  
    if( rhs.dirn(0) == ArrayInfo::to ){
      for( int i = rhs.get_bounds(0).left();
	   i <= rhs.get_bounds(0).right();
	   i++, j++ ){
	retval[j] = rhs[i];
      }
    }
    else {
      for( int i = rhs.get_bounds(0).left();
	   i >= rhs.get_bounds(0).right();
	   i--, j++ ){
	retval[j] = rhs[i];
      }
    }
  }
  
  return retval;
}

ArrayType
savantConcatenate(const ArrayType &lhs, const VHDLType &rhs) {
  bool lhs_is_null_range = lhs.get_bounds(0).is_null_range();
  
  if( lhs_is_null_range == true) {
    int left_bounds = 0;
    ArrayInfo::ArrayDirn_t new_direction = ArrayInfo::to;
    ArrayType retval (ObjectBase::VARIABLE, lhs.getArrayTypeInfo(), lhs.getResolutionFunctionId(), left_bounds, new_direction, left_bounds, 0, NULL);
    retval[left_bounds] = rhs;
    return retval;
  }
  else {
    int new_length       = lhs.get_bounds(0).length() + 1;
    int new_left_bounds  = 0;
    int new_right_bounds = new_left_bounds + new_length - 1;
    
    ArrayType retval(ObjectBase::VARIABLE, lhs.getArrayTypeInfo(), lhs.getResolutionFunctionId(), new_left_bounds,  ArrayInfo::to, new_right_bounds, 0, NULL);
    
    int i = 0, j = new_left_bounds;
    if(lhs.dirn(0) == ArrayInfo::to) {
      for(i = lhs.get_bounds(0).left(); i <= lhs.get_bounds(0).right(); i++) {
        retval[j] = lhs[i];
        j++;
      }
    }
    else {
      for(i = lhs.get_bounds(0).left(); i >= lhs.get_bounds(0).right(); i--) {
        retval[j] = lhs[i];
        j++;
      }
    }
    retval[j] = rhs;
    return retval;       
  }
}

ArrayType
savantConcatenate(const VHDLType& lhs, const ArrayType& rhs)  {
  int new_length       = rhs.get_bounds(0).length() + 1;
  int new_left_bounds  = 0;
  int new_right_bounds = new_left_bounds + new_length - 1;
  
  ArrayType retval(ObjectBase::VARIABLE, rhs.getArrayTypeInfo(), rhs.getResolutionFunctionId(), new_left_bounds,  rhs.getArrayTypeInfo().getRangeInfo(0)->get_direction(), new_right_bounds, 0, NULL);
  
  int i, j = retval.get_bounds(0).left();

  retval[j] = lhs;
  
  if(rhs.dirn(0) == ArrayInfo::to) {
    for(i = rhs.get_bounds(0).left(); i <= rhs.get_bounds(0).right(); i++) {
      j++;
      retval[j] = rhs[i];
    }
  }
  else {
    for(i = rhs.get_bounds(0).left(); i >= rhs.get_bounds(0).right(); i--) {
      j++;
      retval[j] = rhs[i];
    }
  }
  
  return retval;         
}

ArrayType
savantConcatenate(const VHDLType &lhs, const VHDLType &rhs, const ArrayTypeInfo &aInfo) {
  ASSERT ( aInfo.get_dimensions() == 1 );

  ArrayInfo bounds = aInfo.getBounds(0);
  int index2 = bounds.left() + ((bounds.dirn() == ArrayInfo::to) ? 1 : -1);

  ArrayType retval(ObjectBase::VARIABLE, aInfo, aInfo.getResolutionFunctionId(),
		   bounds.left(), bounds.dirn(), index2, 0, NULL);
  retval[bounds.left()] = lhs;
  retval[index2]        = rhs;

  return retval;
}

ostream&
operator<<(ostream &os, const ArrayType &at) {
  if( at.get_bounds(0).isAscending() ){
    for( int i = at.get_bounds(0).left();
	 i <= at.get_bounds(0).right();
	 i++ ){
      os << at[i];
    }
  }
  else{
    for( int i = at.get_bounds(0).left(); 
	 i >= at.get_bounds(0).right();
	 i-- ){
      os << at[i];
    }
  }

  return os;
}
