// Copyright (c) 1995-1999 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.


//
//---------------------------------------------------------------------------

// This file contains the methods in VHDLProcess class that relate to signal
// attributes.  This file has been created to increase the readability of
// both VHDLProcess and SignalAttributes.

#include "VHDLProcess.hh"
#include "SignalBase.hh"
#include "Variable.hh"
#include "UniversalBoolean.hh"
#include "UniversalReal.hh"



// Attribute implementation: The attribute is called with a VHDLType and
// corresponding parameter as arguments.  This is the function defined below.
// This function finds the "kind" of the type using the "get_kind()" member
// function and correspondingly calls another function.  This function
// extracts the signal information from the type and calls yet another
// function which computes the attribute info.
//     This elaborate mechanism is to make the attribute evaluation
// transparent to scalar types, array types and record types.

// Note: The method "get_kind()" MUST NOT be overloaded by a derived type.
// This is to ensure that the type is recognized correctly.

EnumerationType
VHDLProcess::locateQuietAttribute(const VHDLType *sig, const VHDLVTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(0), SavantbooleanType_info);

  ASSERT(sig != NULL);
  ASSERT(time >= VHDLVTime::getVHDLVTimeZero());

  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateQuietAttribute((const ScalarType*)sig, time);
    break;
  case ARRAY_TYPE:
    retval = locateQuietAttribute(((const ArrayType *)sig), time);
    break;
  case RECORD_TYPE:
    retval = locateQuietAttribute((const RecordType*)sig, time);
    break;
  default:
    cerr << "Unknown Type!! Asking for Quiet Attribute, Sssh, Keep QUIET ;-)"
	 << endl;
    break;
  }
  return retval;
}

EnumerationType
VHDLProcess::locateQuietAttribute(const ScalarType *sig, const VHDLVTime time) {
  ASSERT(sig != NULL);
  ASSERT(time >= VHDLVTime::getVHDLVTimeZero());

  return locateQuietAttribute((const SignalBase*)sig->getObject(), time);
}			     

EnumerationType
VHDLProcess::locateQuietAttribute(const ArrayType *sig, const VHDLVTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  int noOfElements = sig->length(0);

  retval = locateQuietAttribute(&(sig->get_element(0)), time);
  for(int i = 1; (i < noOfElements); i++) {
    tempval = locateQuietAttribute(&(sig->get_element(i)), time);
    retval  = retval.vhdlAnd(tempval);
  }
  
  return retval;
}

EnumerationType
VHDLProcess::locateQuietAttribute( const RecordType* sig, const VHDLVTime time ){
  int noOfElements = sig->get_number_of_fields();

  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);

  for (int i=1; i<= noOfElements; i++) {
    tempval = locateQuietAttribute((const VHDLType*)&(sig->get_field(i)),time);
    retval = retval.vhdlAnd(tempval);
  }
  return retval;
}

EnumerationType
VHDLProcess::locateQuietAttribute( const SignalBase *sig,
				   const VHDLVTime time ){
  vector<Attribute *> attribute;
  Attribute *retval = NULL;
  UniversalLongLongInteger lastActiveTime;
  int i;

  ASSERT(sig != NULL);

  attribute = const_cast<SignalBase *>(sig)->getAttributeList();
  for (i = 0; i < sig->getNumAttributes(); i++ ){
    if( attribute[i]->attrib == LAST_ACTIVE ){
      retval = attribute[i];
    }
  }
  ASSERT((retval != NULL && retval->attrib == LAST_ACTIVE));
  lastActiveTime = ((ScalarType *)retval->value)->getObject()->readVal();
 
  attribute = const_cast<SignalBase *>(sig)->getAttributeList();
  for (i = 0; i <sig->getNumAttributes(); i++) {
    if(attribute[i]->attrib == QUIET) 
      retval = attribute[i];
  }
  ASSERT((retval != NULL && retval->attrib == QUIET));

  if(time == VHDLVTime::getVHDLVTimeZero()) {
    if( lastActiveTime < UniversalLongLongInteger(getTimeNow().getMajor()) ){
      updateAttribute(retval, UNIVERSAL_TRUE);
    } 
    else {
      updateAttribute(retval, UNIVERSAL_FALSE);
    }
  } 
  else {
    if( lastActiveTime < UniversalLongLongInteger(getTimeNow().getMajor() - time.getMajor()) ){
      updateAttribute(retval, UNIVERSAL_TRUE);
    } else {
      updateAttribute(retval, UNIVERSAL_FALSE);
    }
  }
    
  return (*((EnumerationType*)retval->value));
}

EnumerationType
VHDLProcess::locateStableAttribute(const VHDLType *sig, const VHDLVTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, SavantbooleanType_info);
  ASSERT(sig != NULL);
  ASSERT(time >= VHDLVTime::getVHDLVTimeZero());
  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateStableAttribute((const ScalarType*)sig, time);
    break;
  case ARRAY_TYPE:
    retval = locateStableAttribute((const ArrayType *) sig, time);
    break;
  case RECORD_TYPE:
    retval = locateStableAttribute((const RecordType*)sig, time);
    break;

  default:
    cerr << "Unknown Type!! Asking for Stable Attribute, Sssh, Keep QUIET ;-)"
	 << endl;
    break;
  }
  return retval;
}

EnumerationType
VHDLProcess::locateStableAttribute(const ScalarType *sig, const VHDLVTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  ASSERT(sig != NULL);
  ASSERT(time >= VHDLVTime::getVHDLVTimeZero());

  retval = locateStableAttribute((const SignalBase*)sig->getObject(), time);

  return retval;
}			     

EnumerationType
VHDLProcess::locateStableAttribute(const ArrayType *sig, const VHDLVTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  int noOfElements = sig->length(0);

  retval = locateStableAttribute(&(sig->get_element(0)), time);
  for(int i = 0; (i < noOfElements); i++) {
    tempval = locateStableAttribute(&(sig->get_element(i)), time);
    retval = retval.vhdlAnd(tempval);
  }
  
  return retval;
}

EnumerationType
VHDLProcess::locateStableAttribute(const RecordType* sig, const VHDLVTime time) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, UniversalBoolean(1), SavantbooleanType_info);
  int noOfElements = sig->get_number_of_fields();

  for(int i=1; i<= noOfElements; i++) {
    tempval = locateStableAttribute((const VHDLType*)&(sig->get_field(i)),time);
    retval = retval.vhdlAnd(tempval);
  }
  return retval;
}

EnumerationType
VHDLProcess::locateStableAttribute(const SignalBase *sig, 
					 const VHDLVTime time) {
  vector<Attribute *> attribute;
  Attribute *retval = 0;
  UniversalLongLongInteger lastEventTime;
  int i;

  ASSERT(sig != NULL);

  attribute = const_cast<SignalBase *>(sig)->getAttributeList();
  for (i = 0; i <sig->getNumAttributes(); i++) {
    if(attribute[i]->attrib == LAST_EVENT){
      retval = attribute[i];
    }
  }

  ASSERT((retval != NULL && retval->attrib == LAST_EVENT));
  lastEventTime = ((ScalarType *)retval->value)->getObject()->readVal();

  attribute = const_cast<SignalBase *>(sig)->getAttributeList();
  for (i = 0; i <sig->getNumAttributes(); i++) {
    if(attribute[i]->attrib == STABLE) 
      retval = attribute[i];
  }

  ASSERT((retval != NULL && retval->attrib == STABLE));
  UniversalLongLongInteger lvt(getTimeNow().getMajor());
  UniversalLongLongInteger timeExpr(time.getMajor());
  if (time == VHDLVTime::getVHDLVTimeZero()) {
    if( lastEventTime < lvt ){
      updateAttribute(retval, UNIVERSAL_TRUE);
    } else {
      updateAttribute(retval, UNIVERSAL_FALSE);
    }
  } 
  else {
    if( lastEventTime <= lvt.vhdlMinus(timeExpr) ){
      updateAttribute(retval, UNIVERSAL_TRUE);
    } 
    else {
      updateAttribute(retval, UNIVERSAL_FALSE);
    }
  }

  return (*(EnumerationType *) retval->value);
}

EnumerationType 
VHDLProcess::locateEventAttribute(const VHDLType *sig) {
  EnumerationType retval(ObjectBase::VARIABLE, SavantbooleanType_info);

  ASSERT(sig != NULL);

  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateEventAttribute((const ScalarType *)sig);
    break;
  case ARRAY_TYPE:
    retval = 
      locateEventAttribute((const ArrayType *) sig);
    break;
  case RECORD_TYPE:
    retval = locateEventAttribute((const RecordType*)sig);
    break;
  default:
    cerr << "Unknown Type!! Asking for Event Attribute" << endl;
    break;
  }
  return retval;
}

EnumerationType
VHDLProcess::locateEventAttribute(const ScalarType *sig) {
  EnumerationType retval(ObjectBase::VARIABLE, SavantbooleanType_info);

  ASSERT(sig != NULL);

  retval = locateEventAttribute((SignalBase*)sig->getObject());

  return retval;
}			     

EnumerationType 
VHDLProcess::locateEventAttribute(const ArrayType *sig) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(0), SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, UniversalBoolean(0), SavantbooleanType_info);
  int noOfElements = sig->length(0);

  for(int i = 0; (i < noOfElements); i++) {
    tempval = locateEventAttribute(&(sig->get_element(i)));
    retval = retval.vhdlOr(tempval);
  }
  
  return retval;
}

EnumerationType 
VHDLProcess::locateEventAttribute(const RecordType* sig) {
  EnumerationType retval(ObjectBase::VARIABLE, UniversalBoolean(0), SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, UniversalBoolean(0), SavantbooleanType_info);
  int noOfElements = sig->get_number_of_fields();

  if(noOfElements >= 1) {
    retval = locateEventAttribute((const VHDLType*)&(sig->get_field(1)));
  }

  for(int i=2; i< noOfElements; i++) {
    tempval = locateEventAttribute((const VHDLType*)&(sig->get_field(i)));
    retval = retval.vhdlOr(tempval);
  }
  return retval;
}

EnumerationType 
VHDLProcess::locateEventAttribute(SignalBase *sig) {
  UniversalLongLongInteger lastEventTime;

    lastEventTime = sig->getAllAttributes().last_event;

    // taking the default stuff
    if ( lastEventTime == UniversalLongLongInteger(getTimeNow().getMajor() )){ 
      sig->getAllAttributes().event = true;
      return SAVANT_BOOLEAN_TRUE;
    } 
    else  {
      sig->getAllAttributes().event = false;
      return SAVANT_BOOLEAN_FALSE;
    }    
}

EnumerationType *
VHDLProcess::locateTransactionAttribute(const VHDLType *sig) {
  EnumerationType *retval = 0;
  ASSERT(sig != NULL);
  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateTransactionAttribute((const ScalarType*)sig);
    break;
  case ARRAY_TYPE:
    retval = locateTransactionAttribute((const ArrayType *) sig);
    break;
  case RECORD_TYPE:
    retval = locateTransactionAttribute((const RecordType*)sig);
    break;
  default:
    cerr << "Unknown Type!! Asking for Transaction Attribute !"
	 << endl;
    break;
  }
  return retval;
}

EnumerationType*
VHDLProcess::locateTransactionAttribute(const ScalarType *sig) {

 EnumerationType *retval;

  ASSERT(sig != NULL);

  retval = locateTransactionAttribute((const SignalBase*)sig->getObject());

  return retval;
}			     

EnumerationType *
VHDLProcess::locateTransactionAttribute(const ArrayType *sig) {
  EnumerationType *retval = 0;
  EnumerationType *tempval;
  int noOfElements = sig->length(0);

  for(int i = 0; (i < noOfElements); i++) {
    tempval = locateTransactionAttribute(&(sig->get_element(i)));
    *retval = retval->vhdlAnd(*tempval);
    delete tempval;
  }
  
  return retval;
}

EnumerationType *
VHDLProcess::locateTransactionAttribute(const RecordType* sig) {
  EnumerationType *retval = 0;
  EnumerationType *tempval;
  int noOfElements = sig->get_number_of_fields();

  for(int i=1; i<= noOfElements; i++) {
    tempval = locateTransactionAttribute((const VHDLType*)&(sig->get_field(i)));
    *retval = retval->vhdlAnd(*tempval);
    delete tempval;
  }
  return retval;
}

EnumerationType *
VHDLProcess::locateTransactionAttribute(const SignalBase *sig) {

  vector<Attribute *> attribute;
  Attribute *retval = NULL;
  
  attribute = const_cast<SignalBase *>(sig)->getAttributeList();
  for (int i = 0; i <sig->getNumAttributes(); i++) {
    if(attribute[i]->attrib == TRANSACTION){
      retval = attribute[i];
    }
  }
  ASSERT((retval != NULL && retval->attrib == TRANSACTION));
  return ((EnumerationType *)retval->value);
}  

// ***************** Active Attributes **********************************
 
EnumerationType
VHDLProcess::locateActiveAttribute(const VHDLType *sig) {
  EnumerationType retval(ObjectBase::VARIABLE, SavantbooleanType_info);

  ASSERT(sig != NULL);

  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateActiveAttribute((const ScalarType*)sig);
    break;
  case ARRAY_TYPE:
    retval = locateActiveAttribute((const ArrayType *) sig);
    break;
  case RECORD_TYPE:
    retval = locateActiveAttribute((const RecordType*)sig);
    break;
  default:
    cerr << "Unknown Type!! Asking for Active Attribute !"
	 << endl;
    break;
  }
  return retval;
}


EnumerationType
VHDLProcess::locateActiveAttribute(const ScalarType *sig) {
  EnumerationType retval(ObjectBase::VARIABLE, SavantbooleanType_info);

  ASSERT(sig != NULL);

  retval = locateActiveAttribute((const SignalBase*)sig->getObject());

  return retval;
}			     

EnumerationType
VHDLProcess::locateActiveAttribute(const ArrayType *sig) {
  EnumerationType retval(ObjectBase::VARIABLE, SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, SavantbooleanType_info);
  int noOfElements = sig->length(0);
  
  for(int i = 0; (i < noOfElements); i++) {
    tempval = locateActiveAttribute(&(sig->get_element(i)));
    retval = retval.vhdlOr( tempval );
  }
  
  return retval;
}


EnumerationType 
VHDLProcess::locateActiveAttribute(const RecordType* sig) {
  EnumerationType retval(ObjectBase::VARIABLE, SavantbooleanType_info);
  EnumerationType tempval(ObjectBase::VARIABLE, SavantbooleanType_info);
  int noOfElements = sig->get_number_of_fields();

  for(int i=1; i<= noOfElements; i++) {
    tempval = locateActiveAttribute((const VHDLType*)&(sig->get_field(i)));
    retval = retval.vhdlOr( tempval );
  }
  return retval;
}

EnumerationType
VHDLProcess::locateActiveAttribute(const SignalBase *sig) {
  vector<Attribute *> attribute;
  Attribute *retval = NULL;
  UniversalLongLongInteger lastActiveTime;
  int i;

  ASSERT(sig != NULL);

  attribute = const_cast<SignalBase *>(sig)->getAttributeList();
  for(i = 0; i <sig->getNumAttributes(); i++) {
    if(attribute[i]->attrib == LAST_ACTIVE){
      retval = attribute[i];
    }
  }
  ASSERT((retval != NULL && retval->attrib == LAST_ACTIVE));
  lastActiveTime = (UniversalLongLongInteger &) ((PhysicalType *)retval->value)->getObject()->readVal();
 
  attribute = const_cast<SignalBase *>(sig)->getAttributeList();
  for(i = 0; i <sig->getNumAttributes(); i++) {
    if(attribute[i]->attrib == ACTIVE) 
      retval = attribute[i];
  }
  ASSERT((retval != NULL && retval->attrib == ACTIVE));

  if ( lastActiveTime == UniversalLongLongInteger(getTimeNow().getMajor() )){
    updateAttribute(retval, UNIVERSAL_TRUE);
    return (*(EnumerationType *) retval->value);
  } else {
    updateAttribute(retval, UNIVERSAL_FALSE);
    return (*(EnumerationType *) retval->value);
  }

}

// ***************** last_event Attributes **********************************
 
const PhysicalType
VHDLProcess::locateLastEventAttribute(const VHDLType *sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::getMax(), SavanttimeType_info);

  ASSERT(sig != NULL);

  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateLastEventAttribute((const ScalarType*)sig);
    break;
  case ARRAY_TYPE:
    retval = locateLastEventAttribute((const ArrayType *) sig);
    break;
  case RECORD_TYPE:
    retval = locateLastEventAttribute((const RecordType*)sig);
    break;
  default:
    cerr << "Unknown Type!! Asking for last_event Attribute !"
	 << endl;
    break;
  }
  return retval;
}


const PhysicalType
VHDLProcess::locateLastEventAttribute(const ScalarType *sig) {
  ASSERT(sig != NULL);

  return locateLastEventAttribute((const SignalBase*)sig->getObject());
}			     

const PhysicalType
VHDLProcess::locateLastEventAttribute(const ArrayType *sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::getMax(), SavanttimeType_info);
  PhysicalType tempval(ObjectBase::VARIABLE, UniversalLongLongInteger::getMax(), SavanttimeType_info);
  int noOfElements = sig->length(0);

  for(int i = 0; (i < noOfElements); i++) {
    tempval = locateLastEventAttribute(&(sig->get_element(i)));
    if( tempval <= retval ){
      retval = tempval;
    }
  }
  
  return retval;
}


const PhysicalType
VHDLProcess::locateLastEventAttribute(const RecordType* sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::getMax(), SavanttimeType_info);
  PhysicalType tempval(ObjectBase::VARIABLE, UniversalLongLongInteger::getMax(), SavanttimeType_info);
  int noOfElements = sig->get_number_of_fields();

  for(int i=1; i<= noOfElements; i++) {
    tempval = locateLastEventAttribute((const VHDLType*)&(sig->get_field(i)));
    if( tempval < retval ){
      retval = tempval;
    }
  }
  return retval;
}

const PhysicalType
VHDLProcess::locateLastEventAttribute(const SignalBase *sig) {
  ASSERT(sig != NULL);

  UniversalLongLongInteger lastEventTime = const_cast<SignalBase *>(sig)->getAllAttributes().last_event;
  lastEventTime = getTimeNow().getMajor() - lastEventTime;

  return PhysicalType(ObjectBase::VARIABLE, lastEventTime, SavanttimeType_info);
}

// ***************** last_active Attributes **********************************
 
PhysicalType
VHDLProcess::locateLastActiveAttribute(const VHDLType *sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::getMax(), SavanttimeType_info);

  ASSERT(sig != NULL);

  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateLastActiveAttribute((const ScalarType*)sig);
    break;
  case ARRAY_TYPE:
    retval = locateLastActiveAttribute((const ArrayType *) sig);
    break;
  case RECORD_TYPE:
    retval = locateLastActiveAttribute((const RecordType*)sig);
    break;
  default:
    cerr << "Unknown Type!! Asking for last_active Attribute !"
	 << endl;
    break;
  }
  return retval;
}

PhysicalType
VHDLProcess::locateLastActiveAttribute(const ScalarType *sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::getMax(), SavanttimeType_info);

  ASSERT(sig != NULL);

  retval = locateLastActiveAttribute((const SignalBase*)sig->getObject());

  return retval;
}			     

PhysicalType
VHDLProcess::locateLastActiveAttribute(const ArrayType *sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::getMax(), SavanttimeType_info);
  PhysicalType tempval(ObjectBase::VARIABLE, UniversalLongLongInteger::getMax(), SavanttimeType_info);
  int noOfElements = sig->length(0);

  for(int i = 0; (i < noOfElements); i++) {
    tempval = locateLastActiveAttribute((const VHDLType*)&(sig->get_element(i)));
    if( tempval < retval ){
      retval = tempval;
    }
  }
  
  return retval;
}


PhysicalType
VHDLProcess::locateLastActiveAttribute(const RecordType* sig) {
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::getMax(), SavanttimeType_info);
  PhysicalType tempval(ObjectBase::VARIABLE, UniversalLongLongInteger::getMax(), SavanttimeType_info);
  int noOfElements = sig->get_number_of_fields();

  for(int i=0; i< noOfElements; i++) {
    tempval = locateLastActiveAttribute((const VHDLType*)&(sig->get_field(i)));
    if( tempval < retval ){
      retval = tempval;
    }
  }
  return retval;
}

PhysicalType
VHDLProcess::locateLastActiveAttribute(const SignalBase *sig) {
  vector<Attribute *> attribute;
  Attribute *attribval = NULL;
  UniversalLongLongInteger lastActiveTime;
  PhysicalType retval(ObjectBase::VARIABLE, UniversalLongLongInteger::getMax(), SavanttimeType_info);

  ASSERT(sig != NULL);

  attribute = const_cast<SignalBase *>(sig)->getAttributeList();
  for (int i = 0; i < sig->getNumAttributes(); i++) {
    if(attribute[i]->attrib == LAST_ACTIVE){
      attribval = attribute[i];
      break;
    }
  }

  ASSERT(attribval != NULL && attribval->attrib == LAST_ACTIVE);

  lastActiveTime = attribval->value->getInt64Value();
  lastActiveTime = getTimeNow().getMajor() - lastActiveTime;
  retval = PhysicalType(ObjectBase::VARIABLE, lastActiveTime, SavanttimeType_info);

  return retval;
}

// ***************** last_value Attributes **********************************
 
VHDLType *
VHDLProcess::locateLastValueAttribute(const VHDLType *sig) {
  VHDLType *retval = 0;

  ASSERT(sig != NULL);

  switch(sig->get_kind()) {
  case INTEGER_TYPE:
  case REAL_TYPE:
  case ENUMERATION_TYPE:
  case PHYSICAL_TYPE:
    retval = locateLastValueAttribute((const ScalarType*)sig);
    break;
  case ARRAY_TYPE:
    retval = locateLastValueAttribute((const ArrayType*) sig);
    break;
  case RECORD_TYPE:
    retval = locateLastValueAttribute((const RecordType*)sig);
    break;
  default:
    cerr << "Unknown Type!! Asking for last_value Attribute !"
	 << endl;
    break;
  }
  return retval;
}

VHDLType *
VHDLProcess::locateLastValueAttribute(const ScalarType *sig) {
  VHDLType *retval;
  static ScalarType* ptr = NULL;
  ObjectBase* objectPtr  = NULL;
  ASSERT(sig != NULL);

  retval = locateLastValueAttribute((const SignalBase*)sig->getObject());
  if(retval == NULL) {
    switch(((SignalBase*)sig->getObject())->getAllAttributes().sigtype) {
    case VHDLData::UNIVERSAL_BOOLEAN:
    case VHDLData::UNIVERSAL_INTEGER:
      objectPtr = (ObjectBase *) new Variable<UniversalInteger>;
      objectPtr->updateVal(UniversalInteger(((SignalBase*)sig->getObject())->getAllAttributes().last_value.inttype));
      break;
    case VHDLData::UNIVERSAL_REAL:
      objectPtr = (ObjectBase *) new Variable<UniversalReal>;
      objectPtr->updateVal(UniversalReal(((SignalBase*)sig->getObject())->getAllAttributes().last_value.floattype));
      break;
    case VHDLData::UNIVERSAL_LONG_LONG_INTEGER:
      objectPtr = (ObjectBase *) new Variable<UniversalLongLongInteger>;
      objectPtr->updateVal(UniversalLongLongInteger((LONG) ((SignalBase*)sig->getObject())->getAllAttributes().last_value.physicaltype));
      break;
    default:
      objectPtr = NULL;
      break;
    }
    if(ptr != NULL) {
      delete ptr;
    }
    ptr = new ScalarType(ObjectBase::VARIABLE);
    ptr->setObject( objectPtr );
    retval = ptr;
  }
  return retval;
}			     

VHDLType *
VHDLProcess::locateLastValueAttribute(const ArrayType* /* sig */) {
  VHDLType *retval = NULL ;
  //VHDLType *tempval;
  //int noOfElements = sig->numElems;
  //for(int i=0; i< noOfElements; i++) {
  //  tempval = locateLastValueAttribute((VHDLType*)&(sig->get_element(i)));
  //}

  cerr << "VHDLProcess<State>::locateLastValueAttribute() : This will not work" 
       << endl; 
  abort();

  return retval;
}


VHDLType *
VHDLProcess::locateLastValueAttribute(const RecordType* /* sig */) {
  VHDLType *retval = NULL;

  cerr << "VHDLProcess<State>::locateLastValueAttribute():This will not work" 
       << endl; 
  abort();
  return retval;
}

// Caution: Does not return a new'ed pointer.  Retruns the same poointer
// as in the attribute.

VHDLType*
VHDLProcess::locateLastValueAttribute(const SignalBase *sig) {
  vector<Attribute *> attribute;
  Attribute *attribval = NULL;

  ASSERT(sig != NULL);

  attribute = const_cast<SignalBase *>(sig)->getAttributeList();
  for (int i = 0; i < sig->getNumAttributes(); i++) {
    if(attribute[i]->attrib == LAST_VALUE){
      attribval = attribute[i];
    }
  }

  //  ASSERT(attribval != NULL && attribval->attrib == LAST_VALUE);
  if(attribval != NULL && attribval->attrib == LAST_VALUE) {
    return attribval->value;
  }
  else {
    return NULL;
  }
}
