/* ==================================================== ======== ======= *
 *
 *  ucall.hpp: callback objects
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _ucall_hpp_
#define	_ucall_hpp_
//pragma ident	"@(#)ucall.hpp	ubit:03.06.04"
#include <ubit/ucond.hpp>

/** <b>UCall</b> and its template subclasses define a family of classes for 
 * <em><b>adding callback functions and methods</b></em> to other bricks.
 *
 * UClass is never instanciated nor used directly. Instead, you must use 
 * a variety of <b>ucall( )</b> or <b>uset( )</b> functions as explained below.
 *
 * <p><h3>1. Adding callbacks to UGroup / UBox / UWin objets (and subclasses)</h3>
 * <p>
 * <b>a) Calling non-member functions</b>
 * <pre><tt>
 *      ubutton("Click Me" + UOn::action / <b>ucall</b>(arg, <em>func1</em>))
 *      ubutton("Click Me" + <b>ucallref</b>(arg1, arg2, <em>func2</em>))
 * </tt></pre>
 *    - <b>UOn::xxx</b> is an <b>Event Condition</b> such as:
 *      UOn::action, UOn::select, UOn::mpress, etc. (see class UOn). 
 *    - if no condition is given (2nd example), UOn::action is taken as the default
 *    - <em>func1, func2</em> are <b>non-member functions</b> that can have 
 *      one or two arguments and can return any type including 'void'.
 *    - <b>ucall( )</b> <b>copies</b> the arguments while <b>ucallref( )</b> 
 *      passes them by <b>reference</b> (<b>CAUTION:</b> do not destroy these
 *      arguments somewhere else in the later case!)
 *<p>
 *<b>b) Calling member functions (ie. "methods") </b>
 *<pre><tt>
 *      ucheckbox("Select Me" + UOn::select / <b>ucall</b>(obj, &Myclass::<em>memfunc1</em>))
 *      uhbox("Press Me" + UOn::mpress / <b>ucallref</b>(obj, arg1, arg2, &Myclass::<em>memfunc2</em>)
 *</tt></pre>   
 *    - <em>memfunc1, memfunc2</em> are <b>member functions</b> that can have 
 *      zero, one or two arguments and can return any type including 'void'.
 *    - <em>obj</em> must be a pointer to an object whose class is 
 *      compatible with <em>Myclass</em> (ie. same class or subclass)
 *<p>
 *<b>c) Adding callbacks after object creation</b>
 *<p>
 *      This can be done by using the <b>add( )</b> method:
 *<pre><tt>
 *      UBox& b = ubutton("Click Me");
 *      b.add(UOn::mclick / ucall(....));
 *</tt></pre>   
 *<p>
 *<b>d) Callback Prototypes</b>
 *<p>
 *     They must have the same arguments as those that are passed to ucall().
 *     Note that the types must be IDENTICAL.
 *     Prototypes can also have an optional UEvent& argument (which must be
 *     the first argument when present).
 *
 *     Exemples:
 *<pre><tt>
 *      void func1(bool state) { ...}
 *     or:
 *      void func1(UEvent& e, bool state) { ...}
 *
 *      ucheckbox("Select Me" 
 *                + UOn::select   / <b>ucall</b>(true, func1)
 *                + UOn::unselect / <b>ucall</b>(false, func1)
 *                );
 *
 *
 *      void MyClass::memfunc2(UColor& fg, UBgolor& bg) { ...}
 *     or:
 *      void MyClass::memfunc2(UEvent& e, UColor& fg, UBgolor& bg) { ...}
 *
 *      MyClass*  obj = new MyClass();
 *      UColor&   col1 = ...;
 *      UBgcolor& col2 = ...;
 *      ubutton("Click Me" + <b>ucallref</b>(obj, col1, col2, &Myclass::memfunc2);
 *</tt></pre>   
 *
 *<p>     
 *<h3>2. Predefined Callback objects</h3>
 *
 *    - <b>uset(a,b)</b> will perform:  <em>a->set(b)</em>
 *      <br>when the condition is verified.
 *
 *    - <b>uassign(a,b)</b> will perform:  <em>a = b</em>
 *      <br>when the condition is verified.
 *
 *    - <b>usetref(a,b)</b> and <b>uassignref(a,b)</b> are similar but 
 *      <em>passes 'b' by reference</em>
 *
 *    Examples:
 * <pre>
 *     UStr s1;
 *     UStr s2 = "toto";
 *
 *     ubutton("Press Me" + UOn::mpress / uset(&a, &b))
 *   or:
 *     ubutton("Press Me" + UOn::mpress / usetref(&a, b))
 * </pre>
 *
 *    - ucloseWin(status) closes the window (either a UMenu or a UDialog)
 *      that contains this object. 
 *      'status' must be 0 or a positive value (see: UGroup::closeWin() and
 *      UWin::close())
 *
 *     Example: the dialog will be closed when the "Close" button is clicked
 *     or released:
 * <pre>
 *          udialog( ... + ubutton("Close" + ucloseWin()) + ... )
 *          udialog( ... + ubutton("Close" + UOn::mrelease / ucloseWin()) + ... )
 * </pre>
 */
class UGenCallImpl: public UBrick {
  virtual void addingTo(class ULink *selflink, UGroup *parent);
  virtual void removingFrom(class ULink *selflink, UGroup *parent);
  // NOTE that removingFrom() requires a destructor to be defined.
public:
  UGenCallImpl() {}
  virtual ~UGenCallImpl() {destructs();}
};

template <class EE> 
class UGenCall: public UGenCallImpl {
public:
  virtual void call(EE&) = 0;
};

typedef UGenCall<class UEvent> UCall;
typedef UGenCall<class UBrick> UListCall;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

template <class CC, class VAL, class EE>
class UCall_show : public UGenCall<EE>{
  CC  obj;
  VAL val;
public:
  UCall_show(CC _obj, VAL _val) : UCall(), obj(_obj), val(_val) {}
  void call(EE&) {obj->show(val);}
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// ushow(a,b) will call 'a->show(b)'
// 'a' must have a show() method that can take 'b' as an argument

template <class CC, class VAL> 
UCall& ushow(CC* obj, VAL val) 
{return *(new UCall_show<CC*,VAL,UEvent>(obj, val));}

template <class CC, class VAL> 
UCall& ushowref(CC* obj, VAL& val) 
{return *(new UCall_show<CC*,VAL&,UEvent>(obj, val));}

/* ==================================================== ======== ======= */

template <class CC, class VAL, class EE>
class UCall_set : public UGenCall<EE>{
  CC  obj;
  VAL val;
public:
  UCall_set(CC _obj, VAL _val) : UCall(), obj(_obj), val(_val) {}
  void call(EE&) {obj->set(val);}
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// uset(a,b) will call 'a->set(b)'
// 'a' must have a set() method that can take 'b' as an argument

template <class CC, class VAL> 
UCall& uset(CC* obj, VAL val) 
{return *(new UCall_set<CC*,VAL,UEvent>(obj, val));}

template <class CC, class VAL> 
UCall& usetref(CC* obj, VAL& val) 
{return *(new UCall_set<CC*,VAL&,UEvent>(obj, val));}

/* ==================================================== ======== ======= */

template <class CC, class VAL, class EE>
class UCall_assign : public UGenCall<EE>{
  CC  obj;
  VAL val;
public:
  UCall_assign(CC _obj, VAL _val) : UCall(), obj(_obj), val(_val) {}
  void call(EE&) {*obj = val;}
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// uassign(a,b) will call 'a = b'

template <class CC, class VAL> 
UCall& uassign(CC* obj, VAL val) 
{return *(new UCall_assign<CC*,VAL,UEvent>(obj, val));}

template <class CC, class VAL> 
UCall& uassignref(CC* obj, VAL& val) 
{return *(new UCall_assign<CC*,VAL&,UEvent>(obj, val));}

// ==================================================== [Elc:03] ======= 
// ==================================================== ======== =======
// Callback methods with NO argument:

template <class CC, class RR, class MM, class EE> 
class UCall_M_0a : public UGenCall<EE>{
  CC obj;
  RR (MM::*member)();
public:
  UCall_M_0a(CC _obj, RR(MM::*mm)()) : obj(_obj), member(mm) {}
  void call(EE&) {(obj->*member)();}
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

template <class CC, class RR,  class MM, class EE> 
class UCall_M_e0a : public UGenCall<EE> {
  CC obj;
  RR (MM::*member)(EE&);
public:
  UCall_M_e0a(CC _obj, RR(MM::*mm)(EE&)) : obj(_obj), member(mm) {}
  void call(EE& e) {(obj->*member)(e);}
};

// ==================================================== ======== ======= 

template <class CC, class RR, class MM> 
UCall& ucall(CC* obj, RR(MM::*mm)()) 
{return *(new UCall_M_0a<CC*,RR,MM,UEvent>(obj, mm));}

template <class CC, class RR, class MM> 
UCall& ucall(CC* obj, RR(MM::*mm)(UEvent&)) 
{return *(new UCall_M_e0a<CC*,RR,MM,UEvent>(obj, mm));}

template <class CC, class RR, class MM> 
UListCall& ucall(CC* obj, RR(MM::*mm)(UBrick&)) 
{return *(new UCall_M_e0a<CC*,RR,MM,UBrick>(obj, mm));}


// ==================================================== [Elc:03] =======
// ==================================================== ======== =======
// Callback methods with ONE argument:

template <class CC, class A1, class RR, class MM, class EE> 
class UCall_M_1a : public UGenCall<EE> {
  CC obj;
  RR (MM::*member)(A1);
  A1 a1;
public:
  UCall_M_1a(CC _obj, RR(MM::*mm)(A1), A1 _a1) : obj(_obj), member(mm), a1(_a1) {}
  void call(EE&) {(obj->*member)(a1);}
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

template <class CC, class A1, class RR, class MM, class EE> 
class UCall_M_e1a : public UGenCall<EE> {
  CC obj;
  RR (MM::*member)(EE&, A1);
  A1 a1;
public:
  UCall_M_e1a(CC _obj, RR(MM::*mm)(EE&,A1), A1 _a1) : obj(_obj), member(mm), a1(_a1) {}
  void call(EE& e) {(obj->*member)(e, a1);}
};

// ==================================================== ======== ======= 
// passage par VALEUR:
// NNN nouvelle version qui evite les ambiguites NNN :
// methode a la fin / type retour RR / passage UEvent& ou UBrick& 

template <class CC, class A1, class RR, class MM> 
UCall& ucall(CC* obj, A1 a1, RR(MM::*mm)(A1)) 
{return *(new UCall_M_1a<CC*,A1,RR,MM,UEvent>(obj, mm, a1));}

template <class CC, class A1, class RR, class MM> 
UCall& ucall(CC* obj, A1 a1, RR(MM::*mm)(UEvent&, A1)) 
{return *(new UCall_M_e1a<CC*,A1,RR,MM,UEvent>(obj, mm, a1));}

template <class CC, class A1, class RR, class MM> 
UListCall& ucall(CC* obj, A1 a1, RR(MM::*mm)(UBrick&, A1)) 
{return *(new UCall_M_e1a<CC*,A1,RR,MM,UBrick>(obj, mm, a1));}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// passage par REFERENCE:

template <class CC, class A1, class RR, class MM> 
UCall& ucallref(CC* obj, A1& a1, RR(MM::*mm)(A1&))
{return *(new UCall_M_1a<CC*,A1&,RR,MM,UEvent>(obj, mm, a1));}

template <class CC, class A1, class RR, class MM> 
UCall& ucallref(CC* obj, A1& a1, RR(MM::*mm)(UEvent&, A1&)) 
{return *(new UCall_M_e1a<CC*,A1&,RR,MM,UEvent>(obj, mm, a1));}

template <class CC, class A1, class RR, class MM> 
UListCall& ucallref(CC* obj, A1& a1, RR(MM::*mm)(UBrick&, A1&)) 
{return *(new UCall_M_e1a<CC*,A1&,RR,MM,UBrick>(obj, mm, a1));}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// passage par CONST REFERENCE:

template <class CC, class A1, class RR, class MM> 
UCall& ucallref(CC* obj, const A1& a1, RR(MM::*mm)(const A1&))
{return *(new UCall_M_1a<CC*,const A1&,RR,MM,UEvent>(obj, mm, a1));}

template <class CC, class A1, class RR, class MM> 
UCall& ucallref(CC* obj, const A1& a1, RR(MM::*mm)(UEvent&, const A1&)) 
{return *(new UCall_M_e1a<CC*,const A1&,RR,MM,UEvent>(obj, mm, a1));}

template <class CC, class A1, class RR, class MM> 
UListCall& ucallref(CC* obj, const A1& a1, RR(MM::*mm)(UBrick&, const A1&)) 
{return *(new UCall_M_e1a<CC*,const A1&,RR,MM,UBrick>(obj, mm, a1));}


// ==================================================== [Elc:03] ======= 
// ==================================================== ======== ======= 
// Callback Methods with TWO arguments

template <class CC, class A1, class A2, class RR, class MM, class EE> 
class UCall_M_2a : public UGenCall<EE> {
  CC obj;
  RR (MM::*member)(A1, A2);
  A1 a1; A2 a2;
public:
  UCall_M_2a(CC _obj, RR(MM::*mm)(A1,A2), A1 _a1, A2 _a2)
    : obj(_obj), member(mm), a1(_a1), a2(_a2) {} 
  void call(EE&) {(obj->*member)(a1, a2);}
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

template <class CC, class A1, class A2, class RR, class MM, class EE> 
class UCall_M_e2a : public UGenCall<EE> {
  CC obj;
  RR (MM::*member)(EE&, A1, A2);
  A1 a1; A2 a2;
public:
  UCall_M_e2a(CC _obj, RR(MM::*mm)(EE&,A1,A2), A1 _a1, A2 _a2)
    : obj(_obj), member(mm), a1(_a1), a2(_a2) {} 
  void call(EE& e) {(obj->*member)(e, a1, a2);}
};

// ==================================================== ======== ======= 
// passage par VALEUR:

template <class CC, class A1, class A2, class RR, class MM> 
UCall& ucall(CC* obj, A1 a1, A2 a2, RR(MM::*mm)(A1,A2)) 
{return *(new UCall_M_2a<CC*,A1,A2,RR,MM,UEvent>(obj, mm, a1, a2));}

template <class CC, class A1, class A2, class RR, class MM> 
UCall& ucall(CC* obj, A1 a1, A2 a2, RR(MM::*mm)(UEvent&,A1,A2)) 
{return *(new UCall_M_e2a<CC*,A1,A2,RR,MM,UEvent>(obj, mm, a1, a2));}

template <class CC, class A1, class A2, class RR, class MM> 
UListCall& ucall(CC* obj, A1 a1, A2 a2, RR(MM::*mm)(UBrick&,A1,A2)) 
{return *(new UCall_M_e2a<CC*,A1,A2,RR,MM,UBrick>(obj, mm, a1, a2));}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// passage par REFERENCE:

template <class CC, class A1, class A2, class RR, class MM> 
UCall& ucallref(CC* obj, A1& a1, A2& a2, RR(MM::*mm)(A1&,A2&)) 
{return *(new UCall_M_2a<CC*,A1&,A2&,RR,MM,UEvent>(obj, mm, a1, a2));}

template <class CC, class A1, class A2, class RR, class MM> 
UCall& ucallref(CC* obj, A1& a1, A2& a2, RR(MM::*mm)(UEvent&,A1&,A2&)) 
{return *(new UCall_M_e2a<CC*,A1&,A2&,RR,MM,UEvent>(obj, mm, a1, a2));}

template <class CC, class A1, class A2, class RR, class MM> 
UListCall& ucallref(CC* obj, A1& a1, A2& a2, RR(MM::*mm)(UBrick&,A1&,A2&)) 
{return *(new UCall_M_e2a<CC*,A1&,A2&,RR,MM,UBrick>(obj, mm, a1, a2));}


/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// Callback Function with ONE argument

template <class A1, class RR, class EE> 
class UCall_F_1a : public UGenCall<EE> {
  RR (*fun)(A1);
  A1 a1;
public:
  UCall_F_1a( RR(*f)(A1), A1 _a1) : fun(f), a1(_a1) {}
  void call(EE&) {(*fun)(a1);}
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

template <class A1, class RR, class EE> 
class UCall_F_e1a : public UGenCall<EE> {
  RR (*fun)(EE&, A1);
  A1 a1;
public:
  UCall_F_e1a( RR(*f)(EE&,A1), A1 _a1) : fun(f), a1(_a1) {}
  void call(EE& e) {(*fun)(e, a1);}
};

/* ==================================================== ======== ======= */
// passage par VALEUR:

template <class A1, class RR> 
UCall& ucall(A1 a1, RR(*f)(A1)) 
{return *(new UCall_F_1a<A1,RR,UEvent>(f, a1));}

template <class A1, class RR> 
UCall& ucall(A1 a1, RR(*f)(UEvent&,A1)) 
{return *(new UCall_F_e1a<A1,RR,UEvent>(f, a1));}

template <class A1, class RR> 
UListCall& ucall(A1 a1, RR(*f)(UBrick&,A1)) 
{return *(new UCall_F_e1a<A1,RR,UBrick>(f, a1));}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// passage par REFERENCE:

template <class A1, class RR> 
UCall& ucallref(A1& a1, RR(*f)(A1&)) 
{return *(new UCall_F_1a<A1&,RR,UEvent>(f, a1));}

template <class A1, class RR> 
UCall& ucallref(A1& a1, RR(*f)(UEvent&,A1&)) 
{return *(new UCall_F_e1a<A1&,RR,UEvent>(f, a1));}

template <class A1, class RR> 
UListCall& ucallref(A1& a1, RR(*f)(UBrick&,A1&)) 
{return *(new UCall_F_e1a<A1&,RR,UBrick>(f, a1));}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// passage par CONST REFERENCE:

template <class A1, class RR> 
UCall& ucallref(const A1& a1, RR(*f)(const A1&)) 
{return *(new UCall_F_1a<const A1&,RR,UEvent> (f, a1));}

template <class A1, class RR> 
UCall& ucallref(const A1& a1, RR(*f)(UEvent&,const A1&)) 
{return *(new UCall_F_e1a<const A1&,RR,UEvent> (f, a1));}


/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// Callback Function with TWO arguments

template <class A1, class A2, class RR, class EE> 
class UCall_F_2a : public UGenCall<EE>{
  RR (*fun)(A1, A2);
  A1 a1; A2 a2;
public:
  UCall_F_2a( RR(*f)(A1,A2), A1 _a1, A2 _a2) : fun(f), a1(_a1), a2(_a2) {}
  void call(EE&) {(*fun)(a1, a2);}
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

template <class A1, class A2, class RR, class EE> 
class UCall_F_e2a : public UGenCall<EE>{
  RR (*fun)(EE&, A1, A2);
  A1 a1; A2 a2;
public:
  UCall_F_e2a( RR(*f)(EE&,A1,A2), A1 _a1, A2 _a2) : fun(f), a1(_a1), a2(_a2) {}
  void call(EE& e) {(*fun)(e, a1, a2);}
};

/* ==================================================== ======== ======= */
// passage par VALEUR:

template <class A1, class A2, class RR> 
UCall& ucall(A1 a1, A2 a2, RR(*f)(A1,A2)) 
{return *(new UCall_F_2a<A1,A2,RR,UEvent>(f, a1, a2));}

template <class A1, class A2, class RR> 
UCall& ucall(A1 a1, A2 a2, RR(*f)(UEvent&,A1,A2)) 
{return *(new UCall_F_e2a<A1,A2,RR,UEvent>(f, a1, a2));}

template <class A1, class A2, class RR> 
UListCall& ucall(A1 a1, A2 a2, RR(*f)(UBrick&,A1,A2)) 
{return *(new UCall_F_e2a<A1,A2,RR,UBrick>(f, a1, a2));}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// passage par REFERENCE:

template <class A1, class A2, class RR> 
UCall& ucallref(A1& a1, A2& a2, RR(*f)(A1&,A2&))
{return *(new UCall_F_2a<A1&,A2&,RR,UEvent>(f, a1, a2));}

template <class A1, class A2, class RR> 
UCall& ucallref(A1& a1, A2& a2, RR(*f)(UEvent&,A1&,A2&))
{return *(new UCall_F_e2a<A1&,A2&,RR,UEvent>(f, a1, a2));}

template <class A1, class A2, class RR> 
UListCall& ucallref(A1& a1, A2& a2, RR(*f)(UBrick&,A1&,A2&))
{return *(new UCall_F_e2a<A1&,A2&,RR,UBrick>(f, a1, a2));}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
// passage par CONST REFERENCE:

template <class A1, class A2, class RR> 
UCall& ucallref(const A1& a1, const A2& a2, RR(*f)(const A1&,const A2&))
{return *(new UCall_F_2a<const A1&, const A2&,RR,UEvent>(f, a1, a2));}

template <class A1, class A2, class RR> 
UCall& ucallref(const A1& a1, const A2& a2, RR(*f)(UEvent&,const A1&,const A2&))
{return *(new UCall_F_e2a<const A1&, const A2&,RR,UEvent>(f, a1, a2));}


#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
