/* ==================================================== ======== ======= *
 *
 *  uchoice.hpp
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _uchoice_hpp_
#define	_uchoice_hpp_
//pragma ident	"@(#)uchoice.hpp		ubit:03.06.00"
#include <ubit/uprop.hpp>


/** Exclusive selector for making UListbox.
 */
class UChoice : public UProp {
public:
  UChoice(UBox* container = null);
  ///< constructor.
  
  friend UChoice& uchoice() {return *(new UChoice());}
  ///< creator shortcut (see constructor for details).

  virtual ~UChoice();
  ///< destructor.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void setCanUnselectMode(bool);
  virtual bool isCanUnselectMode() const;
  ///< clicking on a selected item will unselect it if true (default is false).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual UBox* getItem() const;
  ///< returns the selected item (null if none selected).

  virtual UBox* getLastItem() const;
  virtual int   getLastIndex() const;

  virtual UBox* setItem(UBox*, bool update = true);
  virtual UBox* setItem(UBox&, bool update = true);
  virtual void select(UBox&);
  /**< selects this item (deselects all items if UBox* is null).
   * setItem() returns the selected item (if any)
  */
  
  virtual void unselect();
  ///< unselects all items

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual int getIndex() const;
  ///< returns the index of the selected item (-1 if nothing selected; 0 if first item selected).

  virtual UBox* setIndex(int, bool update = true);
  virtual void select(int);
  /**< selects the item at this index in the list (0 is first item, -1 last item).
    * setIndex() returns the selected item (if any)
    */
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void update();
  ///< updates graphics (this function must be called when adding or removing items).

  //EX: virtual void changed(bool update_now);
  void changed(bool update_now, UBox* target);
  /**< called when the selection is changed.
   * This function:
   * - updates grahics (if arg is true)
   * - first fires object's UOn::change callbacks
   * - then fires parents' UOn::select callbacks
   */

#ifndef NO_DOC
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  virtual void putProp(UContext*, class UCtrl*);

  virtual void addingTo(ULink *selflink, UGroup *parent);
  // NB: removingFrom() requires a destructor to be defined
  virtual void removingFrom(ULink *selflink, UGroup *parent);

  UBox* itemChanged(UEvent&);  // return the selcetd item if any

protected:
  int   index;
  bool  can_unselect_mode;
  uptr<UBox> container;
#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Exclusive selector for selecting boxes (eg. UCheckbox, UButton, etc.).
 *  <p>This selector must be a child of the items it selects.
 *  These items are automatically made selectable (for instance an UButton
 *  will becone selectable if an USelect is added to it as a child). 
 *  <pre>
 *  Exemple:
 *    URadioSelect& sel = uradioSelect();
 *    ubox( ubutton(sel + "One" + UMode::selected)
 *          + ucheckbox(sel + "Two")
 *          + utextbox(sel + "Three")
 *        ) 
 *  </pre>
 */
class URadioSelect : public UProp {
public:
  URadioSelect();
  URadioSelect(UIntg& index);
  /**< constructor.
   * Notes: 
   * - this selector must be a child of the items it selects
   * - selection is exclusive
   * - no item is initially selected
   */

  friend URadioSelect& uradioSelect() {return *(new URadioSelect());}
  ///< creator shortcut (see constructor for details).

  virtual ~URadioSelect();
  ///< destructor.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void setCanUnselectMode(bool);
  virtual bool isCanUnselectMode() const;
  ///< clicking on a selected item will unselect it if true (default is false).

  virtual UGroup* getItem() const;
  ///< returns the selected item (null if none selected)

  virtual void setItem(UGroup*);
  virtual void setItem(UGroup&);
  virtual void select(UGroup&);
  ///< selects this item (deselects all items if null given as an arg).

  virtual void unselect();
  ///< unselects all items

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  UIntg& index() const {return *pindex;}
  virtual int  getIndex() const;
  ///< returns the index of the selected item (-1 if nothing selected; 0 if first item selected).

  virtual void setIndex(int);
  virtual void setIndex(const UIntg&);
  virtual void select(int);
  ///< selects the item at this index in the list (0 is first item, -1 last item).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void update();
  ///< updates graphics (this function must be called when adding or removing items).

  virtual void changed(bool update_now);
  /**< called when the selection is changed.
   * This function:
   * - updates grahics (if arg is true)
   * - first fires object's UOn::change callbacks
   * - then fires parents' UOn::select callbacks
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

#ifndef NO_DOC

  virtual void putProp(UContext*, class UCtrl*);

  virtual void addingTo(ULink *selflink, UGroup *parent);
  // NB: removingFrom() requires a destructor to be defined
  virtual void removingFrom(ULink *selflink, UGroup *parent);

protected:
  void setIndexImpl();

  bool  can_unselect_mode;
  uptr<UIntg> pindex;
  uptr<UCall> pselect_callback;
  void itemChanged(UEvent&);
#endif
};


#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
