#ifndef STATE_MANAGER_IMPLEMENTATION_BASE_H
#define STATE_MANAGER_IMPLEMENTATION_BASE_H

// Copyright (c) 1994-1996 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY 
// FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT 
// PARTIES PROVIDE THE PROGRAM "AS IS" WITHOUT WARRANTY OF ANY KIND, 
// EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE 
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
// PURPOSE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE 
// PROGRAM IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME 
// THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION. 

// IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING 
// WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR 
// REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR 
// DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL 
// DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE THE PROGRAM 
// (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED 
// INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE OF 
// THE PROGRAM TO OPERATE WITH ANY OTHER PROGRAMS), EVEN IF SUCH HOLDER 
// OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES. 

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

#include "warped.h"
#include "SetObject.h"
#include "StateManager.h"
#include "SimulationManager.h"

#include <set>
using std::multiset;

/** The StateManagerImplementationBase class.

    The StateManagerImplementationBase class is an implementation of
    the abstract StateManager class. It contains the methods and class
    attributes common to the PeriodicStateManager and the
    AdaptiveStateManager.

*/
class StateManagerImplementationBase : public StateManager {
public:

   /**@name Public Class Methods of StateManagerImplementationBase. */
   //@{
  
   /** Constructor.

       @param simMgr a handle to the simulation manager
       @param period State-saving period.
   */
   StateManagerImplementationBase(SimulationManager *simMgr,
                                  unsigned int period = 0);

   /// Default Destructor.
   virtual ~StateManagerImplementationBase();

   /** Save the state of the object at the specified time.
      
       @param currentTime Time at which the state is saved.
       @param object The object whose state is saved.
   */
   virtual void saveState(const VTime &currentTime, SimulationObject* object);

   /// return the state saving period
   unsigned int getStatePeriod();
   
   /** Rollback and restore the state of the object at the specified time.

       @param rollbackTime Rollback to this time and restore state.
       @param SimulationObject The object whose state should be restored.
       @return the time of restored state
   */
   virtual const VTime &restoreState(const VTime &rollbackTime,
                                     SimulationObject* object);

   /** Fossil collect up to the specified simulation time.

       @param garbageCollectTime Garbage collect upto this time.
       @param object Object whose states are gbg collected.
       @return lowest time-stamped state in state queue
   */
   virtual const VTime &garbageCollect(const VTime &garbageCollectTime,
				       SimulationObject* object);
  
   /** Print state queue.

       @param currentTime Print statequeue upto current time.
       @param object Object whose statequeue is printed.
   */
   virtual void printStateQueue(const VTime &currentTime,
                                SimulationObject *object, ostream &out);
   
   //@} // End of Public Class Methods of StateManagerImplementationBase.

protected:
   
   /**@name Protected Class Attributes of StateManagerImplementationBase. */
   //@{

   /// handle to the simulation manager
   SimulationManager *const mySimulationManager;
   
   /// The state period - if infrequent, then statePeriod != 0.
   unsigned int statePeriod;

   /// The state saving period of each object
   vector<unsigned int> objectStatePeriod;
   
   /// Time since the last saved state.
   vector<int> periodCounter;

   /// A state queue for every object on this simulation manager.
   multiset< SetObject<State> > *myStateQueue;

   //@} // End of Protected Class Attributes of StateManagerImplementationBase.
};

#endif






