package WebminLdapSkolelinux;
use base 'CGI::Application';
use strict;
use Data::Dumper;
use URI::Escape;
use HTML::FromText;
use Unicode::String qw(utf8 latin1);
use HTML::Template;
use File::Temp qw(tempfile);
use Digest::MD5 qw(md5_hex);
use Storable qw(store_fd retrieve);
#use diagnostics;

require '/usr/share/webmin/web-lib.pl';
use vars qw(%text %config $tb $cb $remote_user);

require '/usr/share/webmin/ldap-users/ldap-users.pl'
  ;    # This should become a .pm of its own
require '/usr/share/webmin/ldap-users/functions.pl';

my %global;

# this happens before we enter any runmodes. we try to detect state
sub cgiapp_prerun {
    my ( $self, $rm ) = @_;    #XXX
    my $q = $self->query();

    my $can_run = ldap_check_capabilities(
        {   #we could run without the capabilities field itself.
            capabilities => 1,    
            nextID       => 1,
#            ageGroup     => 2,
            groupType    => 1,
            aclGroup     => 1,
            attic        => 1,
        }
    );
    $self->prerun_mode('UpgradeBackend') unless $can_run;

    my $filename = $q->cookie('filename');
    my $hash     = $q->cookie('hash');
    restore_state( $filename, $hash );

    if ( $q->param("query_string") ) {
	$global{state}{shortcut_back} = $q->param("query_string");
    }

    if ( $rm eq 'EnterManyUsers' ) {
        if ( check_form_ok($q) and $q->param('CreateUsers') ) {

            # note that check form will be executed unconditionally! 
            $self->prerun_mode('CreateUsers');
        }
        else {
            $self->prerun_mode('EnterManyUsers');
        }
    }

    elsif ( $rm eq 'AdminSearch' ) {
        if ( $q->param('operation_add_users') ) {
            $self->prerun_mode('AddUsers');
        }
        elsif ($q->param('operation_add_groups')) {
            $self->prerun_mode('AddGroups');
        }
        else {
            $self->prerun_mode('AdminSearch');
        }
    }

    elsif ( $rm eq 'CreateUsers' ) {
        my $data_quality = check_data_sufficent($q);
        if ( $data_quality eq "ok" ) {
            unless ( extract_rows_from_raw_data($q) ) {
                $self->prerun_mode('AddUsers');
            }
            $self->prerun_mode('CreateUsers');
        }
        else {
            $self->prerun_mode('Fileimport');
            $global{warning} = $data_quality;
        }
    }

    elsif ( $rm eq 'AddGroups' ) {
        if ( check_form_ok($q) and ( $q->param("CreateGroups") ) ) {
            $self->prerun_mode('CreateGroups');
        }
        else {
            $self->prerun_mode('AddGroups');
        }
    }
}

sub setup {
    my $self = shift;
    $self->mode_param('rm');

    $self->run_modes( [
            qw/
            UpgradeBackend

            AddUsers
            Fileimport
            EnterManyUsers
            CreateUsers

            AddGroups
            CreateGroups

            AdminSearch
            EditUserData
            EditUserClasses
            EditClassData
            EditClassUsers
            /
        ]
    );

    # Do stuff common to all modes
    init_config();            # gives me %config, $cb, tb, $text
    read_adduser_config();    # puts adduser config into %config
    read_miniserv_config();   # pulls in miniserv config for cookie sttings


    my $username;
    if ( $remote_user eq "root" ){
	$username = $config{'rootdn'};
    }
    else {
	$username = "uid=$remote_user,ou=People,$config{'basedn'}"
    }

    ldap_connect(             # Connect to the ldap server.
        $config{'server'}, $username,
        $config{'basedn'}, $config{'FIRST_UID'} || 10000,
        $config{'LAST_UID'} || 29999,
    );

    # set the default mode for users
    if ( is_silly( $remote_user ) ) {
	$self->start_mode('EditUserData');
	$self->query->param( -name  => "user_name" , 
			     -value => $remote_user );
    }
    else {
	$self->start_mode('AdminSearch');
    }
}

sub cgiapp_postrun {

    # $output is a reference to the results of a run mode.
    my ( $self, $output ) = @_;
    my $q = $self->query;

    my ( $hash, $filename ) = dump_state();

    my $file_cookie = $q->cookie(
        -name    => 'filename',
        -value   => $filename,
        -expires => "+" . $config{'logouttime'} . "m",
        -path    => '/'
    );
    my $hash_cookie = $q->cookie(
        -name    => 'hash',
        -value   => $hash,
        -expires => "+" . $config{'logouttime'} . "m",
        -path    => '/'
    );
    
    $self->header_props(
        -cookie  => [ $file_cookie, $hash_cookie ],
        -type    => 'text/html',
        -charset => 'utf-8'
    );
}

sub UpgradeBackend {
    my $self = shift;
    my $q    = $self->query;
    my $output;
    my $template = $self->load_tmpl('UpgradeBackend.include');
    fill_in_template($template);
    $output .= $template->output;
    return $output;
}

sub EditUserClasses {
    my $self = shift;
    my $q    = $self->query;
    my %par  = $q->Vars;

    my $user_login = $par{user_name};
    my $rootpw     = get_root_password($q);

    for my $group_name ( split ( /\0/, $par{"remove_from"} ) ) {
        ldap_del_user_from_group( $user_login, $group_name, $rootpw );
    }
    for my $group_name ( split ( /\0/, $par{"add_to"} ) ) {
        ldap_add_user_to_group( $user_login, $group_name, $rootpw );
    }

    #create one list of all classes a user is in and one he is not in
    my @member_list = ldap_get_member_grouplist( $user_login, "school_class" );
    my @member_web_list = ldap2web( \@member_list, [ "cn", "description" ] );
    my @not_member_list =
      ldap_get_not_member_grouplist( $user_login, "school_class" );
    my @not_member_web_list =
      ldap2web( \@not_member_list, [ "cn", "description" ] );

    my $entry          = ldap_get_user($user_login);
    my $user_full_name = $entry->get_value('cn');
    my $name           = text2html "$user_full_name ($user_login)";
    my $output;
    
    
    my $template = $self->load_tmpl('EditUserClasses.include');
    fill_in_template($template);
    $template->param(
        RemoveUserFromGroups => text( "removeuserfromgroups", $name ),
        AddUserToGroups      => text( "addusertogroups",      $name ),
        edituserclasses      => text( "edituserclasses",      $name ),
        username             => $user_login,
        MemberLoop           => \@member_web_list,
        NoMemberLoop         => \@not_member_web_list,
    );

    $output .= $template->output;
    return $output;
}

sub EditUserData {
    my $self = shift;
    my $q    = $self->query;
    my %par  = $q->Vars;
    my ($output, $warning);

    my $unauthorized = check_authorisation();    #

    my ( @edit_items, @display_items );
    if ( $unauthorized ) {
	@edit_items = qw/userPassword passwordreconfirm/;
	
	@display_items = qw/cn uid uidNumber gidNumber shadowFlag 
	    loginShell homeDirectory mailMessageStore description/;
    }
    else {
	@edit_items = qw/cn userPassword passwordreconfirm loginShell
	    homeDirectory mailMessageStore description/;
	
	@display_items = qw/uid uidNumber gidNumber shadowFlag/;
    }
    
    my $user_name = $par{user_name};
    my $rootpw    = get_root_password($q);

    # write the changes to ldap
    my %i;
    for my $item (@edit_items) {
        $i{$item} = $par{$item} if ( $par{$item} );
    }
    my (@passwd_warning, $passwd_warning);
    if ( $par{userPassword} ne '' ) {

        push @passwd_warning, "passwds_not_equal"
	    unless ( $par{userPassword} eq $par{passwordreconfirm} );

	$passwd_warning = 
	    check_passwd_quality( $par{userPassword}, 
				  \@passwd_warning );


	unless ( $passwd_warning ) { # the password seems fine!  

	    change_samba( $user_name, $par{userPassword}, $rootpw) 
		if $config{"sambasync"};
            $i{userPassword} = "{crypt}" . gen_crypt( $par{userPassword} );

        }
        else {
            delete $i{"userPassword"};
	    $warning .= $passwd_warning;
        }
    }
    delete $i{passwordreconfirm} if $i{passwordreconfirm};

    if( keys %i ) {
	debug(\%i);
	my $ldap_warning =
	    ldap_modify_user( $user_name, \%i, $rootpw );
	$ldap_warning = "EditUserDataSuccessT" 
	    unless ( $ldap_warning or $warning );
	$warning .= text($ldap_warning)?text($ldap_warning):$ldap_warning;
    }

    my @member_list = ldap_get_member_grouplist( $user_name, "*" );
    my $groups;
    for my $entry (@member_list) {
        next unless ($entry);
        my $cn          = $entry->get_value("cn");
        my $description = $entry->get_value("description");
        $groups .= "$cn";
        $groups .= " ($description)" if $description;
        $groups .= ",  ";
    }

    # get the values from ldap 
    my $user           = ldap_get_user($user_name);
    my $user_full_name = $user->get_value('cn');
    my $name           = "$user_full_name ($user_name)";
    my @items;

    for my $item (@display_items) {
        my $value = $user->get_value($item);
        if ( $item eq "shadowFlag" ) {
            $value = $value ? text("disabled") : text("enabled");
        }
        push @items,
          {
            "valueN"    => $item,
            "valueV"    => $value,
            "valueT"    => text($item),
            "canteditF" => "1"
        };
    }
    push @items,
      {
        "valueN"    => "UserMemberOf",
        "valueV"    => $groups,
        "valueT"    => text("classes"),
        "canteditF" => "1"
    };
    for my $item (@edit_items) {
        my $value = $user->get_value($item) unless ( $item eq "userPassword" );
        push @items,
          {
            "valueN"    => $item,
            "valueV"    => $value,
            "valueT"    => text($item),
            "canteditF" => "",
	    "passwdF"   => ( $item eq "userPassword" || 
			     $item eq "passwordreconfirm" ),
        };
    }

    my $template = $self->load_tmpl('EditUserData.include');
    fill_in_template($template);    
    $template->param(
        EditUserDataT => text( "EditUserDataT", text2html $name ),
        user_name     => $user_name,
        ItemLoop      => \@items,
	Warning       => $warning,
    );
    debug(Dumper($warning));
    if($unauthorized){
	$template->param(enterpwt => text('oldpw'));
    }
    
    $output .= $template->output;
    return $output;
}

sub EditClassData {
    my $self = shift;
    my $q    = $self->query;
    my %par  = $q->Vars;
    my $output;

    my @edit_items = qw/description/;

    my @display_items = qw /cn gidNumber/;

    my $cn     = $par{cn};
    my $rootpw = get_root_password($q);

    # write the changes to ldap
#    my @ageGroups;
#    if ( $par{"ageGroup"} ) {
#        @ageGroups = split ( /\0/, $par{"ageGroup"} );
#    }
#    ldap_update_age_groups( $cn, \@ageGroups, $rootpw );

    my %i;
    for my $item (@edit_items) {
        $i{$item} = $par{$item} if ( $par{$item} );
    }
    ldap_modify_group( $cn, \%i, $rootpw ) if keys %i;

    my $group       = ldap_get_group($cn);
    my $description = $group->get_value('description');
    my $name        = "$cn ($description)";

    my @member_list =
      ldap_get_member_userlist( $group->get_value('gidNumber') );
    my $users;
    for my $entry (@member_list) {
        next unless ($entry);
        my $cn  = $entry->get_value("cn");
        my $uid = $entry->get_value("uid");
        $users .= "$cn, ($uid)  ";
    }

    # get the values from ldap, who knows what happend inbetween? 
    my @items;
    for my $item (@display_items) {
        my $value = $group->get_value($item);
        push @items,
          {
            "valueN"    => $item,
            "valueV"    => $value,
            "valueT"    => text($item),
            "canteditF" => "1"
        };
    }
    push @items,
      {
        "valueN"    => "ClassMemberOf",
        "valueV"    => $users,
        "valueT"    => text("users"),
        "canteditF" => "1"
    };
    for my $item (@edit_items) {
        my $value = $group->get_value($item);
        push @items,
          {
            "valueN"    => $item,
            "valueV"    => $value,
            "valueT"    => text($item),
            "canteditF" => ""
        };
    }
#    my @selected_age_groups;
#    for ( ldap_find_age_groups($cn) ) {
#        push @selected_age_groups, $_->get_value("cn");
#    }

#    my %r;
#    for ( ldap_get_groups("age_group") ) { $r{ $_->get_value("cn") } = 1 }
#    for (@selected_age_groups) { delete $r{$_} }
#    my @rest_age_groups = keys %r;
#    push @rest_age_groups, " * none * ";

#    @selected_age_groups = list2web( \@selected_age_groups, "cn" );
#    @rest_age_groups     = list2web( \@rest_age_groups,     "cn" );

    my $template = $self->load_tmpl('EditClassData.include');
    fill_in_template($template);
    $template->param(
        EditClassDataT => text( "EditClassDataT", $name ),
        cn             => $cn,
        ItemLoop       => \@items,
#        SelectedLoop   => \@selected_age_groups,
#        RestLoop       => \@rest_age_groups,
    );
    $output .= $template->output;
    return $output;
}

sub EditClassUsers {
    my $self = shift;
    my $q    = $self->query;
    my %par  = $q->Vars;
    my $output;

    #   $output = text2html( Dumper( \%par ), pre => 1);
    #   return $output;

    my $cn        = $par{cn};
    my $group     = ldap_get_group($cn);
    my $gidNumber = $group->get_value('gidNumber');
    my $rootpw    = get_root_password($q);

    for my $user_login ( split ( /\0/, $par{"remove_from"} ) ) {
        ldap_del_user_from_group( $user_login, $cn, $rootpw );
    }
    for my $user_login ( split ( /\0/, $par{"add_to"} ) ) {
        ldap_add_user_to_group( $user_login, $cn, $rootpw );
    }

    #create one list of all users a class consists of and one the rest
    my @member_list         = ldap_get_member_userlist($gidNumber);
    my @member_web_list     = ldap2web( \@member_list, [ "cn", "uid" ] );
    my @not_member_list     = ldap_get_not_member_userlist($gidNumber);
    my @not_member_web_list = ldap2web( \@not_member_list, [ "cn", "uid" ] );

    my $entry             = ldap_get_groupname($gidNumber);
    my $group_description = ldap_get_groupdescription($gidNumber);
    my $name              = "$group_description ($cn)";
    my $template          = $self->load_tmpl('EditClassUsers.include');
    fill_in_template($template);
    $template->param(
        RemoveUsersFromGroup => text( "removeusersfromgroup", text2html $cn ),
        AddUsersToGroup      => text( "adduserstogroup",      text2html $cn ),
        EditClassUsersT      => text( "EditClassUsersT",      text2html $cn ),
        cn                   => $cn,
        MemberLoop           => \@member_web_list,
        NoMemberLoop         => \@not_member_web_list,
    );

    $output .= $template->output;
    return $output;
}

sub AddGroups {
    my $self = shift;
    my $q    = $self->query;
    my %par  = $q->Vars;
    my ( @group_loop_data, @typ_rest_data, 
#         @age_rest_data 
	 );
    my %typs_hash = (
        "privilege_group" => 1,
        "school_class"    => 1,
#        "age_group"       => 1,
        "authority_group" => 1
    );
#    my %ages_hash;
    my $output;

    unless ( $global{rows} ) {
        check_form_ok($q);
    }

#    compose_all_groups_of_type( \%ages_hash, "age_group" );

    my $group_count = compose_group_loop_data( \@group_loop_data, \%par );

    my ( $typ_recent_ref, 
#         $age_recent_ref 
         );
    if ( $group_loop_data[-1] ) {
        $typ_recent_ref = $group_loop_data[-1]->{TypLoop};
#        $age_recent_ref = $group_loop_data[-1]->{AgeLoop};
    }
    else {
        $typ_recent_ref = [ { typ => ( sort keys %typs_hash )[-1] } ];
#        $age_recent_ref = [ { age => ( sort keys %ages_hash )[-1] } ];
    }
    translate_typs( $typ_recent_ref, "typ" );

    #    $output .= text2html( Dumper( [ sort keys %typs_hash ]  ));
#    compose_rest_data( \@age_rest_data, $age_recent_ref, "age", \%ages_hash );
    compose_rest_data( \@typ_rest_data, $typ_recent_ref, "typ", \%typs_hash );

    translate_typs( \@typ_rest_data, "typ" );

    my $template = $self->load_tmpl('AddGroups.include');
    fill_in_template($template);    # this must come first, so the loop
                                    # variables can overwrite them
                                    # where appropriate
                                    #    my $warning = $global{warning};

    $template->param(
        ListGroupsLoop  => \@group_loop_data,
        TypRecentLoop   => $typ_recent_ref,
        TypRestLoop     => \@typ_rest_data,
        typ_var         => "typ$group_count",
#        AgeRecentLoop   => $age_recent_ref,
#        AgeRestLoop     => \@age_rest_data,
#        age_var         => "age$group_count",
        group_name_var  => "group_name$group_count",
        description_var => "description$group_count",
        Warning         => $global{warning},
    );

    $output .= $template->output;

#    $output .= text2html( Dumper( \%par ), pre => 1 );
    return $output;
}

sub AddUsers {    #XXX this mostly duplicates code from EnterManyUsers,
        #XXX put this into a singel function
    my $self = shift;
    my $q    = $self->query;
    my %par  = $q->Vars;
    my (
        @user_loop_data,      @class_rest_data,
        @authority_rest_data, #@age_rest_data
    );
    my ( %authority_groups, %classes, 
#	 %age_groups 
	 );
    my $output;

    check_form_ok($q);

    #    $output .= text2html( Dumper( \%par ), pre => 1 );
    my $user_count = compose_user_loop_data( \@user_loop_data, \%par );

    compose_all_groups_of_type( \%classes,          "school_class" );
    compose_all_groups_of_type( \%authority_groups, "authority_group" );
#    compose_all_groups_of_type( \%age_groups,       "age_group" );

    my ( $class_recent_ref, $authority_recent_ref, 
#$age_recent_ref 
	 );
    $class_recent_ref = [ { class => ( sort keys %classes )[0] } ];
    $authority_recent_ref =
      [ { authority => ( sort keys %authority_groups )[0] } ];
#    $age_recent_ref = [ { age => ( sort keys %age_groups )[0] } ];
    translate_typs( $authority_recent_ref, "authority" );

    compose_rest_data( \@class_rest_data, $class_recent_ref, "class",
        \%classes );
    compose_rest_data(
        \@authority_rest_data, $authority_recent_ref,
        "authority",           \%authority_groups
    );
#    compose_rest_data( \@age_rest_data, $age_recent_ref, "age", \%age_groups );
    translate_typs( \@authority_rest_data, "authority" );

    #    my $warning = $global{warning};

    my $template = $self->load_tmpl('AddUsers.include');
    fill_in_template($template);

    $template->param(
        ListUsersLoop       => \@user_loop_data,
        ClassRecentLoop     => $class_recent_ref,
        AuthorityRecentLoop => $authority_recent_ref,
#        AgeRecentLoop       => $age_recent_ref,
        ClassRestLoop       => \@class_rest_data,
        AuthorityRestLoop   => \@authority_rest_data,
#        AgeRestLoop         => \@age_rest_data,
        class_var           => "class$user_count",
        authority_var       => "authority$user_count",
#        age_var             => "age$user_count",
        first_name_var      => "first_name$user_count",
        last_name_var       => "last_name$user_count",
        Warning             => $global{warning},
    );

    # the fileimport part does not need any special settings
    $output .= $template->output;

    return $output;
}

sub Fileimport {
    my $self = shift;
    my $q    = $self->query;
    my %par  = $q->Vars;
    my $output;

    my $users_ref;
    my $max_col = 0;

    # this is done so you can use the back button in fileimport
    if ( $par{"myfile"} ) {      # use the uploaded file
	( $users_ref , $max_col ) = preprocess_upload_file( $q, "myfile" );
	$global{state}{user_ref} = $users_ref;
    }
    elsif ( $global{state}  and  # hopefulle the file is saved in state allready.  
	    $global{state}{user_ref}  ) {
	$users_ref = $global{state}{user_ref}; 
	for my $user (  @{ $users_ref } ) {
	    my $col =  $#$user;
	    $max_col = $col if $col > $max_col;
	} 
    }

    my ( $header_loop_data, $user_raw_loop_data ) =
      matrix_to_loop_data( $users_ref, $max_col );

    my ( %authority_groups, %classes, 
#	 %age_groups 
	 );
    compose_all_groups_of_type( \%classes,          "school_class" );
    compose_all_groups_of_type( \%authority_groups, "authority_group" );
#    compose_all_groups_of_type( \%age_groups,       "age_group" );

    my ( @class_data, @authority_data, 
#	 @age_data 
	 );
    compose_rest_data( \@class_data, undef, "class", \%classes );
    compose_rest_data( \@authority_data, undef, "authority",
        \%authority_groups );
#    compose_rest_data( \@age_data, undef, "age", \%age_groups );
    translate_typs( \@authority_data, "authority" );

    my $template = $self->load_tmpl('FileContent.include');

    $global{warning} = $text{ $global{warning} };
    fill_in_template($template);
    
    my $continue_flag;
    if ($par{continue_despite_warning} eq "false") {
	$continue_flag = "true"	
    }

    $template->param(
        header            => $header_loop_data,
        res               => $user_raw_loop_data,
        ClassRestLoop     => \@class_data,
        AuthorityRestLoop => \@authority_data,
#        AgeRestLoop       => \@age_data,
	continue_despite_warning => $continue_flag,
    );
    $output .= $template->output;

    return $output;
}

sub CreateGroups {
    my $self = shift;
    my $q    = $self->query;
    my %rows = %{ $global{"rows"} };
    my @group_results_loop_data;
    my $rootpw = get_root_password($q);

    my $output;
    for my $group_key ( sort { $a <=> $b } keys %rows ) {
        my %group = %{ $rows{$group_key} };

        # create the group
#        my @ages = split /\0/, $group{age}[0];
        my ($error, $out) = create_single_group(
            $group{'group_name'}, $group{'typ'}[0],
#            \@ages,               
	    $group{'description'},
            $rootpw,
        );

        # create the output for the webpage
        my $html_output;
        while ( @{$out} ) {
            $html_output .= text2html( shift ( @{$out} ), pre => 1 );
        }

        my %row_data;
        $row_data{'message'}     = $html_output;
        $row_data{'description'} = $group{'description'};
        $row_data{'group_name'}  = $group{'group_name'};

        push ( @group_results_loop_data, \%row_data );
    }

    #work around the html::template problem
    add_key_value_to_loop( \@group_results_loop_data, 'cb', $cb );

    my $template = $self->load_tmpl('CreateGroupResults.include');
    fill_in_template( $template, \%text );
    $template->param( GroupResultLoop => \@group_results_loop_data );
    $output .= $template->output;

    return $output;
}

sub CreateUsers {
    my $self = shift;
    my $q    = $self->query;
    my %par  = $q->Vars;

    my %rows = %{ $global{"rows"} };
    my @user_results_loop_data;
    my $output;

    my $rootpw = get_root_password($q);
    $global{'auto_create_groups'} = $par{'auto_create_groups'};
    
    system ("/etc/init.d/nscd stop >/dev/null 2>&1"  ) if( -x "/etc/init.d/nscd" );
    
    for my $user_key ( sort { $a <=> $b } keys %rows ) {
        my %user   = %{ $rows{$user_key} };
        my $userpw =
          generate_passwd( $par{'kpwd'}, $par{'fpass'}, $user{'userpw'} );
        my ( $out, $username ) 
	    = create_single_user(
			$rootpw,
			$user{'first_name'},
				 $user{'last_name'},
			$user{username},
			\@{ $user{"class"} },
			\@{ $user{authority} },
			# \@ ...age...
			$userpw,
			$user{'userpw_crypt'},
			$user{'userpw_smbLM'},
			$user{'userpw_smbNT'}
				 );
        my $html_output;
        while ( @{$out} ) {
            unless ( $out->[0] ) { shift @{$out}; next }
            $html_output .= text2html( shift ( @{$out} ) . "\n" , 
				       paras => 1, 
				       blockquotes => 1, 
				       );
        }

        my %row_data;
        $row_data{'messagev'} = $html_output;
        $row_data{'userpwv'}  = $userpw;

        # if passwd-hashes are given then show them in the output too
        for (qw( userpw_crypt userpw_smbLM userpw_smbNT )) {
            if ( defined($user{$_}) ) {
                $row_data{'userpwv'} .=  " :&nbsp;" . $user{$_};
            }
        }

        if ( $user{'last_name'} ) {
            $row_data{'full_namev'} =
              $user{'first_name'} . " " . $user{'last_name'};
        }
        else {
            $row_data{'full_namev'} = $user{'first_name'};
        }
        $row_data{'user_namev'} = $username;

        push ( @user_results_loop_data, \%row_data );
    }
    
    system ("/etc/init.d/nscd start >/dev/null 2>&1") if( -x "/etc/init.d/nscd" );
    
    add_key_value_to_loop( \@user_results_loop_data, 'cb', $cb );
    my $template = $self->load_tmpl('CreateUserResults.include', global_vars => 1);
    fill_in_template( $template, \%text );
    $template->param( UserResultLoop => \@user_results_loop_data );

    $output .= $template->output;

    delete $global{state}{user_ref}; #we dont need this anymore, now.
    
    return $output;
}

sub EnterManyUsers {
    my $self = shift;
    my $q    = $self->query();
    my %par  = $q->Vars;
    my (
        @user_loop_data,      @class_rest_data,
        @authority_rest_data, 
#	@age_rest_data
    );
    my ( %authority_groups, %classes, 
#	 %age_groups 
	 );
    my $output;

    unless ( $global{rows} ) {
        check_form_ok($q);
    }

    #    $output .= text2html( Dumper( \%par ), pre => 1 );
    my $user_count = compose_user_loop_data( \@user_loop_data, \%par );

    compose_all_groups_of_type( \%classes,          "school_class" );
    compose_all_groups_of_type( \%authority_groups, "authority_group" );
#    compose_all_groups_of_type( \%age_groups,       "age_group" );

    my ( $class_recent_ref, $authority_recent_ref, 
#	 $age_recent_ref 
	 );
    if ( $user_loop_data[-1] ) {
        $class_recent_ref     = $user_loop_data[-1]->{ClassLoop};
        $authority_recent_ref = $user_loop_data[-1]->{AuthorityLoop};
#        $age_recent_ref       = $user_loop_data[-1]->{AgeLoop};
    }
    else {
        $class_recent_ref = [ { class => ( sort keys %classes )[0] } ];
        $authority_recent_ref =
          [ { authority => ( sort keys %authority_groups )[0] } ];
#        $age_recent_ref = [ { age => ( sort keys %age_groups )[0] } ];
        translate_typs( $authority_recent_ref, "authority" );
    }

    compose_rest_data( \@class_rest_data, $class_recent_ref, "class",
        \%classes );
    compose_rest_data(
        \@authority_rest_data, $authority_recent_ref,
        "authority",           \%authority_groups
    );
#    compose_rest_data( \@age_rest_data, $age_recent_ref, "age", \%age_groups );
    translate_typs( \@authority_rest_data, "authority" );

    my $template = $self->load_tmpl('EnterManyUsers.include');
    fill_in_template($template);    # this must come first, so the loop
                                    # variables can overwrite them
                                    # where appropriate

    $template->param(
        ListUsersLoop       => \@user_loop_data,
        ClassRecentLoop     => $class_recent_ref,
        AuthorityRecentLoop => $authority_recent_ref,
#        AgeRecentLoop       => $age_recent_ref,
        ClassRestLoop       => \@class_rest_data,
        AuthorityRestLoop   => \@authority_rest_data,
#        AgeRestLoop         => \@age_rest_data,
        class_var           => "class$user_count",
        authority_var       => "authority$user_count",
#        age_var             => "age$user_count",
        first_name_var      => "first_name$user_count",
        last_name_var       => "last_name$user_count",
        Warning             => $global{warning},
    );

    $output .= $template->output();
    return $output;
}

sub AdminSearch {
    my $self = shift;
    my $q    = $self->query();
    my %par  = $q->Vars;
    my $output;

    my $template = $self->load_tmpl( 'Admin.include', global_vars => 1 );

    my ($item_list, $executed_operation) = admin_operations($self);

    my ( $mesg, $result_ref ) = get_search_results($q);  # if $par{search_flag};
    
    my $result_type = get_search_type($q);
    $template->param( $result_type . "_flagV" => 1 );

    fill_in_search_box    ( $template, $q,          $mesg );
    fill_in_search_results( $template, $result_ref, $result_type );
    fill_in_operations    ( $template, $q,          $result_type ) if ( $mesg eq "OK" );

    my $operation_notice;
    if ($result_type and  $item_list) {
	$operation_notice = text( $result_type . $executed_operation, $item_list);
	$operation_notice = text2html( $operation_notice );
	$template->param( operation_notice => $operation_notice );
    }

    delete $global{state}{shortcut_back} if $global{state}{shortcut_back};
    my $query_string = $q->query_string();
    $template->param( query_string => uri_escape($query_string) );    

    fill_in_template($template);    
    $output .= $template->output();

    #    my @pa=$template->param();
    #    $output .= text2html(Dumper(ldap_err_output()));
    return $output;
}

sub admin_operations {
    my ($self) = @_;
    my $q   = $self->query;
    my %par = $q->Vars;
    
    my ($list, $operation);
    if ( $par{operation_delete_flag} ) {
	$list = operation_delete($q);
	$operation = "_delete";
    }
    elsif ( $par{operation_enable_login_flag} ) {
        $list = operation_enable_login($q);
	$operation = "_enable_login";
    }
    elsif ( $par{operation_disable_login_flag} ) {
        $list = operation_disable_login($q);
	$operation = "_disable_login";
    }
    return ($list, $operation);
}

sub operation_disable_login {
    my ($q) = @_;

    my $rootpw               = get_root_password($q);
    my $mode                 = get_search_type($q);
    my $items_to_disable_ref = get_marked_items($q);

    my @users;
    my $list;
    if ( $mode eq "user" ) {
        for my $item ( @{$items_to_disable_ref} ) {
            $list .= " $item" 
		if ldap_disable_user_login( $item, $rootpw );
        }
    }
    elsif ( $mode eq "class" ) {
        for my $item ( @{$items_to_disable_ref} ) {
	    my $entry = ldap_get_group($item);
	    push @users, $entry->get_value("memberUid");
        }
        for my $item ( unique(@users) ) {
            $list .= " $item" 
		if ldap_disable_user_login( $item, $rootpw );
        }
    }
    return $list;
}

sub operation_enable_login {
    my ($q) = @_;

    my $rootpw              = get_root_password($q);
    my $mode                = get_search_type($q);
    my $items_to_enable_ref = get_marked_items($q);

    my @users;
    my $list;
    if ( $mode eq "user" ) {
        for my $item ( @{$items_to_enable_ref} ) {
            $list .= " $item" 
		if ldap_enable_user_login( $item, $rootpw );
        }
    }
    elsif ( $mode eq "class" ) {
        for my $item ( @{$items_to_enable_ref} ) {
	    my $entry = ldap_get_group($item);
	    push @users, $entry->get_value("memberUid");
        }
        for my $item ( unique(@users) ) {
            $list .= " $item" 
		if ldap_enable_user_login( $item, $rootpw );
        }
    }
    return $list;
}

sub ldap2web {
    my ( $ldap_list_ref, $value_list_ref ) = @_;
    my @web_list;

    for my $entry ( @{$ldap_list_ref} ) {
        my %i;
        next unless ($entry);
        for my $value ( @{$value_list_ref} ) {
            $i{$value} = text2html $entry->get_value($value);
        }
        push ( @web_list, \%i );
    }

    return @web_list;
}

sub list2web {
    my ( $list_ref, $name ) = @_;
    my @web_list;

    for my $entry (@$list_ref) {
        next unless ($entry);
        push @web_list, { $name => $entry };
    }
    return @web_list;
}

sub get_marked_items {
    my ($q) = @_;
    my %par = $q->Vars;

    my @items = split ( /\0/, $par{"selected_item"} );
    return \@items;
}

sub get_root_password {
    my ($q) = @_;

    my $rootpw = $q->param("rootpw");

    # was no password given?
    if ( !$rootpw or $rootpw eq "" ) {

        # do i have a saved password ?
        if ( $global{state} and $global{state}{passwd} ) {

            # use that one
            $rootpw = $global{state}{passwd};
        }
        else {

            # warn about missing password
            $global{warning} .= text2html( $text{prompt_passwd} );
        }
    }
    else {

        #save the password and username
        $global{state}{passwd}   = $rootpw;
	$global{state}{username} = $remote_user;
    }
    return $rootpw;
}

sub delete_single_user {
    my ($user_name , $rootpw) = @_;
    
    my @groups = ldap_get_membergroups($user_name);
    for my $group (@groups) {
	next unless $group;
	my $gid = $group->get_value("cn");
	ldap_del_user_from_group( $user_name, $gid, $rootpw );
    }
    ldap_disable_user_login( $user_name, $rootpw );
    return " $user_name"
	if ( ! remove_group     ( $user_name, $rootpw) and
	     ! ldap_remove_user ( $user_name, $rootpw ) 
	     );
}


sub delete_all_users_in_group {
    my ($group_name , $rootpw) = @_;

    return "" unless $group_name;
    my $group = ldap_get_group($group_name);
    return "" unless $group;
    my $gid = $group->get_value("gidNumber");
    my @member_list = ldap_get_member_userlist( $gid );
    # iterate over all users
    my $deleted_users;
    for my $member ( @member_list ) {
	next unless $member;
	my $user_name = $member->get_value("uid");
	$deleted_users .= delete_single_user( $user_name , $rootpw );
    }
    return $deleted_users;
}

sub operation_delete {
    my ($q) = @_;

    my $rootpw              = get_root_password($q);
    my $mode                = get_search_type($q);
    my $items_to_delete_ref = get_marked_items($q);

    my $deleted_items;

    # XXX check for delete options once we add them
    # like delete homedir, backup data, backup mail, send mesg (what for?)    

    for my $item ( @{$items_to_delete_ref} ) {

        #  everything gets more complex if we delete groups 
        #    with all their users.
        #should we
        #  orphane all files that user owned and shared with the group?
        #  remove user from all groups he was part of? 

        if ( $mode eq "user" ) {
	    $deleted_items .= delete_single_user ( $item, $rootpw )
        }
        elsif ( $mode eq "class" ) {
	    if( $q->param("operation_delete_all_users_flag") ){
		$deleted_items .= delete_all_users_in_group( $item, $rootpw );
	    }
            $deleted_items .= " $item" 
		unless ( ldap_delete_group( $item, $rootpw ) );
        }
    }
    return $deleted_items;
}

sub fixed_header {
    open FOO, "-|", header( $text{title1}, '', undef, 1 );
    my $output .= $_ while <FOO>;
    close FOO;
    return $output;
}

sub fixed_footer {
    open FOO, "-|", footer( "/", "Index" );
    my $output .= $_ while <FOO>;
    close FOO;
    return $output;
}

sub fill_in_template {
    my ($template) = @_;

    my @parameter_names = $template->param();

    for my $name (@parameter_names) {

        if ( $text{$name} ) {
            $template->param( $name => $text{$name} );
        }
    }
    $template->param( JavaScriptStatus => generate_status() )
      if ( $template->query( name => "JavaScriptStatus" ) );
    $template->param( cb => $cb )
      if ( $template->query( name => "cb" ) );
    $template->param( tb => $tb )
      if ( $template->query( name => "tb" ) );
    $template->param( CgiName => "index.cgi" )
      if ( $template->query( name => "CgiName" ) );
    $template->param( Warning => $global{warning} ? $global{warning} : " " )
      if ( $template->query( name => "Warning" ) );
    $template->param( known_passwd => $global{state}{passwd} ? $global{state}{passwd} : "" )
      if ( $template->query( name => "known_passwd" ) );
    $template->param( shortcut_back => $global{state}{shortcut_back} ? 
		      $global{state}{shortcut_back} : "" )
      if (  $global{state} and $global{state}{shortcut_back} and $template->query( name => "shortcut_back" ) );
    
    
}

sub generate_status {
    my $status = "\"";
    $status .= $ENV{'ANONYMOUS_USER'} ? "Anonymous user" : $remote_user;
    $status .= $ENV{'SSL_USER'} ? " (SSL certified)"
      : $ENV{'LOCAL_USER'} ? " (Local user)"
      : "";
    $status .= " logged into " . $text{'programname'};
    $status .= " " . get_webmin_version();
    $status .= " on " . get_system_hostname();
    $status .= "(" . $ENV{'HTTP_USER_AGENT'};
    $status .= ")\"";

    return $status;
}

sub compose_group_loop_data {
    my ($group_data_ref) = @_;
    my $counter = 0;
    my %rows    = %{ $global{"rows"} };

    for my $row_key ( sort { $a <=> $b } keys %rows ) {
        my %row_data;
        my %row = %{ $rows{$row_key} };

        my $typs_ref = compose_selected_loop_data( $row{"typ"}, "typ" );
#        my $ages_ref = compose_selected_loop_data( $row{"age"}, "age" );
        translate_typs( $typs_ref, "typ" );

        # these are the value="" strings and/or texts in the webpage 
        $row_data{TypLoop}     = $typs_ref;
#        $row_data{AgeLoop}     = $ages_ref;
        $row_data{group_name}  = $row{group_name};
        $row_data{description} = $row{description};

        # these are the variable names used in the webpage
        $row_data{group_name_var_loop}  = "group_name$row_key";
        $row_data{description_var_loop} = "description$row_key";

        # work around a bug in HTML::Template which requires 
        # to have all vars local
        add_key_value_to_loop( $typs_ref, 'typ_var_loop', "typ$row_key" );
#        add_key_value_to_loop( $ages_ref, 'age_var_loop', "age$row_key" );

        # mark fields with invalid input
        for my $entry ("group_name") {
            if ( $row{flaws}->{$entry} ) {
                $row_data{ $entry . "_invalid" } = "bgcolor=\"red\""; # XXX HTML
            }
            else {
                $row_data{ $entry . "_invalid" } = "";
            }
        }
        push ( @{$group_data_ref}, \%row_data );

        $counter = $row_key + 1;
    }
    return $counter;
}

sub compose_user_loop_data {
    my ($user_data_ref) = @_;
    my $counter = 0;
    my %rows    = %{ $global{"rows"} };

    for my $row_key ( sort { $a <=> $b } keys %rows ) {
        my %row_data;
        my %row = %{ $rows{$row_key} };

        my $authoritys_ref =
          compose_selected_loop_data( $row{"authority"}, "authority" );
        translate_typs( $authoritys_ref, "authority" );
        my $classes_ref = compose_selected_loop_data( $row{"class"}, "class" );
#        my $ages_ref    = compose_selected_loop_data( $row{"age"},   "age" );

        # these are the value="" strings and/or texts in the webpage 
        $row_data{ClassLoop}     = $classes_ref;
#        $row_data{AgeLoop}       = $ages_ref;
        $row_data{AuthorityLoop} = $authoritys_ref;
        $row_data{first_name}    = $row{first_name};
        $row_data{last_name}     = $row{last_name};

        # these are the variable names used in the webpage
        $row_data{first_name_var_loop} = "first_name$row_key";
        $row_data{last_name_var_loop}  = "last_name$row_key";

        # work around a bug in HTML::Template which requires 
        # to have all vars local
        add_key_value_to_loop( $authoritys_ref, 'authority_var_loop',
            "authority$row_key" );
        add_key_value_to_loop( $classes_ref, 'class_var_loop',
            "class$row_key" );
#        add_key_value_to_loop( $ages_ref, 'age_var_loop', "age$row_key" );

        # mark fields with invalid input
        for my $entry ( "last_name", "first_name", "authority", "class", 
#			"age" 
			)
        {
            if ( $row{flaws}->{$entry} ) {
                $row_data{ $entry . "_invalid" } = "bgcolor=\"red\""; # XXX HTML
            }
            else {
                $row_data{ $entry . "_invalid" } = "";
            }
        }
        push ( @{$user_data_ref}, \%row_data );

        $counter = $row_key + 1;
    }
    return $counter;
}

sub compose_all_groups_of_type {
    my ( $hash_ref, $group_type ) = @_;

    my @all_groups = ldap_get_groups($group_type);
    for my $entry (@all_groups) {
        $hash_ref->{ $entry->get_value('cn') } = 1;
    }
}

sub compose_rest_data {
    my ( $rest_ref, $selected_ref, $key, $hash_ref ) = @_;

    for my $selection ( @{$selected_ref} ) {
        delete $hash_ref->{ $selection->{$key} };
    }
    for my $typ ( sort keys %{$hash_ref} ) {
        my %row_data;
        $row_data{$key} = $typ;
        push ( @{$rest_ref}, \%row_data );
    }
}

sub compose_selected_loop_data {
    my ( $src_list_ref, $name ) = @_;
    my @list;

    foreach my $entry ( @{$src_list_ref} ) {
        my %row_data;

        $row_data{$name} = $entry;
        push ( @list, \%row_data );
    }
    return \@list;
}

# this cute little function adds an entry holding the 
# translated text
sub translate_typs {
    my ( $list_ref, $key ) = @_;
    for my $counter ( 0 .. $#{$list_ref} ) {
        if ( $text{ $list_ref->[$counter]{$key} } ) {
            $list_ref->[$counter]{ $key . "_text" } =
              $text{ $list_ref->[$counter]{$key} };
        }
        else {
            $list_ref->[$counter]{ $key . "_text" } =
              $list_ref->[$counter]{$key};
        }
    }
}

sub add_key_value_to_loop {
    my ( $list_ref, $key, $value ) = @_;
    for my $counter ( 0 .. $#{$list_ref} ) {
        $list_ref->[$counter]{$key} = $value;
    }
}

sub sanitize_name {
    my ($name) = @_;
    
    $name = utf8($name)->latin1
	if ( $name =~ /[\xC2\xC3][\x80-\xBF]/ );

    $name = conv_name($name);
    $name = substr( $name, 0, 8 );
    return $name;
}

sub create_single_group {
    my ( $group_name, $group_type, 
#	 $group_ages_ref, 
	 $description, $rootpw ) =
      @_;
    my @out;
    
    $group_name = sanitize_name($group_name);
    my $gidNumber = ldap_get_newid($rootpw);

    my ($error, $out) =
        create_new_group(
            $group_name,     $gidNumber, $group_type,
#	    $group_ages_ref, 
	    $rootpw,    $description
        );
    if ( $error ) {
	push ( @out, $out )
	}
    else {
	push ( @out, "OK" )
	}
    return ($error, \@out);
}

sub create_single_user {
    my (
        $rootpw,
        $first_name,
	$last_name,
        $prefered_login,
        $classes_ref,
        $authgroup_ref,
#        $age_ref,
        $userpw,
        $userpw_crypt,
        $userpw_smbLM,
        $userpw_smbNT
      )
      = @_;
    my @out;

    my $cn = make_cn( $first_name, $last_name );
    $cn = latin1($cn)->utf8 unless ( $cn =~ /[\xC2\xC3][\x80-\xBF]/ );

    $first_name = utf8($first_name)->latin1
      if ( $first_name and $first_name =~ /[\xC2\xC3][\x80-\xBF]/ );
    $last_name  = utf8($last_name )->latin1
      if ( $last_name  and $last_name  =~ /[\xC2\xC3][\x80-\xBF]/ );


    if ( $prefered_login ) {
	$prefered_login = sanitize_name( $prefered_login )
	}
    my $username =
      make_unique_username( $first_name, $last_name, $prefered_login );

    push @out, ldap_err_output();
    push @out, ldap_dbg_output() if $config{debug};

    my $groupname = $username;

    my $uidNumber = ldap_get_newid($rootpw);
    push @out, ldap_err_output();
    push @out, ldap_dbg_output() if $config{debug};
    return ( \@out, $username ) unless ($uidNumber);

    my $gidNumber = $uidNumber;

    my $homedir = $config{'homeprefix'} . "/$username";
    my $maildir = $config{'mailprefix'} . "/$username";

    my $result = ldap_add_user(
        $cn,        $username,  $userpw,
        $uidNumber, $uidNumber, $rootpw,
        $homedir,   $maildir,   $userpw_crypt
    );

    push @out, ldap_err_output();
    push @out, ldap_dbg_output() if $config{debug};

    if ( $result->code() ) {
        push ( @out,
            print_ldap_error( text( "addfailed", $username ), $result ) );
        goto rollback_useraccount;
    }

    my ( $error, $out ) =
        create_user_sambaaccount(
            $rootpw, $userpw, $username, $userpw_smbLM, $userpw_smbNT
        );
    if ($error) {
        push @out, $out;
        goto rollback_user_sambaaccount;
    } 
    elsif ( $config{debug} ) {
        push @out, $out;
    }
    
    ( $error, $out, my $homedir_creation_time ) =
        create_user_home( $uidNumber, $gidNumber, $homedir );
    if ($error) {
        push @out, $out;
        goto rollback_user_home;
    } 
    elsif ( $config{debug} ) {
        push @out, $out; 
    }

    ( $error, $out ) =
        create_new_group( $username, $gidNumber, "private", $rootpw );
    if ($error) {
        push @out, $out;
        goto rollback_group;
    } 
    elsif ( $config{debug} ) {
        push @out, $out; 
    }
    
    ( $error, $out ) =
        add_user_to_groups( [$username], $username, "private", $rootpw );
    if ($error) {
        push @out, $out;
        goto rollback_user_from_private_groups;
    } 
    elsif ( $config{debug} ) {
        push @out, $out; 
    }
    
    ( $error, $out ) =
        add_user_to_groups( $classes_ref, $username, "school_class", $rootpw );
    if ($error) {
        push @out, $out;
        goto rollback_user_from_class_groups;
    } 
    elsif ( $config{debug} ) {
        push @out, $out; 
    }
    
    ( $error, $out ) =
        add_user_to_groups( $authgroup_ref, $username, "authority_group", $rootpw );
    if ($error) {
        push @out, $out;
        goto rollback_user_from_auth_groups;
    } 
    elsif ( $config{debug} ) {
        push @out, $out; 
    }

    push @out, "OK";
    return ( \@out, $username );

  rollback_user_from_auth_groups:    
    remove_user_from_groups( $authgroup_ref, $username, $rootpw );

  rollback_user_from_class_groups:    
    remove_user_from_groups( $classes_ref, $username, $rootpw );
    
  rollback_user_from_private_groups:    
    remove_user_from_groups( [$username], $username, $rootpw );

  rollback_group:
    remove_group($username, $rootpw);

  rollback_user_home:
    remove_user_home($homedir, $homedir_creation_time);

  rollback_user_sambaaccount:
    delete_user_sambaaccount( $username );

  rollback_useraccount:    
    ldap_delete_user ($username, $rootpw);
    
    push @out, "Error; User account not created";
    
    return ( \@out, $username );
}

sub create_user_sambaaccount {
    my ( $rootpw, $userpw, $username, $userpw_smbLM, $userpw_smbNT ) = @_;
    my ( $error,  $out );

    # check if sambasync is set, add user if it is
    if ( $config{'sambasync'} ) {
        
        unless ( $config{'samba_root'} ) {

            # hack to make sure root has a samba-account (first time run)
            my $root_exists = ldap_get_user("root");
            unless ($root_exists) {
                ( $error, $out ) =
                    change_samba ("root", $rootpw, $rootpw );
		return ($error, $out) if $error; 
                $config{'samba_root'} = 1;
            }
        }
        
        # add the new user to samba
        ( $error, $out ) =
            change_samba( $username, $userpw, $rootpw );
        unless ( $error ) {
            $out = $text{"added", $username};
            # if the userpw-sambahashes have been used, take them into
            # account, too --- in every sense of it ;-)
            if ( defined($userpw_smbLM) or defined($userpw_smbNT) ) {
                my %user_attr;
                if ( defined ($userpw_smbLM) ) {
                     $user_attr{sambaLMPassword} = $userpw_smbLM;
                }
                if ( defined ($userpw_smbNT) ) {
                     $user_attr{sambaNTPassword} = $userpw_smbNT;
                }
                my $warning =
                    ldap_modify_user( $username, \%user_attr, $rootpw );
                if ( $warning ) {
                    $out .= " $text{'added_smbHashes_failed'} $warning";
                }
                else {
                    $out .= " $text{'added_smbHashes_success'}";
                }
            }
        }

    }
    return ($error, $out );
}

sub delete_user_sambaaccount {
    my ( $username ) = @_;
    my ( $error,  $out );
    
    # check if sambasync is set, delete user if it is
    if ( $config{'sambasync'} ) {
        
        system ("/usr/bin/smbpasswd -x $username  >/dev/null 2>&1") 
            if ( -x "/usr/bin/smbpasswd" );
    }
    return ($error, $text{ "deleted", $username, $out });
}

sub create_user_home {
    my ( $uidNumber, $gidNumber, $homedir ) = @_;
    
    my $homedir_creation_time = time;
    
    my $error = undef;
    my ( $res, $out ) = create_dir( $homedir, $uidNumber, $gidNumber );
    if ($res) {
        $out .= text( "addHomedirSuccess", $homedir );
    }
    else {
        $out .= text( "addHomedirFailed", $homedir );
        $error = "error";
    }
    return ($error, $out, $homedir_creation_time);
}

sub remove_user_home {
    my ( $homedir, $homedir_creation_time ) = @_;

    my $error = undef;
    my $out;
    if ( -d $homedir ) {
        my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
            $atime,$mtime,$ctime,$blksize,$blocks)
            = stat($homedir);
        if( $mtime >= $homedir_creation_time) {
            system( "rm -rf $homedir >/dev/null 2>&1" );
            $error = $?;
        }
        else {
            $error = "not my homedir";
        }
    }
    return ($error, $out);
}

sub create_new_group {
    my (
        $groupname,      $gidNumber, $groupType,
#        $group_ages_ref, 
        $rootpw,    $description
      )
      = @_;
    my ($error, $output);

#    if ( $groupType eq "private" or $groupType eq "age_group" ) {
#        @{$group_ages_ref} = ();
#    }

    my $result = ldap_add_group(
        $groupname, $gidNumber,      $rootpw,
        $groupType, 
#        $group_ages_ref, 
        $description
    );
    if ( $result->code() ) {
        $output =
          print_ldap_error( text( "addGroupFailed \n", $groupname ), $result );
        $output .= ldap_err_output();
	$output .= ldap_dbg_output() if $config{debug};
	$error = "error";
    }
    else {
        $output = text( "addedGroup", $groupname );
    }
    return ($error, $output);
}

sub add_user_to_groups {
    my ( $groups_ref, $username, $group_type, $rootpw ) = @_;

    my ($error, @out, $out);

    foreach my $groupname ( @{$groups_ref} ) {
	next unless $groupname;
        my $res = ldap_add_user_to_group( $username, $groupname, $rootpw );
        push @out, text( $res, $username, $groupname );
        if ( ( $res eq "groupdoesntexist" )
            and ( $global{'auto_create_groups'} eq "yes" ) )
        {
            my $gidNumber = ldap_get_newid($rootpw);
            ($error, $out) =
                create_new_group( $groupname, $gidNumber, $group_type,
                                  $rootpw );
            if ($error) {
                push @out, $out;
                return ($error, @out);
            }
            $res = ldap_add_user_to_group( $username, $groupname, $rootpw );
            push @out, text( $res, $username, $groupname );
        }
    }
    return ($error, @out);
}

sub remove_group {
    my ( $groupname, $rootpw ) = @_;

    return ldap_delete_group( $groupname, $rootpw );
}


sub remove_user_from_groups {
    my ( $groups_ref, $username, $rootpw ) = @_;

    foreach my $groupname ( @{$groups_ref} ) {
        ldap_del_user_from_group( $username, $groupname, $rootpw );
    }
}

sub generate_passwd {
    my ( $password, $flag, $userpw ) = @_;

    if ( $password ){
	return $password unless ( check_passwd_quality( $password ) ); 
                          #and ( $flag eq "yes" ) 
			  # it seems people enter single users 
                          # and want to determine the passwd themselfs, 
                          # and then the common password question 
                          # does not make sense.
	error_msg(text("weak_passwd"));
	$password = undef;
    }
    if ($userpw) {
	return $userpw   unless ( check_passwd_quality( $userpw ) ); 
	error_msg(text("weak_passwd"));
    }
    
    my @chars =
      ( 'a' .. 'z', 'A' .. 'Z', '0' .. '9', ';', ':', '%' ); # what other chars?
    my $count = $#chars;
    for ( 1 .. 6 ) {
        $password .= $chars[ rand $count ];
    }
    return $password;
}

sub compose_groups_data {
    my ( $string, $list_ref ) = @_;

    my @entries = split ( /\0/, $string );
    foreach my $entry (@entries) {
        push ( @{$list_ref}, $entry );
    }

}

sub check_form_ok {
    my ($q) = @_;
    my $form_ok = "ok";

    my %rows;
    extract_data_from_query( $q, \%rows );

    my $form_type = $q->param("rm");
    for my $row_key ( keys %rows ) {
        unless ( check_row_ok( $rows{$row_key}, $form_type ) ) {
            if ( $rows{$row_key}{flaws}{delete} ) {
                delete $rows{$row_key};
            }
            else {
                $form_ok = undef;    # form is not ok
            }
        }
    }

    $global{warning} .= $text{'completemarkedfields'} unless $form_ok;
    
    if (  $q->param("kpwd") ) {
	my $password_warning = check_passwd_quality ( $q->param("kpwd") );
	if ( $password_warning ) {
	    $global{warning} = $password_warning;
	    $form_ok = undef;
	}
    }
    

    $global{'form_ok'} = $form_ok;
    $global{"rows"}    = \%rows;     # XXX UGLY!!! # save result for later use
    return $form_ok;
}

sub extract_data_from_query {
    my ( $q, $rows_ref ) = @_;
    my %par             = $q->Vars;
    my @names           = $q->param;
    my $number_of_param = $#names;

    for my $key ( "authority", "class", 
#                  "age", 
                  "typ" ) {
        for my $count ( 0 .. $number_of_param ) {
            if ( exists $par{"$key$count"} ) {
                my @entries = split ( /\0/, $par{"$key$count"} );
                unless ( exists $entries[0] ) {
                    @entries = undef;
                }
                $rows_ref->{$count}{$key} = \@entries;
            }
        }
    }

    for my $key ( "last_name", "first_name", "group_name", "description" ) {
        for my $count ( 0 .. $number_of_param ) {
            if ( exists $par{"$key$count"} ) {
                $rows_ref->{$count}{$key} = cleanup( $par{"$key$count"} );
            }
        }
    }

    for my $key ("group_name") {
        for my $count ( 0 .. $number_of_param ) {
            if ( exists $par{"$key$count"} ) {
                $rows_ref->{$count}{$key} = 
		    sanitize_name( $par{"$key$count"}  );
            }
        }
    }

}

sub cleanup {
    my ($string) = @_;

    if ( $string =~ m/^\s*(\S+.*\S+)\s*$/ ) {  # XXX stricter? does \w match ?
        $string = $1;
    }
    else {
        $string = undef;
    }
    return $string;
}

sub check_row_ok {
    my ( $row_ref, $form_type ) = @_;
    my $row_ok = "ok";
    my %flaws;

    # these are only the essential fields. there could be other,  optional ones
    my @user_fields = ( "last_name", "first_name", 
			#"authority", "class" 
			);
    my @group_fields = ( "group_name", "typ" );
    my @fields;

    if    ( $form_type eq "EnterManyUsers" ) { @fields = @user_fields }
    elsif ( $form_type eq "AddGroups" )      { @fields = @group_fields }
    else { return undef }

    # check if all fields are filled in
    for my $entry (@fields) {
        unless ( $row_ref->{$entry} ) {
            $flaws{$entry} = "empty";
            $row_ref->{$entry} = undef;
        }
    }

    # if names were not entered or deleted afterwards, delete row.
    if ( keys %flaws ) {

        if ( $form_type eq "EnterManyUsers" ) {

            if ( ( $flaws{"first_name"} eq "empty" )
                and ( $flaws{"last_name"} eq "empty" ) )
            {
                $flaws{delete} = 1;
            }
        }
        elsif ( $form_type eq "AddGroups" ) {
            if ( $flaws{"group_name"} eq "empty" ) {
                $flaws{delete} = 1;
            }
        }
    }

    # fail, if we have flaws
    if ( keys %flaws ) {
        $row_ok = undef;
        $row_ref->{flaws} = \%flaws;
    }
    return $row_ok;
}

sub preprocess_upload_file {
    my ( $q, $file_name ) = @_;
    my @users;
    my $max_col = 0;

    my $fh = $q->upload($file_name);
    while (<$fh>) {
	next if /^\#/;
	next if /^\s*$/;
        my @user;
        for my $entry ( split ( /;/, $_ ) ) {
            push ( @user, cleanup( latin1($entry)->utf8 ) );
        }
        my $cols = $#user;
        if ( $cols > $max_col ) { $max_col = $cols }
        push ( @users, [@user] );
    }
    return ( \@users, $max_col );
}

sub matrix_to_loop_data {
    my ( $users_ref, $max_col ) = @_;

    my @list;
#    for (qw (first_name last_name cn course authoritygroup age
    for (qw( first_name last_name cn course authoritygroup
        username userpw userpw_crypt userpw_smbLM userpw_smbNT ))
    {
        my %i = (
            'column'      => "$_",
            'column_text' => $text{$_}
        );
        push ( @list, \%i );
    }
    
    my @selected_cols;
    if ($global{column_content}) {
	for my $selected_header ( keys %{  $global{ column_content }  } ) {
	    for my $maked_col ( keys %{$global{ column_content }{ $selected_header }} ){
		$selected_cols[$maked_col] = $selected_header;
	    }
	}
    }
	
    my @header;
    for my $column_nr ( 0 .. $max_col ) {
	unless ($selected_cols[$column_nr]) {
	    $selected_cols[$column_nr] = "ignore";
	}
	
        my %r = (
            'select_column_name_loop' => \@list,
            'column_nr_var'           => "column_$column_nr",
            'selected_column'         => $selected_cols[$column_nr],
            'selected_column_text'    => $text{$selected_cols[$column_nr]}
        );
        push ( @header, \%r );
    }

    my @bug_matrix;
    if ( $global{bug_matrix} ) {
	@bug_matrix = @{ $global{bug_matrix} };    
    }

    my @res;
    my $row_count = 0;
    for my $row ( @{$users_ref} ) {
        my @col;
        my $col_count = 0;
        for my $cv ( @{$row} ) {
	    my $color = "bgcolor=";
	    my $cell= $bug_matrix[$row_count][$col_count];
	    if ( $cell ) {
		$color .= sprintf "\"0x%lx\"", $cell;
	    }
	    else {
		$color .= "\"white\"";
	    }
            my %c = ( 'cv' => $cv,
		      'cv_warn' => $color );
            push ( @col, \%c );
            $col_count++;
        }

        # this makes it possible to have rows of different lenght
        for ( $col_count .. $max_col ) {
            my %c = ( 'cv' => "" );
            push ( @col, \%c );    #adds elements till the row is complete
        }
        my %r = ( 'rv' => \@col );
        push ( @res, \%r );
	$row_count++;
    }
    return ( \@header, \@res );
}

sub dump_state {

    unless ( $global{state} ) { return ( undef, undef ) }
    
    my %state = %{ $global{state} };
    my ( $fh, $filename, $digest );

    # did the state change?
    if ( $global{state_hash} and md5_hex(%state) eq $global{state_hash} ) {
        $digest   = $global{tmp_hash};
        $filename = $global{tmp_file};
    }
    else {

        # create a new state file and hash
        ( $fh, $filename ) = tempfile( DIR => "/tmp", SUFFIX => ".wlus.state" );
        store_fd( \%state, $fh );
        binmode($fh);
        seek( $fh, 0, 0 );
        my $md5 = Digest::MD5->new();
        $md5->addfile($fh);
        $md5->add( $config{'secret'} );
        $digest = $md5->hexdigest;
        close $fh;

        #delete the old file
        unlink $global{tmp_file} if $global{tmp_file};
    }
    return ( $digest, $filename );
}

sub restore_state {
    my ( $filename, $old_digest ) = @_;
    my $fh;

    ( $filename and $old_digest ) or return undef;

    open( $fh, "<", $filename ) or return undef;

    binmode($fh);
    seek( $fh, 0, 0 );
    my $md5 = Digest::MD5->new();
    $md5->addfile($fh);
    $md5->add( $config{'secret'} );
    my $new_digest = $md5->hexdigest;

    return undef unless ( $old_digest eq $new_digest );

    my $state_ref = retrieve $filename ;

    unless ( $remote_user eq $state_ref->{username} ) {
	delete $state_ref->{passwd}
    }

    $global{state} = $state_ref;

    $global{tmp_file}   = $filename;
    $global{tmp_hash}   = $old_digest;
    $global{state_hash} = md5_hex( %{$global{state}} );
    
    

}

sub check_data_sufficent {
    my ($q) = @_;
    my %par       = $q->Vars;
    my $column_nr = 0;
    my %column_content;
    my $data_quality = "ok";
    
    while ( exists $par{"column_$column_nr"} ) {
        $column_content{ $par{"column_$column_nr"} }{$column_nr} = 1
          unless ( $par{"column_$column_nr"} eq "ignore" );
        ;
        $column_nr++;
    }

    $global{column_content} = \%column_content;

    unless ( exists $column_content{cn}
        or ( exists $column_content{first_name}
            and exists $column_content{last_name} ) )
    {
        $global{data} = "insufficient";
        return "insufficient";
    }
    for my $key (qw(cn first_name last_name userpw username)) {
        next unless exists $column_content{$key};
        if ( scalar keys( %{ $column_content{$key} } ) > 1 ) {
            $global{data} = "ambigous";
            return "ambigous";
        }
    }
    
    if ( exists $column_content{username} and
	 $par{continue_despite_warning} and 
	 $par{continue_despite_warning} eq "false"){
	$data_quality .= mass_check_usernames( keys %{$column_content{username}} );
    }

    if ( exists $column_content{userpw} and
	 $par{continue_despite_warning} and 
	 $par{continue_despite_warning}  eq "false"){
	$data_quality .= mass_check_passwd_quality( keys %{$column_content{userpw}} );
    }    
    $data_quality =~ s/^ok(.+)/$1/;
    $data_quality =~ s/^ok(.+)/$1/;
    $data_quality =~ s/(.+)ok$/$1/;
    
    
    return $data_quality;
}

sub mass_check_usernames {
    my ( $user_col ) = @_;
    
    my ( %usernames, $count, %existing_usernames, 
	 %double_usernames) ;

    $count = 0;
    for my $user (  @{ $global{state}{user_ref} } ) {
	my $username = $user->[$user_col];
	
	if (  exists $usernames{ $username }  ) {
	    push @{ $usernames{ $username } }, $count; 
	    $double_usernames{ $username } = 
		    $usernames{ $username };
	} 
	else {
	    push @{ $usernames{ $username } }, $count; 
	}
	$count++;
    }
    # ceck if any of the usernames are taken allready
    for my $username (  keys %usernames  ) {
	if (  ldap_get_user ( $username ) or 
		  ldap_get_old_user ( $username ) ) {
	    $existing_usernames{ $username } = 
		$usernames{ $username };
	}
    }
    
    # Now we have all the bogus usernames in either
    # %existing_usernames or %double_usernames, 
    # complete with their row number in the table.
    
    # Now we need to integrate it in the displayed table.
    my @bug_matrix;
    if ( $global{bug_matrix} ) {
	@bug_matrix = @{ $global{bug_matrix} };    
    }
    
    my $bogus_name_flag;
	for  my $username ( keys %double_usernames ) {
	    for my $row ( @{$double_usernames{ $username } } ) {
		$bug_matrix[$row][$user_col] += 0x00FF00;
		$bogus_name_flag=1
		}
	}
    for  my $username ( keys %existing_usernames ) {
	for my $row (  @{ $existing_usernames{$username} }  ) {
	    $bug_matrix[$row][$user_col] += 0xFF0000;
	    $bogus_name_flag=1
	    }
    }
    $global{bug_matrix}= \@bug_matrix;
    return "bogus_usernames" if $bogus_name_flag;
    return "ok";
}

sub mass_check_passwd_quality{
    my ( $userpw_col ) = @_;


    my ( %passwords, %weak_passwords ) ;

    my $count = 0;
    for my $user (  @{ $global{state}{user_ref} } ) {
	my $password = $user->[$userpw_col];
	
	push @{ $passwords{ $password } }, $count;
	$count++;
    }
    # ceck if any of the passwords are weak
    for my $password (  keys %passwords  ) {
	if (  check_passwd_quality ( $password )  ) {
	    $weak_passwords{ $password } = 
		$passwords{ $password };
	}
    }
    
    # Now we have all the bogus passwords in 
    # %weak_passwords, 
    # complete with their row number in the table.
    
    # Now we need to integrate it in the displayed table.
    
    my @bug_matrix;
    if ( $global{bug_matrix} ) {
	@bug_matrix = @{ $global{bug_matrix} };    
    }
    
    my $bogus_userpw_flag;
    for  my $password ( keys %weak_passwords ) {
	for my $row ( @{$weak_passwords{ $password } } ) {
	    $bug_matrix[$row][$userpw_col] += 0x00FFFF;
	    $bogus_userpw_flag=1
	    }
    }
    $global{bug_matrix}= \@bug_matrix;

    return "weak_passwords" if $bogus_userpw_flag;
    return "ok";
}

sub extract_rows_from_raw_data {
    my ($q) = @_;
    my %par            = $q->Vars;
    my %column_content = %{ $global{column_content} };

    my $user_ref = $global{state}{user_ref};
    unless ($user_ref) {
        return undef;
    }
    $global{dump} = $user_ref;

    my %rows;
    my $user_count = 0;
    my %default;
#    my @age_default       = split ( /\0/, $par{age_default} );
    my @authority_default = split ( /\0/, $par{authority_default} ) 
	if $par{authority_default};
    my @class_default     = split ( /\0/, $par{class_default} )
	if $par{class_default};
    for my $row ( @{$user_ref} ) {
#         my @ages = @age_default;
#         for my $index ( keys( %{ $column_content{age} } ) ) {
#             push ( @ages, $row->[$index] );
#         }
        my @classes = @class_default;
        for my $index (
            keys( %{ $column_content{"class"} } ),
            keys( %{ $column_content{"course"} } )
          )
        {
            push ( @classes, $row->[$index] );
        }
        my @authoritys = @authority_default;
        for my $index (
            keys( %{ $column_content{"authority"} } ),
            keys( %{ $column_content{"authoritygroup"} } )
          )
        {
            push ( @authoritys, $row->[$index] );
        }
        $rows{$user_count}{class}     = \@classes;
#        $rows{$user_count}{age}       = \@ages;
        $rows{$user_count}{authority} = \@authoritys;
        if ( exists $column_content{cn} ) {
            $rows{$user_count}{first_name} =
              $row->[ ( keys %{ $column_content{"cn"} } )[0] ];
        }
        else {
            $rows{$user_count}{"last_name"} =
              $row->[ ( keys %{ $column_content{"last_name"} } )[0] ];
            $rows{$user_count}{"first_name"} =
              $row->[ ( keys %{ $column_content{"first_name"} } )[0] ];
        }

        for (qw( username userpw userpw_crypt userpw_smbLM userpw_smbNT )) {
            if ( exists $column_content{$_} ) {
                $rows{$user_count}{$_} =
                  $row->[ ( keys %{ $column_content{$_} } )[0] ];
            }
        }

        $user_count++;
    }
    $global{rows} = \%rows;
    return "ok";
}

sub fill_in_search_box {
    my ( $template, $q, $mesg ) = @_;

    my $typSelected = $q->param("typV")
      if $q->param("typV");
    my $SearchExpressionV = $q->param("SearchExpressionN")
      if $q->param("SearchExpressionN");
    my @typ_list;

    for (qw (user_cn user_login class_cn)) {
        next if ( $typSelected and $_ eq $typSelected );
        my %i = (
            'typ'  => "$_",
            'typN' => $text{$_}
        );
        push ( @typ_list, \%i );
    }
    $mesg = " " if($mesg and $mesg eq "OK");

    $template->param(
        typLoop           => \@typ_list,
        typSelected       => $typSelected,
        typSelectedT      => $text{$typSelected},
        SearchExpressionV => $SearchExpressionV,
        SearchErrorV      => $mesg,
    );
}

sub get_search_results {
    my ($q) = @_;

    my $typ        = $q->param("typV");
    my $search_exp = $q->param("SearchExpressionN");
    my ( $msg, @list ) = ldap_search( $typ, $search_exp ) if ($typ);
    return ( $msg, \@list );
}

sub fill_in_search_results {
    my ( $template, $results_ref, $result_type ) = @_;
    my @out_list;

    if ( $result_type and $result_type eq "user" ) {
        for my $entry ( @{$results_ref} ) {
            next unless $entry;
            my $user_name = $entry->get_value('uid');

            # special case for admin, who does not have a uid                  
            $user_name = " " unless $user_name;
            my $login_status = $entry->get_value('shadowFlag') ? 0 : 1;
            my %i = (
                full_name    => text2html( $entry->get_value('cn') ),
                user_name    => $user_name,
                login_status => $login_status,
            );

            push @out_list, \%i;
        }
    }
    elsif ( $result_type and $result_type eq "class" ) {
        for my $entry ( @{$results_ref} ) {
            next unless $entry;
	    my $cn = $entry->get_value('cn');
	    my ($login_enabled, $login_disabled) 
		= ldap_get_group_login_status( $cn );
	    my %i = (
                full_name       => text2html( $cn ),
                cn              => text2html( $cn ),
		login_enabledf  => $login_enabled,
		login_disabledf => $login_disabled,
	    );
            push @out_list, \%i;
        }
    }
    $template->param(
        SearchResultLoop => \@out_list,

        #        debug_outputV    => text2html( ldap_err_output(), pre => 1 ),
    );
}

sub fill_in_operations {
    my ( $template, $q, $result_type ) = @_;

    # these flags disable the respective buttons in the webapp:
    # the "disable login" and the "enable login" button
    my $disable_enable_login  = find_disabled_user_logins($q);
    my $disable_disable_login = find_enabled_user_logins($q);

    #XXX missing: printing and internet access
    my $unauthorized = check_authorisation();    #
         #find out if we do a user or class operation
    $template->param(
        OperationsV        => 1,
        DisabledDisableLV  => $disable_disable_login,
        DisabledEnableLV   => $disable_enable_login,
        DisabledDeleteV    => $unauthorized,
        DisabledAddusersV  => $unauthorized,
        DisabledAddgroupsV => $unauthorized,
    );
}

sub check_authorisation {

    my $unauthorized = "root" ne $remote_user;
    $unauthorized = $unauthorized ? "1" : "0";
    return $unauthorized;
}

sub get_search_type {
    my ($q) = @_;
    my $result_type = undef;

    unless ( $q->param("typV") ) { return undef }

    #setting flags for special class oder user administration
    if ( ( $q->param("typV") eq "user_cn" )
        or ( $q->param("typV") eq "user_login" ) )
    {
        $result_type = "user";
    }
    elsif ( $q->param("typV") eq "class_cn" ) {
        $result_type = "class";
    }

    return $result_type;
}

sub find_enabled_user_logins {
    my ($q) = @_;

    my $enabled_logins = undef;

    my $typ        = $q->param("typV");
    my $search_exp = $q->param("SearchExpressionN");
    my ( $msg, @list ) = ldap_search_user_enabled_logins( $typ, $search_exp );

    if ( $msg eq "OK" ) { $enabled_logins = 1 }

    return $enabled_logins;
}

sub find_disabled_user_logins {
    my ($q) = @_;

    my $disabled_logins = undef;

    my $typ        = $q->param("typV");
    my $search_exp = $q->param("SearchExpressionN");
    my ( $msg, @list ) = ldap_search_user_disabled_logins( $typ, $search_exp );
    if ( $msg eq "OK" ) { $disabled_logins = 1 }

    return $disabled_logins;
}


sub check_passwd_quality {
    my ($p, $pw_warning_ref) = @_;
    
    return "no password" unless $p;

    my ( @passwd_warning, $passwd_warning );
    if ( $pw_warning_ref ) {
	@passwd_warning = @{$pw_warning_ref};
    }
    
    if($config{passwd_capital} > 0){
	push @passwd_warning, "passwd_no_capital"
	    unless ( $p =~ /[A-Z]/ );
    }
    if($config{passwd_lowercase} > 0){
	push @passwd_warning, "passwd_no_lowercase"
	    unless ( $p =~ /[a-z]/ );
    }
    if($config{passwd_number} > 0){
	push @passwd_warning, "passwd_no_number"
	    unless ( $p =~ /\d/ );
    }
    if($config{passwd_length} > 0){
	push @passwd_warning, "passwd_short"
	    unless ( $p =~ /.{$config{passwd_length},}/ );
    }

    unless ( $#passwd_warning  == -1 ) {
	$passwd_warning .= text2html( text('new_passwd_warning') . "\n",
				      paras => 1, 
				      blockquotes => 1, 
				      );
	for my $item (@passwd_warning) {
	    if($item ne "passwd_short"){
		$passwd_warning .= text2html( text($item) . "\n",
					      paras => 1, 
					      blockquotes => 1, ) 
		}
	    else {
		$passwd_warning .= text2html( text($item , $config{passwd_length} ) . "\n",
					      paras => 1, 
					      blockquotes => 1, ) 
		}
	}
    }
    return $passwd_warning;   
}

sub is_silly {
    my ( $uid ) = @_;

    return undef if $uid eq "root";

    my @auth_groups = ldap_get_member_grouplist ( $uid, "authority_groups" );
    
    for my $auth_group (  @auth_groups ) {
	return undef if ( $auth_group eq "jradmins" or
			  $auth_group eq "admins" );
    }
    return "user is probably silly";
}

sub debug {

    my @list = @_;
    my @c    = caller(1);
    unshift @list, [ "Line $c[2]", "Function $c[3]" ];
    my $fh;
    my $file = "/var/log/webmin/miniserv.error";
    open $fh, ">>$file" || return;
    my $dump = Dumper(@list);
    print $fh $dump;
    close $fh;
}



1;    # Eaer good perl module should have this #%D

