/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>

#include <stdlib.h>

#include "wfmetahost.h"
#include "wfindent.h"
#include "list1.h"
#include "defs.h"


bool
wf_metahost::add(const wf_metahost& metahost) {
  list<wf_metahost_elem>::const_iterator first = metahost.elems.begin(),
    last = metahost.elems.end();
  for (; first != last; ++first) {
    if (add(*first) == false)
      return false;
  }
  return true;
}

bool
wf_metahost::add(const wf_metahost_elem& elem) {
  if (elem.gettype() == WF_METAHOST_NONE)
    return false;

  elems.push_back(elem);
  return true;
}

void
wf_metahost::del(const wf_metahost_elem& elem) {
  elems.remove(elem);
}

void
wf_metahost::clear() {
  name = "";
  comment = "";
  elems.clear();
}

void
wf_metahost::substitute(const string& substname, const wf_metahost& metahost) {
  if (name == substname) {
    *this = metahost;
    name = "";
    /* This is a hack: we use this to make everyone believe it is not
       a reference, and thus output substituted value instead of name. */
  }
  else {
    list<wf_metahost_elem>::iterator first = elems.begin(), last = elems.end();
    for (; first != last; ++first) {
      if (first->gettype() == WF_METAHOST_METAHOST)
	first->get_metahost()->substitute(substname, metahost);
    }
  }
}

bool
wf_metahost::isreference() const {
  return (name.empty() == false && elems.empty());
}

unsigned int
wf_metahost::count_hosts() const {
  unsigned int count = 0;
  
  list<wf_metahost_elem>::const_iterator first = elems.begin(),
    last = elems.end();
  for (; first != last; ++first) {
    if (first->gettype() == WF_METAHOST_METAHOST)
      count += first->get_metahost()->count_hosts();
    else
      count++;
  }
  return count;
}

bool
wf_metahost::isnulladdr() const {
  if (name.empty() && elems.size() == 0)
    return true;
  if (elems.size() == 1) {
    const wf_metahost_elem& elem = elems.front();
    if (elem.gettype() == WF_METAHOST_NETWORK) {
      if (elem.get_network()->isnull())
	return true;
    }
    else if (elem.gettype() == WF_METAHOST_METAHOST)
      return elem.get_metahost()->isnulladdr();
  }
  return false;
}

ostream&
wf_metahost::print_value(ostream& os) const {
  list1_print(elems, 0, os);
  return os;
}

ostream&
wf_metahost::print(ostream& os) const {
  if (name.empty())
    list1_print(elems, 0, os);
  else
    os << '$' << name;
  return os;
}

ostream&
wf_metahost::output_xml(ostream& os, unsigned int indent_level) const {
  os << wf_indent(indent_level++) << "<metahost>\n";
  if (name.empty() == false)
    os << wf_indent(indent_level) << "<name>" << name << "</name>\n";

  list<wf_metahost_elem>::const_iterator first = elems.begin(),
    last = elems.end();
  for (; first != last; ++first)
    first->output_xml(os, indent_level);

  if (comment.empty() == false)
    os << wf_indent(indent_level) << "<comment>" << comment << "</comment>\n";
  return os << wf_indent(--indent_level) << "</metahost>" << endl;
}

ostream&
wf_metahost::debugprint(ostream& os) const {
  os << _("Metahost:") << endl;

  os << _("name:\t\t") << name << endl;

  os << _("elems:\t\t");
  /* replace 0 (always_parenthesis) by config variable RV@@4 */
  list1_print(elems, 0, os);
  os << endl;

  os << _("comment:\t") << comment << endl;

  return os;
}


ostream&
operator<<(ostream& os, const wf_metahost& metahost) {
  return metahost.print(os);
}
