/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#ifndef _WF_METAHOST_H
#define _WF_METAHOST_H

#include <list>
#include <string>

#include "wfhost_base.h"
#include "wfipaddr.h"
#include "wfnetwork.h"
#include "wfipaddr_range.h"

/*
 * metahost
 */
class wf_metahost;

class wf_metahost_elem {
 public:
  wf_metahost_elem();
  wf_metahost_elem(const wf_metahost& metahost);
  wf_metahost_elem(const wf_ipaddr& ipaddr);
  wf_metahost_elem(const wf_network& network);
  wf_metahost_elem(const wf_ipaddr_range& ipaddr_range);
  virtual ~wf_metahost_elem();
  wf_metahost_elem(const wf_metahost_elem&);
  wf_metahost_elem& operator=(const wf_metahost_elem&);

  bool set(const wf_metahost& metahost);
  bool set(const wf_ipaddr& ipaddr);
  bool set(const wf_network& network);
  bool set(const wf_ipaddr_range& ipaddr_range);

  const wf_metahost* get_metahost() const;
  const wf_ipaddr* get_ipaddr() const;
  const wf_network* get_network() const;
  const wf_ipaddr_range* get_ipaddr_range() const;
  wf_metahost* get_metahost();
  wf_ipaddr* get_ipaddr();
  wf_network* get_network();
  wf_ipaddr_range* get_ipaddr_range();

  ostream& print(ostream& os) const;
  ostream& output_xml(ostream& os, unsigned int indent_level = 0) const;
  ostream& debugprint(ostream& os) const;

  enum metahost_elem_type gettype() const {
    return (host_base != NULL) ? host_base->gettype() : WF_METAHOST_NONE;
  }

  friend ostream& operator<<(ostream& os, const wf_metahost_elem& elem);
  friend bool operator==(const wf_metahost_elem& elem1,
			 const wf_metahost_elem& elem2);

 private:
  wf_host_base* host_base;
};

class wf_metahost : public wf_host_base {
 public:
  wf_metahost() :
    name(),
    comment(),
    elems() {}
  virtual ~wf_metahost() {}
  // wf_metahost(const wf_metahost& metahost) {} implicit copy constructor
  virtual wf_host_base* VClone() const;

  bool add(const wf_metahost& metahost);
  bool add(const wf_metahost_elem& elem);
  void del(const wf_metahost_elem& elem);
  void clear();

  void substitute(const string& substname, const wf_metahost& metahost);

  bool isreference() const;
  unsigned int count_hosts() const;
  bool isnulladdr() const;
  bool empty() const { return elems.empty(); }
  
  virtual enum metahost_elem_type gettype() const { return WF_METAHOST_METAHOST; }
  virtual ostream& print(ostream& os) const;
  virtual ostream& print_value(ostream& os) const;
  virtual ostream& output_xml(ostream& os, unsigned int indent_level = 0) const;
  ostream& debugprint(ostream& os) const;

  friend ostream& operator<<(ostream& os, const wf_metahost& metahost);

 public:
  string name;
  string comment;
  list<wf_metahost_elem> elems;
};


inline wf_host_base*
wf_metahost::VClone() const {
  return new wf_metahost(*this);
}

inline const wf_metahost*
wf_metahost_elem::get_metahost() const {
  return (wf_metahost*)host_base;
}

inline const wf_ipaddr*
wf_metahost_elem::get_ipaddr() const {
  return (wf_ipaddr*)host_base;
}

inline const wf_network*
wf_metahost_elem::get_network() const {
  return (wf_network*)host_base;
}

inline const wf_ipaddr_range*
wf_metahost_elem::get_ipaddr_range() const {
  return (wf_ipaddr_range*)host_base;
}

inline wf_metahost*
wf_metahost_elem::get_metahost() {
  return (wf_metahost*)host_base;
}

inline wf_ipaddr*
wf_metahost_elem::get_ipaddr() {
  return (wf_ipaddr*)host_base;
}

inline wf_network*
wf_metahost_elem::get_network() {
  return (wf_network*)host_base;
}

inline wf_ipaddr_range*
wf_metahost_elem::get_ipaddr_range() {
  return (wf_ipaddr_range*)host_base;
}

#endif
