/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "sdlhandler.h"

#include <cstdlib>
#include <SDL/SDL_error.h>

#include "focus.h"
#include "mouse.h"
#include "rootwindow.h"
#include "joystick.h"

SigC::Signal1<bool,const SDL_UserEvent&,wftk::BoolMarshal> wftk::SDLHandler::userEvent;

static const std::string sdl_fatal_spacer = ": ";

wftk::SDLFatal::SDLFatal(const std::string& reason) :
  Fatal(reason + sdl_fatal_spacer + SDL_GetError())
{
  // this is a bit of a hack to get around the fact that Fatal
  // has to get initialized before sdl_error_

  // don't call SDL_GetError() twice, but still need to initialize sdl_error_;
  unsigned prefix_len = reason.size() + sdl_fatal_spacer.size();
  sdl_error_ = what() + prefix_len; // truncate prefix
}

wftk::SDLHandler::SDLHandler(const SDL_Event& event)
{
  memcpy(&event_, &event, sizeof(SDL_Event));
}

void wftk::SDLHandler::operator()()
{
  switch(event_.type) { 
    case SDL_KEYDOWN:
    case SDL_KEYUP:
      Focus::instance()->handleEvent(&event_);
      break;
    case SDL_MOUSEMOTION:
    case SDL_MOUSEBUTTONDOWN:
    case SDL_MOUSEBUTTONUP:
      Mouse::instance()->handleEvent(&event_);
      break;
    case SDL_QUIT:
      Application::instance()->quit();
      break;
    case SDL_ACTIVEEVENT:
      if(event_.active.state & SDL_APPMOUSEFOCUS)
        Mouse::instance()->handleEvent(&event_);
      if(event_.active.state & SDL_APPINPUTFOCUS)
        Focus::instance()->handleEvent(&event_);
      if(event_.active.state & SDL_APPACTIVE)
        RootWindow::handleEvent(&event_);
      break;
    case SDL_VIDEOEXPOSE:
    case SDL_VIDEORESIZE:
      RootWindow::handleEvent(&event_);
      break;
    case SDL_JOYAXISMOTION:
    case SDL_JOYBALLMOTION:
    case SDL_JOYHATMOTION:
    case SDL_JOYBUTTONUP:
    case SDL_JOYBUTTONDOWN:
      Joystick::handleEvent(&event_);
      break;
    case SDL_SYSWMEVENT:
      break; // unhandled
    default:
      assert(event_.type >= SDL_USEREVENT);
      assert(event_.type < SDL_NUMEVENTS);
      userEvent.emit(event_.user);
      break;
  }
}

void wftk::SDLHandler::queueEvents()
{
  // a reasonable number of events, but at least one
  const unsigned buff_size = 1024;
  const unsigned max_events = sizeof(SDL_Event) < buff_size ?
	buff_size / sizeof(SDL_Event) : 1;
  const Uint32 event_mask = SDL_ALLEVENTS - SDL_SYSWMEVENTMASK;
  SDL_Event events[max_events];
  int nevents;

  // drive the event loop
  SDL_PumpEvents();

  do {
    // don't pump the event loop, we already did that
    nevents = SDL_PeepEvents(events, max_events, SDL_GETEVENT, event_mask);

    if(nevents < 0) {
      // error -- FIXME error handling
      break;
    }

    assert((unsigned) nevents <= max_events);

    for(const SDL_Event* event = events; event < events + nevents; ++event)
      Application::instance()->pushEvent(new SDLHandler(*event));

  } while((unsigned) nevents == max_events); // there may be more events
}
