/*
 *   Copyright (C) 2002-2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "JT44Receive.h"
#include "JT44Lookups.h"
#include "JT44Defs.h"
#include "JT44App.h"

#include "common/Exception.h"
#include "common/SFFT.h"

#include <wx/datetime.h>
#include <wx/debug.h>
#include <wx/log.h>

#include <cmath>
using namespace std;

CJT44Receive::CJT44Receive(const wxString& name, EWho who) :
CReceive(name, who),
m_audioSamples(NULL),
m_correlations(NULL),
m_samplesCount(0),
m_fft(JT44_FFT_LENGTH)
{
	m_audioSamples = new double[JT44_MAX_AUDIO_DATA];

	int corrCount = (JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST) * (6 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR;
	m_correlations = new CCorrelation[corrCount];
}

CJT44Receive::~CJT44Receive()
{
	delete[] m_audioSamples;
	delete[] m_correlations;
}

void CJT44Receive::reset(bool firstTime)
{
	int corrCount = (JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST) * (6 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR;
	for (int i = 0; i < corrCount; i++)
		m_correlations[i].clear();

	m_samplesCount = 0;
}

void CJT44Receive::run()
{
	wxString id = createId();

	bool end = false;

	openSoundDevice();

	// The main loop ends if we have been asked to end by the controller
	// isStopped() is true, or we have enough samples, or our time is
	// at an end. At around 29.8 or 59.8 on the clock.
	while (!isStopped() && !end && m_samplesCount < (JT44_MAX_AUDIO_DATA - JT44_SOUNDBUF_LENGTH)) {
		int len = JT44_SOUNDBUF_LENGTH;

		getSoundDevice()->read(m_audioSamples + m_samplesCount, len);

		if (len <= 0) break;

		m_samplesCount += len;

		::wxGetApp().showAudio(m_audioSamples[m_samplesCount - 100], getWho());

		end = getEndTime();
	}

	closeSoundDevice();

	recordAudio(id, m_audioSamples, m_samplesCount);

	double bins[JT44_FFT_LENGTH];

	for (int samplesCount = 0; samplesCount < m_samplesCount; samplesCount++) {
		if (samplesCount >= JT44_FFT_LENGTH && (samplesCount % JT44_SKIP_FACTOR) == 0) {
			m_fft.process(m_audioSamples + samplesCount - JT44_FFT_LENGTH, bins);
			storeCorrelations(bins, samplesCount);
		}
	}

	// Create the data for the GUI graphs and send it to the GUI
	createReceiveData();

	// Find the approximate correlation time and exact bin number
	int syncBin, timeOffset;
	findCorrelation(syncBin, timeOffset);

	// Find the exact correlation time
	correlate(syncBin, timeOffset);

	// Create the message carrier object with the raw FFT outputs
	// for the FFT bins.
	decode(id, syncBin, timeOffset);
}

/*
 * "You are not expected to understand this"
 *
 * This horrible piece of code is the approximate correlator that
 * slides the reference signal past the incoming data and performs the
 * appropriate correlation calculation and stores it in the appropriate
 * CCorrelation object.
 *
 * "This code was hard to write, therefore it should be hard to understand"
 */
void CJT44Receive::storeCorrelations(double* data, int samplesCount)
{
	wxASSERT(m_correlations != NULL);
	wxASSERT(data != NULL);
	wxASSERT(samplesCount >= 0);

	CJT44Lookups lookups;

	int sampleNo = samplesCount / JT44_SKIP_FACTOR;

	for (int t = 0; t < (6 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR; t++) {
		int n = sampleNo - t + (1 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR;

		int pos = n / JT44_SKIP_RATE;

		if (pos >= 0 && (n % JT44_SKIP_RATE) == 0 && pos < 135) {
			double mult = lookups.lookupSync(pos) ? JT44_SYNC_POS_COEFF : JT44_SYNC_NEG_COEFF;

			for (int bin = 0; bin < (JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST); bin++) {
				int index = getCorrelationsIndex(bin, t);
				m_correlations[index].addProduct(mult, data[bin + JT44_SYNCBIN_FIRST]);
			}
		}
	}
}

/*
 * This is simpler than it looks. It simply searches a 2D array (stored as a
 * 1D array) for the best correlation value and returns it. These are
 * only approximate correlations, the exact FFT bin is returned but only
 * the approximate timing.
 */
void CJT44Receive::findCorrelation(int& syncBin, int& offset) const
{
	wxASSERT(m_correlations != NULL);

	double quality = 0.0;
	int maxI = 0, maxJ = 0;

	// For each potential FFT bin containing the synchonisation data
	for (int i = 0; i < (JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST); i++) {

		// For each of the correlation times
		for (int j = 0; j < (6 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR; j++) {
			int n = getCorrelationsIndex(i, j);

			double value;
			bool ret = m_correlations[n].getRawValue(value);

			// Find the best correlation
			if (ret && value > quality) {
				quality = value;
				maxJ    = j;
				maxI    = i;
			}
		}
	}

	offset  = maxJ * JT44_SKIP_FACTOR - JT44_SAMPLE_RATE;
	syncBin = maxI + JT44_SYNCBIN_FIRST;
}

/*
 * This method finds the exact correlation time within the already
 * detected synchronisation bin. Because of the previous
 * approximate correlation, we only have to search within a relatively
 * narrow time range, thus saving a large amount of time.
 */
void CJT44Receive::correlate(int syncBin, int& offset) const
{
	wxASSERT(m_audioSamples != NULL);
	wxASSERT(m_samplesCount > 0);
	wxASSERT(syncBin >= JT44_SYNCBIN_FIRST);

	CJT44Lookups lookup;

	double* syncSamples = new double[m_samplesCount];

	CSFFT sfft(JT44_FFT_LENGTH, syncBin, syncBin + 1);

	double bins[JT44_FFT_LENGTH];

	for (int i = 0; i < m_samplesCount; i++) {
		sfft.process(m_audioSamples[i], bins);
		syncSamples[i] = bins[syncBin];
	}

	double bestCorrelation = 0.0;

	for (int timeOffset = offset - JT44_SKIP_FACTOR; timeOffset < (offset + JT44_SKIP_FACTOR); timeOffset++) {
		CCorrelation correlation;

		for (int i = 0; i < 135; i++) {
			int dataIndex = timeOffset + (i * JT44_SYMBOL_LENGTH);

			if (dataIndex >= 0 && dataIndex < m_samplesCount) {
				if (lookup.lookupSync(i))
					correlation.addProduct(JT44_SYNC_POS_COEFF, syncSamples[dataIndex]);
				else
					correlation.addProduct(JT44_SYNC_NEG_COEFF, syncSamples[dataIndex]);
			}
		}

		double value;
		bool ret = correlation.getRawValue(value);

		if (ret && value > bestCorrelation) {
			bestCorrelation = value;
			offset          = timeOffset;
		}
	}

	delete[] syncSamples;
}

/*
 * At this stage we do the decoding of the FFT bins that contain the message
 * data. We have the timing and location of these FFT bins so it is a
 * quick and efficient piece of code. We simply store the message data and
 * make no decision about which character is represented where, this is
 * done later in the CJT44Message class.
 */
void CJT44Receive::decode(const wxString& id, int syncBin, int offset)
{
	wxASSERT(m_audioSamples != NULL);
	wxASSERT(m_samplesCount > 0);
	wxASSERT(syncBin >= JT44_SYNCBIN_FIRST);

	CJT44Message* message = new CJT44Message();

	message->setId(id);
	message->setDt(calculateTime(offset));
	message->setDF(calculateFrequency(syncBin));

	double bins[JT44_FFT_LENGTH];

	for (int i = 0; i < 135; i++) {
		int index = offset + (i * JT44_SYMBOL_LENGTH) - JT44_FFT_LENGTH;

		if (index >= 0 && index < (m_samplesCount - JT44_FFT_LENGTH)) {
			m_fft.process(m_audioSamples + index, bins);

			double noise = 0.0;
			noise += bins[syncBin - 3];
			noise += bins[syncBin - 2];
			noise += bins[syncBin + 2];
			noise += bins[syncBin + 3];

			message->setSync(i, bins[syncBin]);
			message->setNoise(i, noise / 4.0);

			for (int j = 0; j < JT44_ALPHABET_COUNT; j++) {
				int binNo = syncBin + (j * 2) + 6;

				message->setData(j, i, bins[binNo]);
			}
		}
	}

	// Send the message data to the GUI
	::wxGetApp().receiveMessage(message, getWho());
}

/*
 * Prepare the results of the approximate correlation so that they may
 * be displayed on the graph in the GUI.
 */
void CJT44Receive::createReceiveData() const
{
	CJT44Levels* levels = new CJT44Levels();

	levels->setAudioData(m_audioSamples, m_samplesCount);

	double* maxDF = new double[JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST];
	for (int i = 0; i < (JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST); i++)
		maxDF[i] = 0.0;

	double* maxDt = new double[(6 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR];
	for (int i = 0; i < (6 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR; i++)
		maxDt[i] = 0.0;

	for (int bin = 0; bin < (JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST); bin++) {
		for (int offset = 0; offset < (6 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR; offset++) {
			int index = getCorrelationsIndex(bin, offset);

			double value;
			bool ret = m_correlations[index].getNormalisedValue(value);

			if (ret) {
				if (value > maxDF[bin])
					maxDF[bin] = value;

				if (value > maxDt[offset])
					maxDt[offset] = value;
			}
		}
	}

	levels->setDFData(maxDF, JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST);

	levels->setDtData(maxDt, (6 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR);

	delete[] maxDF;
	delete[] maxDt;

	::wxGetApp().showLevels(levels, getWho());
}

int CJT44Receive::getCorrelationsIndex(int bin, int offset) const
{
	int nBins   = JT44_SYNCBIN_LAST - JT44_SYNCBIN_FIRST;
	int nOffset = (6 * JT44_SAMPLE_RATE) / JT44_SKIP_FACTOR;

	wxASSERT(bin >= 0 && bin < nBins);
	wxASSERT(offset >= 0 && offset < nOffset);

	return bin * nOffset + offset;
}

int CJT44Receive::getSymbolsIndex(int letter, int pos) const
{
	wxASSERT(letter >= 0 && letter < JT44_ALPHABET_COUNT);
	wxASSERT(pos >= 0 && pos < JT44_MESSAGE_LENGTH);

	return letter * JT44_MESSAGE_LENGTH + pos;
}

double CJT44Receive::calculateTime(int offset) const
{
	return double(offset - JT44_SAMPLE_RATE) / double(JT44_SAMPLE_RATE);
}

int CJT44Receive::calculateFrequency(int bin) const
{
	return (bin - 236) * JT44_SAMPLE_RATE / JT44_SYMBOL_LENGTH;
}
