/*
 * Copyright (C) 2003 the xmms-kde team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#ifndef __XMMS_KDE_H__
#define __XMMS_KDE_H__

#include <config.h>

#include <kaboutapplication.h>
#include <kaboutdata.h>
#include <kapp.h>
#include <kapplication.h>
#include <kglobalaccel.h>
#include <kglobal.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpanelapplet.h>
#include <kprocess.h>
#include <ksimpleconfig.h>
#include <kstddirs.h>

#include <math.h>

#include <qcolor.h>
#include <qdir.h>
#include <qdragobject.h>
#include <qfont.h>
#include <qfontinfo.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qpoint.h>
#include <qpopupmenu.h>
#include <qsize.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <qvariant.h>

#include <stdio.h>

#include "noatunplayer.h"
#include "osdframe.h"
#include "playerinterface.h"
#include "configdialog.h"
#include "smpegplayer.h"
#include "xmms-kdedb.h"
#include "xmmsplayer.h"


//! A widget-theme class for xmms-kde.
/*!
  This class contains pixmaps to render the visuals of xmms-kde in the panel,
  information about the fonts to use, and the positions of the song title,
  the buttons and sliders.
*/
class XmmsKdeTheme {

public:
  //! the constuctor.
  XmmsKdeTheme();

  //! the destructor.
  ~XmmsKdeTheme();

  //! load a new theme from a .tar.gz archive.
  /*!
    Erase the old theme, and load a new one.
    \param themeName  the name of the archive
    \param dataDirs   the directories in which to look for the theme
   */
  int loadTheme(QString themeName, KStandardDirs *dataDirs);

public: 
  //! the name of the theme.
  QString themeName;
  
  //! the color of the displayed time.
  QColor timeColor;
  
  //! the font to display the time.
  QFont  timeFont;

  //! the position where the elapsed time is displayed.
  QPoint timePlayedPos;

  //! the position where the total time is displayed.
  QPoint timeTotalPos;

  //! the main font.
  QFont  mainFont;

  //! the color of the title.
  QColor titleColor;

  //! the font to use for the title.
  QFont  titleFont;
  
  //! the position of the title.
  int    titleYPos;      

  //! the clipping area of the title.
  QRect  titleClipRect;
  
  //! the pixmap of the volume slider.
  QPixmap *volumePixmap;

  //! the pixmap of the seek slider.
  QPixmap *seekPixmap;

  //! the background pixmap 
  QPixmap *mainPixmap;

  //! the pixmap for the pressed back button.
  QPixmap *backPixmap;

  //! the pixmap for the pressed next button.
  QPixmap *nextPixmap;

  //! the pixmap for the pressed pause button.
  QPixmap *pausePixmap;

  //! the pixmap for the pressed stop button.
  QPixmap *stopPixmap;

  //! the pixmap for the pressed play button.
  QPixmap *playPixmap;

  //! the pixmap for the pressed shuffle button.
  QPixmap *shuffleOnPixmap;

  //! the pixmap for the pressed repeat button.
  QPixmap *repeatOnPixmap;
  
  //! the areas that react to mouse clicks.
  QRect rectBack, rectPlay, rectPause, rectStop, rectSkip, rectVolume, rectSeek;

  //! mouse click areas for shuffle and repeat.
  QRect *rectShuffle, *rectRepeat;
  
  //! the width and height of the theme.
  int myWidth, myHeight;

private:
  //! load a simple theme.
  /*!
    \param file              the filename of the theme archive.
    \param mainBitmapFile    the filename of the background pixmap.
    \param volumeBitmapFile  the filename of the volume slider pixmap.
    \param seekBitmapFile    the filename of the seek slider pixmap.
   */
  bool loadBasicSkin(QString file, 
		     QString mainBitmapFile, 
		     QString volumeBitmapFile, 
		     QString seekBitmapFile);

  //! load an extended theme.
  /*!
    \param file              the filename of the theme archive.
    \param backBitmapFile    the filename of the back button pixmap.
    \param nextBitmapFile    the filename of the next button pixmap.
    \param pauseBitmapFile   the filename of the pause button pixmap.
    \param stopBitmapFile    the filename of the stop button pixmap.
    \param playBitmapFile    the filename of the play button pixmap.
    \param shuffleBitmapFile the filename of the shuffle button pixmap.
    \param repeatBitmapFile  the filename of the repeat button pixmap.
   */
  bool loadExtendedSkin(QString file, 
			QString &backBitmapFile,
			QString &nextBitmapFile,
			QString &pauseBitmapFile,
			QString &stopBitmapFile, 
			QString &playBitmapFile,
			QString &shuffleBitmapFile,
			QString &repeatBitmapFile);

  //! release the resources held by this theme.  
  void deleteOld();
};


//! the main class of xmms-kde.
/*!
  This is the main class of the panel applet. It is mainly responsible for 
  user interaction.
*/
class XmmsKde : public KPanelApplet, QToolTip {
  Q_OBJECT
public:
  //! constuct a new panel applet based on a given configuration file.
  /*!
    \param configFile  the configuration file.
    \param t           the type of the applet.
    \param actions     the actions that the applet can do.
    \param parent      the parent widget of the applet.
    \param name        the name of the applet in the widget tree.
  */
  XmmsKde(const QString& configFile, Type t = Normal, int actions = 0,
                QWidget *parent = 0, const char *name = 0);

  //! destructor of the applet.
  virtual ~XmmsKde();

  //! the width of the applet.
  int widthForHeight(int height) const;

  //! the height of the applet.
  int heightForWidth(int width) const;

  //! display an about dialog.
  void about();

  //! save the current configuration to disk.
  void saveConfig(); 

public slots:
  //! launch the help browser.
  void help();

protected:
  //! repaint the controls
  void paintEvent(QPaintEvent*);

  //! resize the applet.
  void resizeEvent(QResizeEvent*);

  //! reacts on mouse press events.
  void mousePressEvent(QMouseEvent*);

  //! reacts on mouse release events.
  void mouseReleaseEvent(QMouseEvent*);

  //! reacts on mouse move events.
  void mouseMoveEvent(QMouseEvent*);

  //! reacts on drag and drop events.
  void dropEvent(QDropEvent*);

  //! reacts on drag and drop events.
  void dragEnterEvent(QDragEnterEvent*);

  //! display a tool tip.
  virtual void maybeTip(const QPoint&);

  //! adjust the volume on mouse wheel events.
  virtual void wheelEvent( QWheelEvent * );
  
private:
  //! start the timers for communication with the player and title scrolling.
  void setTimers(bool);
  
public slots:
  //! show a configuration dialog.
  void preferences();

protected slots:

private slots:  

  //! repaint the applet
  void paint();

  //! scroll the title.
  void scroll();

  //! ask the player for information about the played song.
  void receive();

  //! load a new theme.
  void loadTheme(QString);

  //! return to the previous song in the playlist.
  void previous();

  //! start playing.
  void play();

  //! start playing if paused, pause if playing.
  void playpause();

  //! pause playing.
  void pause();

  //! stop playing.
  void stop();

  //! jump to the next song in the playlist.
  void next();

  //! turn the volume up.
  void volp();

  //! turn the volume down.
  void volm();

  //! show the database dialog.
  void showDB();

  //! exit the player.
  void exitPlayer();

  //! the user toggled the shuffle button.
  void toggleShuffle();

  //! the user toggled the repeat button.
  void toggleRepeat();

  //! the user tries to seek forward in the current song.
  void seekForward();

  //! the user tries to seek backwards in the current song.
  void seekBackward();

  
 private:
  // values constantly updated from player.

  //! the current volume.
  int volume;

  //! the length of the song.
  int trackTime;

  //! the number of milliseconds to skip when seeking in the song.
  int seekTime;

  //! the current elapsed time.
  int currentTime;

  //! the position of seek handle if in drag move!!
  int currentSeekTime; 

  //! the current position in the playlist.
  int playListPos;

  //! status of the repeat button.
  bool repeatOn;

  //! status of the shuffle button.
  bool shuffleOn;

  //! status of the player.
  bool running;

  //! does the user try to change the volume?
  bool inVolumeChange;

  //! does the user try to seek currently?
  bool inSeekChange;  

  //! status of the pause button.
  bool pausePressed;

  //! status of the next button.
  bool nextPressed;

  //! status of the back button.
  bool backPressed;

  //! status of the play button.
  bool playPressed;

  //! status of the stop button.
  bool stopPressed;

  //! has the title looped once in loop once mode?
  bool loopedOnce;

  //! the current x position of the title.
  int titleXPosition;

  //! the current speed of the title scrolling.
  int titleScrollSpeed;

  //! the default speed of the title scrolling.
  int titleDefaultScrollSpeed;

  //! the scroll mode of the title.
  int titleScrollMode;

  //! the width of the title.
  int titleWidth;

  //! the current title position in sinus scroll mode.
  float titleSinArg;
  
  //! is the elapsed time displayed?
  int displayElapsedTime;  

  //! the timer for the title scrolling.
  QTimer *scrollTimer;

  //! the timer for the player communication.
  QTimer *receiveTimer;

  //! the title of the current song.
  QString title;

  //! the directories to search for themes.
  KStandardDirs *dataDirs;

  //! the popupmenu for user interaction.
  QPopupMenu *themeMenu;

  //! the list of all available themes.
  QStringList *themeList;

  //! the playlist of the smpeg player.
  QStringList smpegPlayList;

  //! shall the spmeg player repeat the playlist?
  bool smpegRepeat;

  //! shall the spmeg player shuffle the playlist?
  bool smpegShuffle;

  //! the main configuration object.
  KConfig *mainConfig;

  //! the database.
  XmmsKdeDB *db;

  //! the database query dialog.
  XmmsKdeDBQuery *query;

  //! the on screen display.
  OSDFrame *osd;

  //! the object that controls keyboard shortcuts.
  KGlobalAccel  *globalAccel;

  //! the currently selected theme.
  XmmsKdeTheme theme;

  //! the currently selected player.
  PlayerInterface *player;

  // xmms player configuration.
  bool hideXmms;
  bool readID3Tags;
  bool convertUnderscores;

  // smpeg player configuration.
  QSize smpegPlayListSize;
  QPoint smpegPlayListPos;

  // the icon for the smpeg playlist.
  QPixmap *playListIcon;

  //! the contents of the about dialog.
  KAboutData *aboutData;
};


#endif

