/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <string.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>
#include <xmms/util.h>

#include "singit_debug.h"
#include "singit/song.h"
#include "singit_song_private.h"

#include "singit_main.h"

#include "singit_file_info.h"

extern VisPlugin singit_vp;

SingitFileInfo* singit_file_info_new(gchar *filename, gboolean hash)
{
	SingitFileInfo *result =
		(SingitFileInfo*) g_malloc(sizeof(SingitFileInfo));

	if (filename != NULL) {
		result->name = g_strdup(filename);
		singit_file_info_reset(result, hash);
	}
	else { result->name = NULL; }

	return result;
}

void singit_file_info_free(SingitFileInfo *info)
{
	if (info == NULL) { return; }

	if (info->name != NULL) { g_free(info->name); }
	g_free(info);
}

void singit_file_info_free_list(SingitFileInfo **infos)
{
	gint i;

	if (infos == NULL) { return; }

	for (i = 0; infos[i] != NULL; i++)
		singit_file_info_free(infos[i]);
	g_free(infos);
}

gboolean singit_file_info_reset(SingitFileInfo *info, gboolean hash)
{
	FILE *file;
	SHA_INFO sha;

	if ((info == NULL) || (info->name == NULL))
		{ return FALSE; }

	if (stat(info->name, &info->stats) == -1) { return FALSE; }
	if ((file = fopen(info->name, "r")) == NULL) { return FALSE; }

	if (hash == TRUE)
		{ sha_stream(info->sha_digest, &sha, file); }

	fclose(file);

	return TRUE;
}

gboolean singit_file_info_reinit(SingitFileInfo *info, gchar *filename, gboolean hash)
{
	FILE *file;
	SHA_INFO sha;

	if (info == NULL)
		{ return FALSE; }

	if (info->name != NULL) { g_free(info->name); }
	if (filename != NULL)
		{ info->name = g_strdup(filename); }
	else { info->name = NULL; }

	if (stat(info->name, &info->stats) == -1) { return FALSE; }

	if ((file = fopen(info->name, "r")) == NULL) { return FALSE; }

	if (hash == TRUE)
		{ sha_stream(info->sha_digest, &sha, file); }

	fclose(file);

	return TRUE;
}

gint singit_file_info_changed(SingitFileInfo *info, struct stat *stats, FILE *handle, gboolean hash)
{
	FILE *file = handle;
	struct stat in_stats;
	SHA_INFO sha;
	guchar new_sha_digest[SHA_DIGESTSIZE];
	gint i;

	if ((info == NULL) || (info->name == NULL))
		{ return -1; }

	if (stats == NULL) {
		if (stat(info->name, &in_stats) == -1)
			{ return -1; }
		stats = &in_stats;
	}

	if (info->stats.st_size != stats->st_size)
		{ return 1; }

	if (info->stats.st_mtime != stats->st_mtime)
		{ return 1; }

	if (hash == TRUE) {
		if ((handle == NULL) && ((file = fopen(info->name, "r")) == 0))
			{ return -1; }

		memcpy(&new_sha_digest[0], &info->sha_digest[0], SHA_DIGESTSIZE);
		sha_stream(info->sha_digest, &sha, file);
		if (handle == 0) { fclose(file); }

		for (i = 0; i < 20; i++)
			if (new_sha_digest[i] != info->sha_digest[i]) {
#ifdef CODEDEBUG
				DEBUG(DLV_ALL, ("singit_song.c [singit_song_lyrics_changed] : SHA Changed\n"));
#endif
				return 1;
		}
	}

	return 0;
}

SingitFileInfo* singit_file_info_any_changed(SingitFileInfo **infos, gboolean hash)
{
	gint i = 0;
	SingitFileInfo *result = NULL;

	g_return_val_if_fail(infos != NULL, NULL);

	while ((infos[i] != NULL) && (result == NULL)) {
		if (singit_file_info_changed(infos[i], NULL, NULL, hash) > 0)
			{ result = infos[i]; }
		i++;
	}

	return result;
}
