#include <string>
#include <sstream>
#include <iostream>
#include <gtkmm/messagedialog.h>
#include <time.h>

using namespace std;
#include "gtklibrary.h"
#include "musiccatalog.h"
#include "playlist.h"
#include "metadata.h"
#include "preferences.h"
#include "gtkmusicbrowser.h"
#include "gtktracklist.h"
#include "gtkalbumlist.h"
#include "gtklibrarytree.h"
#include "player.h"
#include "cataloginfo.h"

GTKLibrary::GTKLibrary(FAContext *context, PlaylistManager *plm, GTKBrowserMusic *musicbrowser) 
{
	//Init Variables
	m_PlayImmediately = false;
	m_last_change = time(NULL);
	m_context = context;
	m_plm = plm;
	m_musicbrowser = musicbrowser;
	
	m_TrackList = NULL;
	m_AlbumList = NULL;
	m_LibraryTree = NULL;

	//Init CatalogInfo Cache
	CatalogInfo::init(m_context, m_musicbrowser);
	
	//Set Style
	/*set_label("Music Library");
	set_shadow_type(Gtk::SHADOW_NONE);
	set_border_width(6);*/
	
	//Read Prefrences
	ReadPrefs();
	
	//Create Menu
	Gtk::Menu::MenuList& MenuList = m_catalogPopupMenu.items();
	
	MenuList.push_back( Gtk::Menu_Helpers::MenuElem("Play Now", SigC::slot(*this, &GTKLibrary::on_menu_play)) );
	MenuList.push_back( Gtk::Menu_Helpers::MenuElem("Add to Playlist", SigC::slot(*this, &GTKLibrary::on_menu_add)) );
	MenuList.push_back( Gtk::Menu_Helpers::MenuElem("Add and Play Now", SigC::slot(*this, &GTKLibrary::on_menu_add_play)) );
	MenuList.push_back( Gtk::Menu_Helpers::SeparatorElem() );
	MenuList.push_back( Gtk::Menu_Helpers::MenuElem("Remove", SigC::slot(*this, &GTKLibrary::on_menu_remove)) );
	menu_loc_remove = 4;
	MenuList.push_back( Gtk::Menu_Helpers::SeparatorElem() );
	MenuList.push_back( Gtk::Menu_Helpers::MenuElem("Edit Info", SigC::slot(*this, &GTKLibrary::on_menu_edit)) );
	menu_loc_edit = 6;
}

GTKLibrary::~GTKLibrary()
{
	//Delete Created Widgets
	delete m_AlbumList;
	delete m_TrackList;
	delete m_LibraryTree;
	
	//Delete Selection
	std::vector<CatalogInfo *>::iterator item;
	for(item = selected_info.begin(); item != selected_info.end(); item++)
		delete *item;
}

void GTKLibrary::ReadPrefs()
{
	m_context->prefs->GetPrefBoolean(kPlayImmediatelyPref, m_PlayImmediately);
}

Error GTKLibrary::AcceptEvent(Event* event)
{
	switch(event->Type()) {
		case INFO_PrefsChanged:
			ReadPrefs();
			break;
		
		case INFO_MusicCatalogTrackAdded:
		case INFO_MusicCatalogTrackRemoved:
		case INFO_MusicCatalogPlaylistAdded:
		case INFO_MusicCatalogPlaylistRemoved:
		case INFO_MusicCatalogStreamAdded:
		case INFO_MusicCatalogStreamRemoved:
		case INFO_MusicCatalogCleared:
		case INFO_MusicCatalogTrackChanged:
			m_last_change = time(NULL);
			if(m_LibraryTree) m_LibraryTree->Refresh();
			if(m_AlbumList) m_AlbumList->Refresh();
			if(m_TrackList) m_TrackList->Refresh();
			m_musicbrowser->StatusText("Music Library Changed", 1);
			break;
			
		default:
			break;
	}
	
	return kError_NoErr;
}

void GTKLibrary::DoEditTrack()
{
	//Display Edit Dialog For Currently Selected Tracks
	on_menu_edit();
}

void GTKLibrary::DoAddTrack()
{
	//Add Currently Selected Tracks To Playlist
	on_menu_add();
}

bool GTKLibrary::has_focus()
{
	bool focus = false;
	if(m_TrackList) focus = focus || m_TrackList->has_focus();
	if(m_AlbumList) focus = focus || m_AlbumList->has_focus();
	if(m_LibraryTree) focus = focus || m_LibraryTree->has_focus();
	
	return focus;
}

void GTKLibrary::row_activated(CatalogInfo *catalog_info, bool force_playimmediately, bool force_onlyadd)
{
	//Put Data In Vector
	std::vector<CatalogInfo *> catalog_infos;
	catalog_infos.push_back(catalog_info);
	
	rows_activated(catalog_infos, force_playimmediately, force_onlyadd);
}

void GTKLibrary::rows_activated(std::vector<CatalogInfo *> catalog_infos, bool force_playimmediately, bool force_onlyadd)
{
	bool PlaylistCleared = false;
	
	//Clear Playlist If About To Add Files And PlayImmediatly
	if((m_PlayImmediately || force_playimmediately) && !force_onlyadd) { 
		m_plm->RemoveAll();
        m_plm->clear();
		PlaylistCleared = true;
	}
	
	std::vector<CatalogInfo *>::iterator catalog_info;
	vector<string> tracks;

	for(catalog_info = catalog_infos.begin(); catalog_info != catalog_infos.end(); catalog_info++) {
        (*catalog_info)->get_tracks(tracks);
        Playlist_AddItems(tracks);        
	}
	
	if(PlaylistCleared)
		m_musicbrowser->StartPlaying();
}

void GTKLibrary::Playlist_AddItems(const std::vector<std::string>& URLs)
{
	m_plm->AddItems(URLs);
    m_plm->insert(m_plm->end(), URLs.begin(), URLs.end());
}

void GTKLibrary::Playlist_AddItem(const std::string& URL)
{
	m_plm->AddItem(URL);
    m_plm->insert(m_plm->end(), URL);
}

void GTKLibrary::Playlist_ReadPlaylist(const std::string& Playlist)
{
	m_plm->ReadPlaylist(Playlist.c_str());
}

void GTKLibrary::set_layout(MusicLayout *layout)
{
	//See Which Widgets Are Needed
	if(layout->need_library_tree()) {
		if(!m_LibraryTree)
			m_LibraryTree = new GTKLibraryTree(this);
		layout->set_library_tree(m_LibraryTree);
	} else {
		delete m_LibraryTree;
		m_LibraryTree = NULL;
	}
	
	if(layout->need_album_list()) {
		if(!m_AlbumList)
			m_AlbumList = new GTKAlbumList(this);
		layout->set_album_list(m_AlbumList);
	} else {
		delete m_AlbumList;
		m_AlbumList = NULL;
	}
	
	if(layout->need_track_list()) {
		if(!m_TrackList)
			m_TrackList = new GTKTrackList(this);
		layout->set_track_list(m_TrackList);
	} else {
		delete m_TrackList;
		m_TrackList = NULL;
	}
}

void GTKLibrary::set_selection(std::vector<CatalogInfo *>& new_selection)
{
	//Delete Old Selection
	std::vector<CatalogInfo *>::iterator item;
	for(item = selected_info.begin(); item != selected_info.end(); item++)
		delete *item;
	
	//Set New Selection
	selected_info = new_selection;
}

void GTKLibrary::on_menu_play()
{
	rows_activated(selected_info, true, false);
}

void GTKLibrary::on_menu_add()
{
	rows_activated(selected_info, false, true);
}

void GTKLibrary::on_menu_add_play()
{
	//KGK uint32_t start_index = m_plm->CountItems(); //Added items will start at...
	uint32_t start_index = m_plm->size(); //Added items will start at...
	rows_activated(selected_info, false, true);
	m_musicbrowser->StartPlaying(true, start_index);
}

void GTKLibrary::on_menu_remove()
{
	//Get Selected URLs
	std::vector<std::string> URLs;
	std::vector<CatalogInfo *>::iterator selected_item;
	for(selected_item = selected_info.begin(); selected_item != selected_info.end(); selected_item++) {
		std::vector<std::string> temp_urls;
		(*selected_item)->get_tracks(temp_urls);
		URLs.insert(URLs.end(), temp_urls.begin(), temp_urls.end());
	}
	
	if(URLs.size() > 0) {
		//Check With User Before Removing
		Gtk::MessageDialog Dialog("Are you sure you want to remove these items from the library?", Gtk::MESSAGE_WARNING, Gtk::BUTTONS_YES_NO);
		if(Dialog.run() == Gtk::RESPONSE_YES) {
			//Remove URLs
			std::vector<std::string>::iterator url;
			for(url = URLs.begin(); url != URLs.end(); url++) {
				m_context->catalog->Remove((*url).c_str());
			}
		}
	}
}

void GTKLibrary::on_menu_edit()
{
	//Get All Selected Tracks
	std::vector<std::string> URLs;
	std::vector<CatalogInfo *>::iterator selected_item;
	for(selected_item = selected_info.begin(); selected_item != selected_info.end(); selected_item++) {
		if((*selected_item)->is_playlist()) {
			//Is Playlist
			m_musicbrowser->OpenPlaylist((*selected_item)->item_url());
			continue;
		}
		
		//Add Selected Items URLs To List
		std::vector<std::string> child_urls;
		(*selected_item)->get_tracks(child_urls);
		URLs.insert(URLs.end(), child_urls.begin(), child_urls.end());
	}
	
	if(URLs.size() > 0)
		m_musicbrowser->TracksToEdit(&URLs);
}
/* arch-tag: bd669691-1957-4544-b2f4-321702ca504a */
