/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.explain.view;

import java.awt.event.*;
import java.awt.Image;
import java.util.HashMap;
import javax.swing.ImageIcon;
import javax.swing.tree.TreeModel;
import com.redhat.rhdb.treedisplay.TreeDisplay;
import com.redhat.rhdb.treedisplay.TreeLayoutNode;
import com.redhat.rhdb.explain.ExplainTreeNode;

/**
 * Extension of TreeDisplay specialized for drawing EXPLAIN
 * trees. TreeDisplay has extended with the addition of a couple of
 * new properties (large icons and image directory). The icon for a
 * given explain tree node type can be retrieved with {@link
 * #getIcon}.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 * @see com.redhat.rhdb.explain.view.ExplainTreeLayoutModel
 * @see com.redhat.rhdb.explain.view.ExplainTreeLayoutNode
 * @see com.redhat.rhdb.explain.view.ExplainTreeRenderer
 */

public class ExplainTreeDisplay extends TreeDisplay {

	/** Large icon property */
	public static final String LARGE_ICON_PROPERTY = "largeIcons";

	/** Image directory property */
	public static final String IMAGE_DIR_PROPERTY = "imageDir";

	private boolean largeicon;
	private HashMap icons, licons;
	private String imagedir = "/com/redhat/rhdb/explain/view/images";
	private ExplainTreeLayoutModel etlm;

	/**
	 * Create a new <code>ExplainTreeDisplay</code> instance.
	 */
	public ExplainTreeDisplay()
	{
		super();
		init();
	}

	/**
	 * Create a new <code>ExplainTreeDisplay</code> instance.
	 *
	 * @param tm a <code>TreeModel</code> value
	 */
	public ExplainTreeDisplay(TreeModel tm)
	{
		super(tm);
		init();
	}

	/**
	 * Are large icons being used instead of normal sized ones?
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isUseLargeIcons()
	{
		return largeicon;
	}

	/**
	 * Turn on/off usage of large icons.
	 *
	 * @param b a <code>boolean</code> value
	 */
	public void setUseLargeIcons(boolean b)
	{
		boolean old = largeicon;

		if (b != old)
		{
			largeicon = b;
			firePropertyChange(LARGE_ICON_PROPERTY, old, largeicon);
			etlm.layout();
			resizeAndRepaint();
		}
	}

	/**
	 * Get the icon associated with a given node type.
	 *
	 * @param nodetype an <code>int</code> value
	 * @return an <code>Image</code> value
	 * @see com.redhat.rhdb.explain.ExplainTreeNode
	 */
	public Image getIcon(int nodetype)
	{
		if (largeicon)
			return (Image)licons.get(new Integer(nodetype));
		else
			return (Image)icons.get(new Integer(nodetype));
	}

	/**
	 * Get the current directory where icon files are stored.
	 *
	 * @return a <code>String</code> value
	 */
	public String getImageDir()
	{
		return imagedir;
	}

	/**
	 * Set the icon directory.
	 *
	 * @param s a <code>String</code> value
	 */
	public void setImageDir(String s)
	{
		String old = imagedir;
		imagedir = s;
		firePropertyChange(IMAGE_DIR_PROPERTY, old, imagedir);
	}

	/**
	 * Get the tool tip text for a specific location.
	 *
	 * @param e a <code>MouseEvent</code> value
	 * @return a <code>String</code> value
	 */
	public String getToolTipText(MouseEvent e)
	{
		String text = null;
		TreeLayoutNode node = getNode(e.getPoint());

		if (node != null)
			text = ((ExplainTreeNode) node.getData()).getToolTipText();
		return text;
	}

	//
	// private methods
	//

	private void init()
	{
		createIconHash();
		etlm = new ExplainTreeLayoutModel(this);
		etlm.setTreeLayoutNodeClass(ExplainTreeLayoutNode.class);
		setTreeLayoutModel(etlm);
		setTreeRenderer(new ExplainTreeRenderer());
	}

	private Image getIcon(String file)
	{
		if (imagedir == null)
			return null;

		String path = imagedir + "/" + file;
		
		ImageIcon i = new ImageIcon(getClass().getResource(path));

		return i.getImage();
	}

	private void createIconHash()
	{
		// normal icons
		icons = new HashMap(25);
		icons.put(new Integer(ExplainTreeNode.NODE_UNKNOWN), getIcon("unknown32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_RESULT), getIcon("result32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_APPEND), getIcon("append32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_JOIN), getIcon("join32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_SCAN), getIcon("scan32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_MATERIALIZE), getIcon("materialize32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_SORT), getIcon("sort32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_GROUP), getIcon("group32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_AGGREGATE), getIcon("aggregate32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_UNIQUE), getIcon("unique32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_SETOP), getIcon("setop32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_LIMIT), getIcon("limit32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_HASH), getIcon("hash32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_INITPLAN), getIcon("initplan32.gif"));
		icons.put(new Integer(ExplainTreeNode.NODE_SUBPLAN), getIcon("subplan32.gif"));

		// large icons
		licons = new HashMap(25);
		licons.put(new Integer(ExplainTreeNode.NODE_UNKNOWN), getIcon("unknown48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_RESULT), getIcon("result48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_APPEND), getIcon("append48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_JOIN), getIcon("join48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_SCAN), getIcon("scan48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_MATERIALIZE), getIcon("materialize48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_SORT), getIcon("sort48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_GROUP), getIcon("group48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_AGGREGATE), getIcon("aggregate48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_UNIQUE), getIcon("unique48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_SETOP), getIcon("setop48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_LIMIT), getIcon("limit48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_HASH), getIcon("hash48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_INITPLAN), getIcon("initplan48.gif"));
		licons.put(new Integer(ExplainTreeNode.NODE_SUBPLAN), getIcon("subplan48.gif"));
	}
}// ExplainTreeDisplay
