/**********************************************************************
 ** Area_Dbase class: This class manages the area database. The area
 **                   database is a database that contains all mudobjects
 **                   from a particular area
 **
 ** Reviewed through: version 0.10
 **    
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef AREA_DBASE_C
#define AREA_DBASE_C


#include "sysdep.h"
#include "area_dbase.h"
#include "location.h"
#include "marker.h"
#include "food.h"
#include "boat.h"
#include "mobile.h"
#include "weapon.h"
#include "wearable.h"
#include "indflags.h"
#include "doorflags.h"
#include "areaflags.h"
#include "text.h"
#include "global.h"
#include "action.h"
#include "chatline.h"
#include "newfuncts.h"
#include "objtype.h"
#include "readobj.h"
#include "adminflags.h"
#include "mask.h"
#include "race.h"
#include "inclination.h"
#include "talent.h"
#include "memchk.h"
#include "utils.h"
#include "linkedlist.h"

/***********************************************************************
 ** Area_Dbase (constructor) - initializes the area database
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Area_Dbase::Area_Dbase(char *areaname)
{
   area_name = areaname;
   valid = 1;
   reload_list = NULL;
   reload_interval = num_obj_loaded = 0;

   area_flags = new Flags(2);

   // Set the database to delete all MudObjects on delection of this database
   the_dbase.set_del_data();
}


/***********************************************************************
 ** Area_Dbase (destructor) - destroys the area database
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Area_Dbase::~Area_Dbase()
{
   delete area_flags;
   the_dbase.clr_tree();
}


/***********************************************************************
 ** get_areaobject - gets an object from the area database
 **
 ** Parameters: the_object - name of the object to get
 **
 ** Returns: pointer to the object requested
 **          NULL if error
 **
 ***********************************************************************/

Entity *Area_Dbase::get_areaobject(char *the_object)
{
   return the_dbase.find(the_object);
}


/***********************************************************************
 ** add_areaobject - adds an object to the area database
 **
 ** Parameters: the_object - pointer to the object to add
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/

int Area_Dbase::add_areaobject(Entity *the_object)
{
   int the_result;

   the_result = the_dbase.add(the_object->get_name(), the_object);
   if (the_result > 0)
      num_obj_loaded++;
 
   return the_result;
}


/***********************************************************************
 ** delete_areaobject - deletes an object from the database
 **
 ** Parameters: the_obj - the object name to delete from the database
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/

int Area_Dbase::delete_areaobject(char *the_obj)
{
   int the_result;

   the_result = the_dbase.del(the_obj);
   if (the_result > 0)
      num_obj_loaded--;
 
   return the_result;
}

/***********************************************************************
 ** remove_areaobject - removes an object from the database, does not delete
 **
 ** Parameters: the_obj - the object name to remove from the database
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/

int Area_Dbase::remove_areaobject(Entity *the_obj)
{
   int the_result;

   the_result = the_dbase.rem(the_obj->get_name());
   if (the_result > 0)
      num_obj_loaded--;
 
   return the_result;
}


/***********************************************************************
 ** delete_areaobject - deletes an object from the area database
 **
 ** Parameters: the_object - pointer to the object to delete
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/

int Area_Dbase::delete_areaobject(Entity *the_object)
{
   int the_result;

   the_result = the_dbase.del(the_object->get_name());
   if (the_result > 0)
      num_obj_loaded--;
 
   return the_result;
}


/***********************************************************************
 ** reset_list - resets the dbase object list to the beginning
 **
 ** Parameters: None
 **
 ** Returns:  1 if success
 **
 ***********************************************************************/

void Area_Dbase::reset_list()
{
   the_dbase.reset_current();
}
   

/***********************************************************************
 ** get_next_obj - gets the next object in the database
 **
 ** Parameters: None
 **
 ** Returns:  a pointer to the object if a success
 **           NULL if at the end of the list
 **
 ***********************************************************************/

Entity *Area_Dbase::get_next_obj()
{
   return the_dbase.get_next();
}


/***********************************************************************
 ** get_read_deny - gets the deny list for read priviledges to this area
 **
 ** Parameters: Nothing
 **
 ** Returns: a pointer to the space delineated list
 **
 ***********************************************************************/

char *Area_Dbase::get_read_deny()
{
   return read_deny.str_show();
}


/***********************************************************************
 ** get_write_allow - gets the allow list for write priviledges to this area
 **
 ** Parameters: Nothing
 **
 ** Returns: a pointer to the space delineated list
 **
 ***********************************************************************/

char *Area_Dbase::get_write_allow()
{
   return write_allow.str_show();
}


/***********************************************************************
 ** set_read_deny - sets the deny list for read priviledges
 **
 ** Parameters: the_list - the string we are setting the list to
 **
 ***********************************************************************/

void Area_Dbase::set_read_deny(char *the_list)
{
   read_deny = the_list;
}


/***********************************************************************
 ** set_write_allow - sets the allow list for write priviledges
 **
 ** Parameters: the_list - the string we are setting the list to
 **
 ***********************************************************************/

void Area_Dbase::set_write_allow(char *the_list)
{
   write_allow = the_list;
}


/***********************************************************************
 ** describe - describes the area to the builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ** Returns:  Nothing
 **
 ***********************************************************************/

void Area_Dbase::describe(Builder *the_builder)
{
   the_builder->send_bldr(_("\n&+GAreaName: \t\t&+M%s&*\n"), 
                                              the_builder->get_areaname());
   the_builder->send_bldr(_("&+GWriteAllow: \t\t&+G%s&*\n"), get_write_allow());
   the_builder->send_bldr(_("&+GReadDeny: \t\t&+R%s&*\n"), get_read_deny());
   the_builder->send_bldr(_("&+GReloadInterval: \t&+W%ld&*\n"), 
                                                     get_reload_interval());
   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** can_read - returns if the builder can read this area file
 **
 ** Parameters: the_builder - the person we are checking
 **
 ** Returns:  1 if they can read it, 0 if not
 **
 ***********************************************************************/

int Area_Dbase::can_read(Builder *the_builder)
{   
   return !((read_deny.find_in_str(the_builder->get_name())) ||
            (read_deny.find_in_str(_("all"))));
}


/***********************************************************************
 ** can_write - returns if the builder can write to this area file
 **
 ** Parameters: the_builder - the person we are checking
 **
 ** Returns:  1 if they can read it, 0 if not
 **
 ***********************************************************************/

int Area_Dbase::can_write(Builder *the_builder)
{
   return ((write_allow.find_in_str(the_builder->get_name())) ||
           (write_allow.find_in_str("all")));
}


/***********************************************************************
 ** get_areaname - returns the areaname string
 **
 **
 ***********************************************************************/

char *Area_Dbase::get_areaname()
{
   return area_name.str_show();
}


/***********************************************************************
 ** set_areaname - changes the area name.  WARNING: only to be used on
 **                the builder port!  This will royally screw up the
 **                binary search tree
 **
 ** Parameters: new_name - what we change the new name to
 **
 ***********************************************************************/

void Area_Dbase::set_areaname(char *new_name)
{
   if ((new_name == NULL) || (strlen(new_name) == 0))
      return;

   area_name = new_name;
}

/***********************************************************************
 ** load_file - loads an area into this area database from a file
 **
 ** Parameters: areafile -  pointer to the file we are loading from
 **             error_log - where to record errors in loading
 **             for_builder - are we loading this for the builder port?
 **             build_format - are we loading a .bldr file?
 **
 ** Returns:   1 if it loaded without errors
 **            0 if an error was found
 **           -1 if a fatal error and could not load
 **
 ***********************************************************************/

int Area_Dbase::load_area(FILE *areafile, ErrLog *error_log, int build_format,
                                                     int for_build_port)
{
   token_record *the_token;
   Strings      holder;
   int valid = 1;
   Entity       *next_obj;
   int          is_actions = 0;
   int          is_chatline = 0;
   int          is_abilities = 0;
   int          is_quests = 0;
   int          is_race = 0;
   int          is_mask = 0;
   int          is_levels = 0;
   int          is_inclination = 0;
   int          is_talent = 0;
   Strings filename;

   /* determine if this is a special type of area so we will need to take
      special steps when loading it */
   if (!STRCASECMP(get_areaname(), _("actions")))
      is_actions = 1;
   else if (!STRCASECMP(get_areaname(), _("abilities")))
      is_abilities = 1;
   else if (!STRCASECMP(get_areaname(), _("chatlines")))
      is_chatline = 1;
   else if (!STRCASECMP(get_areaname(), _("levels")))
      is_levels = 1;
   else if (!STRCASECMP(get_areaname(), _("quests")))
      is_quests = 1;
   else if (!STRCASECMP(get_areaname(), _("races")))
      is_race = 1;
   else if (!STRCASECMP(get_areaname(), _("inclinations")))
      is_inclination = 1;
   else if (!STRCASECMP(get_areaname(), _("talents")))
      is_talent = 1;
   else if (!STRCASECMP(get_areaname(), _("masks")))
      is_mask = 1;

   /* first read in the version number and compare, raise an error if
      the area version doesn't match the current version */
   the_token = get_token(areafile, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_header(get_areaname(), "load_area");
     valid = 0;
   }
   the_token = get_token(areafile, '^');
   if (STRCASECMP(FULLVERSION, the_token->the_string))
   {
     Strings filename;
     Strings directory;
     Strings ext;

     xfclose(areafile, "load_area_file");
     areafile = NULL;

#ifdef WIN32
	 return -2;
#endif

     if ((!is_actions) && (!is_abilities) && (!is_quests) && (!is_levels) &&
         (!is_race) && (!is_mask) && (!is_inclination) && (!is_talent) &&
         (!is_chatline))
     {
        ext = "area";
        directory.sprintf("%s/%s", the_config.basedir.str_show(), AREADIR);
     }
     else
     {
        ext = "dat";
        directory.sprintf("%s/%s", the_config.basedir.str_show(), DATA_DIR);
     }
     filename.sprintf("%s.%s", get_areaname(), ext.str_show());

     if (prompt_user_convert(filename.str_show(), directory.str_show()) != 1)
     {
       holder.sprintf(_("Invalid version '%s' for area '%s'.  "
                     "Should be version '%s'.\n"), the_token->the_string,
                                               get_areaname(), FULLVERSION);
       error_log->log_err(holder.str_show(), "load_area");
       return -1;
     }
     else
     {
       filename.sprintf("%s/%s.%s", directory.str_show(), get_areaname(), 
			                                  ext.str_show());
       if ((areafile = xfopen(filename.str_show(), "r", "load_area_file")) 
	                                                              == NULL)
       {
         holder.sprintf("Error - could not open area %s "
                        "file after convert.\n", get_areaname());
         error_log->log_err(holder.str_show(), "Object_List");
 	     return 0;
       }
       return load_area(areafile, error_log, build_format, for_build_port);
     }
   }

   /* get the readdeny, preventing builders from reading this area */
   the_token = get_token(areafile, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_header(get_areaname(), "load_area");
     valid = 0;
   }
   the_token = get_token(areafile, '^');
   set_read_deny(the_token->the_string);

   /* read in the writeallow list, allowing certain builders to write */
   the_token = get_token(areafile, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_header(get_areaname(), "load_area");
   }
   the_token = get_token(areafile, '^');
   set_write_allow(the_token->the_string);

   /* if it is a regular area file, read in some header information */
   if ((!is_actions) && (!is_abilities) && (!is_quests) && (!is_levels) &&
       (!is_race) && (!is_mask) && (!is_inclination) && (!is_talent) &&
       (!is_chatline))
   {
      /* read in the area flags */
      if (area_flags->read_flags(areafile, error_log) <= 0)
      {
         holder.sprintf(_("Error reading areaflags for area '%s'"), 
                                                             get_areaname()); 
         error_log->log_err(holder.str_show(), "load_area");
         xfclose(areafile, "load_area_file");

         return -1;
      }   

      /* Set reload interval */
      the_token = get_token(areafile, '\0');     
      if (the_token->token_type != T_NUMERICAL)
      {
         holder.
            sprintf(_("Invalid format for reload interval in area %s"), 
                                                         get_areaname());
         error_log->log_err(holder.str_show(), "load_area");
         xfclose(areafile, "load_area_file");
         return -1;
      }
      set_reload_interval(atol(the_token->the_string));

   }

   the_token = get_token(areafile, '\0');

   /* while we have not reached an error or the end of the file, 
      keep getting tokens.  We take action with those tokens based
      on the area file we are reading in */
   while (the_token->token_type > 0)
   {
      if ((!is_actions) && (!is_abilities) && (!is_mask))
      {
         next_obj = read_mudobject(areafile, the_token, get_areaname(), 
                                    error_log, build_format, for_build_port);
      }
      else if (is_abilities)
      {
         next_obj = read_ability(areafile, the_token, error_log, 
                                                             build_format);
      }
      else if (is_mask)
      {
	 next_obj = read_mask(areafile, the_token, error_log, build_format);
      }
      else
      {
         next_obj = (Entity *) new_Action(the_token->the_string);
         if (((Action *) next_obj)->
                        load_action(areafile, error_log, build_format) < 0)
	 {
            delete_Action((Action *) next_obj);
            next_obj = NULL;
         }
      }

      if (next_obj != NULL)
      {
	 add_areaobject(next_obj);
      }

      the_token = get_token(areafile, '\0');
   }
   xfclose(areafile, "load_area_file");
   areafile = NULL;

   return valid;
}

/***********************************************************************
 ** load_area - loads an area into this area database from a file
 **
 ** Parameters: the_dir - the directory we read from
 **             pattern - the pattern of files to search for
 **             error_log - where to write the errors to
 **             for_builder - are we loading this for the builder port?
 **             build_format - are we loading a .bldr file?
 **
 ** Returns:   1 if it loaded without errors
 **            0 if an error was found
 **           -1 if a fatal error and could not load
 **
 ***********************************************************************/

int Area_Dbase::load_area(DirRead *the_dir, char *pattern, ErrLog *error_log, 
                                      int build_format, int for_build_port)
{
   Strings      holder;
   int valid = 1;
   Entity       *next_obj;
   int          text_type = 0;
   int          is_bulletin = 0;
   Strings      filename;

   if (!STRCASECMP(get_areaname(), _("bulletins")))
      is_bulletin = 1;

   else if (!STRCASECMP(get_areaname(), _("info")))
   {
      text_type = TEXT_TYPE_INFO;
   }
   else if (!STRCASECMP(get_areaname(), _("help")))
   {
      text_type = TEXT_TYPE_HELP;
   }
   else if (!STRCASECMP(get_areaname(), _("banner")))
   {
      text_type = TEXT_TYPE_BANNER;
   }

   /* if this is a text type, then we need to load a bunch of files instead
      of just one file */
   if (text_type || is_bulletin)
   {
      char    areaname[129];
      char    *charptr;
      char    *prevptr;

      /* read through the help filenames and load them one by one */
      while ((charptr = the_dir->get_next(pattern)) != NULL)
      {
         /* format the string to a correct filename */
		 strncpy(areaname, charptr, 128);
         charptr = &areaname[0];
         prevptr = charptr;
         while ((*charptr) && ((*charptr != '\n') && (*charptr != '\0') && 
                                                     (*charptr != '.')))
         {
            if (*charptr == '/')
               prevptr = charptr+1;
            charptr++;
         }
         if (*charptr)
            *charptr = '\0';

         if (text_type)
	 {
            next_obj = (Entity *) new_Text(prevptr, text_type);
            if (((Text *) next_obj)->load_text() < 0)
	    {
               mainstruct->delete_entity(next_obj);
               next_obj = NULL;
            }
	 }
         else
	 {
            next_obj = (Entity *) new_Bulletin(prevptr, for_build_port);
            if (((Bulletin *) next_obj)->load_bulletin(error_log) < 0)
			{
               mainstruct->delete_entity(next_obj);
               next_obj = NULL;
            }
		 }

         if (next_obj != NULL)
         {
	        add_areaobject(next_obj);
         }
      }
      return valid;
   }
   return 0;
}

/***********************************************************************
 ** read_mudobject - Reads in a mudobject from the area file 
 **
 ** Parameters: read_file - the file we are reading the mudobject from
 **             the_tok - the tokens that we get to read
 **             areaname - the area we are reading from
 **             error_log - the log we write any errors we encounter to
 **
 ** Returns: a pointer to the mudobject we read in
 **          NULL if an error is found
 **
 ***********************************************************************/

Entity *Area_Dbase::read_mudobject(FILE *read_file, token_record *the_tok, 
                         char *areaname, ErrLog *error_log, int build_format, 
                                                    int for_build_port)
{
   Strings holder;

   switch(the_tok->token_type)
   {
      case T_LOCATION:
         return read_location(read_file, areaname, error_log, build_format);
         break;
      case T_MARKER:
         return read_marker(read_file, areaname, error_log, build_format);
         break;
      case T_DOOR:
         return read_door(read_file, areaname, error_log, build_format);
         break;
      case T_MOVEABLE:
	 return read_moveable(read_file, areaname, error_log, build_format);
         break;
      case T_WEARABLE:
	 return read_wearable(read_file, areaname, error_log, build_format);
         break;
      case T_WEAPON:
	 return read_weapon(read_file, areaname, error_log, build_format);
         break;
      case T_MERGER:
	 return read_merger(read_file, areaname, error_log, build_format);
         break;
      case T_MONEY:
	 return read_money(read_file, areaname, error_log, build_format);
         break;
      case T_FOOD:
	 return read_food(read_file, areaname, error_log, build_format);
         break;
      case T_KEY:
         return read_key(read_file, areaname, error_log, build_format);
         break;
      case T_BOAT:
         return read_boat(read_file, areaname, error_log, build_format);
         break;
      case T_BOOK:
         return read_book(read_file, areaname, error_log, build_format);
         break;
      case T_MOBILE:
         return read_mobile(read_file, areaname, error_log, build_format, 
                                           for_build_port);
         break;
      case T_ROPE:
         return read_rope(read_file, areaname, error_log, build_format);
         break;
      case T_SPECIAL:
         return read_specials(read_file, areaname, error_log, build_format, 
                                           for_build_port);
         break;
      case T_LEVEL:
         return read_level(read_file, error_log, build_format);
         break;
      case T_QUEST:
         return read_quest(read_file, error_log, build_format);
         break;
      case T_RACE:
         return read_race(read_file, error_log, build_format);
         break;
      case T_INCLINAT:
         return read_inclination(read_file, error_log, build_format);
         break;
      case T_TALENT:
         return read_talent(read_file, error_log, build_format);
         break;
      case T_MASK:
         return read_mask(read_file, the_tok, error_log, build_format);
         break;
      case T_CHATLINE:
         return read_chatline(read_file, error_log, build_format);
         break;
      default:
         break;
   }
   return NULL;
}


/***********************************************************************
 ** read_marker - Reads in a marker from the area files
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Marker MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/

Entity *Area_Dbase::read_marker(FILE *read_file, char *areaname,
                      ErrLog *error_log, int build_format)
{
   token_record  *the_token;
   Marker        *temp_mark;  /* holds a marker for loading in */
   Strings       holder;      /* holds temp strings */


   /* get the name of the location */
   the_token = get_token(read_file, '\0');

   temp_mark = new_Marker(the_token->the_string, areaname);

   if (temp_mark->read_mudobject_attrib(read_file, get_areaname(), 
                                              build_format, error_log) <= 0)
   {
      delete_Marker(temp_mark);
      return NULL;
   }
   if (temp_mark->read_item_attrib(read_file, get_areaname(), error_log) <= 0)
   {
      delete_Marker(temp_mark);
      return NULL;
   }
     
   return (Entity *) temp_mark;
}


/***********************************************************************
 ** read_location - Reads in a location from the area files
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Location MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/

Entity *Area_Dbase::read_location(FILE *read_file, char *areaname,
                    ErrLog *error_log, int build_format)
{
   token_record  *the_token;
   Location      *temp_loc;   /* holds a location for loading in */
   Strings       holder;      /* holds temp strings */

   /* get the name of the location */
   the_token = get_token(read_file, '\0');

   temp_loc = new_Location(the_token->the_string, areaname);

   if (temp_loc->read_mudobject_attrib(read_file, get_areaname(), 
                                          build_format, error_log) <= 0)
   {
      delete_Location(temp_loc);
      return NULL;
   }

   if (temp_loc->read_location_attrib(read_file, areaname, error_log) <= 0)
   {
      delete_Location(temp_loc);
      return NULL;
   }
   return (Entity *) temp_loc;
}


/***********************************************************************
 ** read_door - reads in a door object
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Marker MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/

Entity *Area_Dbase::read_door(FILE *read_file, char *areaname,
                   ErrLog *error_log, int build_format)
{
   token_record  *the_token;
   Door          *temp_door;  /* holds a marker for loading in */
   Strings       holder;      /* holds temp strings */

   /* get the name of the location */
   the_token = get_token(read_file, '\0');

   temp_door = new_Door(the_token->the_string, areaname);

   if (temp_door->read_mudobject_attrib(read_file, get_areaname(), build_format,
                                                           error_log) <= 0)
   {
      delete_Door(temp_door);
      return NULL;
   }

   if (temp_door->read_item_attrib(read_file, get_areaname(), 
                                                           error_log) <= 0)
   {
      delete_Door(temp_door);
      return NULL;
   }

   if (temp_door->read_door_attrib(read_file, error_log) <= 0)
   {
      delete_Door(temp_door);
      return NULL;
   }
   return (Entity *) temp_door;
}


/***********************************************************************
 ** read_moveable - reads in a moveable object
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Moveable MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_moveable(FILE *read_file, char *areaname,
                   ErrLog *error_log, int build_format)
{
   Moveable     *the_moveable;
   token_record *the_token;
   Strings      holder;
    

   the_token = get_token(read_file,'\0');

   the_moveable = new_Moveable(the_token->the_string, areaname);
   
   if (the_moveable->read_mudobject_attrib(read_file, get_areaname(), 
                                                build_format, error_log) <= 0)
   {
      delete_Moveable(the_moveable);
      return NULL;
   }

   if (the_moveable->read_item_attrib(read_file, get_areaname(), 
                                                           error_log) <= 0)
   {
      delete_Moveable(the_moveable);
      return NULL;
   }

   if (the_moveable->read_moveable_attrib(read_file, error_log) <= 0)
   {
      delete_Moveable(the_moveable);
      return NULL;
   }

   return (Entity *)the_moveable;
}


/***********************************************************************
 ** read_wearable - reads in a wearable object
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Wearable MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_wearable(FILE *read_file, char *areaname,
                   ErrLog *error_log, int build_format)
{
    Wearable     *the_wearable;
    token_record *the_token;
    Strings      tmp_strholder;
    Strings      holder;

    the_token = get_token(read_file,'\0');

    the_wearable = new Wearable(the_token->the_string, areaname);
   
    if (the_wearable->read_mudobject_attrib(read_file, get_areaname(), 
                                               build_format, error_log) <= 0)
    {
       delete the_wearable;
       return NULL;
    }

    if (the_wearable->read_item_attrib(read_file, get_areaname(), 
                                                           error_log) <= 0)
    {
       delete the_wearable;
       return NULL;
    }

    if (the_wearable->read_moveable_attrib(read_file, error_log) <= 0)
    {
       delete the_wearable;
       return NULL;
    }

    if (the_wearable->read_wearable_attrib(read_file, error_log) <= 0)
    {
       delete the_wearable;
       return NULL;
    }
    return (Entity *)the_wearable;
}


/***********************************************************************
 ** read_rope - reads in a rope object
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Wearable MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_rope(FILE *read_file, char *areaname,
                   ErrLog *error_log, int build_format)
{
    Rope         *the_rope;
    token_record *the_token;
    Strings      tmp_strholder;
    Strings      holder;

    the_token = get_token(read_file,'\0');

    the_rope = new_Rope(the_token->the_string, areaname);
   
    if (the_rope->read_mudobject_attrib(read_file, get_areaname(), 
                                                build_format, error_log) <= 0)
    {
       delete_Rope(the_rope);
       return NULL;
    }

    if (the_rope->read_item_attrib(read_file, get_areaname(), 
                                                           error_log) <= 0)
    {
       delete_Rope(the_rope);
       return NULL;
    }

    if (the_rope->read_moveable_attrib(read_file, error_log) <= 0)
    {
       delete_Rope(the_rope);
       return NULL;
    }

    if (the_rope->read_rope_attrib(read_file, error_log) <= 0)
    {
       delete_Rope(the_rope);
       return NULL;
    }
    return (Entity *)the_rope;
}


/***********************************************************************
 ** read_weapon - reads in a weapon object
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Weapon MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_weapon(FILE *read_file, char *areaname,
                   ErrLog *error_log, int build_format)
{   Weapon       *the_weapon;
    token_record *the_token;
    Strings      holder;

    the_token = get_token(read_file,'\0');

    the_weapon = new_Weapon(the_token->the_string, areaname);
   
    if (the_weapon->read_mudobject_attrib(read_file, get_areaname(), 
                                               build_format, error_log) <= 0)
    {
       delete_Weapon(the_weapon);
       return NULL;
    }

    if (the_weapon->read_item_attrib(read_file, get_areaname(), 
                                                           error_log) <= 0)
    {
       delete_Weapon(the_weapon);
       return NULL;
    }

    if (the_weapon->read_moveable_attrib(read_file, error_log) <= 0)
    {
       delete_Weapon(the_weapon);
       return NULL;
    }

    if (the_weapon->read_weapon_attrib(read_file, error_log) <= 0)
    {
       delete_Weapon(the_weapon);
       return NULL;
    }

    return (Entity *)the_weapon;
}


/***********************************************************************
 ** read_merger - reads in a moveable object
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Moveable MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_merger(FILE *read_file, char *areaname,
                   ErrLog *error_log, int build_format)
{   Merger       *the_merger;
    token_record *the_token;
    Strings      holder;

    the_token = get_token(read_file,'\0');

    the_merger = new_Merger(the_token->the_string, areaname);
   
    if (the_merger->read_mudobject_attrib(read_file, get_areaname(), 
                                                  build_format, error_log) <= 0)
    {
       delete_Merger(the_merger);
       return NULL;
    }

    if (the_merger->read_item_attrib(read_file, get_areaname(), 
                                                           error_log) <= 0)
    {
       delete_Merger(the_merger);
       return NULL;
    }

    if (the_merger->read_moveable_attrib(read_file, error_log) <= 0)
    {
       delete_Merger(the_merger);
       return NULL;
    }

    if (the_merger->read_merger_attrib(read_file, error_log) <= 0)
    {
       delete_Merger(the_merger);
       return NULL;
    }
    return (Entity *)the_merger;
}


/***********************************************************************
 ** read_key - reads in a key object
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Key MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_key(FILE *read_file, char *areaname,
                   ErrLog *error_log, int build_format)
{   Key          *the_key;
    token_record *the_token;
    Strings      holder;

    the_token = get_token(read_file,'\0');

    the_key = new_Key(the_token->the_string, areaname);
   
    if (the_key->read_mudobject_attrib(read_file, get_areaname(), 
                                               build_format, error_log) <= 0)
    {
       delete_Key(the_key);
       return NULL;
    }

    if (the_key->read_item_attrib(read_file, get_areaname(), error_log) <= 0)
    {
       delete_Key(the_key);
       return NULL;
    }

    if (the_key->read_moveable_attrib(read_file, error_log) <= 0)
    {
       delete_Key(the_key);
       return NULL;
    }
    return (Entity *)the_key;
}


/***********************************************************************
 ** read_boat - reads in a boat object
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Boat MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_boat(FILE *read_file, char *areaname,
                   ErrLog *error_log, int build_format)
{   Boat         *the_boat;
    token_record *the_token;
    Strings      holder;

    the_token = get_token(read_file,'\0');

    the_boat = new_Boat(the_token->the_string, areaname);
   
    if (the_boat->read_mudobject_attrib(read_file, get_areaname(), 
                                                build_format, error_log) <= 0)
    {
       delete_Boat(the_boat);
       return NULL;
    }

    if (the_boat->read_item_attrib(read_file, get_areaname(), error_log) <= 0)
    {
       delete_Boat(the_boat);
       return NULL;
    }

    if (the_boat->read_moveable_attrib(read_file, error_log) <= 0)
    {
       delete_Boat(the_boat);
       return NULL;
    }

    if (the_boat->read_boat_attrib(read_file, get_areaname(), error_log) <= 0)
    {
       delete_Boat(the_boat);
       return NULL;
    }
    return (Entity *)the_boat;
}


/***********************************************************************
 ** read_book - reads in a book
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Book MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_book(FILE *read_file, char *areaname,
                   ErrLog *error_log, int build_format)
{   Book         *the_book;
    token_record *the_token;
    Strings      holder;

    the_token = get_token(read_file,'\0');

    the_book = new_Book(the_token->the_string, areaname);
   
    if (the_book->read_mudobject_attrib(read_file, get_areaname(), 
                                                 build_format, error_log) <= 0)
    {
       delete_Book(the_book);
       return NULL;
    }

    if (the_book->read_item_attrib(read_file, get_areaname(), error_log) <= 0)
    {
       delete_Book(the_book);
       return NULL;
    }

    if (the_book->read_moveable_attrib(read_file, error_log) <= 0)
    {
       delete_Book(the_book);
       return NULL;
    }

    if (the_book->read_book_attrib(read_file, error_log) <= 0)
    {
       delete_Book(the_book);
       return NULL;
    }
    return (Entity *)the_book;
}


/***********************************************************************
 ** read_mobile - reads in a mobile from the file passed in
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Mobile MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/

Entity *Area_Dbase::read_mobile(FILE *read_file, char *areaname,
                   ErrLog *error_log, int build_format, int for_build_port)
{
   token_record  *the_token;
   Mobile        *temp_mob;   /* holds a mobile for loading in */
   Strings       holder;      /* holds temp strings */


   /* get the name of the mobile */
   the_token = get_token(read_file, '\0');

   temp_mob = new_Mobile(the_token->the_string, areaname);

   if (temp_mob->read_mudobject_attrib(read_file, get_areaname(), 
                                                  build_format, error_log) <= 0)
   {
      delete_Mobile(temp_mob);
      return NULL;
   }

   if (temp_mob->read_mobile_attrib(read_file, error_log, 
                                                         for_build_port) <= 0)
   {
      delete_Mobile(temp_mob);
      return NULL;
   }
   return (Entity *) temp_mob;
}


/***********************************************************************
 ** read_money - reads in a money object
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Money MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_money(FILE *read_file, char *areaname,
                   ErrLog *error_log, int build_format)
{   Money        *the_money;
    token_record *the_token;
    Strings      holder;

    the_token = get_token(read_file,'\0');

    the_money = new_Money(the_token->the_string, areaname);
   
    if (the_money->read_mudobject_attrib(read_file, get_areaname(), 
                                              build_format, error_log) <= 0)
    {
       delete_Money(the_money);
       return NULL;
    }

    if (the_money->read_item_attrib(read_file, get_areaname(), 
                                                           error_log) <= 0)
    {
       delete_Money(the_money);
       return NULL;
    }

    if (the_money->read_moveable_attrib(read_file, error_log) <= 0)
    {
       delete_Money(the_money);
       return NULL;
    }

    if (the_money->read_merger_attrib(read_file, error_log) <= 0)
    {
       delete_Money(the_money);
       return NULL;
    }

    if (the_money->read_money_attrib(read_file, error_log) <= 0)
    {
       delete_Money(the_money);
       return NULL;
    }
    return (Entity *)the_money;
}


/***********************************************************************
 ** read_food - reads in a food object
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Food MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/

Entity *Area_Dbase::read_food(FILE *read_file, char *areaname,
                   ErrLog *error_log, int build_format)
{   Food         *the_food;
    token_record *the_token;
    Strings      holder;

    the_token = get_token(read_file,'\0');

    the_food = new_Food(the_token->the_string, areaname);
   
    if (the_food->read_mudobject_attrib(read_file, get_areaname(), 
                                                build_format, error_log) <= 0)
    {
       delete_Food(the_food);
       return NULL;
    }

    if (the_food->read_item_attrib(read_file, get_areaname(), 
                                                           error_log) <= 0)
    {
       delete_Food(the_food);
       return NULL;
    }

    if (the_food->read_moveable_attrib(read_file, error_log) <= 0)
    {
       delete_Food(the_food);
       return NULL;
    }

    if (the_food->read_merger_attrib(read_file, error_log) <= 0)
    {
       delete_Food(the_food);
       return NULL;
    }

    if (the_food->read_food_attrib(read_file, error_log) <= 0)
    {
       delete_Food(the_food);
       return NULL;
    }

    return (Entity *)the_food;
}


/***********************************************************************
 ** read_special - reads in a specials object
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns: a pointer to the Special MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/

Entity *Area_Dbase::read_specials(FILE *read_file, char *areaname,
                     ErrLog *error_log, int build_format, int for_build_port)
{   Specials     *the_special;
    token_record *the_token;
    Strings      holder;

    the_token = get_token(read_file,'\0');

    the_special = new Specials(the_token->the_string, areaname);
   
    if (the_special->read_specials_attrib(read_file, error_log, build_format,
                                                       for_build_port) <= 0)
    {
       delete the_special;
       return NULL;
    }

    return (Entity *)the_special;
}


/***********************************************************************
 ** read_level - reads in a level object
 **
 ** Parameters: read_file - the file to read in from
 **             error_log - the error log to write any errors to
 **             build_format - is this read in in builder format?
 **
 ** Returns: a pointer to the Level that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_level(FILE *read_file,
                   ErrLog *error_log, int build_format)
{
   Level *next_obj;
   token_record *the_token;   
 
   the_token = get_token(read_file,'\0');
 
   next_obj = new Level(the_token->the_string);
   if (next_obj->load_level(read_file, error_log, build_format) < 0)
   {
      delete next_obj;
      return NULL;
   }
   return (Entity *) next_obj;
}


/***********************************************************************
 ** read_quest - reads in a quest object
 **
 ** Parameters: read_file - the file to read in from
 **             error_log - the error log to write any errors to
 **             build_format - is this read in in builder format?
 **
 ** Returns: a pointer to the Quest that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_quest(FILE *read_file,
                                         ErrLog *error_log, int build_format)
{
   Quest *next_obj;
   token_record *the_token;   
 
   the_token = get_token(read_file,'\0');

   next_obj = new_Quest(the_token->the_string);
   if (next_obj->load_quest(read_file, error_log, build_format) < 0)
   {
      delete_Quest(next_obj);
      return NULL;
   }
   return (Entity *) next_obj;
}


/***********************************************************************
 ** read_race - reads in a race object
 **
 ** Parameters: read_file - the file to read in from
 **             error_log - the error log to write any errors to
 **             build_format - is this read in in builder format?
 **
 ** Returns: a pointer to the Race that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_race(FILE *read_file,
                                         ErrLog *error_log, int build_format)
{
   Race *next_obj;
   token_record *the_token;   
 
   the_token = get_token(read_file,'\0');

   next_obj = new Race(the_token->the_string);
   if (next_obj->load_race(read_file, error_log, build_format) < 0)
   {
      delete next_obj;
      return NULL;
   }
   return (Entity *) next_obj;
}


/***********************************************************************
 ** read_inclination - reads in an inclination object
 **
 ** Parameters: read_file - the file to read in from
 **             error_log - the error log to write any errors to
 **             build_format - is this read in in builder format?
 **
 ** Returns: a pointer to the Inclination that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_inclination(FILE *read_file,
                                         ErrLog *error_log, int build_format)
{
   Inclination *next_obj;
   token_record *the_token;   
 
   the_token = get_token(read_file,'\0');

   next_obj = new Inclination(the_token->the_string);
   if (next_obj->load_inclination(read_file, error_log, build_format) < 0)
   {
      delete next_obj;
      return NULL;
   }
   return (Entity *) next_obj;
}


/***********************************************************************
 ** read_talent - reads in a talent object
 **
 ** Parameters: read_file - the file to read in from
 **             error_log - the error log to write any errors to
 **             build_format - is this read in in builder format?
 **
 ** Returns: a pointer to the Talent that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_talent(FILE *read_file,
                                         ErrLog *error_log, int build_format)
{
   Talent *next_obj;
   token_record *the_token;   
 
   the_token = get_token(read_file,'\0');

   next_obj = new Talent(the_token->the_string);
   if (next_obj->load_talent(read_file, error_log, build_format) < 0)
   {
      delete next_obj;
      return NULL;
   }
   return (Entity *) next_obj;
}


/***********************************************************************
 ** read_mask - reads in a mask object
 **
 ** Parameters: read_file - the file to read in from
 **             the_token - the token with the mask name
 **             error_log - the error log to write any errors to
 **             build_format - is this read in in builder format?
 **
 ** Returns: a pointer to the Mask that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_mask(FILE *read_file, token_record *the_token,
                                         ErrLog *error_log, int build_format)
{
   Mask *next_obj;

   next_obj = new_Mask(the_token->the_string);
   if (next_obj->load_mask(read_file, error_log, build_format) < 0)
   {
      delete_Mask(next_obj);
      return NULL;
   }
   return (Entity *) next_obj;
}


/***********************************************************************
 ** read_chatline - reads in a chatline object
 **
 ** Parameters: read_file - the file to read in from
 **             error_log - the error log to write any errors to
 **             build_format - is this read in in builder format?
 **
 ** Returns: a pointer to the ChatLine that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *Area_Dbase::read_chatline(FILE *read_file,
                                         ErrLog *error_log, int build_format)
{
   ChatLine *next_obj;
   token_record *the_token;   
 
   the_token = get_token(read_file,'\0');

   next_obj = new_ChatLine(the_token->the_string);
   if (next_obj->load_chatline(read_file, error_log, build_format) < 0)
   {
      delete_ChatLine(next_obj);
      return NULL;
   }
   return (Entity *) next_obj;
}



/***********************************************************************
 ** add_dual_reboot_info - adds a reboot info to the list for a rope or door,
 **                        which means it adds two locations to the object
 **                        since doors and ropes can be in two locs
 **
 ** Parameters: new_obj - the object to set dual reboot on
 **             inside_loc - the inside location of this object
 **             outside_loc - the outside location of this object
 **
 ** Returns: -1 for error, 1 for good
 **         
 **
 ***********************************************************************/

int Area_Dbase::add_dual_reboot_info(MudObject *new_obj, 
                                                    MudObject *inside_loc,
                                                    MudObject *outside_loc)
{
   reload_obj_info *tmp_info;
   MudObject       *container;
   MudObject       *cur_loc;
   int             i;

   if (new_obj == NULL)
   {
      fault(_("add_dual_reboot_info lacked new_obj, possibly fatal.\n"));
      return -1;
   }

   /* loop through inside and out locs, saving their info */
   cur_loc = inside_loc;          // first do inside loc
   for (i=0; i<2; i++)
   {

      /* as long as we have a location, save it in a reload_obj_info 
         struct */
      if (cur_loc != NULL)
      {
         tmp_info = new_reload_obj_info();
         tmp_info->the_obj = new_obj;
         tmp_info->spec_holder = NULL;
         if ((container = new_obj->get_contained_by()) != NULL)
         {   
            tmp_info->last_loc_name = cur_loc->get_name();
            tmp_info->last_loc_area = cur_loc->get_area();
         }
         else
         {
            printf(_("Could not find container for dual_reboot_info, obj '%s@%s'!\n"),
		   new_obj->get_name(), new_obj->get_area());
	    fault("blah");
         }

         if (reload_list == NULL)
         {
            reload_list = tmp_info;
            tmp_info->next_info = NULL;
            return 1;
         }
         tmp_info->next_info = reload_list;
         reload_list = tmp_info;
      }
      cur_loc = outside_loc;         // next do the outside loc 
   }
   return 1;

}


/***********************************************************************
 ** add_reboot_info - adds a reboot info to the list
 **
 ** Parameters: new_obj - the object to add to reboot info
 **
 ** Returns: -1 for errors, 1 for good
 **
 ***********************************************************************/

int Area_Dbase::add_reboot_info(MudObject *new_obj)
{
   reload_obj_info *tmp_info;
   MudObject       *tmp_loc;

   if ((new_obj == NULL) || (new_obj->get_type() == OBJ_TYPE_LOCATION))
   {
      fault(_("No object for reboot_info submitted! Possibly Fatal.\n"));
      return -1;
   }

   /* get the object's location */
   if ((tmp_loc = new_obj->get_contained_by()) == NULL)
   {
     Strings holder;
     holder.sprintf(_("could not get loc for object: %s!\n"), new_obj->get_title());
     fault(holder.str_show());
      return -1;
   }

   /* if it is a player in the boat, the loc actually is the boat's loc */
   if ((tmp_loc->get_type() == OBJ_TYPE_BOAT) && 
       (new_obj->get_type() == OBJ_TYPE_PLAYER))
   {
      tmp_loc = ((Player *) new_obj)->get_loc(); 
   }

   /* now create the new reboot info, collecting location info */
   tmp_info = new_reload_obj_info();
   tmp_info->the_obj = new_obj;
   tmp_info->spec_holder = NULL;
   if (tmp_loc != NULL)
   {
      tmp_info->last_loc_name = tmp_loc->get_name();
      tmp_info->last_loc_area = tmp_loc->get_area();
   }
   else
   {
      fault(_("no location found for object for saving reboot_info!\n"));
   }

   if (reload_list == NULL)
   {
      reload_list = tmp_info;
      tmp_info->next_info = NULL;
      return 1;
   }
   tmp_info->next_info = reload_list;
   reload_list = tmp_info;
   return 1;
}


/***********************************************************************
 ** add_spec_reboot_info - adds the specials to the reboot info list
 **                        that are from the area being reloaded
 **
 ** Parameters: new_obj - reads in the new object
 **             the_area - the area that is getting reloaded
 **
 ***********************************************************************/

void Area_Dbase::add_spec_reboot_info(MudObject *new_obj, char *the_area)
{
   reload_obj_info *tmp_info;
   special_holder  *tmp_spec;
   special_holder  *prev_spec;
   Specials        *the_special;

   
   /* loop through all specials in the object */
   prev_spec = NULL;
   tmp_spec = new_obj->get_special_list();
   while (tmp_spec != NULL)
   {
      the_special = tmp_spec->the_special;

      /* if this special is in the area being reloaded, save its info */
      if (!STRCASECMP(the_area, the_special->get_area()))
      {
         if (prev_spec == NULL)
	 {
            new_obj->set_special_list(tmp_spec->next_special);
         }
         else
	 {
            prev_spec->next_special = tmp_spec->next_special;
         }
         tmp_info = new_reload_obj_info();
         tmp_info->the_obj = NULL;
         tmp_info->spec_holder = tmp_spec;
         tmp_spec = tmp_spec->next_special;
         tmp_info->spec_holder->next_special = NULL;
         tmp_info->last_loc_name = new_obj->get_name();
         tmp_info->last_loc_area = new_obj->get_area();
         tmp_info->spec_name = the_special->get_name();
         if (reload_list == NULL)
         {
            reload_list = tmp_info;
            tmp_info->next_info = NULL;
         }
         else
	 {
            tmp_info->next_info = reload_list;
            reload_list = tmp_info;
         }
      }
      else
      {
         prev_spec = tmp_spec->next_special;
         tmp_spec = tmp_spec->next_special;
      } 
   }
   return;
}


/***********************************************************************
 ** add_inv_reboot_info - adds the objects to the reboot info list
 **                       that are from the area being reloaded and
 **                       are in the given object's inventory
 **
 ** Parameters: new_obj - the inventory reboot object to add
 **             the_area - the area that is getting reloaded
 **             the_dbase - the object database
 **
 ** Returns: 
 **
 ***********************************************************************/

int Area_Dbase::add_inv_reboot_info(MudObject *new_obj, char *the_area, 
                                                  Object_List *the_dbase)
{
   Inventory *the_inv;
   MudObject  *tmp_obj;   

   /* loop through all objects in the inventory */
   the_inv = new_obj->get_inventory();
   the_inv->reset_current();

   /* we only care about those objects in the area being reloaded that
      is in the inventory, so loop and find them */
   tmp_obj = the_inv->get_next();
   while (tmp_obj != NULL)
   {

      /* if this object is in the area being reloaded, save its info */
      if (!STRCASECMP(tmp_obj->get_area(), the_area))
      {
         add_reboot_info(tmp_obj);
         add_spec_reboot_info(tmp_obj, the_area);
         add_inv_reboot_info(tmp_obj, the_area, the_dbase);
         new_obj->remove_contained(tmp_obj);
         the_dbase->remove_obj(tmp_obj);
         the_inv->reset_current();
      }
      tmp_obj = the_inv->get_next();
   }
   return 1;
}


/***********************************************************************
 ** reboot_door - code to reboot the door so that it does it smoothly
 **
 ** Parameters: tmp_obj - the object that is marked a door
 **             error_log - the error log to write to
 **             in_zone - are we searching the reboot zone now?
 **
 **
 ** Basic Algorithm:
 **
 **   *If this door has a rope tied to it
 **      - get the rope object
 **      * If the rope is tied and neither the rope nor door are members
 **        of the area being reloaded
 **         * If either the inside our outside loc is sitting in the area
 **           being reloaded
 **            - Remove both sides of the door from their locations and
 **              remember where they were
 **      * If the rope tied to the door is in the area being rebooted
 **         * If the door is in the area being reloaded
 **            - Save the door reboot info
 **         * else return 0
 **      *If the door is marked as tied but rope is not in location
 **         -Raise an error
 **      *If the door is not in the zone being reloaded
 **         -Save the door info
 **      -Remove the door and rope from the location
 **      -Set the door as no rope tied to it
 **      
 **   *else
 **      *If the door is not in the area being reloaded
 **         -store the door info and remove it from the loc for adding
 **          later on
 **      *else
 **         -just remove the door from its location
 **         *If the rope is tied to the door and neither are members of
 **          the reloading area
 **            *If the inside or outside loc of either rope or door is
 **             sitting in the area being reloaded
 **               -Remove both sides from their location and store info
 **         *If rope is in area being reloaded
 **            *If the door is in the area being reloaded
 **               -Remove door and rope from locations 
 **         *else 
 **            -Return 0
 **      
 **           
 ** Returns: 1 for success
 **
 ***********************************************************************/

int Area_Dbase::reboot_door(MudObject *tmp_obj, ErrLog *error_log, 
                                                       int in_zone)
{
   MudObject *tmp_rope;
   Door      *the_door;
   Flags     *tmp_itemflags;
   Flags     *tmp_doorflags;
   Flags     *rope_itemflags;
   Rope      *the_rope;
   MudObject *remove_loc;
   Strings     holder;
   Object_List *obj_dbase;
   MudObject *inside_loc;
   MudObject *outside_loc;

   obj_dbase = mainstruct->get_dbase();
   the_door = (Door *) tmp_obj;
   tmp_itemflags = the_door->get_itemflags();
   tmp_doorflags = the_door->get_doorflags();

   inside_loc = obj_dbase->get_mudobject(the_door->get_area(), 
                                            the_door->get_location());

   outside_loc = obj_dbase->get_mudobject(the_door->get_area(), 
                                             the_door->get_outside_loc());

   /* if the door being rebooted is a rope door and has a rope tied to it */
   if ((tmp_doorflags->get_flag(DOORFLAG_ROPETIE)) &&
       (the_door->get_door_state() != DOOR_CLOSED))
   {

      /* get the rope name that is tied to the door */
      if (the_door->get_rope_name() == NULL)
      {
         holder.sprintf(_("Rope Door %s marked as tied but no rope given."), 
                                                    the_door->get_name());
         error_log->log_err(holder.str_show(), "reboot_door");
         return -1;
      }

      /* find the rope based on the name */
      if ((tmp_rope = obj_dbase->get_mudobject
          (the_door->get_area(), the_door->get_rope_name())) == NULL)
      {
         holder.sprintf(_("Door %s rope link '%s' doesn't seem to exist."), 
                          the_door->get_name(), the_door->get_rope_name()); 
         error_log->log_err(holder.str_show(), "reboot_door");
         return -1;
      }

      /* ensure what we found is a rope */
      if (tmp_rope->get_type() != OBJ_TYPE_ROPE)
      {
         holder.sprintf(_("Door %s rope link '%s' not a rope."), 
                         the_door->get_name(), tmp_rope->get_name()); 
         error_log->log_err(holder.str_show(), "reboot_door");
         return -1;
      }

      the_rope = (Rope *) tmp_rope;

      rope_itemflags = the_rope->get_itemflags();

      /* if the rope is tied and it is not a member of the reloading area,
         and the door is not a member, then do a check */
      if ((rope_itemflags->get_flag(ITEMFLAG_ISTIED)) &&
          (STRCASECMP(the_rope->get_area(), get_areaname())) &&
          (STRCASECMP(the_door->get_area(), get_areaname())))
      {

	 /* if either the inside loc or the outside loc is sitting in an
            area that is going to be reloaded, yank out both sides */
         if (((inside_loc != NULL) && 
              (!STRCASECMP(inside_loc->get_area(), get_areaname()))) ||
             ((outside_loc != NULL) && 
              (!STRCASECMP(outside_loc->get_area(), get_areaname()))))
	 {
            add_dual_reboot_info(the_rope, inside_loc, outside_loc);
            add_spec_reboot_info(the_rope, get_areaname());

            if (inside_loc != NULL)
               inside_loc->remove_contained(the_rope);

            if (outside_loc != NULL)
               outside_loc->remove_contained(the_rope);

            /* do the same to the door */
            add_reboot_info(the_door);   // doors don't need add_dual_reboot as
	                                 // they already store the needed info 
            add_spec_reboot_info(the_door, get_areaname());

            if (inside_loc != NULL)
               inside_loc->remove_contained(the_door);

            if (outside_loc != NULL)
               outside_loc->remove_contained(the_door);
         }
      }    
      /* if the rope area is the same as the area we are reloading (same
         as the door area), and we are in the zone that is reloading, then
         handle it.  If the areas are not the same, and the door is not
         from the area we are reloading, we return 0 */
      if (!STRCASECMP(the_rope->get_area(), get_areaname()))
      {

	 /* if the door is in the area that is reloading */
         if (in_zone)
         {
            if (inside_loc != NULL)
	    {
               inside_loc->remove_contained(the_rope);
               inside_loc->remove_contained(the_door);
            }
            if (outside_loc != NULL)
	    {
               outside_loc->remove_contained(the_rope);
               outside_loc->remove_contained(the_door);
            }
            return 0;
         }
      }
      else
         if (!in_zone)
            return 0;

      /* if the rope is not in the door's location but the door is marked
         as tied by it, tis a bad thing */
      if ((the_door->get_door_state() != DOOR_LOCKED) &&
          ((STRCASECMP(the_rope->get_location(), 
                                             the_door->get_location())) &&
           (STRCASECMP(the_rope->get_location(), 
                                             the_door->get_outside_loc()))))
      {
         holder.sprintf(_("Door %s rope link '%s' not in door location."), 
                                the_door->get_name(), the_rope->get_name()); 
         error_log->log_err(holder.str_show(), "reboot_door");
         return -1;
      }

      /* save the door info if not in the area being reloaded so it can 
         survive a reload if necessary */ 
      if (!in_zone)
      {
         the_door->set_door_state(DOOR_CLOSED);
         add_reboot_info(the_door);
         add_spec_reboot_info(the_door, get_areaname());
      }

      /* the rope disappears if we reload the door */
      remove_loc = obj_dbase->get_mudobject(the_door->get_area(), 
                                            the_door->get_outside_loc());
      if (remove_loc != NULL)
      {
         remove_loc->remove_contained(the_rope);
         if (!in_zone)
            remove_loc->remove_contained(the_door);
      }
      remove_loc = obj_dbase->get_mudobject(the_door->get_area(), 
                                                the_door->get_location());
      if (remove_loc != NULL)
      {
         remove_loc->remove_contained(the_rope);
         if (!in_zone)
            remove_loc->remove_contained(the_door);
      }
      the_rope->remove_from_loc();
      the_rope->set_untied();
   }
   else 
   {
      if (!in_zone)
      {
         /* if either the inside loc or the outside loc is sitting in an
            area that is going to be reloaded, yank out both sides */
         if (((inside_loc != NULL) && 
              (!STRCASECMP(inside_loc->get_area(), get_areaname()))) ||
             ((outside_loc != NULL) && 
              (!STRCASECMP(outside_loc->get_area(), get_areaname()))))
         {
            /* ensure the door will return at a later date */
            add_reboot_info(the_door);
            add_spec_reboot_info(the_door, get_areaname());

            if (inside_loc != NULL)
              inside_loc->remove_contained(the_door);

            if (outside_loc != NULL)
               outside_loc->remove_contained(the_door);
         }
      }
      else
      {
         if (inside_loc != NULL)
            inside_loc->remove_contained(the_door);

         if (outside_loc != NULL)
            outside_loc->remove_contained(the_door);
      }


   }
   return 1;
}

/***********************************************************************
 ** reload_area - reloads the area, taking into account anything that may
 **               be affected by the reload, like destroying objects that
 **               are sitting around and the like
 **
 ** Parameters: the_file - the file we read the data from
 **
 ** Returns: 1 for success
 **
 **
 ** Basic Algorithm:
 **
 **   *While we haven't gone through all areas, loop through them
 **      *While we haven't looped through all objects in the area
 **         *If the object is a MudObject class or subclass of
 **            *If the current area is not the area being reloaded
 **               --Store specials reboot info for this object
 **            *If current area is area being reloaded and object is
 **             not a location type
 **               *If object is carried by a player or a boat players
 **                are standing in
 **                  *If this is not already cloned
 **                     -Clone the object and remove old object from loc
 **                     *If it is a wearable object
 **                        *If the object is worn by the player
 **                           -Remove old and wear newly cloned item
 **                        *Else
 **                           - Just add the clone to the player's inv
 **                     -Swap contained inventories of object with clone
 ** Finish this later
 **
 ***********************************************************************/

int Area_Dbase::reload_area(FILE *the_file)
{
   Entity          *tmp_entity = NULL;
   MudObject       *tmp_obj;   
   MudObject       *container;
   MudObject       *new_clone;
   Object_List     *obj_dbase;
   Area_Dbase      *tmp_area;
   Player          *tmp_player;
   reload_obj_info *tmp_reload;
   reload_obj_info *del_reload;
   Strings         holder;

#ifdef DEBUG_LOAD
   printf("reloading: %s\n", get_areaname());
#endif

   obj_dbase = mainstruct->get_dbase();

   /* loop through all areas, looking for reload issues to handle */
   obj_dbase->reset_list();
   tmp_area = obj_dbase->get_next_area();
   while (tmp_area != NULL)
   {
      /* loop through all objects in this area, checking them */
      tmp_area->reset_list();

      tmp_entity = tmp_area->get_next_obj();
      while (tmp_entity != NULL)
      {
	 // First, if this is a rope and it is tied, we handle it when we get
	 // to the door it is tied to
	 if ((tmp_entity->get_type() == OBJ_TYPE_ROPE) &&
	     (((Rope *) tmp_entity)->get_itemflags())->get_flag(ITEMFLAG_ISTIED))
	 {
	    tmp_entity = tmp_area->get_next_obj();
	    continue;
	 }

         if (tmp_entity->is_a_mudobject())
	 {

            tmp_obj = (MudObject *) tmp_entity;

            /* first handle all specials that point to a special in the 
               area being reloaded */
            if (tmp_area != this)
 	    {
               add_spec_reboot_info(tmp_obj, get_areaname());
            }

	    /* if we are searching the area that is being reloaded, and the
               object is not a location */
            if ((tmp_area == this) && 
                (tmp_obj->get_type() != OBJ_TYPE_LOCATION))
            {
               container = tmp_obj->get_contained_by();

               /* if a player has this object, let's make sure it doesn't go
                  away on them, or if they are in this object (boat) make
                  sure it doesn't dissapear and send them into the drink */
               if ((container != NULL) && 
                   tmp_obj->carried_by_player() ||
                   ((tmp_obj->get_type() == OBJ_TYPE_BOAT) &&
                    (((Boat *)tmp_obj)->contains_players())))
	       {
                  Inventory *old_inventory;
                  MudObject  *tmp_objholder;

	          /* if this is not a cloned object, clone it */
                  if (tmp_obj->get_parent() == NULL)
	          {
                     new_clone = obj_dbase->clone_object(tmp_obj);
                     container->remove_contained(tmp_obj);
		     container->add_contained(new_clone);
                     if (tmp_obj->get_type() == OBJ_TYPE_WEARABLE)
		     {
                        Wearable *the_wearable;
                        Individual *wearing_ind;

                        the_wearable = (Wearable *) tmp_obj;
                        wearing_ind = (Individual *) container;
                        if (wearing_ind->is_worn(the_wearable))
			{
                           wearing_ind->remove_item(the_wearable);
                           wearing_ind->add_contained(new_clone);
                           wearing_ind->wear_item((Wearable *) new_clone);
                        }
                        else
                           container->add_contained(new_clone);
		     }
                  
                     /* swap the inventories */
                     update_reboot_info(tmp_obj->get_name(),
                                                     new_clone->get_name());
                     old_inventory = tmp_obj->get_inventory();
                     old_inventory->reset_current();
                     tmp_objholder = old_inventory->get_next();
                     while (tmp_objholder != NULL)
		     {
                        if (tmp_objholder->get_type() == OBJ_TYPE_PLAYER)
			{
                           holder.sprintf("%s@%s", new_clone->get_name(), 
                                                   new_clone->get_area());
                           ((Player *) tmp_objholder)->set_location(new_clone);
			}
                        tmp_objholder = old_inventory->get_next(); 
                     }
                     tmp_obj = new_clone;
                  }
                  obj_dbase->remove_obj(tmp_obj);

                  add_reboot_info(tmp_obj);

                  /* add the inventory of this object that is of this area */
                  add_inv_reboot_info(tmp_obj, get_areaname(), obj_dbase);

                  /* add the special info for this object, since we are
                     saving it */
                  add_spec_reboot_info(tmp_obj, get_areaname());

                  container->remove_contained(tmp_obj);
               }

               /* if this is a mobile and it is a clone, then check if it
                  is fighting */
               else if ((container != NULL) && 
                   (tmp_obj->get_type() == OBJ_TYPE_MOBILE))
	       {
                  Mobile *the_mobile;
                  Flags  *indflags;

                  the_mobile = (Mobile *) tmp_obj;
                  indflags = the_mobile->get_indflags();         

                  /* if they are fighting, we can't just make them go away */
                  if (indflags->get_flag(INDFLAG_FIGHTING))
		  {
		     /* if they are cloned, we store them, then re-add them,
                        then let them continue fighting */
                     if (tmp_obj->get_parent() != NULL)
		     {

                        obj_dbase->remove_obj(tmp_obj);
                        add_reboot_info(tmp_obj);
                        add_spec_reboot_info(tmp_obj, get_areaname());
                     }

                     /* if they are not cloned, we stop them from fighting,
                        and then do the reboot */
                     else
		     {
                        mainstruct->remove_fighter(the_mobile);
                        mainstruct->clear_fighter(the_mobile);
                        the_mobile->stop_fighting();
                     }
                  }
		  container->remove_contained(tmp_obj);
               }

               /* else if this is a rope door, check to see if the rope is of
                  this zone as well and if not, untie the rope */
               else if ((container != NULL) && 
                        (tmp_obj->get_type() == OBJ_TYPE_DOOR))
	       {
                  reboot_door(tmp_obj, mainstruct->get_log(), 1);
               }

               else if (container != NULL)
	       {
                  container->remove_contained(tmp_obj);
               }
            }

            /* if this is not the area being reloaded */
            else if ((tmp_area != this) && 
                     (tmp_obj->get_type() != OBJ_TYPE_LOCATION))
	    {

              if (tmp_obj->is_a_moveable())
	      {
	        /* if it is an object that has been moved, we keep it, else
                   delete it */

                if (((Moveable *)tmp_obj)->is_moved())
		{
	          /* if it is in an object that is in the reloading area, 
                     remove and save the location so we can put it back 
                     after reload */

                  if ((container = tmp_obj->get_contained_by()) != NULL)
	          {

                    if (!STRCASECMP(container->get_area(),get_areaname()))
	            {
                      add_reboot_info(tmp_obj);
                      container->remove_contained(tmp_obj);
                    }
                  }
                  add_spec_reboot_info(tmp_obj, get_areaname());
                }
                else
		{
                  if (((container = tmp_obj->get_contained_by()) != NULL) &&
	              (!STRCASECMP(container->get_area(),get_areaname())) &&
                      (!tmp_obj->carried_by_player()))
	          {
                    container->remove_contained(tmp_obj);

                    /* if this is a clone, delete it */
                    if (tmp_obj->get_parent() != NULL)
	  	    {
                      mainstruct->delete_object(tmp_obj);
                    }
		  }
                }
	      }

              /* else this is not a moveable object so we dont have to
                  do as many special checks on it */
              else
	      {
                if ((container = tmp_obj->get_contained_by()) != NULL)
	        {
                  if (tmp_obj->get_type() == OBJ_TYPE_DOOR)
	          {
                     reboot_door(tmp_obj, mainstruct->get_log(), 0);
	          }

                  else if (!STRCASECMP(container->get_area(),get_areaname()))
	          {
                     add_reboot_info(tmp_obj);
                     add_spec_reboot_info(tmp_obj, get_areaname());
                     container->remove_contained(tmp_obj);
                  }
                }
                add_spec_reboot_info(tmp_obj, get_areaname());
              }
            }
         }
         tmp_entity = tmp_area->get_next_obj();
      }
      tmp_area = obj_dbase->get_next_area();
   }

   /* check the players next */
   tmp_player = mainstruct->get_first_player();
   while (tmp_player != NULL)
   {
      Location *player_loc;
      MudObject *new_loc;

      player_loc = tmp_player->get_loc();

      /* if they are in the game, and in this area */
      if ((player_loc != NULL) && 
          (!STRCASECMP(player_loc->get_area(), get_areaname())))
      {
         add_reboot_info((MudObject *) tmp_player);
         holder.sprintf("%s@%s", the_config.backup_locname.str_show(), 
                                     the_config.hardcoded_areaname.str_show());
         new_loc = mainstruct->get_object(holder.str_show());
         if ((new_loc == NULL) || (new_loc->get_type() != OBJ_TYPE_LOCATION))
	 {
            mainstruct->
               log_error(_("Hardcoded area is not valid!"), "reload_area");
            return -1;
         }
 
	 tmp_player->set_location(new_loc);
      }
      tmp_player = tmp_player->get_next_player();
   }

   /* now we delete the objects in this area */
   the_dbase.clr_tree();

   if (load_area(the_file, mainstruct->get_log(), 0, 0) <= 0)
      return -1;

   /* coordinate here */

   coord_area1(mainstruct->get_log(), mainstruct->get_dbase());
   coord_area2(mainstruct->get_log(), mainstruct->get_dbase());

   /* re-add stuff to the database that we temporarily pulled out */
   tmp_reload = reload_list;
   while (tmp_reload != NULL)
   {
      if(tmp_reload->the_obj != NULL)
      {
         if (!STRCASECMP((tmp_reload->the_obj)->get_area(), get_areaname()))
          
         {
            add_areaobject(tmp_reload->the_obj);
         }

      }
      tmp_reload = tmp_reload->next_info;
   }

   /* reset reload issues here */
   tmp_reload = reload_list;
   while (tmp_reload != NULL)
   {
      holder.
       sprintf("%s@%s", tmp_reload->last_loc_name.str_show(), 
                        tmp_reload->last_loc_area.str_show());
      if (!STRCASECMP(tmp_reload->last_loc_area.str_show(), "player"))
      {
         if (((container = (MudObject *) mainstruct->
                get_player(tmp_reload->last_loc_name.str_show())) == NULL) &&
              (tmp_reload->spec_holder == NULL))
	 {
            holder.
            sprintf(_("Object '%s' location '%s' does not exist after reload."), 
                  (tmp_reload->the_obj)->get_name(), holder.str_show());
            mainstruct->log_error(holder.str_show(), "reload_area");
            mainstruct->delete_object((MudObject*) tmp_reload->the_obj);
            tmp_reload->the_obj = NULL;
	 }
         else if (tmp_reload->spec_holder != NULL)
	 {
            Specials       *tmp_special;
            special_holder *tmp_holder;

            holder.sprintf("%s@%s", tmp_reload->spec_name.str_show(), 
                                                        get_areaname());
            if ((tmp_special = mainstruct->get_special(holder.str_show()))
                                                         == NULL)
	    {
               holder.
                  sprintf(_("Special '%s' for object "
                    "'%s' does not exist after reload."), 
                    holder.str_show(), tmp_reload->last_loc_name.str_show());
               mainstruct->log_error(holder.str_show(), "reload_area");
            }
            else
	    {
               tmp_holder = tmp_reload->spec_holder;
               tmp_holder->the_special = tmp_special;
               container->add_special_holder(tmp_holder);
	    }
         }
         else if (tmp_reload->spec_holder == NULL)
            container->add_contained(tmp_reload->the_obj);
      }
      else if (((container = 
                          mainstruct->get_object(holder.str_show())) == NULL) &&
               (tmp_reload->spec_holder == NULL))
      {
         holder.
           sprintf(_("Object '%s' location '%s' does not exist after reload."), 
                  (tmp_reload->the_obj)->get_name(), holder.str_show());
         mainstruct->log_error(holder.str_show(), "reload_area");
         mainstruct->delete_object((MudObject *) tmp_reload->the_obj);
         tmp_reload->the_obj = NULL;
      }
      else if (tmp_reload->spec_holder != NULL)
      {
         Specials       *tmp_special;
         special_holder *tmp_holder;

         holder.sprintf("%s@%s", tmp_reload->spec_name.str_show(), 
                                                        get_areaname());
         if ((tmp_special = mainstruct->get_special(holder.str_show()))
                                                         == NULL)
	 {
            holder.sprintf(_("Special '%s' for object "
                    "'%s' does not exist after reload."), 
                    holder.str_show(), tmp_reload->last_loc_name.str_show());
            mainstruct->log_error(holder.str_show(), "reload_area");
         }
         else
	 {
            tmp_holder = tmp_reload->spec_holder;
            tmp_holder->the_special = tmp_special;
            if (container != NULL)
               container->add_special_holder(tmp_holder);
            else
	    {
               delete_special_holder(tmp_holder);
               tmp_reload->spec_holder = NULL;
	    }
	 }
      }
      else if (container != NULL)
      {
         if (((tmp_reload->the_obj)->get_type() == OBJ_TYPE_PLAYER) ||
             ((tmp_reload->the_obj)->get_type() == OBJ_TYPE_MOBILE))
	 {
            if (container->get_type() != OBJ_TYPE_LOCATION) 
	    {
               holder.sprintf(_("Individual '%s' reload location is not set to "
                    "a location"), (tmp_reload->the_obj)->get_name());
               mainstruct->log_error(holder.str_show(), "reload_area");
               if ((tmp_reload->the_obj)->get_type() == OBJ_TYPE_PLAYER)
                  mainstruct->
                       remove_player((tmp_reload->the_obj)->get_name());
               else
                  mainstruct->delete_object((MudObject *)tmp_reload->the_obj);

               tmp_reload->the_obj = NULL;
            }
            else
	    {
               Individual   *this_ind;
               Flags        *tmp_indflags;
 
               this_ind = (Individual *) (tmp_reload->the_obj);
               tmp_indflags = this_ind->get_indflags();
            
	       this_ind->set_location(container);
            }
         }
         else if ((tmp_reload->the_obj)->get_type() == OBJ_TYPE_DOOR)
	 {
            MudObject *inside_loc;
            MudObject *outside_loc;
            MudObject *cur_loc;
            MudObject *tmp_obj;
            Inventory *the_inv;
            Door      *the_door;       
            int i;

            the_door = (Door *) (tmp_reload->the_obj);

            inside_loc = obj_dbase->get_mudobject(the_door->get_area(), 
                                            the_door->get_location());

            outside_loc = obj_dbase->get_mudobject(the_door->get_area(), 
                                             the_door->get_outside_loc());

            cur_loc = inside_loc;
            for (i=0; i<2; i++)
	    {

	       /* for each location, check if the door is not already here
                  and if not, add it */
               if (cur_loc != NULL)
	       {
                  the_inv = cur_loc->get_inventory();
                  the_inv->reset_current();
                  tmp_obj = the_inv->get_next();
                  while (tmp_obj != NULL)
		  {
                     if (tmp_obj == the_door)
                        break;
                     tmp_obj = the_inv->get_next();
		  }
                  if (tmp_obj == NULL)
                     cur_loc->add_contained(the_door);
	       }
               cur_loc = outside_loc;
	    }

	 }
         else
            container->add_contained(tmp_reload->the_obj);

      }
      del_reload = tmp_reload;
      tmp_reload = tmp_reload->next_info;
      delete_reload_obj_info(del_reload);
   }
   reload_list = NULL;

   reload_time.reset_time();
   return 1;
}


/***********************************************************************
 ** coord_area1 - takes the list of objects in this area and checks for 
 **               invalid exit links and also adds objects to other object's
 **               inventories.  First sweep.
 **
 ** Parameters: error_log - the error log to report errors to
 **             obj_dbase - the game's database of objects
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Area_Dbase::coord_area1(ErrLog *error_log, Object_List *obj_dbase)
{
   Entity      *cur_entity;
   MudObject   *cur_obj;
   MudObject   *tmp_obj;
   Strings     holder;
   char        *the_name;
   char        *the_area;
   Flags       *tmp_itemflags;

   reset_list();
   cur_entity = get_next_obj();
   while (cur_entity != NULL)
   {
#ifdef DEBUG_LOAD
      printf("coordinating: %s\n", cur_obj->get_name());
#endif

      if (cur_entity->is_a_mudobject())
      {
         cur_obj = (MudObject *) cur_entity;

         if (cur_obj->get_type() != OBJ_TYPE_LOCATION)
         {
            int i;

            holder = cur_obj->get_location();
            /* loop twice so we can handle doors sits in two locations */
            for (i=0; i<2; i++)
	    {
	       /* if this is not a door, don't go through the code more than
                  once */
               if ((i==1) && (cur_obj->get_type() != OBJ_TYPE_DOOR))
                  break;

               /* set up the name and area pointers for the location */
               if (holder.num_char('@') > 0)
	       {
                  the_name = holder.str_show();
                  the_area = the_name;
                  while (*the_area != '@')
                     the_area++;
                  *the_area = '\0';
                  the_area++;
               }
               else
	       {
                  the_name = holder.str_show();
                  the_area = cur_obj->get_area();
               }

               if (!STRCASECMP(the_name, _("none")))
	       {

               }
               else if ((tmp_obj = 
                      obj_dbase->get_mudobject(the_area, the_name)) == NULL)
	       {
                  holder.sprintf(
                   _("Mudobject '%s' location '%s@%s' doesn't seem to exist."), 
                              cur_obj->get_name(), the_name, the_area); 
                  error_log->log_err(holder.str_show(), "coord_area1");
               }
               else if (tmp_obj == cur_obj)
	       {
		 holder.sprintf(
			_("Mudobject '%s' location can't be set to self."), 
			cur_obj->get_name());
		 error_log->log_err(holder.str_show(), "coord_area1");
		 cur_obj->set_loc_str("none");
	       }
	       else
	       {
                  if (tmp_obj->is_an_item())
                     tmp_itemflags = ((Item *) tmp_obj)->get_itemflags();
                  else
                     tmp_itemflags = NULL;

                  if (cur_obj->get_type() == OBJ_TYPE_MOBILE)
                  {
                     Mobile *tmp_mob;
                     Location *the_loc;

                     if (tmp_obj->get_type() != OBJ_TYPE_LOCATION)
                     {
                        holder.sprintf(_("Mobile '%s' location must be set to "
                                          "an actual location.\n"), 
                                                       cur_obj->get_name());
                        error_log->log_err(holder.str_show(), "coord_area1");
                     }
                     else
                     {
                        tmp_mob = (Mobile *) cur_obj;
                        the_loc = (Location *) tmp_obj;
                        tmp_mob->set_location(the_loc);
			tmp_mob->coord_ranks(obj_dbase);
                     }

                  }
                  else
	          {

                     if ((tmp_itemflags != NULL) && 
                             (!tmp_itemflags->get_flag(ITEMFLAG_CONTAINER)))
		     {
                        holder.sprintf(_("Item '%s' location does not have the "
                                       "container itemflag set.\n"), 
                                                     cur_obj->get_name());
                        error_log->log_err(holder.str_show(), "coord_area1");
                     }
                     else
		     {
		       if ((cur_obj->get_type() == OBJ_TYPE_DOOR) && 
			   (tmp_obj->get_type() != OBJ_TYPE_LOCATION))
		       {
			 holder.sprintf(
			   _("Door '%s@%s' location '%s@%s' is not a location "
			     "object and must be."), cur_obj->get_name(), 
			     cur_obj->get_area(), tmp_obj->get_name(), 
			     tmp_obj->get_area());
			 error_log->log_err(holder.str_show(), "coord_area1");
		       }
		       else if ((!cur_obj->is_a_moveable()) &&
				(tmp_obj->get_type() != OBJ_TYPE_LOCATION))
		       {
			 holder.sprintf(
			   _("Object '%s@%s' location '%s@%s' is not a location "
			     "object and must be."), cur_obj->get_name(), 
			     cur_obj->get_area(), tmp_obj->get_name(), 
			     tmp_obj->get_area());
			 error_log->log_err(holder.str_show(), "coord_area1");
		       }
		       else
		       {
			 tmp_obj->add_contained(cur_obj);

			 if ((cur_obj->is_a_moveable()) &&
			     (tmp_obj->get_type() != OBJ_TYPE_LOCATION))
			   ((Moveable *) cur_obj)->set_moved();
		       }
                     }
                  }

               }
          
               /* we set up to load the outside loc as well for doors */
               if (cur_obj->get_type() == OBJ_TYPE_DOOR)
	       {
                  holder = ((Door *) cur_obj)->get_outside_loc();     
               }
            } 
         }

         /* clone mergers and place them in this object's inventory */
         if (!cur_obj->is_merger())
         {
	    /* add the cloned objects in this object's inventory to it */
            cur_obj->add_clones(obj_dbase, error_log);
         }

         /* add any specials to the object */
	 if (cur_obj->get_special_str() != NULL)
	 {
            Specials *the_special;
            Strings next_word;
            int     i = 1;

            next_word.assign_word(cur_obj->get_special_str(), i);
            while ((next_word.str_show() != NULL) &&
                   (next_word.str_len() > 0))
	    {
               if (next_word.num_char('@') > 0)
	       {
                  the_name = next_word.str_show();
                  the_area = the_name;
                  while (*the_area != '@')
                     the_area++;
                  *the_area = '\0';
                  the_area++;
               }
               else
	       {
                  the_name = next_word.str_show();
                  the_area = cur_obj->get_area();
               }

               if ((the_special = 
                  obj_dbase->get_special_obj(the_area, the_name)) == NULL)
	       {
                  holder.sprintf(
                      _("Mudobject '%s@%s' special '%s' doesn't seem to exist "
                      "or is not a special."), 
                      cur_obj->get_name(), cur_obj->get_area(), the_name); 
                  error_log->log_err(holder.str_show(), "coord_area1");
               }
               else
	       {
                  cur_obj->add_special(the_special);
               }
               i += 1;
               next_word.assign_word(cur_obj->get_special_str(), i);
            }
         }

         /* handle coordination of shops */
         if (cur_obj->get_type() == OBJ_TYPE_MOBILE)
         {
            Mobile *the_mobile;
            Shop *the_shop;
            MudObject *currency_obj;

            the_mobile = (Mobile *) cur_obj;
            if ((the_shop = the_mobile->get_shop()) != NULL)
            { 
               holder = the_shop->get_currency();

               /* set up the name and area pointers for the  */
               if (holder.num_char('@') > 0)
	       {
                  the_name = holder.str_show();
                  the_area = the_name;
                  while (*the_area != '@')
                     the_area++;
                  *the_area = '\0';
                  the_area++;
               }
               else
	       {
                  the_name = holder.str_show();
                  the_area = cur_obj->get_area();
               }

               if ((currency_obj = 
                     obj_dbase->get_mudobject(the_area, the_name)) == NULL)
	       {
                  the_mobile->delete_shop();
                  holder.sprintf(
                      _("Cannot find currency for Mobile '%s@%s' Shop\n"),
                                  cur_obj->get_name(), cur_obj->get_area());
                  error_log->log_err(holder.str_show(), "coord_area1");
               }
               else
	       {
                  if (currency_obj->get_type() != OBJ_TYPE_MONEY)
	          {
                     the_mobile->delete_shop();
                     holder.sprintf(
                        _("Currency must be a Money object for '%s@%s' Shop\n"),
                                 cur_obj->get_name(), cur_obj->get_area());
                     error_log->log_err(holder.str_show(), "coord_area1");
	          }
                  else
		  {
                     holder.sprintf("%s@%s", currency_obj->get_name(), 
                                             currency_obj->get_area());
                     the_shop->set_currency(holder.str_show());
                  }
               }
            }
         }
      }

      /* do a quick check on flags to ensure they don't conflict */
      if (cur_entity->is_an_item())
      {
         tmp_itemflags = ((Item *)cur_entity)->get_itemflags();

         /* lets see if the lockable container flags play nice */
         if ((tmp_itemflags->get_flag(ITEMFLAG_LOCKABLE)) && 
             (tmp_itemflags->get_flag(ITEMFLAG_CLOSEABLE)))
	 {
            holder.sprintf(_("Warning: Lockable and Closeable itemflags "
                           "should not be both set for object %s@%s."), 
                       cur_entity->get_name(), cur_entity->get_area()); 
            error_log->log_err(holder.str_show(), "coord_area1");
            tmp_itemflags->clr_flag(ITEMFLAG_CLOSEABLE);
	 }

         /* lets see if the lockable container flags play nice */
         if ((tmp_itemflags->get_flag(ITEMFLAG_LOCKED)) &&  
             (!tmp_itemflags->get_flag(ITEMFLAG_LOCKABLE)))
	 {
            holder.sprintf(_("Warning: Locked itemflag needs Lockable flag set "
                           "for object %s@%s."), cur_entity->get_name(), 
                                                cur_entity->get_area()); 
            error_log->log_err(holder.str_show(), "coord_area1");
            tmp_itemflags->set_flag(ITEMFLAG_LOCKABLE);
	 }

         /* lets see if the lockable container flags play nice */
         if ((tmp_itemflags->get_flag(ITEMFLAG_CLOSED)) &&  
             (!tmp_itemflags->get_flag(ITEMFLAG_LOCKABLE)) &&
             (!tmp_itemflags->get_flag(ITEMFLAG_CLOSEABLE)))
	 {
            holder.sprintf(_("Warning: Closed itemflag needs either "
                           "Lockable or Closeable flag set "
                           "for object %s@%s."), cur_entity->get_name(),
                                                cur_entity->get_area()); 
            error_log->log_err(holder.str_show(), "coord_area1");
            tmp_itemflags->set_flag(ITEMFLAG_CLOSEABLE);
	 }

         if (tmp_itemflags->get_flag(ITEMFLAG_LOCKED))
            tmp_itemflags->set_flag(ITEMFLAG_CLOSED);
      }
      cur_entity = get_next_obj();
   }
   return 1;
}


/***********************************************************************
 ** coord_area2 - takes the list of objects in this area and wields and
 **               wears the objects when appropriate, second sweep
 **
 ** Parameters: error_log - the error log to report errors to
 **             obj_dbase - the game's database of objects
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Area_Dbase::coord_area2(ErrLog *error_log, Object_List *obj_dbase)
{
   Entity    *cur_entity;
   MudObject *cur_obj;
   MudObject *tmp_obj;
   Strings   holder;
   char      *the_name;
   char      *the_area;

   reset_list();
   cur_entity = get_next_obj();
   while (cur_entity != NULL)
   {

     if (cur_entity->is_a_mudobject())
     {

       cur_obj = (MudObject *) cur_entity;

       /* wield objects that need to be wielded */
       if (cur_obj->get_type() == OBJ_TYPE_MOBILE)
       {

         Mobile    *the_mobile;
 
         /* make them wielding objects */
         the_mobile = (Mobile *) cur_obj;

         /* wield code goes here */
         the_mobile->start_wield(obj_dbase, error_log);        

         the_mobile->start_wear(obj_dbase, error_log);
       }

       /* set up a rope door */
       if (cur_obj->get_type() == OBJ_TYPE_DOOR)
       {
         Door  *the_door;
         Flags *tmp_doorflags;

         the_door = (Door *) cur_obj;
         tmp_doorflags = the_door->get_doorflags();
         if (tmp_doorflags->get_flag(DOORFLAG_ROPETIE))
	 {
	   /* if the rope is tied already */
           if (the_door->get_door_state() != DOOR_CLOSED)
	   {
             if (the_door->get_rope_name() == NULL)
	     {
               holder.sprintf(
                    _("Rope Door %s marked as tied but no rope given."), 
                                                       the_door->get_name());
               error_log->log_err(holder.str_show(), "coord_area2");
                     
	     }
             else
	     {

               holder = the_door->get_rope_name();
               /* set up the name and area pointers for the rope */
               if (holder.num_char('@') > 0)
	       {
                  the_name = holder.str_show();
                  the_area = the_name;
                  while (*the_area != '@')
                     the_area++;
                  *the_area = '\0';
                  the_area++;
               }
               else
	       {
                  the_name = holder.str_show();
                  the_area = cur_obj->get_area();
               }

               if ((tmp_obj = obj_dbase->get_mudobject
                                              (the_area, the_name)) == NULL)
	       {
                 holder.sprintf(
                    _("Door %s rope link '%s' doesn't seem to exist."), 
                         cur_obj->get_name(), the_door->get_rope_name()); 
                  error_log->log_err(holder.str_show(), "coord_area2");
               }
               else
	       {
                 if (tmp_obj->get_type() != OBJ_TYPE_ROPE)
		 {
                   holder.sprintf(
                     _("Door %s rope link '%s' not a rope."), 
                         cur_obj->get_name(), tmp_obj->get_name()); 
                   error_log->log_err(holder.str_show(), "coord_area2");
		 }
                 else
		 {
                   Rope *the_rope;

                   the_rope = (Rope *) tmp_obj;

                   if ((STRCASECMP(the_rope->get_location(), 
                                         the_door->get_location())) &&
                       (STRCASECMP(the_rope->get_location(), 
                                         the_door->get_outside_loc()))) 
		   {
                     holder.sprintf(
                        _("Door %s rope link '%s' not in door location."), 
                            cur_obj->get_name(), tmp_obj->get_name()); 
                     error_log->log_err(holder.str_show(), "coord_area2");
		   }
                   else
		   {
                     the_rope->set_tied();
                     the_rope->set_dual_loc(the_door->get_location(), 
                              the_door->get_outside_loc(), obj_dbase);
		   }
		 }
               }
	     }
	   }
	 }
       }

     }
     cur_entity = get_next_obj();
   }
   return 1;
}


/***********************************************************************
 ** update_reboot_info - resets the location of all objects that are
 **                      set to object y to object z.  This is needed if
 **                      you clone an object, to make sure things get put
 **                      into the clone
 **
 ** Parameters: oldname - the name of the old object
 **             newname - the name we are to set this to
 **
 ** Returns: number changed for success
 **
 ***********************************************************************/

int Area_Dbase::update_reboot_info(char *oldname, char *newname)
{
   reload_obj_info *tmp_info;
   int             count = 0;
   
   tmp_info = reload_list;
   while (tmp_info != NULL)
   {
      if (!STRCASECMP(tmp_info->last_loc_name.str_show(), oldname))
      {
         tmp_info->last_loc_name = newname;
         count++;
      }
      tmp_info = tmp_info->next_info;
   }

   return count;
}


/***********************************************************************
 ** get_reload_interval - gets the interval in seconds that this area will
 **                       get reloaded
 **
 ***********************************************************************/

long Area_Dbase::get_reload_interval()
{
   return reload_interval;
}


/***********************************************************************
 ** set_reload_interval - sets the reload interval to the value passed in
 **
 ***********************************************************************/

void Area_Dbase::set_reload_interval(long new_val)
{
   reload_interval = new_val;
}

/***********************************************************************
 ** get_areaflags - returns a pointer to the flags structure for the
 **                 area flags of this area
 **
 ***********************************************************************/

Flags *Area_Dbase::get_areaflags()
{
   return area_flags;
}

/***********************************************************************
 ** write_area - writes this area to disk
 **
 ** Parameters: oldname - the name of the old object
 **             newname - the name we are to set this to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Area_Dbase::write_area(FILE *the_file, int is_builder_file)
{
   Entity *the_object;
   int    is_action = 0;
   int    is_abilities = 0;
   int    is_quests = 0;
   int    is_race = 0;
   int    is_inclination = 0;
   int    is_talent = 0;
   int    is_mask = 0;
   int    is_chatline = 0;
   int    is_levels = 0;

   reset_list();

   if (!STRCASECMP(get_areaname(), "actions"))
      is_action = 1;
   else if (!STRCASECMP(get_areaname(), "abilities"))
      is_abilities = 1;
   else if (!STRCASECMP(get_areaname(), "levels"))
      is_levels = 1;
   else if (!STRCASECMP(get_areaname(), "quests"))
      is_quests = 1;
   else if (!STRCASECMP(get_areaname(), "races"))
      is_race = 1;
   else if (!STRCASECMP(get_areaname(), "inclination"))
      is_inclination = 1;
   else if (!STRCASECMP(get_areaname(), "talent"))
      is_talent = 1;
   else if (!STRCASECMP(get_areaname(), "masks"))
      is_mask = 1;
   else if (!STRCASECMP(get_areaname(), "chatlines"))
      is_chatline = 1;

   if (the_file != NULL)
   {
      fprintf(the_file, "^%s^\n", FULLVERSION);

      /* write the area file header */
      fprintf(the_file, "^%s^\n", ((get_read_deny() == NULL) ? "" : 
                                                      get_read_deny()));
      fprintf(the_file, "^%s^\n\n\n", ((get_write_allow() == NULL) ? "" :
                                                     get_write_allow()));

      if ((!is_action) && (!is_abilities) && (!is_quests) && (!is_levels) &&
          (!is_race) && (!is_mask) && (!is_inclination) && (!is_talent) &&
          (!is_chatline))
      {
         area_flags->write_flag(the_file);
         fprintf(the_file, "%ld\n\n", get_reload_interval());
      }

   }

   /* go through all objects in the database, writing each one */
   while ((the_object = get_next_obj()) != NULL)
   {
      the_object->write_object(the_file, is_builder_file);
   }
   return 1;
}


/***********************************************************************
 ** try_reload - checks if we can reload yet, and if so reloads the area
 **
 ** Parameters: None
 **
 ** Returns: 1 for reloaded, 0 for no reload, -1 for failure
 **
 ***********************************************************************/

int Area_Dbase::try_reload()
{
   Player   *tmp_player;
   Location *cur_loc;
   Strings  holder;

   if (!get_reload_interval())
      return 0;

   if (reload_time.elapsed_time() >= get_reload_interval())
   {
      if (!area_flags->get_flag(AREAFLAG_IGNORE_PLAYERS))
      {
         tmp_player = mainstruct->get_first_player();
         while (tmp_player != NULL)
         {
            cur_loc = tmp_player->get_loc();
            if ((cur_loc != NULL) && 
                (!STRCASECMP(cur_loc->get_area(), get_areaname())))
               return 0;

            tmp_player = tmp_player->get_next_player();
         }
      }
      mainstruct->reload_area(get_areaname());
      holder.sprintf(_("&+Y[&*Area &+M%s&* autoreloaded&+Y]&*\n"),
                                           get_areaname());

      mainstruct->send_all_players(holder.str_show(), NULL, 
                                                      ADMINFLAG_SEERELOADS);

      reload_time.reset_time();
      return 1;
   }
   return 0;
}

/***********************************************************************
 ** display_brief_data - displays a small amount of data for a particular 
 **                      area
 **                
 ** Parameters: the_player - who we send the messages to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Area_Dbase::display_brief_data(Player *the_player)
{
   
   Entity    *tmp_entity;
   int       areanamelen;
   int       size = 0;

   int num_obj = 0;
   int num_cloned = 0;
   
   the_dbase.reset_current();
   tmp_entity = the_dbase.get_next();
   while (tmp_entity != NULL)
   {
      size += tmp_entity->get_mem_size();
      if (tmp_entity->is_a_mudobject())
      {
         MudObject *tmp_obj;
         tmp_obj = (MudObject *) tmp_entity;

         num_obj++;
         if (tmp_obj->get_parent() != NULL)
         num_cloned++;
      }
      tmp_entity = the_dbase.get_next();
   }
  
   areanamelen = strlen(get_areaname());

   the_player->send_plr("%-13s %5d %7d %10ld %10ld %10d\n", 
                    get_areaname(),  
                    num_obj, num_cloned, reload_time.elapsed_time(), 
                        get_reload_interval(), size);
   return 1;
}


/***********************************************************************
 ** get_num_obj_loaded - gets the number of objects loaded in this area
 **                
 ***********************************************************************/

int Area_Dbase::get_num_obj_loaded()
{
   return num_obj_loaded;
}


void Area_Dbase::save_state(void)
{
   the_dbase.push_current();
}

void Area_Dbase::restore_state(void)
{
   the_dbase.pop_current();
}



/***********************************************************************
 ** set_time_till_reset - sets the time till the reset occurs to a new
 **                       value
 **
 ** Parameters: new_minutes - new number of seconds till reset
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Area_Dbase::set_time_till_reset(int new_secs)
{
   reload_time.set_time(time(0) + new_secs);
}


/***********************************************************************
 ** get_time_till_reset - gets the time till the reset occurs
 **
 ** Parameters: None
 **
 ** Returns: the time in seconds till the zone resets
 **
 ***********************************************************************/

long Area_Dbase::get_time_till_reset()
{
   return (reload_interval - reload_time.elapsed_time()); 
}


/***********************************************************************
 ** get_stored_mem - gets the total used memory by all objects in the
 **                  database
 **
 ** Returns: number of bytes used
 **
 ***********************************************************************/

int Area_Dbase::get_stored_mem()
{
   Entity *tmp_entity;
   int    size = 0;

   reset_list();
   tmp_entity = get_next_obj();
   while (tmp_entity != NULL)
   {
     size += tmp_entity->get_mem_size();
     tmp_entity = get_next_obj();
   }
   return size;
}


/***********************************************************************
 ** empty_objects = empties all objects of other objects so we can delete
 **                 them safely
 **
 ** Returns: 1 for success, 0 for failure
 **
 ***********************************************************************/

int Area_Dbase::empty_objects()
{
  Entity *tmp_obj;
  Inventory *the_inv;
  MudObject *the_obj;
  MudObject *remove_obj;
  MudObject *remove_from;

  reset_list();
  while ((tmp_obj = get_next_obj()) != NULL)
  {
    // If it is a mudobject, deconflict it
    if (tmp_obj->is_a_mudobject())
    {
      // First, we empty out the object
      the_obj = (MudObject *) tmp_obj;
      the_inv = the_obj->get_inventory();
      the_inv->reset_current();
      while ((remove_obj = the_inv->get_first()) != NULL)
      {
	     the_obj->remove_contained(remove_obj);
		 the_inv->reset_current();
      }

      // Remove the door from both sides
      if (the_obj->get_type() == OBJ_TYPE_DOOR)
      {
		 if ((remove_from = ((Door *) the_obj)->get_contained_by()) != NULL)
		 {
		    remove_from->remove_contained(the_obj);
		 }

	     if ((remove_from = ((Door *) the_obj)->get_outside_loc_obj()) != NULL)
		 {
	        remove_from->remove_contained(the_obj);
		 }
      }
      else  // Its not a door, just remove it from its loc
      {
	  
	     // Now make sure this object is not contained in something
	     if (the_obj->get_contained_by() != NULL)
		 {
	        the_obj->get_contained_by()->remove_contained(the_obj);
	  
		 }
	     if ((the_obj->get_type() == OBJ_TYPE_ROPE) &&
	         (((Rope *) the_obj)->get_other_loc() != NULL))
	         (((Rope *) the_obj)->get_other_loc())->remove_contained(the_obj);
	  }
    }
  }

  return the_dbase.clr_tree();
}



#endif


