/**********************************************************************
 ** Weapon class:  Has attributes and methods for a weapon object
 **
 ** Last reviewed:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef WEAPON_C
#define WEAPON_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "weapon.h"
#include "objtype.h"
#include "flags.h"
#include "player.h"
#include "specials.h"
#include "code.h"
#include "global.h"

/***********************************************************************
 ** Weapon (constructor)
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Weapon::Weapon(void)
{

}


/***********************************************************************
 ** Weapon (constructor) - creates the item
 **
 ** Parameters: the_name - the name of the marker
 **             the_area - the area the marker belongs to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Weapon::Weapon(char *the_name, char *the_area)
{
   /* this should be the only place that this is set */
   obj_type = OBJ_TYPE_WEAPON;
   
   if (the_name != NULL)
      set_name(the_name);
   
   if (the_area != NULL)
      set_area(the_area);

   size = weight = state = damage = dep_strength = dep_dexterity = 0;
   wield_type = One_Handed;  
}

/***********************************************************************
 ** ~Weapon (destructor) - destroys it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Weapon::~Weapon()
{
}


/***********************************************************************
 ** write_object - writes the weapon to a specified file in builder
 **                file format
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
void Weapon::write_object(FILE *the_file, int build_format)
{
   fprintf(the_file, "\nweapon %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());
   write_mudobject_attrib(the_file);
   write_item_attrib(the_file);
   write_moveable_attrib(the_file);
   fprintf(the_file, "%s\n", (get_weapon_class() == NULL) ? "generic" :
                                                   get_weapon_class());
   fprintf(the_file, "%d\n%d\n%d\n", get_damage(), get_dep_strength(), 
                                                   get_dep_dex());
   fprintf(the_file, "%d\n", get_wield_type());
}

/***********************************************************************
 ** describe - describes the weapon to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Weapon::describe(Builder *the_builder)
{
   the_builder->send_bldr("\n&+GWeapon: \t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_builder->send_bldr("&+GAltnames: \t&+g%s&*\n", get_altname());
   the_builder->send_bldr("&+GClones: \t&+g%s&*\n", get_clones());
   the_builder->send_bldr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_builder->send_bldr("&+GGuards: \t&+g%s&*\n", get_guards());
   the_builder->send_bldr("&+GLocation: \t&+M%s&*\n", get_location());
   the_builder->send_bldr("&+GClass:\t\t&+w%s&*\n", get_weapon_class());
   the_builder->send_bldr("&+GWieldType:\t&+w%s&*\n", 
             (get_wield_type() == One_Handed) ? "OneHanded" : "TwoHanded");
   the_builder->send_bldr("&+GSize: \t\t&+w%d&*\n", get_size());
   the_builder->send_bldr("&+GWeight:\t\t&+w%d&*\n", get_weight());
   if (itemflags->get_flag(ITEMFLAG_CONTAINER))
      the_builder->send_bldr("&+GCapacity:\t&+w%d&*\n", get_capacity());
   the_builder->send_bldr("&+GDamage:\t\t&+w%d&*\n", get_damage());
   the_builder->send_bldr("&+GDepStr: \t&+w%d&*\n", get_dep_strength());
   the_builder->send_bldr("&+GDepDex: \t&+w%d&*\n", get_dep_dex());
   the_builder->send_bldr("&+GBrief0:\n&*%s\n", get_brief(0));
   the_builder->send_bldr("&+GBrief1:\n&*%s\n", get_brief(1));
   the_builder->send_bldr("&+GDesc:&*\n%s\n", get_desc());

   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the weapon to a player
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void Weapon::describe(Player *the_player)
{
   MudObject *tmp_container;
   Strings   container_name;

   the_player->send_plr("\n&+GWeapon: \t&+M%s&*\n", get_name());
   the_player->send_plr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_player->send_plr("&+GAltnames: \t&+g%s&*\n", get_altname());
   the_player->send_plr("&+GClones: \t&+g%s&*\n", get_clones());
   the_player->send_plr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_player->send_plr("&+GGuards: \t&+g%s&*\n", get_guards());
   the_player->send_plr("&+GStartLoc: \t&+M%s&*\n", get_location());

   tmp_container = get_contained_by();
   if (tmp_container == NULL)
      container_name = "nowhere";
   else
      container_name.sprintf("%s@%s", tmp_container->get_name(), 
                                         tmp_container->get_area());

   the_player->send_plr("&+GCurrentLoc: \t&+M%s&*\n", 
                                                 container_name.str_show());
   the_player->send_plr("&+GClass:\t\t&+w%s&*\n", get_weapon_class());
   the_player->send_plr("&+GWieldType:\t&+w%s&*\n", 
             (get_wield_type() == One_Handed) ? "OneHanded" : "TwoHanded");
   the_player->send_plr("&+GSize: \t\t&+w%d&*\n", get_size());
   the_player->send_plr("&+GWeight:\t\t&+w%d&*\n", get_weight());
   if (itemflags->get_flag(ITEMFLAG_CONTAINER))
      the_player->send_plr("&+GCapacity:\t&+w%d&*\n", get_capacity());
   the_player->send_plr("&+GDamage:\t\t&+w%d&*\n", get_damage());
   the_player->send_plr("&+GDepStr: \t&+w%d&*\n", get_dep_strength());
   the_player->send_plr("&+GDepDex: \t&+w%d&*\n", get_dep_dex());
   the_player->send_plr("&+GBrief0:\n&*%s\n", get_brief(0));
   the_player->send_plr("&+GBrief1:\n&*%s\n", get_brief(1));
   the_player->send_plr("&+GDesc:&*\n%s\n", get_desc());
   the_player->send_plr("&+YSize: \t\t\t&+W%d&*\n", get_mem_size());

   list_specials(the_player);

   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Weapon::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
            send_bldr("You can set the following attributes on a weapon.\n"
               "   title, weight, size, altnames, desc, brief0, brief1,\n"
               "   guards, itemflags, location, specials, merger, class,\n"
               "   wieldtype, damage, depstr, depdex, and capacity\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "title",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_title(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "weight",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_weight(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "size",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_size(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "altnames",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_altnames(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "clones",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_clones(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "desc", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_desc(the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "brief0", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 0);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "itemflags",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_itemflags(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "guards",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_guard(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "specials", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_special(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "brief1", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 1);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "location", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_loc(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "class",
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
               send_bldr("You need to specify a weapon class to set to.\n");
         return -1;
      }
      holder.assign_word(the_parsed->get_speech(), 1);
      set_weapon_class(holder.str_show());

      the_builder->send_bldr("Class on %s set to: %s\n", get_name(), 
                                                     get_weapon_class());
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "wieldtype",
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;      

      if ((the_parsed->get_speech() == NULL) || 
          (strlen(the_parsed->get_speech()) <= 0))
      {
         the_builder->
              send_bldr("You need to specify a wield type to set to.\n"
                        "Either OneHanded or TwoHanded\n");
         return -1;
      }
      holder.assign_word(the_parsed->get_speech(), 1);
      if (!STRNCASECMP(holder.str_show(), "onehanded", holder.str_len()))
         set_wield_type(One_Handed);
      else if (!STRNCASECMP(holder.str_show(), "twohanded", holder.str_len()))
         set_wield_type(Two_Handed);
      else
      {
         the_builder->send_bldr("Invalid wield type.\n"
                                "Valid types are: OneHanded and TwoHanded\n");
         return -1;
      }

      the_builder->send_bldr("Wield Type on %s set to: %s\n", get_name(), 
             (get_wield_type() == One_Handed) ? "OneHanded" : "TwoHanded");
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "damage",
                               strlen(the_parsed->get_target1())))
   {
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as damage.\n");
         return -1;
      }

      set_damage(atoi(the_parsed->get_speech()));
      the_builder->send_bldr("Damage set to %d on weapon object %s.\n",
                                          get_damage(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "depstr",
                               strlen(the_parsed->get_target1())))
   {
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as depstr.\n");
         return -1;
      }

      set_dep_strength(atoi(the_parsed->get_speech()));
      the_builder->send_bldr("DepStr set to %d on weapon object %s.\n",
                                          get_dep_strength(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "depdex",
                               strlen(the_parsed->get_target1())))
   {
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as DepDex.\n");
         return -1;
      }

      set_dep_dex(atoi(the_parsed->get_speech()));
      the_builder->send_bldr("DepDex set to %d on weapon object %s.\n",
                                          get_dep_dex(), get_name());
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "capacity", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_capacity(the_parsed, the_builder);
   }

   the_builder->send_bldr("The attribute '%s' is not a weapon attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** copy_object - copies the weapon to an object of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Weapon::copy_object(Entity *copy_obj)
{
   Weapon *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_WEAPON)
      return 0;

   copy_from = (Weapon *) copy_obj;

   /******* set the mudobject attributes *****/
   copy_mudobject_attrib((MudObject *) copy_from);

   /****** set the item attributes ******/
   copy_item_attrib((Item *) copy_from);

   /****** set the moveable attributes ******/
   copy_moveable_attrib((Moveable *) copy_from);

   /****** set up weapon attributes ******/
   set_weapon_class(copy_from->get_weapon_class());
   set_damage(copy_from->get_damage());
   set_wield_type(copy_from->get_wield_type());
   set_dep_strength(copy_from->get_dep_strength());
   set_dep_dex(copy_from->get_dep_dex());

   return 1;
}

/***********************************************************************
 ** set_weapon_class - sets the class of weapon this will be, like
 **                    sword, blunt, polearm, bow, etc
 **
 ** Parameters: new_class - the class to set this to
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/

int Weapon::set_weapon_class(char *new_class)
{
   if (new_class == NULL)
      return -1;

   weapon_class = new_class;
   return 1;
}


/***********************************************************************
 ** get_weapon_class - returns the class that this weapon will be, for
 **                    example it could be shortsword, bow, dagger, etc
 **
 ** Parameters: Nothing
 **
 ** Returns:  A pointer to the class string
 **
 ***********************************************************************/

char *Weapon::get_weapon_class(void)
{
   return weapon_class.str_show();
}


/***********************************************************************
 ** set_damage - sets the damage this weapon will cause
 **
 ** Parameters: new_damage - the damage we are setting this to
 **
 ** Returns:  Nothing
 **
 ***********************************************************************/

void Weapon::set_damage(int new_damage)
{
   damage = new_damage;
}


/***********************************************************************
 ** get_damage - returns the damage that this weapon can inflict
 **
 ** Parameters: Nothing
 **
 ** Returns:  The damage for this weapon
 **
 ***********************************************************************/

int Weapon::get_damage(void)
{
   return damage;
}


/***********************************************************************
 ** set_wield_type - sets the wield type for this weapon, passes in an
 **                  enumerated type of either One_Handed or Two_Handed
 **
 ** Parameters: the_type - the type to set it to
 **
 ** Returns:  Nothing
 **
 ***********************************************************************/
 
void Weapon::set_wield_type(hands the_type)
{
   wield_type = the_type;
}


/***********************************************************************
 ** get_wield_type - gets the wield type of this weapon, which is an 
 **                  enumerated type of either One_Handed or Two_Handed
 **
 ** Parameters: Nothing
 **
 ** Returns:  Either One_Handed or Two_Handed
 **
 ***********************************************************************/

hands Weapon::get_wield_type()
{
   return wield_type;
}


/***********************************************************************
 ** set_dep_strength - sets the dependancy this weapon has on the wielder's
 **                    strength, from 0 to 100 where 0 is not at all
 **                    dependant, and 100 is fully dependant when it comes
 **                    to chance of hitting
 **
 ** Parameters: the_strength - the new strength to set this to
 **
 ** Returns:  1 for success, -1 for error
 **
 ***********************************************************************/

int Weapon::set_dep_strength(int the_strength)
{
   if ((the_strength < 0) || (the_strength > 100))
      return -1;

   dep_strength = the_strength;
   return 1;
}


/***********************************************************************
 ** get_dep_strength - gets the dependancy this weapon has on the user's
 **                    strength, from 0 to 100 where 0 is not dependant, 
 **                    and 100 is fully dependant
 **
 ** Parameters: Nothing
 **
 ** Returns:  The dependancy value
 **
 ***********************************************************************/

int Weapon::get_dep_strength(void)
{
   return dep_strength;
}



/***********************************************************************
 ** set_dep_dex - sets the dependancy this weapon has on the wielder's
 **               dexterity, from 0 to 100 where 0 is not at all 
 **               dependant, and 100 is fully dependant when it comes
 **               to chance of hitting
 **
 ** Parameters: the_dexterity - the new dexterity to set this to
 **
 ** Returns:  1 for success, -1 for error
 **
 ***********************************************************************/

int Weapon::set_dep_dex(int the_dexterity)
{
   if ((the_dexterity < 0) || (the_dexterity > 100))
      return -1;

   dep_dexterity = the_dexterity;
   return 1;
}


/***********************************************************************
 ** get_dep_dex - gets the dependancy this weapon has on the user's
 **               dexterity, from 0 to 100 where 0 is not dependant, 
 **               and 100 is fully dependant
 **
 ** Parameters: Nothing
 **
 ** Returns:  The dependancy value
 **
 ***********************************************************************/

int Weapon::get_dep_dex(void)
{
   return dep_dexterity;
}


/***********************************************************************
 ** operator = - copies an object to this object
 **
 ** Parameters: None
 **
 ** Returns: a pointer to this object copied to
 **
 ***********************************************************************/

Weapon *Weapon::operator = (Weapon *copy_from)
{
   if (!STRCASECMP(copy_from->get_name(), get_name()))
      return NULL;

   copy_object(copy_from);
   return this;
}



/***********************************************************************
 ** read_weapon_attrib - reads in weapons attributes from the file
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns:  1 for successful read
 **          -1 for errors in the read
 **
 ***********************************************************************/

int Weapon::read_weapon_attrib(FILE *read_file, ErrLog *error_log)
{
   token_record *the_token;
   Strings      holder;

   /* get the weapon class */
   the_token = get_token(read_file, '\0');
   set_weapon_class(the_token->the_string);
   
   /* Set damage */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute damage in weapon %s", 
                                                            get_name());
      error_log->log_err(holder.str_show(), "read_weapon_attrib");
      return -1;
   }
   set_damage(atoi(the_token->the_string));

   /* Set dep strength */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute depstr in weapon %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "read_weapon_attrib");
      return -1;
   }
   set_dep_strength(atoi(the_token->the_string));

   /* Set dep dexterity */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.
          sprintf("Invalid format for attribute dep dexterity in weapon %s", 
                                                                 get_name());
      error_log->log_err(holder.str_show(), "read_weapon_attrib");
      return -1;
   }
   set_dep_dex(atoi(the_token->the_string));

   /* Set wield type */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute wield type in weapon %s", 
                                                                 get_name());
      error_log->log_err(holder.str_show(), "read_weapon_attrib");
      return -1;
   }
   set_wield_type((hands) atoi(the_token->the_string));

   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Weapon::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Weapon::get_mem_size_dynamic()
{
   int  size = 0;

   size += weapon_class.get_mem_size_dynamic();

   size += get_mem_size_moveable();
   size += get_mem_size_item();
   size += get_mem_size_mudobj();
   size += get_mem_size_entity();

   return size;
}


/***********************************************************************
 ** swing - uses the weapon and if a range weapon, decrements one ammo
 **
 ** Returns: 1 for success, 0 for no ammo available
 **
 ***********************************************************************/

int Weapon::swing(Individual *attacker, Individual *target, Merger **proj_return)
{
   Inventory  *weapon_inv;
   MudObject  *projectile;
   special_holder  *proj_special;
   in_params  fill_param;
   Player     *tmp_player = NULL;
   int        results;

   /* does the attacker wield a long range weapon (bow, slingshot, etc) */
   if (STRCASECMP(get_weapon_class(), "bow"))
   {
     *proj_return = NULL;
     return 1;
   }

   weapon_inv = get_inventory();
   weapon_inv->reset_current();
   projectile = weapon_inv->get_next();

   while (projectile != NULL)
   {
      if ((proj_special = projectile->find_special("on_launch")) != NULL)
      {
         break;
      }
      projectile = weapon_inv->get_next();
   }

   // They have nothing loaded
   if ((projectile == NULL) || (projectile->get_type() != OBJ_TYPE_MERGER))
   {
     return 0;
   }

   // We pass back the projectile name
   (*proj_return) = (Merger *) projectile;

   // Now we run the fire special which should decrement the number of
   // projectiles and display the fire message 

   if (attacker->get_type() == OBJ_TYPE_PLAYER)
      tmp_player = (Player *) attacker;

   fill_param.primary_obj = (MudObject *) target;
   fill_param.secondary_obj = (MudObject *) attacker;
   fill_param.this_obj = (MudObject *) projectile;

   results = (proj_special->the_special)->
             run_special(tmp_player, &fill_param, &proj_special->environment);
   if (results == 0)
      return 1;
   if (results == 2)
   {
      attacker->send_plr("&+RYou have run out of %s!&*\n", 
                                            projectile->get_title());
      remove_contained(projectile);
      mainstruct->delete_object(projectile);
      return 0;
   }

   return 1;
}


#endif




