// pkg_tree.h      -*-c++-*-
//
//  Copyright 1999,2000 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Uses the vs_tree classes to display a tree containing packages.  A generic
// version of this, suitable only for displaying the current state of the
// package cache, is provided; it can be extended as needed.

#ifndef PKG_TREE_H
#define PKG_TREE_H

#include "menu_redirect.h"

#include <vscreen/vs_tree.h>
#include <vscreen/vs_editline.h>

#include "generic/matchers.h"

class OpProgress;

class pkg_grouppolicy;
class pkg_grouppolicy_factory;
class pkg_sortpolicy;
class pkg_tree_node;
class undo_group;

/** Provides some routines that can be used to connect to the Package
 *  menu.
 */
class pkg_menu_tree:public vs_tree, public menu_redirect
{
  /** Return the selected node, if any, or \b NULL if no node is selected. */
  pkg_tree_node *selection();

  /** A precompiled matcher representing the last search that was performed. */
  pkg_matcher *last_search_matcher;

  /** The string that was compiled to produce the above matcher. */
  std::string last_search_term;

  /** \b true if an incremental search is in progress. */
  bool doing_incsearch;

  /** The iterator that was selected prior to the incremental search. */
  vs_treeiterator pre_incsearch_selected;

  void do_search(std::string s);
  void do_incsearch(std::string s);
  void do_cancel_incsearch();

  /** Execute the given action; this is needed because some "wrapper"
   *  code is used to handle undo and so on.
   */
  bool package_action(void (pkg_tree_node::* action)(undo_group *));

  /** \return \b true if a package or a package version is selected. */
  bool pkg_or_ver_selected();

  static vs_editline::history_list search_history;
protected:
  /** Reset all information about the incremental search.  This must be
   *  performed whenever the root is changed.
   */
  void reset_incsearch();
public:
  pkg_menu_tree();
  ~pkg_menu_tree();

  /** \return \b true iff a pkg_node descendant is currently selected. */
  bool package_enabled();

  /** If a pkg_node is currently selected, execute its "install" operation. */
  bool package_install();

  /** If a pkg_node is currently selected, execute its "remove" operation. */
  bool package_remove();

  /** If a pkg_node is currently selected, execute its "purge" operation. */
  bool package_purge();

  /** If a pkg_node is currently selected, execute its "keep" operation. */
  bool package_keep();

  /** If a pkg_node is currently selected, execute its "hold" operation. */
  bool package_hold();

  /** If a pkg_node is currently selected, execute its "set auto" operation. */
  bool package_mark_auto();

  /** If a pkg_node is currently selected, execute its "set manual" operation. */
  bool package_unmark_auto();

  /** \return \b true if a package or a package version is selected. */
  bool package_forbid_enabled();

  /** If a package or a version is selected, perform a "forbid"
   *  operation on it.
   */
  bool package_forbid();

  /** \return \b true if a package or a package version is selected. */
  bool package_changelog_enabled();

  /** If a package or version is selected, show its changelog. */
  bool package_changelog();

  /** \return \b true if a package or a package version is selected. */
  bool package_information_enabled();

  /** If a package or version is selected, show its information. */
  bool package_information();

  /** \return \b true; all package trees know how to search. */
  bool find_search_enabled();

  /** Execute the 'search' menu command. */
  bool find_search();

  /** \return \b true if there is a "previous search". */
  bool find_research_enabled();

  /** Execute the 're-search' menu command. */
  bool find_research();

  /** \return \b false. */
  bool find_limit_enabled();

  /** Does nothing. */
  bool find_limit();

  /** \return \b false. */
  bool find_reset_limit_enabled();

  /** Does nothing. */
  bool find_reset_limit();

  /** \return \b true if this view is active. */
  bool find_broken_enabled();

  /** Find the next broken package (searches for '~b'). */
  bool find_broken();

  bool handle_char(chtype ch);
};

class pkg_tree:public pkg_menu_tree
{
  pkg_grouppolicy_factory *grouping;
  std::string groupingstr;
  pkg_sortpolicy *sorting;

  pkg_matcher *limit;
  std::string limitstr;
  // Defines the limits on the display (what packages will be allowed
  // to be displayed)  This could be a grouping policy, but hardcoding the
  // filter here makes it easier to alter from the UI.

  static vs_editline::history_list limit_history, grouping_history,
    sorting_history;

  void handle_cache_close();

  void init(const char *limitstr,
	    OpProgress &progress);
protected:
  virtual bool handle_char(chtype ch);
public:
  pkg_tree(const char *groupingstr,
	   pkg_grouppolicy_factory *_grouping,
	   const char *limitstr,
	   OpProgress &progress);

  pkg_tree(const char *groupingstr,
	   pkg_grouppolicy_factory *_grouping,
	   const char *limitstr);

  // Should you be able to just pass in a string (?)
  ~pkg_tree();

  /** Rebuilds the tree.  If the new tree would be empty, keeps the
   *  current tree and returns \b false.
   *
   *  \param progress a progress bar with which to show the progress
   *  of the rebuild.
   *
   *  \return true iff the rebuild was successful.
   */
  bool build_tree(OpProgress &progress);

  /** Rebuilds the tree.  If the new tree would be empty, keeps the
   *  current tree and returns \b false.  Generates a new progress bar
   *  using gen_progress_bar().
   *
   *  \return true iff the rebuild was successful.
   */
  bool build_tree();

  void set_grouping(pkg_grouppolicy_factory *_grouping);
  void set_grouping(std::string s);
  void set_sorting(pkg_sortpolicy *_sorting);
  void set_sorting(std::string s);
  void set_limit(string _limit);
  // Selects a new limit and rebuilds the tree.
  string get_limit_str() {return limitstr;}

  /** Return \b true. */
  bool find_limit_enabled();

  /** Pop up the limit selection dialog. */
  bool find_limit();

  /** Return \b true if a limit is already set. */
  bool find_reset_limit_enabled();

  /** Reset the limit. */
  bool find_reset_limit();

  // Local bindings:
  static keybindings *bindings;
  static void init_bindings();

  SigC::Signal2<void, const pkgCache::PkgIterator &, const pkgCache::VerIterator &> selected_signal;
  SigC::Signal1<void, std::string> selected_desc_signal;
};

#endif
