#!/usr/bin/perl -w

# ArchZoom - a web-based Arch browser, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

use FindBin '$Bin';
use lib
	"$Bin/../perllib",  # replaced during installation
	"$Bin/archzoom-data/perllib",
	"$Bin/../archzoom-data/perllib", 
	"$Bin/../../archzoom-data/perllib";

use ArchZoom;
use ArchZoom::Util;
use ArchZoom::TemplateEngine;
use ArchZoom::MimeTypes;
use ArchZoom::LiteCGI;
use Arch::Util qw(run_tla load_file standardize_date adjacent_revision);
use Arch::Session;
use Arch::Library;
use Arch::Tarball;
use Arch::RunLimit;
use Arch::SharedCache;
use Arch::RevisionBunches;
use Arch::FileHighlighter;

my $cgi = ArchZoom::LiteCGI->new(add_header => $ArchZoom::Config{http_headers});
my $template_engine = "Too early,\n\tno template engine created yet";

# catch all errors and show a nice page with meaningful error message
$SIG{__DIE__} = sub {
	# ignore errors inside eval block
	for (1 .. 20) { return if caller($_) && (caller($_))[3] =~ /^\(eval/ }
	my $msg = shift;
	print $cgi->header_once;
	eval { print $template_engine->parse("error", { "message" => $msg }) };
	print "<pre>\nInternal error: $msg\nIn addition: $@" if $@;
};

$template_engine = new ArchZoom::TemplateEngine(
	dir => $ArchZoom::Config{template_dir},
	set => $ArchZoom::Config{template_set},
);

# ---- parse url params ------------------------------------------------------

# store all "?word" url flags in %flags
my %flags = $cgi->param_hash;
my $download = $flags{download} || 0;
my $expand = $flags{expanded} || $flags{expand} || 0;
my $library_mode = $flags{library} || 0;
my $nocache = $flags{nocache} && $ArchZoom::Config{nocache_enabled};

# this is a temporary hack, will be replaced later
$download = 1 if ($ENV{HTTP_REFERER} || "") =~ /\?download$/;

apply_config(%flags);
$cgi->content_charset($ArchZoom::Config{content_charset});
my $browser_js_support = $cgi->browser_js_support;
$ArchZoom::Config{javascript_expanding} &&= $browser_js_support;
$ArchZoom::Config{javascript_tooltip}   &&= $browser_js_support;

if ($flags{debug}) {
	die "Sorry, debug mode is explicitly disabled\n"
		unless $ArchZoom::Config{debug_enabled};
	$cgi->start_debug_mode;
}

my $run_limit = new Arch::RunLimit(
	limit   => $ArchZoom::Config{run_limit_number},
	timeout => $ArchZoom::Config{run_limit_timeout},
	file    => $ArchZoom::Config{run_limit_file},
);
if ($run_limit->exceeded) {
	sleep(1);
	my $limit = $ArchZoom::Config{run_limit_number};
	die "Sorry, limit of $limit simultaneous invocations exceeded, try later\n";
}

die "Unexpected '$Arch::Backend::EXE --version' output\n"
	unless (run_tla("--version") || "") =~ /\b(Tom Lord|Arch|Bazaar)\b/;

# ---- parse url path --------------------------------------------------------

my $session = new Arch::Session;
my $library = new Arch::Library;
my $storage = $session;
if ($library_mode) {
	$storage = $library;
	$ArchZoom::Config{library_lookup} = 1;
	$ArchZoom::Config{auto_temp_trees} = 0;
	$ArchZoom::Config{auto_library_updating} = 0;
}
my ($archive, $category, $branch, $version, $revision);
my ($globcats, $version_to_filter);
my ($tarball, $is_changeset_tarball);
my $filepath;
my $pathinfo = $cgi->path_info;
$pathinfo =~ s!/$!!;

if ($pathinfo =~ m!^/+(\*|[^/]+@[^/]+)(.*)$!) {
	($archive, $pathinfo) = ($1, $2);
	if ($archive eq '*') {
		die "Sorry, global-categories view is disabled\n"
			unless $ArchZoom::Config{globcats_enabled};
		$globcats = 1;
		$archive = undef;
	} else {
		$storage->working_names($archive);
	}
	if ($pathinfo =~ m!^/+([^/]+?)(?:--\*--([^/]+?))?((?:\.patches)?)((?:\.tar\.gz)?)(|/.*)$!) {
		$pathinfo = $5;
		$version_to_filter = $2;
		$storage->working_name("$archive/$1");
		($archive, $category, $branch, $version, $revision) = $storage->working_names;
		$tarball = $revision && $4 && $ArchZoom::Config{tarball_downloading};
		$is_changeset_tarball = $3;
		if ($storage->{name_alias} && $ArchZoom::Config{name_alias_redirection}) {
			my $tarball_suffix = ($3 || "") . ($4 || "");
			$cgi->path_info("/" . $storage->working_name
				. $tarball_suffix . $pathinfo);
			print $cgi->redirect();
			exit(0);
		}
		if ($pathinfo =~ m!^/+(.+)$!) {
			$filepath = $1;
		}
	}
}

if ($archive) {
	die "Can't process your request, because library_archives disabled\n"
		if $library_mode && !$ArchZoom::Config{library_archives};
	die "Can't process your request, because regular_archives disabled\n"
		if !$library_mode && !$ArchZoom::Config{regular_archives};
	die "Archive $archive is not managed\n"
		unless &is_archive_managed($archive);
} else {
	$library_mode = 0 unless $globcats;
}
$expand = 1 if $ArchZoom::Config{auto_archive_expanding} && defined $archive
	&& !defined $version && !exists $flags{expand};
$expand = 1 if defined $version_to_filter && !defined $version;

$template_engine->add_default_stash(
	archive  => $archive,
	category => $category,
	branch   => $branch,
	version  => $version,
	revision => $revision,
	globcats => $globcats,
	expand   => $expand,
	can_expand => !$expand && defined $archive && !$revision,
	can_unexpand => 0,
	can_tarball  => $revision && $ArchZoom::Config{tarball_downloading},
	library_mode => $library_mode,
	globcats_enabled => !$globcats && !$revision && $ArchZoom::Config{globcats_enabled} || undef,
);

# ---- determine and prepare page --------------------------------------------

my $page_name = "archives";
$page_name = "globcats"   if $globcats;
$page_name = "categories" if defined $archive;
$page_name = "branches"   if defined $category;
$page_name = "versions"   if defined $branch;
$page_name = "abrowse"    if defined $archive && $expand;
$page_name = "revisions"  if defined $version;
$page_name = "tree"       if defined $revision;
$page_name = "ancestry"   if defined $revision && $flags{ancestry};
$page_name = "log"        if defined $revision && $flags{log};
$page_name = "diff"       if defined $filepath && $flags{diff};
$page_name = "history"    if defined $filepath && $flags{history};
$page_name = "annotate"   if defined $filepath && $flags{annotate};
$page_name = "tarball"    if $tarball;

$filepath = "." if $revision && !defined $filepath;
$filepath =~ s/\.diff$// if $flags{diff} && $flags{diff} =~ /^1|-1|-2|-3$/;

my ($full_revision, $tree_root, $library_tree_found);
$full_revision = $storage->working_name if defined $revision;
if (defined $revision && $ArchZoom::Config{library_lookup}) {
	$library->init(working_revision => $full_revision);

	$library->fallback_dir($ArchZoom::Config{fallback_revision_library})
		unless $flags{log};  ### FIXME
	my $may_add = $ArchZoom::Config{auto_library_updating};
	$tree_root = $library->find_tree($may_add);
	die "Internal error, unexisting directory $tree_root returned\n"
		if defined $tree_root && !-d $tree_root;
	if (defined $tree_root) {
		$library_tree_found = 1;
	} else {
		$session->working_name($full_revision);
	}
}

$template_engine->add_default_stash(
	path => $filepath,
	full_revision => $full_revision,
);

print $cgi->header_once unless $download || $tarball;

# ---- functions -------------------------------------------------------------

sub is_archive_managed ($;$) {
	my $archive = shift;
	my $skip_check = shift;
	return 0 if $ArchZoom::Config{disabled_archive_regexp}
		&& $archive =~ $ArchZoom::Config{disabled_archive_regexp};
	return 0 unless $skip_check || $storage->is_archive_managed($archive);
	return 1;
}

sub is_revision_managed ($) {
	my $revision = shift;
	$revision =~ m!^([^/]+)/! || return 0;
	return is_archive_managed($1);
}

sub is_nonvoid ($) {
	my $value = shift;
	return defined $value && ($value || $value eq '0');
}

sub repeat ($$) {
	my $string = shift || "";
	my $number = shift || 0;
	return $string x $number;
}

sub get_file_stash ($$) {
	my $filepath = shift;
	my $ondisk_file = shift;

	my ($dir, $file, $slash) = $filepath;
	if ($dir =~ s@(/)([^/]+)$@@) {
		$slash = $1; $file = $2;
	} else {
		$dir = $slash = ""; $file = $filepath;
	}

	my $is_dir = -d $ondisk_file;
	my $is_binary = !$is_dir && $ondisk_file ne "/dev/null" && !-T _;
	my $size = -s _;
	my $mime_type = $is_dir? "filesystem/directory":
		get_file_mime_type($filepath, $is_binary);

	lstat($ondisk_file);
	my $is_link = -l _;
	# this is only good for unexisting symlinks; TODO
	unless (defined $size) {
		die "Size undefined and not symlink...\n" unless $is_link;
		$is_binary = 0;
		$size = -s _;
		$mime_type = "text/x-unix-symlink";
	}

	my $is_image = $mime_type =~ m!^image/!;
	$is_image = 0 if $mime_type =~ m!^image/(x-xpixmap|xpm)$!;
	my $can_annotate = $ArchZoom::Config{annotate_enabled}
		&& !$is_dir && !$is_binary && !$is_link;

	return {
		dir       => $dir,
		slash     => $slash,
		file      => $file,
		is_dir    => $is_dir,
		size      => $size,
		is_binary => $is_binary,
		is_link   => $is_link,
		mime_type => $mime_type,
		is_image  => $is_image,
		can_annotate => $can_annotate,
	};
}

sub get_changeset () {
	my $changeset = $library_tree_found? $library->get_changeset:
		$ArchZoom::Config{auto_temp_trees}? $session->get_changeset: undef;

	unless ($changeset) {
		die "Weird bug, no expected changeset from library\n"
			if $library_tree_found;
		die "Weird bug, no expected changeset from archive\n"
			if $ArchZoom::Config{auto_temp_trees};
		die "Revision $full_revision is not found in library,"
			. " and fetching of changeset is disabled in config, sorry\n";
	}
	return $changeset;
}

sub get_tree () {
	my $tree;
	$tree = Arch::Tree->new($tree_root) if $library_tree_found;
	if (!$library_tree_found && $ArchZoom::Config{auto_temp_trees}) {
		$tree = $session->get_tree({ no_greedy_add => 1 });
		$tree_root = $tree->{dir}
	}

	unless ($tree_root && -d $tree_root) {
		die "Weird bug, no expected library tree\n"
			if $library_tree_found;
		die "Weird bug, get from archive returned non directory ($tree_root)\n"
			if $ArchZoom::Config{auto_temp_trees} && $tree_root;
		die "Weird bug, get from archive returned nothing\n"
			if $ArchZoom::Config{auto_temp_trees};
		die "Revision $full_revision is not found in library,"
			. " and fetching is disabled in config, sorry\n";
	}
	$tree->{own_logs} = 1;
	return $tree;
}

sub get_processed_patch ($$$$) {
	my $changeset = shift;
	my $filepath = shift;
	my $type = shift;
	my $hint_asis = shift;

	my ($patch_content, $ondisk_file, $change_type, $asis) =
		$changeset->get_patch($filepath, $type, $hint_asis);

	return ($patch_content, $ondisk_file, $change_type, $asis) if $download;

	if ($change_type eq "patch" || !$asis) {
		$patch_content = htmlize($patch_content);
		my $prev_revision = $changeset->ancestor;

		$patch_content =~ s!^(--- )orig(.*)!<span class="patch_orig">$1$prev_revision$2</span>!m;
		$patch_content =~ s!^(\+\+\+ )mod(.*)!<span class="patch_mod">$1$full_revision$2</span>!m;
		$patch_content =~ s!^(@@.*)!<span class="patch_line">$1</span>!mg;
		$patch_content =~ s!^(-.*(\n^-.*)*)!<span class="patch_del">$1</span>!mg;
		$patch_content =~ s!^(\+.*(\n^\+.*)*)!<span class="patch_add">$1</span>!mg;
		$patch_content =~ s!(</span>)(\n)!$2$1!sg;
	}

	return ($patch_content, $ondisk_file, $change_type, $asis);
}

sub get_revision_bunches_stash ($%) {
	my $revision_descs = shift;
	my %extra_params = @_;

	my $collapse = defined $flags{expand} && !$expand;
	my $bunch_size = $flags{rbsize} || $ArchZoom::Config{revision_bunch_size};
	my $max_sumlen = $flags{sumlen} || $ArchZoom::Config{revision_summary_max_length};
	my $additive   = $flags{rbadd}  || $ArchZoom::Config{revision_bunch_additive} || 0;
	my $number     = $flags{rbnum};
	my $reverse    = $ArchZoom::Config{revisions_reverse};
	$reverse = $flags{reverse} if defined $flags{reverse};
	die "No revision_bunch_size defined\n" unless $bunch_size > 0;

	my $revision_bunches = Arch::RevisionBunches->new(
		bunch_size => $bunch_size,
		max_sumlen => $max_sumlen,
		%extra_params,
	);
	$revision_bunches->add_revision_descs($revision_descs);
	$revision_bunches->reverse_revision_descs if $reverse;
	my $bunches = $revision_bunches->get;

	$number = ($reverse ? 0 : @$bunches - 1)
		unless defined $number && $number ne "";

	# calculate which bunches are non empty
	my @non_empty_flags;
	unless ($expand) {
		@non_empty_flags = (0) x @$bunches;
		my @bad_edge = (0, 0);
		my ($step, $sign) = (0, 1);
		my $total = 0;
		until ($bad_edge[0] && $bad_edge[1]) {
			$number += ($sign *= -1) * ($step++);
			my $bunch = $bunches->[$number];
			my $sign_idx = $sign > 0? 1: 0;
			goto EDGE if $bad_edge[$sign_idx]
				|| $number < 0 || $number >= @$bunches
				|| $total + $bunch->{size} > $bunch_size + $additive;
			$non_empty_flags[$number] = 1;
			$total += $bunch->{size};
			last if $collapse;
			redo;
			EDGE:
			$bad_edge[$sign_idx] = 1;
		}
	}

	my $count = 0;
	my $non_count = 0;
	foreach my $bunch (@$bunches) {
		my $non_empty = $expand || $non_empty_flags[$count];
		$bunch->{non_empty} = $non_empty;
		$bunch->{non_count} = $non_count + 1;
		$bunch->{is_version_managed} = !$bunch->{version}
			|| is_revision_managed($bunch->{version});
		$non_count++ if $non_empty;
		$count++;
	}

	return {
		revision_bunches => $bunches,
		multi_filepath => @{$revision_bunches->filepaths} > 1 || undef,
		can_expand     => !$expand && @$bunches > $non_count,
		can_unexpand   =>  $expand,
		can_collapse   => !$collapse && $non_count > 1,
		can_uncollapse =>  $collapse,
		can_reverse    => !$reverse && @$revision_descs > 1,
		can_unreverse  =>  $reverse && @$revision_descs > 1,
	};
}

sub highlight_file_content ($\$) {
	my $file_name = shift;
	my $content_ref = shift;

	$main::highlighter ||= new Arch::FileHighlighter(
		$ArchZoom::Config{file_syntax_highlighting}
	);
	my $highlighter = $main::highlighter;
	my $html_ref = $highlighter->highlight($file_name, $content_ref);

	if ($html_ref && length($$html_ref) > $ArchZoom::Config{'inline_file_max_size'}) {
		substr($$html_ref, $ArchZoom::Config{'inline_file_max_size'}) = "";
		$$html_ref =~ s/<[^>]{0,100}$//;
		$$html_ref .= "</span></span></span></i></b>\n<hr>[Large file truncated]\n";
	}
	return $html_ref;
}

sub adjust_complex_name (\$$@) {
	my $name_ref = shift;
	my $regexp = shift;
	my @names = $$name_ref =~ /$regexp/;
	$$name_ref = pop @names if @names;
	@names = reverse @names;
	splice @names, 0, scalar @_, @_;
	return @names;
}

sub apply_url_params ($%) {
	my $url = shift;
	my %flags0 = (
		library  => $library_mode,
		charset  => $flags{charset},
		template => $flags{template},
		layout   => $flags{layout},
		color    => $flags{color},
		@_
	);

	while (my ($key, $value) = each %flags0) {
		next unless defined $value;

		if ($key =~ /^(library|debug|log)$/) {
			$url .= "?$key" if $value;
		} else {
			$url .= "?$key";
			$url .= "=$value" if $value ne "";
		}
	}

	return $url;
}

# ---- selfurl ---------------------------------------------------------------

sub _selfurl (;$) {
	my $pathinfo = shift || "";
	$pathinfo = "/$pathinfo";
	$pathinfo =~ s!/$!!;
	return $cgi->url . $pathinfo;
}

sub _selfurl_archive (@) {
	my $archive0 = shift || $archive;
	return _selfurl($archive0);
}

sub _selfurl_category (@) {
	my $category0 = shift || $category;
	my @names = adjust_complex_name($category0, '(.*)/(.*)', @_);
	return _selfurl_archive(@names) . "/$category0";
}

sub _selfurl_branch (@) {
	my $branch0 = shift;
	$branch0 = $branch unless defined $branch0;
	my @names = adjust_complex_name($branch0, '(?:(.*)/)?(.*?)--(.*)', @_);
	return _selfurl_category(@names) . "--$branch0";
}

sub _selfurl_version (@) {
	# optimization for large versions
	unless (@_) {
		return $::_selfurl_version if defined $::_selfurl_version;
		return $::_selfurl_version = _selfurl_branch() . "--$version";
	}
	my $version0 = shift;
	$version0 = $version unless defined $version0 && $version0 ne '';
	my @names = adjust_complex_name($version0, '(?:(.*)/)?(.*?)--(.*?)--(\d.*)', @_);
	return _selfurl_branch(@names) . "--$version0";
}

sub _selfurl_revision (@) {
	# optimization for large trees
	unless (@_) {
		return $::_selfurl_revision if defined $::_selfurl_revision;
		return $::_selfurl_revision = _selfurl_version() . "--$revision";
	}
	my $revision0 = shift || $revision;
	my @names = adjust_complex_name($revision0, '(?:(.*)/)?(.*?)--(.*?)--(.*?)--(.*)', @_);
	return _selfurl_version(@names) . "--$revision0";
}

sub _selfurl_tree (@) {
	my $filepath0 = @_? shift: "";
	# remove something/.. if any, then add / separator if needed
	$filepath0 =~ s~(?:^|/)(?:[^/\.]|[^/][^/\.]|[^/\.][^/]|[^/]{3,})/\.\.$~~;
	$filepath0 =~ s~^/*([^/])~/$1~;
	return _selfurl_revision(@_) . $filepath0;
}

sub selfurl (;$) {
	return apply_url_params(_selfurl(shift));
}

sub selfurl_globcats () {
	return apply_url_params(_selfurl_archive('*'));
}

sub selfurl_archive (@) {
	return apply_url_params(_selfurl_archive(@_));
}

sub selfurl_category (@) {
	return apply_url_params(_selfurl_category(@_));
}

sub selfurl_branch (@) {
	return apply_url_params(_selfurl_branch(@_));
}

sub selfurl_version (@) {
	return apply_url_params(_selfurl_version(@_));
}

sub selfurl_revision (@) {
	return apply_url_params(_selfurl_revision(@_));
}

sub selfurl_tree (@) {
	return apply_url_params(_selfurl_tree(@_));
}

sub selfurl_tree_type ($@) {
	my $is_dir = shift;
	return apply_url_params(
		_selfurl_tree(@_),
		$is_dir && exists $flags{expand}? (expand => $expand? "": 0): (),
	);
}

sub selfurl_abrowse (@) {
	return apply_url_params(
		_selfurl_archive(@_),
		expand => $ArchZoom::Config{auto_archive_expanding}? undef: "",
	);
}

sub selfurl_revisions_bunch ($) {
	my $number = shift || 0;

	my @base_url_params =
		$flags{history}? (_selfurl_tree($filepath), history => ""):
		$flags{ancestry}? (_selfurl_revision(), ancestry => ""):
		_selfurl_version();
	return &apply_url_params(
		@base_url_params,
		rbnum => $number,
		rbsize => $flags{rbsize},
		rbadd => $flags{rbadd},
		sumlen => $flags{sumlen},
		reverse => $flags{reverse},
	);
}

sub selfurl_ancestry ($@) {
	return apply_url_params(
		_selfurl_revision(@_),
		ancestry => "",
	);
}

sub selfurl_log (@) {
	return apply_url_params(
		_selfurl_revision(@_),
		log => 1,
	);
}

sub selfurl_diff ($$@) {
	my $filepath = shift;
	my $type = shift || 1;
	die "selfurl_diff: incorrect type ($type)\n" unless $type =~ /^-?[123]$/;
	$filepath .= ".diff" if $type <= 1;

	return apply_url_params(
		_selfurl_tree($filepath, @_),
		diff => $type != 1? $type: "",
	);
}

sub selfurl_history ($@) {
	my $filepath0 = shift || $filepath;

	return apply_url_params(
		_selfurl_tree($filepath0, @_),
		history => "",
	);
}

sub selfurl_annotate ($@) {
	my $filepath0 = shift || $filepath;

	return apply_url_params(
		_selfurl_tree($filepath0, @_),
		annotate => "",
	);
}

sub selfurl_this () {
	# memoize
	return $::_selfurl_this if defined $::_selfurl_this;
	return $::_selfurl_this = $cgi->url(info => 1);
}

sub selfurl_debug () {
	my $url = selfurl_this();
	return $url =~ /\?debug\b/? $url: "$url?debug";
}

sub selfurl_download () {
	my $url = selfurl_this();
	$url =~ s/\?annotate(=\w*)?\b//;
	return $download? $url: "$url?download";
}

sub selfurl_unexpand () {
	my $url = selfurl_this();
	$url =~ s/\?expand(=\w*)?\b//g;
	return $url;
}

sub selfurl_expand () {
	my $url = selfurl_unexpand();
	return "$url?expand";
}

sub selfurl_collapse () {
	my $url = selfurl_unexpand();
	return "$url?expand=0";
}

sub selfurl_untranspose () {
	my $url = selfurl_this();
	$url =~ s/\?transpose(=\w*)?\b//;
	return "$url?transpose=0";
}

sub selfurl_transpose () {
	my $url = selfurl_this();
	$url =~ s/\?transpose(=\w*)?\b//;
	return "$url?transpose";
}

# TODO: this may replace other functions above
sub selfurl_flag ($) {
	my $flag = shift;
	my $value = $flag =~ s/^!// ? "=0" : "";
	my $remove = $flag =~ s/^-//;
	my $url = selfurl_this();
	$url =~ s/\?$flag(=\w*)?\b//;
	return $url if $remove;
	return "$url?$flag$value";
}

sub selfurl_tarball_tree (;$@) {
	my $subdir = shift;
	my $url = _selfurl_revision(@_);
	$subdir = "" unless defined $subdir;
	$subdir =~ s!^\.$!!;
	$subdir =~ s!/+$!!;
	my $pathinfo = "";
	if ($subdir ne "") {
		$url =~ m!^.*/(.*)$! || die;
		my $cbvr = $1;
		my $sdir = $subdir;
		$sdir =~ s!/!,!g;
		$pathinfo = "/$subdir/$cbvr,$sdir.tar.gz";
	}
	return "$url.tar.gz$pathinfo";
}

sub selfurl_tarball_changeset (@) {
	my $url = _selfurl_revision(@_);
	return "$url.patches.tar.gz";
}

# ---- download --------------------------------------------------------------

sub genpage_download ($) {
	my $is_binary = substr($_[0], 0, 4096) =~ /\0/;
	my $mime_type = get_file_mime_type($filepath, $is_binary);
	print "Content-Type: $mime_type\n";
	print "Content-Length: " . length($_[0]) . "\n\n";
	print $_[0];
	return "";
}

# ---- archives --------------------------------------------------------------

sub genpage_archives () {
	my @group_data = ();
	my $cache = undef;
	$cache = Arch::SharedCache->new(%{$ArchZoom::Config{archives_cache}})
		if $ArchZoom::Config{archives_cache};

	my @expected_labels = ();
	for my $items (
		[ $session, qw(regular_archives regular Registered) ],
		[ $library, qw(library_archives library Library) ],
	) {
		my ($storage, $config_option, $key, $label) = @$items;
		next unless $ArchZoom::Config{$config_option};
		my $code = sub { $storage->archives };
		my $data = $cache && !$nocache? $cache->fetch_store($code, $key): &$code();
		$cache->store($key, $data) if $cache && $nocache;
		if (@$data) {
			push @group_data, [
				[ grep { is_archive_managed($_, 1) } @$data ],
				$label
			];
		} else {
			push @expected_labels, $label;
		}
	}

	my $error_message = "";
	unless (@group_data) {
		$error_message = @expected_labels
			? sprintf("No %s archives found", join(" or ", @expected_labels))
			: "Both kinds of archives are explicitly disabled";
	}

	my $is_multi_list = @group_data > 1
		&& @{$group_data[0][0]} && @{$group_data[1][0]}
		&& @{$group_data[0][0]} + @{$group_data[1][0]} > 20;
	return $template_engine->parse("archives", {
		archives_group_data => \@group_data,
		multi_list => $is_multi_list,
		error_message => $error_message,
	});
}

# ---- globcats --------------------------------------------------------------

sub genpage_globcats () {
	my $cache = undef;
	$cache = Arch::SharedCache->new(%{$ArchZoom::Config{globcats_cache}})
		if $ArchZoom::Config{globcats_cache};
	my $key = $library_mode? "library": "regular";
	my $code = sub {
		my $archives = $storage->archives;
		my %categories = ();
		foreach my $archive (@{$storage->archives}) {
			push @{$categories{$_} ||= []}, $archive
				foreach @{$storage->categories($archive)};
		}
		[ map { [ $_, $categories{$_} ] } sort keys %categories ];
	};
	my $data = $cache && !$nocache? $cache->fetch_store($code, $key): &$code();
	$cache->store($key, $data) if $cache && $nocache;

	return $template_engine->parse("globcats", {
		list_data => $data,
	});
}

# ---- categories ------------------------------------------------------------

sub genpage_categories () {
	return $template_engine->parse("categories", {
		list_data => $storage->categories,
	});
}

# ---- abrowse ---------------------------------------------------------------

sub genpage_abrowse () {
	my $cache = undef;
	$cache = Arch::SharedCache->new(%{$ArchZoom::Config{abrowse_cache}})
		if $ArchZoom::Config{abrowse_cache};
	my $key = $storage->working_name . ($library_mode? "|library": "");
	my $code = sub { $storage->expanded_archive_info };
	my $data = $cache && !$nocache? $cache->fetch_store($code, $key): &$code();
	$cache->store($key, $data) if $cache && $nocache;

	my $tchangeable = $ArchZoom::Config{abrowse_transpose_changeable};
	my $compact = defined $flags{compact}?
		$flags{compact}: $ArchZoom::Config{abrowse_compact};
	my $transpose = $tchangeable && defined $flags{transpose}?
		$flags{transpose}: $ArchZoom::Config{abrowse_transpose};

	if (defined $version_to_filter) {
		foreach (@$data) {
			my ($c, $cdata) = @$_;
			foreach (@$cdata) {
				my ($b, $bdata) = @$_;
				my $found;
				foreach (@$bdata) {
					my ($v) = @$_;
					$found = $_ if $v eq $version_to_filter;
					last if $found;
				}
				$_ = $found? [ $b, [ $found ] ]: undef;
			}
			@{$_->[1]} = grep { defined } @{$_->[1]};
			$_ = undef unless @{$_->[1]};
		}
		@$data = grep { defined } @$data;
	}

	if ($transpose) {
		my $new_data = [];
		foreach (@$data) {
			my ($c, $cdata) = @$_;
			my %vhash = ();
			foreach (@$cdata) {
				my ($b, $bdata) = @$_;
				foreach (@$bdata) {
					my ($v, @rest) = @$_;
					push @{$vhash{$v} ||= []}, [ $b, @rest ];
				}
			}
			push @$new_data, [ $c, [
				map { [ $_, $vhash{$_} ] } sort keys %vhash
			] ];
		}
		$data = $new_data;
	}

	return $template_engine->parse("abrowse", {
		archive_info => $data,
		compact   => $compact,
		transpose => $transpose,
		can_transpose   => $tchangeable && !$transpose,
		can_untranspose => $tchangeable && $transpose,
	});
}

# ---- branches --------------------------------------------------------------

sub genpage_branches () {
	return $template_engine->parse("branches", {
		list_data => $storage->branches,
	});
}

# ---- versions --------------------------------------------------------------

sub genpage_versions () {
	return $template_engine->parse("versions", {
		list_data => $storage->versions,
	});
}

# ---- revisions -------------------------------------------------------------

sub genpage_revisions () {
	my $cache = undef;
	$cache = Arch::SharedCache->new(%{$ArchZoom::Config{revisions_cache}})
		if $ArchZoom::Config{revisions_cache};
	my $key = $storage->working_name . ($library_mode? "|library": "");
	my $code = sub { $storage->get_revision_descs };
	my $data = $cache && !$nocache? $cache->fetch_store($code, $key): &$code();
	$cache->store($key, $data) if $cache && $nocache;

	my $revision_bunches_stash = get_revision_bunches_stash(
		$data,
		version => $storage->working_name,
	);

	return $template_engine->parse("revisions", {
		%$revision_bunches_stash
	});
}

# ---- log -------------------------------------------------------------------

sub genpage_log () {
	my $log = $library_tree_found? $library->get_log: $session->get_log;
	my $message = $log->get_message;
	my $headers = $log->get_headers;

	# an import log reports new-files, that have no corresponding diffs
	$headers->{imported_files} = delete $headers->{new_files}
		if $headers->{new_files} && $revision eq 'base-0';

	foreach (@{$headers->{modified_files}}, @{$headers->{new_files}}, @{$headers->{removed_files}}) {
		my $name = $_;
		$_ = [ $name, {} ];
	}

	if ($expand) {
		my $changeset = get_changeset();
		my $hint_asis = $download? 0: 1;
		my $type = 1;
		foreach (
			@{$headers->{modified_files}}, undef,
			@{$headers->{new_files}}, undef,
			@{$headers->{removed_files}}
		) {

			my $pair = $_;
			my $filepath = $pair->[0];
			++$type && next unless defined $_;
			my ($patch_content, $patch_file, $change_type, $asis) =
				get_processed_patch($changeset, $filepath, $type, $hint_asis);

			if ($download) {
				$patch_content = "=[$change_type file $filepath can't be included inline]="
					if $asis;
				$message .= "\n$patch_content\n" if $download;
			} else {
				my $filepath_stash = get_file_stash($filepath, $patch_file);
				my $html_ref = highlight_file_content($patch_file, $patch_content)
					if $asis;

				$filepath_stash->{content} = $asis && $html_ref? $$html_ref: $patch_content;
				$_->[0] = $filepath;  # bug somewhere
				$_->[1] = $filepath_stash;
			}
		}
	}

	return genpage_download($message) if $download;

	my $log_revision = "$archive/" . (delete $headers->{revision} || '*undefined*');
	die "Internal error: log revision '$full_revision' vus '$log_revision' mismatch\n"
		unless $full_revision eq $log_revision;
	$headers->{date} = standardize_date($headers->{date});
	$headers->{email} = 'no@email.defined';
	if ($headers->{creator} =~ /^(.*?)\s*<(.*)>$/) {
		($headers->{creator}, $headers->{email}) = ($1, $2);
	}

	return $template_engine->parse("log", {
		%$headers,
		can_expand    => !$expand,
		can_unexpand  => $expand,
		prev_revision => adjacent_revision($full_revision, -1),
		next_revision => adjacent_revision($full_revision, +1),
	});
}

# ---- ancestry --------------------------------------------------------------

sub genpage_ancestry () {
	my $tree = get_tree();

	my $revision_bunches_stash = get_revision_bunches_stash(
		[ reverse @{$tree->get_history_revision_descs} ],
		final_revision => $full_revision,
	);

	return $template_engine->parse("ancestry", {
		%$revision_bunches_stash,
	});
}

# ---- tree ------------------------------------------------------------------

sub genpage_tree () {
	get_tree();
	my $tree_file = "$tree_root/$filepath";

	if (-f $tree_file) {
		my $content = load_file($tree_file);

		return genpage_download($content) if $download;

		my $filepath_stash = get_file_stash($filepath, $tree_file);
		my $html_ref = highlight_file_content($tree_file, $content);

		return $template_engine->parse("file", {
			%$filepath_stash,
			content   => $html_ref? $$html_ref: $content,
		});
	}
	die "No file or directory $filepath in revision $full_revision\n"
		unless -d $tree_file;

	my $collapse = defined $flags{expand} && !$expand;
	my $limit = $collapse? 0: $expand?
		$ArchZoom::Config{explicit_expanded_dir_max_files}:
		$ArchZoom::Config{implicit_expanded_dir_max_files};

	my $dir_data = expanded_dir_data($tree_file, $limit);
	shift @$dir_data if $filepath eq ".";

	my $can_expand     = 0;
	my $can_collapse   = 0;
	foreach (@$dir_data) {
		$can_expand   = 1 if !$expand   && $_->[1] == -1;
		$can_collapse = 1 if !$collapse && $_->[1] == 1;
		last if ($can_expand || $expand) && ($can_collapse || $collapse);
	}

	my $can_unexpand   = !$can_expand && $expand;
	my $can_uncollapse = !$can_collapse && $collapse;
	$can_collapse = $can_uncollapse = 0
		if $ArchZoom::Config{implicit_expanded_dir_max_files} <= 0;
	$can_expand = $can_unexpand = 0
		if $ArchZoom::Config{implicit_expanded_dir_max_files}
		>= $ArchZoom::Config{explicit_expanded_dir_max_files};

	return $template_engine->parse("tree", {
		list_data      => $dir_data,
		dir            => $filepath eq ""? ".": $filepath,
		collapse       => $collapse,
		can_expand     => $can_expand,
		can_collapse   => $can_collapse,
		can_unexpand   => $can_unexpand,
		can_uncollapse => $can_uncollapse,
		prev_revision  => adjacent_revision($full_revision, -1),
		next_revision  => adjacent_revision($full_revision, +1),
	});
}

# ---- diff ------------------------------------------------------------------

sub genpage_diff () {
	my $changeset = get_changeset();

	my $type = $flags{diff} || die "genpage_diff: no diff flag\n";
	$type =~ s/^(-?)([123])$/$2/ || die "genpage_diff: unknown diff type ($type)\n";
	my ($patch_content, $patch_file, $change_type, $asis) =
		get_processed_patch($changeset, $filepath, $type, !$1);
	return genpage_download($patch_content) if $download;

	my $filepath_stash = get_file_stash($filepath, $patch_file);
	my $html_ref = highlight_file_content($patch_file, $patch_content)
		if $asis;
	my $prev_revision = $changeset->ancestor;

	return $template_engine->parse("diff", {
		%$filepath_stash,
		content       => $asis && $html_ref? $$html_ref: $patch_content,
		prev_revision => $prev_revision,
		change_type   => $change_type,
	});
}

# ---- history ---------------------------------------------------------------

sub genpage_history () {
	my $tree = get_tree();
	my $tree_file = "$tree_root/$filepath";
	my $filepath_stash = get_file_stash($filepath, $tree_file);
	my $file_or_dir_label = $filepath_stash->{is_dir}? "dir": "file";

	my $revision_bunches_stash = get_revision_bunches_stash(
		[ reverse @{$tree->get_history_revision_descs($filepath)} ],
		final_revision => $full_revision,
		final_filepath => $filepath,
	);

	return $template_engine->parse("history", {
		%$revision_bunches_stash,
		%$filepath_stash,
		file_or_dir_label => $file_or_dir_label,
	});
}

# ---- annotate --------------------------------------------------------------

sub genpage_annotate () {
	my $tree = get_tree();
	my $tree_file = "$tree_root/$filepath";
	my $filepath_stash = get_file_stash($filepath, $tree_file);

	die "The 'annotate' view is disabled in configuration. Sorry.\n"
		unless $ArchZoom::Config{annotate_enabled};
	die "Can't annotate binary file or symlink or directory $filepath\n"
		unless $filepath_stash->{can_annotate};

	my %args = (group => 1, highlight => 1);
	my ($lines, $line_rd_indexes, $revision_descs) =
		$tree->get_annotate_revision_descs($filepath, %args);
	my @line_groups = map {
		my $index = shift @$line_rd_indexes;
		my $revision_desc = $revision_descs->[$index];
		$revision_desc->{global_count} = $index + 1;
		[ join("\n", @$_) . "\n", $index + 1, $revision_desc ]
	} @$lines;

	my $revision_bunches_stash = get_revision_bunches_stash(
		$revision_descs,
		final_revision => $full_revision,
		final_filepath => $filepath,
	);

	return $template_engine->parse("annotate", {
		%$revision_bunches_stash,
		%$filepath_stash,
		line_groups => \@line_groups,
	});
}

# ---- tarball ---------------------------------------------------------------

sub genpage_tarball () {
	die "Tarball downloading is disabled\n"
		unless $ArchZoom::Config{tarball_downloading};
	my $tarball = Arch::Tarball->new;
	my $fh;

	if ($is_changeset_tarball) {
		my $changeset = get_changeset();
		my $revision = $changeset->{revision};
		$revision =~ s!.*?/!!;
		$fh = $tarball->create(
			dir => $changeset->{dir},
			base_name => $revision . ".patches",
			pipe => 1,
		);
	} else {
		get_tree();
		my ($subdir, $base_name);
		# support cbvr.tar.gz and cbvr.tar.gz/dir/cbvr,dir.tar.gz
		if ($filepath eq ".") {
			$full_revision =~ s!^.*/(.*)$!! || die;
			$subdir = ".";
			$base_name = $1;
		} elsif ($filepath =~ m!(.*)/(.*)\.tar\.gz$!) {
			$subdir = $1;
			$base_name = $2;
		} else {
			die "File path $filepath is not supported for tarball\n";
		}
		my $tree_dir = "$tree_root/$subdir";
		die "No subdir ($subdir) in tree of revision $full_revision\n"
			unless -d $tree_dir;
		$fh = $tarball->create(
			dir => $tree_dir,
			base_name => $base_name,
			pipe => 1,
		);
	}

	print "Content-Type: application/x-tar\n";
	print "Content-Transfer-Encoding: gzip\n\n";
	my $chunk = "";
	print $chunk while read($fh, $chunk, 1024);
	close($fh);

	return "";
}

# ---- main ------------------------------------------------------------------

my $genpage_func = "genpage_" . $page_name;
no strict 'refs';
print &$genpage_func;
