# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package ArchZoom::Util;

use Exporter;
use vars qw(@ISA @EXPORT);
@ISA = qw(Exporter);
@EXPORT = qw(htmlize urlize break expanded_dir_data);

sub htmlize ($) {
	my $str = shift;
	$str =~ s/&/&amp;/sg;
	$str =~ s/</&lt;/sg;
	$str =~ s/>/&gt;/sg;
	return $str;
}

sub urlize ($) {
	my $str = shift;
	$str =~ s/([^\w\/\.,:;=\@#\$\?-])/sprintf("%%%02x", ord($1))/seg;
	return $str;
}

sub break ($) {
	my $str = shift;
	$str =~ s/\n/<br>\n/g;
	return $str;
}

sub expanded_dir_data ($$) {
	my $dir = shift || die "No dir parameter";
	my $limit = shift || 0;
	require POSIX;

	my @files_details = ([ '.', -1 ]);
	my $num_unexpanded_subdirs = 1;
	my $first_overdraw_subdir = '';
	my $last_index = 0;

	while ($num_unexpanded_subdirs > 0) {
		# find the first unexpanded dir
		my $subdir;
		my $round = 0;
		while ($round <= 1) {
			if ($last_index >= @files_details) {
				$last_index = 0;
				$round++;
			}
			my $file_details = $files_details[$last_index];
			$last_index++;
			if ($file_details->[1] == -1) {
				$subdir = $file_details->[0];
				last;
			}
		}
		die "Endless subdir loop" unless $round <= 1;
		die "Internal, no subdir" unless defined $subdir;
		last if $subdir ne '.' && @files_details > $limit;
		last if $first_overdraw_subdir eq $subdir;

		# open subdir: create a list of file names
		my $full_subdir = "$dir/$subdir";
		opendir DIR, $full_subdir or die "Can't opendir $full_subdir: $!\n";
		my @new_files = ();
		my @new_parent = ();
		push @new_parent, ".." if @files_details == 1;
		my $overdraw = 0;
		while (my $file = readdir DIR) {
			next if $file eq '.' || $file eq '..';
			next if $file =~ /^,,/;
			next if $file eq '{arch}' && $subdir eq '.';
			next if $file eq '.arch-ids' && !$ArchZoom::Config{'arch_ids_listing'};
			if ($subdir ne '.' && @files_details + @new_files > $limit) {
				$overdraw = 1;
				@new_files = ();
				last;
			}
			push @new_files, $file;
		}
		closedir DIR;

		# mark the processed unexpanded directory as expanded
		if ($overdraw) {
			$first_overdraw_subdir = $subdir if $first_overdraw_subdir eq '';
		} else {
			$files_details[$last_index - 1][1] = 1;
			$num_unexpanded_subdirs--;
		}

		# stat and add all subdir files
		foreach (@new_parent, sort @new_files) {
			my $name = $subdir eq '.'? $_: "$subdir/$_";
			my $full_file = "$dir/$name";

			my (undef, undef, $mode, undef, undef, undef, undef,
				$size, undef, $mtime) = lstat($full_file);

			my $is_symlink = -l _;
			my $is_dir = -d _;
			my $type = $is_dir? -1: 0;
			my $symlink = $is_symlink? readlink $full_file: undef;

			my @modes = qw(--- --x -w- -wx r-- r-x rw- rwx);
			$mode = join('', map { $modes[$_] } (split('', sprintf("%o", $mode)))[-3, -2, -1]);
			$mode = ($is_symlink? $is_dir? "L": "l": $is_dir? "d": "-") . $mode;

			my $date = POSIX::strftime('%Y-%m-%d %H:%M:%S %Z', localtime($mtime));

			my $accessible = -r _;
			$accessible = 0 if $is_dir && !-x _;
			my @parts = split('/', $name);
			my $level = @parts - 1;

			if ($is_dir) {
				if ($name eq '..') {
					$type = -2;
				} else {
					$num_unexpanded_subdirs++;
				}
			}

			splice @files_details, $last_index, 0, [
				$name, $type, $mode, $size, $date, $accessible, $symlink, $level,
			];

			$last_index++;
		}
	}

	# exclude the dummy '.' directory entry
	shift @files_details;
#	unshift @files_details, [ '..', -2, '', '', '', 1, undef, 0 ];
	return \@files_details;
}

1;
