/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: route_group.h,v 1.3 2004/11/29 03:04:46 pauld Exp $
*/

#ifndef __ardour_route_group_h__
#define __ardour_route_group_h__

#include <list>
#include <string>
#include <stdint.h>
#include <sigc++/signal_system.h>
#include <ardour/stateful.h>

using std::string;
using std::list;

namespace ARDOUR {

class Route;
class AudioTrack;

class RouteGroup : public Stateful, public SigC::Object {
  public:
    enum Flag {
	    Relative = 0x1,
	    Active = 0x2,
	    Hidden = 0x4,
    };

    RouteGroup(const string &n, Flag f = Flag(0)) : _name (n), _flags (f) {}

    const string& name() { return _name; }

    bool is_active () const { return _flags & Active; }
    bool is_relative () const { return _flags & Relative; }
    bool is_hidden () const { return _flags & Hidden; }
    bool empty() const {return routes.empty();}

    int size() { return routes.size();}
    ARDOUR::Route * first () const { return *routes.begin();}

    void set_active (bool yn, void *src);
    void set_relative (bool yn, void *src);
    void set_hidden (bool yn, void *src);


    int add (Route *);

    int remove (Route *);

    template<class T> void apply (void (Route::*func)(T, void *), T val, void *src) {
	    for (list<Route *>::iterator i = routes.begin(); i != routes.end(); i++) {
		    ((*i)->*func)(val, this);
	    }
    }

    template<class T> void foreach_route (T *obj, void (T::*func)(Route&)) {
	    for (list<Route *>::iterator i = routes.begin(); i != routes.end(); i++) {
		    (obj->*func)(**i);
	    }
    }

    /* to use these, #include <ardour/route_group_specialized.h> */

    template<class T> void apply (void (AudioTrack::*func)(T, void *), T val, void *src);

    void clear () {
	    routes.clear ();
	    changed();
    }

    const list<Route*>& route_list() { return routes; }
    
    SigC::Signal0<void> changed;
    SigC::Signal1<void,void*> FlagsChanged;

    XMLNode& get_state (void);

    int set_state (const XMLNode&);

 private:
    list<Route *> routes;
    string _name;
    uint32_t _flags;

    void remove_when_going_away (Route*);
};

} /* namespace */

#endif /* __ardour_route_group_h__ */
