/* $Id: ArkSpline.cpp,v 1.6 2002/10/28 22:31:17 mrq Exp $
** 
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <Ark/ArkSpline.h>
#include <Ark/ArkTimer.h>


#include <math.h>


namespace Ark
{

   // Weight of the nth point at time t (t in [0;1])
   // for a 4 points cubic spline.
   static scalar bspline_factor(int n, scalar t)
   {
      switch(n)
      {
	 case 0:
	    return (powf(1.f-t, 3.f))/6.f;
	 case 1:
	    return (3.f*powf(t, 3.f)-6.f*powf(t, 2.f)+4.f)/6.f;
	 case 2:
	    return (-3.f*powf(t, 3.f)+3.f*powf(t, 2.f)+3.f*t+1.f)/6.f;
	 case 3:
	    return powf(t, 3.f)/6.f;
      }
      
      return 0.f;
   }

   static inline SplinePoint bspline_pointat (const Spline &spline, int point)
   {
      if (point < 0) return spline.front();
      if ((size_t)point >= spline.size()) return spline.back();

      return spline[point];
   }

   Spline::Spline()
   {
      m_CurPoint = 0;
   }

   bool
   Spline::Evaluate (scalar t, Vector3 &pos)
   {
      // find the part corresponding to t.
      size_t high = m_CurPoint;

      if (m_Interp.empty()) return false;
      if (high >= m_Interp.size() || m_Interp.back().first < t)
      {
	 m_CurPoint = m_Interp.size();
	 pos = m_Interp.back().second; return true;
      }
      
      while(m_Interp[high].first < t)
	 high++;

      m_CurPoint = high;

      if (high == 0 || size() == 1)
      {
	 pos = m_Interp.front().second; return true;
      }

      scalar w = ((t - m_Interp[high-1].first) /
		  (m_Interp[high].first - m_Interp[high-1].first));
      pos = w * m_Interp[high].second + (1.0f-w) * m_Interp[high-1].second;
      return true;
   }

   bool Spline::IsGoalReached()
   {
      return (m_CurPoint >= (int)m_Interp.size());
   }

   void
   Spline::Compute()
   {
      m_Interp.resize(0);
      m_CurPoint = 0;
      scalar cur_t;

      if (size() == 0) return;
      if (size() == 1) { m_Interp.push_back((*this)[0]); return; }

      cur_t = front().first;
   
#define FIXME_HARDCODED_SPEED 1.5f
      for (size_t i = 2; i <= size()+1; i++)
      {
	 for (scalar t = 0.0f; t <= 1.001f; t+=0.1f)
	 {       
            // "Draw" the curve i-3, i-2, i-1, i
	    Vector3 p;
	    for (int f = 0; f < 4; f++)
	       p += (bspline_factor(f, t) *
		     bspline_pointat(*this, i-(3-f)).second);

	    m_Interp.push_back (SplinePoint(cur_t,p));

	    if (m_Interp.size() > 1)
	    {
	       cur_t += ((m_Interp.back().second -
			  m_Interp[m_Interp.size()-2].second).GetMagnitude())/
		  FIXME_HARDCODED_SPEED;
	    }
	 } 
      }
   }
}
