/*
 * Copyright (c) 1995 - 2002 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <nnpfs/nnpfs_locl.h>

RCSID("$Id: nnpfs_vfsops-freebsd.c,v 1.28 2003/06/02 18:26:43 lha Exp $");

#include <nnpfs/nnpfs_common.h>
#include <nnpfs/nnpfs_message.h>
#include <nnpfs/nnpfs_fs.h>
#include <nnpfs/nnpfs_dev.h>
#include <nnpfs/nnpfs_deb.h>
#include <nnpfs/nnpfs_vfsops.h>
#include <nnpfs/nnpfs_vfsops-bsd.h>
#include <nnpfs/nnpfs_vnodeops.h>
#include <nnpfs/nnpfs_node.h>

static vop_t **nnpfs_dead_vnodeop_p;

#if __FreeBSD_version >= 500043
const char *VT_AFS = "afs";
static const char *VT_NON = "dead-afs";
#endif

int
nnpfs_make_dead_vnode(struct mount *mp, struct vnode **vpp)
{
    NNPFSDEB(XDEBNODE, ("make_dead_vnode mp = %lx\n",
		      (unsigned long)mp));

    return getnewvnode(VT_NON, mp, nnpfs_dead_vnodeop_p, vpp);
}

static struct vnodeopv_entry_desc nnpfs_dead_vnodeop_entries[] = {
    {&vop_default_desc, (vop_t *) vop_defaultop},
    {&vop_lookup_desc,	(vop_t *) nnpfs_dead_lookup},
    {&vop_reclaim_desc, (vop_t *) nnpfs_returnzero},
    {&vop_lock_desc,	(vop_t *) vop_nolock},
    {&vop_unlock_desc,	(vop_t *) vop_nounlock},
    {&vop_islocked_desc,(vop_t *) vop_noislocked},
    {NULL, NULL}};

static struct vnodeopv_desc nnpfs_dead_vnodeop_opv_desc =
{&nnpfs_dead_vnodeop_p, nnpfs_dead_vnodeop_entries};

VNODEOP_SET(nnpfs_dead_vnodeop_opv_desc);

extern struct vnodeopv_desc nnpfs_vnodeop_opv_desc;

static int
nnpfs_init(struct vfsconf *conf)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_init\n"));
#if defined(HAVE_KERNEL_VFS_OPV_INIT)
    vfs_opv_init(&nnpfs_vnodeop_opv_desc);
    vfs_opv_init(&nnpfs_dead_vnodeop_opv_desc);
#elif defined(HAVE_KERNEL_VFS_ADD_VNODEOPS) && !defined(KLD_MODULE)
    vfs_add_vnodeops (&nnpfs_vnodeop_opv_desc);
    vfs_add_vnodeops (&nnpfs_dead_vnodeop_opv_desc);
#endif
    return 0;
}

static int
nnpfs_uninit(struct vfsconf *vfc)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_uninit\n"));
    return 0;
}

#if __FreeBSD_version >= 500019
#define VFS_MOUNT_TAKES_KERNEL_POINTERS 1
#endif

#if VFS_MOUNT_TAKES_KERNEL_POINTERS

/* new mount sends in kernel pointers */

static int
nnpfs_mount_freebsd(struct mount *mp, char *path, caddr_t data,
		  struct nameidata *ndp, d_thread_t *p)
{
    return nnpfs_mount_common_sys (mp, path, data, ndp, p);
}

#else

static int
nnpfs_mount_freebsd(struct mount *mp, char *path, caddr_t data,
		  struct nameidata *ndp, d_thread_t *p)
{
    return nnpfs_mount_common (mp, path, data, ndp, p);
}

#endif

#if __FreeBSD_version >= 500032
static int
nnpfs_vget_freebsd(struct mount *mp,
	 ino_t ino,
	 int flags,
	 struct vnode ** vpp)
{
    return nnpfs_vget(mp, ino, vpp);
}
#else
#define nnpfs_vget_freebsd nnpfs_vget
#endif


struct vfsops nnpfs_vfsops = {
    nnpfs_mount_freebsd,
    nnpfs_start,
    nnpfs_unmount,
    nnpfs_root,
    nnpfs_quotactl,
    nnpfs_statfs,
    nnpfs_sync,
    nnpfs_vget_freebsd,
    nnpfs_fhtovp,
#ifdef HAVE_STRUCT_VFSOPS_VFS_CHECKEXP
    nnpfs_checkexp,
#endif
    nnpfs_vptofh,
    nnpfs_init
};

static struct vfsconf nnpfs_vfc = {
    &nnpfs_vfsops,
    "nnpfs",
    0,
    0,
    0
};

#ifndef HAVE_KERNEL_VFS_REGISTER

static int
vfs_register (struct vfsconf *vfs)
{
    int i;

    for (i = 0; i < MOUNT_MAXTYPE; i++)
	if (strcmp(vfsconf[i]->vfc_name, vfs->vfc_name) == 0)
	    return EEXIST;

    for (i = MOUNT_MAXTYPE - 1; i >= 0; --i)
	if (vfsconf[i] == &void_vfsconf)
	    break;

    if (i < 0) {
	NNPFSDEB(XDEBVFOPS, ("failed to find free VFS slot\n"));
	return EINVAL;
    }

    vfs->vfc_index = i;
    vfsconf[i] = vfs;

    vfssw[i] = vfs->vfc_vfsops;
    (*(vfssw[i]->vfs_init)) ();
    return 0;
}

static int
vfs_unregister (struct vfsconf *vfs)
{
    int i = vfs->vfc_index;

    if (vfs->vfc_refcount)
	return EBUSY;

    vfsconf[i] = &void_vfsconf;
    vfssw[i]   = NULL;
    return 0;
}

#endif

#if KLD_MODULE

int
nnpfs_install_filesys(void)
{
    return 0;
}

int
nnpfs_uninstall_filesys(void)
{
    return 0;
}

int
nnpfs_stat_filesys (void)
{
    return 0;
}

#else

int
nnpfs_install_filesys(void)
{
    return vfs_register(&nnpfs_vfc);
}

int
nnpfs_uninstall_filesys(void)
{
    return vfs_unregister(&nnpfs_vfc);
}

int
nnpfs_stat_filesys (void)
{
    return 0;
}

#endif /* KLD_MODULE */
