/*
 * autopilot - Launches a given set of programs automaticly for the palm
 *
 * Copyright (C) 2000 Leigh Morresi (leigh@signal-x.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ********************************************************************
 *
 * Please cruise the man page (autopilot) or the README for more info
 * about using this app.
 * 
 * It was developed under:
 * - Linux kernel 2.2.16
 * - gcc version 2.7.2.3
 * - no-name onboard serial card (16550 UART)
 * - PalmPilot (USRobotics)
 * - a little help from the statserial-1.1 sources.
 *
 * Leigh Morresi (leigh@signal-x.com)
 */

#ifndef PILOTDEVICE
#define PILOTDEVICE "/dev/palm"
#endif
 
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <fcntl.h>

/* global */
char 	*device = PALMDEVICE;   /* default device if none specified on command line */
char		*command;
int		defaultDelay = 2;
int		line = TIOCM_DSR;

/* print command usage and exit */
void usage()
{
  fprintf(stderr, "usage: autopilot [-n seconds] -h -c <command>\n");
  fprintf(stderr, "  -n interval between scan for pilot in seconds. (default %i)\n",defaultDelay);
  fprintf(stderr, "  -c command to run on active ie 'pilot-mail -p /dev/palm'\n");
  fprintf(stderr, "  -h check the RST line (hot sync button) instead.\n");
  exit(1);
}

/* handle command line options */
void parse_args(int argc, char **argv)
{
  const char     *flags = "f?n:c:p:h";
  int             c;
  
  while ((c = getopt(argc, argv, flags)) != EOF) {
    switch (c) {
    case 'n': /* roger wilco */
          printf("seconds: %s\n",argv[optind-1]);
      break;
    case 'c': /* command to run with exec */
          command = argv[optind-1];
      break;
    case 'h': 
          line=TIOCM_CTS;
      break;
    case 'p':
          device=argv[optind-1];
          break;
    case '?':
          usage();
      break;
    }
  }
    
}

/* main program */  
int main(int argc, char *argv[])
{
  int fd;                       /* for serial device */
  int status;                   /* status of system calls */
  unsigned int old_status = 1;  /* value of previous call, presume it was in */
  unsigned int new_status = 0;  /* value of previous call */

  unsigned int arg;             /* value returned by ioctl */

  /* parse command line arguments */
  parse_args(argc, argv);
  /* looking for $PILOTPORT */
  if (getenv("PILOTPORT") != NULL)
     device=getenv("PILOTPORT");
  /* open port */
  printf("autopilot: opening %s\n",device);
  
  fd = open(device, O_RDONLY);  
  if (fd == -1) {
    char s[255];
    snprintf(s, 255,"autopilot: can't open device, check your symlink `%s'", device);
    perror(s);
    exit(1);
  }
  if (!command) {
      fprintf(stderr,"You must specify a command to run with the -c option.\n");
      usage();
      exit(1);
  }
  
  printf("autopilot: launching..\n\n");

  /* loop forever */
  if (fork() == 0) {
    for (;;) {
        /* get serial status info */  
	status = ioctl(fd, TIOCMGET, &arg);
        if (status != 0) {
	  perror("autopilot: TIOCMGET failed");
          exit(1);
        }
	
	new_status=!!(arg & line);

	/* check if the thing has been inserted */	
	/* exec an app to handle that */
	if( (new_status != old_status) && (old_status == 0) ) {
	    if (system(command) == -1) {
		fprintf(stderr,"autopilot: Error running %s, please check it and try again\n",command);
		exit(1);
	    }
	}
	
	old_status=new_status;
        sleep(defaultDelay);
    }
  }
      
  return 0;
}
