#!/usr/bin/php4 -q
<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    axyl-security-update.php                                */
/* Author:      Paul Waite                                              */
/* Description: Security update for Axyl, incoroporating the change     */
/*              in directory structure to the new model.                */
/*                                                                      */
/* This update implements a re-organisation of Axyl, and the websites   */
/* which are installed from it. We now have a different website         */
/* hierarchy featuring more restrictive permissions on files and        */
/* directories where we only allow webserver write access to the 'var'  */
/* hierarchy.                                                           */
/*                                                                      */
/* This script is designed to be run from INSIDE the root directory of  */
/* the Axyl website that you want to convert to the new model. You      */
/* can leave this script in the Axyl installation tree and just run it  */
/* remotely by doing something like the following, as root:             */
/*                                                                      */
/*   cd /your/axyl/website/root                                         */
/*   /path/to/axyl/scripts/update/axyl-security-update.php              */
/*                                                                      */
/* NOTE 1: A pre-requisite for running this script is that you have     */
/* Axyl installed in its new form somewhere on your system. You can     */
/* install it either from debian package, or CVS. If you have done the  */
/* latter, then make sure you have created a valid /etc/axyl/axyl.conf, */
/* by running install/setup-axyl.sh.                                    */
/*                                                                      */
/* NOTE 2: On servers which are running Php < 4.2.0 the php4-cgi        */
/* interpreter will not allow you to run the script by typing the path  */
/* to it as described above. Instead, copy the script to the website    */
/* root of te website to convert and run it from there.                 */
/*                                                                      */
/* ******************************************************************** */
// STDIN File Handle..
$FH = fopen("php://stdin", "r");

// ----------------------------------------------------------------------
// AXYL LIBRARY
// The 'old' configuration has this as either a soft-link to an Axyl
// library somewhere, or a physical directory. Both of these options
// are now deprecated. Library modules are reference via the Apache
// 'include_path' (see httpd.conf) and there is a physical 'lib'
// directory placed in the website root, with two soft-links to the
// Axyl installation 'js' and 'img' directories. Since this script uses
// the old Axyl library, we will be applying the new-look right at the
// end prior to finishing.
//
// First though, find the Axyl library - failure is not an option..
$LIBDIR_POSSIBLES = array(".", "lib", "library", "axyl");
if (!isset($LIBDIR)) {
  foreach ($LIBDIR_POSSIBLES as $testdir) {
    if (file_exists("$testdir/application-defs.php")) {
      $LIBDIR = $testdir;
      break;
    }
  }
}
if (!isset($LIBDIR)) {
  die(
    "\n\nERROR:\nCould not locate your Axyl library './lib/'. Is this\n"
  . "really an Axyl website which has not had the security update applied\n"
  . "to it?\n\n Check this and fix up whatever necessary before re-running\n"
  . "this script."
  );
}
// Set everything up. Note this uses the old form of lib include
// since we are converting an old site here..
include_once("application.php");

// ----------------------------------------------------------------------
// LOGGING
$LOGFILE = "./axyl-security-update.log";
$LOG = new outputfile($LOGFILE);
if (!$LOG->opened) {
  echo "Could not open logfile for writing.";
  exit;
}
else {
  $stamp = timestamp_to_displaydate(NICE_DATETIME, time());
  logit("AXYL SECURITY UPDATE");
  logit("Converting " . APP_NAME . " (" . APP_PREFIX . ")");
  logit("$stamp\n");
}

// ----------------------------------------------------------------------
// FUNCTIONS
// ----------------------------------------------------------------------

// A utility function which makes sure a given directory path
// is made, using mkdir as required. This function assumes
// that permissions are all as needed.
function makepath($path) {
  if (substr($path, 0, 2) != "./" && substr($path, 0, 1) != "/") {
    $path = "./" . $path;
  }
  $result = true;
  clearstatcache();
  if (!is_dir($path)) {
    $dirs = explode("/", $path);
    $testpath = "";
    foreach ($dirs as $dir) {
      if ($dir == "") $testpath = "/";
      elseif ($dir == ".") $testpath = realpath($dir);
      else $testpath .= "/$dir";
      if (!is_dir($testpath)) {
        if (!mkdir($testpath, 0775)) {
          $result = false;
          break;
        }
      }
    }
  }
  return $result;
}

// Function to move a directory. The source directory must exist and
// the destination directory must NOT exist for this to succeed.
function move($sourcedir, $destdir) {
  $result = false;
  if (is_dir($sourcedir) && !is_dir($destdir)) {
    exec("mv $sourcedir $destdir", $output, $retval);
    $result = ($retval == 0);
  }
  return $result;
} // move

function abort($msg) {
  global $LOG;
  $msg = "ERROR: ABORTING SECURITY UPDATE!\n\n$msg\n";
  logit($msg);
  $LOG->closefile();
  exit;
}

function logit($msg) {
  global $LOG;
  $LOG->writeln($msg);
  echo "$msg\n";
}

function connect_pgdb($dbname, $dbuser="", $dbpass="", $dbhost="", $dbport=0) {
  $connstr = " dbname=" . $dbname;
  if ($dbuser != "") $connstr .= " user=" . $dbuser;
  if ($dbpass != "") $connstr .= " password=" . $dbpass;
  if ($dbhost != "") $connstr .= " host=" . $dbhost;
  if ($dbport != 0 ) $connstr .= " port=" . $dbport;
  $connstr = trim($connstr);
  return pg_connect($connstr);
}

function exec_pg_command($cmd, $dbuser="", $dbhost="", $dbport=0, $parms="") {

  $execstring = "$cmd -U $dbuser";
  if ( trim($dbhost) != "" ) {
    $execstring .= " -h $dbhost";
  }
  if ( $dbport != 0 ) {
    $execstring .= " -p $dbport";
  }
  if ( trim($parms) != "" ) {
    $execstring .= " $parms";
  }
  exec($execstring, $dummy, $returnvar);

  return $returnvar;
}

function create_new_field($tablename, $fieldname, $fieldtype) {
  $res = false;
  logit(" -- $tablename create new column '$fieldname ($fieldtype)'..");
  $q  = "SELECT pga.attname as fieldname";
  $q .= "  FROM pg_class pgc,pg_attribute pga";
  $q .= " WHERE pgc.relname='$tablename'";
  $q .= "   AND pga.attrelid=pgc.oid";
  $q .= "   AND pga.attname='$fieldname'";
  $ax = dbrecordset($q);
  if ($ax->hasdata) {
    logit(" -- $tablename.$fieldname already present - good");
    $res = true;
  }
  else {
    $res = dbcommand("ALTER TABLE $tablename ADD COLUMN $fieldname $fieldtype");
    if ($res) {
      logit(" -- created new column $tablename.$fieldname");
    }
    else {
      logit(" -- ERROR: failed to create new column $tablename.$fieldname");
    }
  }
  return $res;
} // create_new_field

// ----------------------------------------------------------------------
// Announce..
echo "Important: Please do not run this script on an Axyl website without\n"
   . "making a complete backup of it AND the database beforehand. If this\n"
   . "script aborts due to it detecting an error part-way through, then\n"
   . "your website may be in an indeterminate state, and will have to be\n"
   . "re-created (including database) from that backup before correcting\n"
   . "the problem and re-running this script.\n"
   . "\n";
echo "Continue? Y or N [N] :";
$ans = rtrim(fgets($FH, 256));
if ($ans != "y" && $ans != "Y") {
  abort("User abort.");
}

// ----------------------------------------------------------------------

// Check we are root..
$uid = shell_exec("id -u");
if ($uid != 0) {
  abort("You must be root to run this script.");
}

// Clear filesyste, stat cache..
clearstatcache();

# Are we in an Axyl root directory?
if (!file_exists("./application.xml")) {
  abort(
    "This script is designed to be run from the root directory of\n"
  . "an Axyl website, yet there is no application.xml file here."
  );
}

# Have we been this way before?
// Check not already run this..
if (is_dir("./var")) {
  abort(
    "A 'var' directory already exists! Have you run the script on\n"
  . "this Axyl website before? If you haven't, then you will have to\n"
  . "edit this script and bypass this check, or remove that directory."
  );
} else {
  // The writeable ./var directory hierarchy..
  if (!makepath("./var")) {
    abort(" -- there was a problem creating directory ./var");
  }
  else {
    logit(" -- created new directory ./var");
  }
}

// A directory which Muct be there..
if (!isset($TEMPLATESDIR) || $TEMPLATESDIR == "") {
  $TEMPLATESDIR = "templates";
}
if (!is_dir($TEMPLATESDIR)) {
  abort(
    "No 'templates' sub-directory found ($TEMPLATESDIR). Is this really\n"
  . "an Axyl website?"
  );
}

// ----------------------------------------------------------------------
// Various sub-directories which might or might not be there..
$catalog_dir_was_missing = false;
if (!isset($CATALOGDIR) || $CATALOGDIR == "") {
  $CATALOGDIR = "catalog";
}
if (!is_dir($CATALOGDIR)) {
  logit("No 'catalog' sub-directory found ($CATALOGDIR).. creating it.");
  $catalog_dir_was_missing = true;
  mkdir($CATALOGDIR, 0775);
}
$cache_dir_was_missing = false;
if (!isset($CACHEDIR) || $CACHEDIR == "") {
  $CACHEDIR = "cache";
}
if (!is_dir($CACHEDIR)) {
  logit("No 'cache' sub-directory found ($CACHEDIR).. creating it.");
  $cache_dir_was_missing = true;
  mkdir($CACHEDIR, 0775);
}
$inc_dir_was_missing = false;
if (!isset($INCDIR) || $INCDIR == "") {
  $INCDIR = "inc";
}
if (!is_dir($INCDIR)) {
  logit("No 'inc' sub-directory found ($INCDIR).. creating it.");
  $inc_dir_was_missing = true;
  mkdir($INCDIR, 0775);
}
// ----------------------------------------------------------------------
// Configuration file..
$AXUSER = "";
$AXYL_HOME = "";
$AXYL_DATA = "";
$LUCENE_PORT = 0;
$LUCENE_INSTALLED = false;

$CONF_FILE = "/etc/axyl/axyl.conf";
$conf = new inputfile($CONF_FILE);
if ($conf->opened) {
  $conf->readall();
  if (preg_match("/AXUSER=([\S]+)\s/", $conf->content, $matches)) {
    $AXUSER = $matches[1];
  }
  if (preg_match("/AXYL_HOME=([\S]+)\s/", $conf->content, $matches)) {
    $AXYL_HOME = $matches[1];
  }
  if (preg_match("/AXYL_DATA=([\S]+)\s/", $conf->content, $matches)) {
    $AXYL_DATA = $matches[1];
  }
  if (preg_match("/LUCENE_PORT=([\S]+)\s/", $conf->content, $matches)) {
    $LUCENE_PORT = $matches[1];
  }

  if ($AXYL_HOME != "") {
    if (!is_dir($AXYL_HOME)) {
      abort("AXYL_HOME is undefined. Check your /etc/axyl/axyl.conf.");
    }
    else {
      // More horse bread checking..
      if (!is_dir("$AXYL_HOME/db/postgres")) {
        abort(
          "Your Axyl installation at $AXYL_HOME doesn't check out.\n"
        . "For example the 'db/postgres' sub-directory is not there."
        );
      }
      elseif (!is_dir("$AXYL_HOME/www")) {
        abort(
          "Your Axyl installation at $AXYL_HOME doesn't check out.\n"
        . "For example the 'www' sub-directory is not there."
        );
      }
    }
  }
  if ($AXYL_DATA != "") {
    if (!is_dir($AXYL_DATA)) {
      abort("AXYL_DATA is undefined. Check your /etc/axyl/axyl.conf.");
    }
    else {
      $LUCENE_INSTALLED = is_dir("$AXYL_DATA/lucene/servers");
    }
  }
}
else {
  abort(
    "File /etc/axyl/axyl.conf was not found. Have you installed Axyl?\n"
  . "If you have, then have you remembered to set it up properly, by\n"
  . "runing the script install/setup-axyl.sh?\n"
  );
}

// Make sure we have an Axyl admin username..
if ($AXUSER == "") {
  abort("AXUSER is undefined. Check your /etc/axyl/axyl.conf.");
}

// Make sure Lucene Server status is consistent..
if ($LUCENE_INSTALLED && $LUCENE_PORT == 0) {
  abort(
    "Inconsistent Lucene Server status: no LUCENE_PORT defined, but\n"
  . "a Lucene servers directory exists at $AXYL_DATA/lucene/servers.\n"
  . "Sort this out, and then get back to me."
  );
}
elseif (!$LUCENE_INSTALLED && $LUCENE_PORT != 0) {
  abort(
    "Inconsistent Lucene Server status: a LUCENE_PORT is defined, as\n"
  . "port $LUCENE_PORT, but no Lucene servers directory exists at\n"
  . "$AXYL_DATA/lucene/servers. Sort this out, and then try again."
  );
}

// ----------------------------------------------------------------------
// WEBSERVER USER AND GROUPS
// These are required for permissions-setting exercises later on.

// Webserver user and group..
$WWWUSER = "www-data";
$WWWGROUP = "www-data";
if ( file_exists("/etc/apache/httpd.conf") ) {
  $APACHE_CONFDIR = "/etc/apache";
} else {
  $APACHE_CONFDIR = "";
  $toexec = `find /etc -maxdepth 2 -type f -name "httpd.conf"`;
  $httpdfiles = explode("\n", $toexec);
  foreach ( $httpdfiles as $conf ) {
    $APACHE_CONFDIR = dirname($conf);
    if ( file_exists("$APACHE_CONFDIR/httpd.conf") ) {
      break;
    } else {
      $APACHE_CONFDIR = "";
    }
  }
}
if ( trim($APACHE_CONFDIR) == "" ) {
  $APACHE_CONFDIR = "/etc/apache";
}
logit(" -- Apache config location: " . $APACHE_CONFDIR);
if (file_exists("$APACHE_CONFDIR/httpd.conf")) {
  $G = trim(`grep "^Group" $APACHE_CONFDIR/httpd.conf | cut -d" " -f2`);
  $U = trim(`grep "^User" $APACHE_CONFDIR/httpd.conf | cut -d" " -f2`);
  if ($G != "") $WWWGROUP = $G;
  if ($U != "") $WWWUSER = $U;
}

// Web hosting administration group..
$WWWADMIN = "wwwadmin";

// Now verify these with the user here..
echo "\nThe following user and group settings are REQUIRED so that we can\n"
   . "properly secure this Axyl website. The suggested settings are most\n"
   . "probably ok, but if necessary please make sure before proceeding.\n"
   . "NB: we require a pre-exsting 'web administrator' group to be named\n"
   . "here. This is recommended to be 'wwwadmin', but you can pick any\n"
   . "existing group on your system.\n\n";

echo "The Webserver runs as which user? [$WWWUSER] :";
$ans = rtrim(fgets($FH, 256));
if ($ans != "") $WWWUSER = $ans;
if (`grep $WWWUSER /etc/passwd | cut -d : -f 3` == "") {
  abort("Sorry, but user '$WWWUSER' is not a valid Unix user");
}

echo "The Webserver group is? [$WWWGROUP] :";
$ans = rtrim(fgets($FH, 256));
if ($ans != "") $WWWGROUP = $ans;
if (`grep $WWWGROUP /etc/group | cut -d : -f 1` == "") {
  abort("Sorry, but group '$WWWGROUP' is not a valid Unix group");
}

echo "Webserver Administrator group is? [$WWWADMIN] :";
$ans = rtrim(fgets($FH, 256));
if ($ans != "") $WWWADMIN = $ans;
if (`grep $WWWADMIN /etc/group | cut -d : -f 1` == "") {
  abort("Sorry, but group '$WWWADMIN' is not a valid Unix group");
}

echo "\nPlease list the Apache virtual host ServerNames, each separated by\n";
echo "a space which will be serving this website. If necessary, go and\n";
echo "have a look in your Apache config files now. This script will have\n";
echo "a go at suggesting one, but it may get it wrong, or you might have\n";
echo "more than one. Don't bother with ServerAlias's. List each ServerName\n";
echo "which identifies a unique Virtual Host entry for this website.\n";
echo "NB: enter 'none' to force no SeverNames specified.\n";
$VHSERVERNAMES = "";
$APACHECONF = "$APACHE_CONFDIR/httpd.conf";
if (file_exists($APACHECONF)) {
  $grepline = exec("grep --ignore-case \"^[\w]*ServerName " . APP_PREFIX . "\" $APACHECONF");
  if ( $grepline != "" ) {
    $bits = explode(" ", $grepline);
    $VHSERVERNAMES = $bits[1];
  }
  else {
    $grepline = exec("grep --ignore-case \"^[\w]*ServerName\" $APACHECONF");
    if ( $grepline != "" ) {
      $bits = explode(" ", $grepline);
      $servername = $bits[1];
      $bits = explode(".", $servername);
      $host = $bits[0];
      if ($host != "" ) {
         $VHSERVERNAMES = str_replace($host, APP_PREFIX, $servername);
      }
    }
  }
}
echo "Website virtual host ServerNames? [$VHSERVERNAMES] :";
$ans = rtrim(fgets($FH, 256));
if ($ans != "") $VHSERVERNAMES = $ans;
if ($ans == "none") $VHSERVERNAMES = "";

// ----------------------------------------------------------------------
// REPORT STATUS
logit("\nINITIAL STATUS:");
$WWW_PATH = getcwd();
logit("the website root directory is: '$WWW_PATH'");
logit("website is served as: $VHSERVERNAMES");
$msg = "local website Axyl lib is found in: '$LIBDIR'";
if (is_link($LIBDIR)) $msg .= " (a link)";
else $msg .= " (physical directory)";
logit($msg);
logit("the Axyl templates are stored under: '$TEMPLATESDIR'");
if ($catalog_dir_was_missing) {
  logit("missing catalog directory was created: '$CATALOGDIR'");
}
else {
  logit("the Axyl catalog content is in: '$CATALOGDIR'");
}
if ($cache_dir_was_missing) {
  logit("missing cache directory was created: '$CACHEDIR'");
}
else {
  logit("the Axyl cached content is in: '$CACHEDIR'");
}
if ($inc_dir_was_missing) {
  logit("missing inc directory was created: '$INCDIR'");
}
else {
  logit("the Axyl include content is in: '$INCDIR'");
}
logit("the Axyl Administrator username is: $AXUSER");
logit("the Axyl system is verified to be in: $AXYL_HOME");
logit("the Axyl data area is: $AXYL_DATA and it exists.");
if ($LUCENE_PORT != 0) {
  logit("the Lucene Server is defined on port: $LUCENE_PORT");
}
else {
  logit("the Lucene Server apparently not installed.");
}
logit("the Webserver user is $WWWUSER and it exists");
logit("the Webserver usergroup is $WWWGROUP, and that exists");
logit("the Web Administrator group is $WWWADMIN, and it exists");

echo "\nContinue with this update? Y or N [Y] :";
$ans = rtrim(fgets($FH, 256));
if ($ans == "n" || $ans == "N") {
  abort("User abort.");
}

logit("\nUPDATING..");
// ----------------------------------------------------------------------
// DATABASE UPGRADE
logit("Axyl database upgrade..");
// Create axyl_core database from latest schema..
$curdb = $RESPONSE->datasource->database[ $RESPONSE->datasource->db_name_default ];

// If Axyl core DB already exists, drop it..
$dbid = connect_pgdb("axyl_core", $curdb->user, $curdb->passwd, $curdb->host, $curdb->port);
if ($dbid !== false) {
  pg_close($dbid);
  $diditwork = exec_pg_command("dropdb axyl_core", $curdb->user, $curdb->host, $curdb->port);
  logit(" -- dropped pre-existing axyl_core database");
}
else {
  logit(" -- axyl core database not found - good");
}
// Create the latest Axyl core..
$diditwork = exec_pg_command("createdb axyl_core", $curdb->user, $curdb->host, $curdb->port);
$diditwork = exec_pg_command("psql -d axyl_core", $curdb->user, $curdb->host, $curdb->port, "-f $AXYL_HOME/db/postgres/axyl_core.sql >/dev/null 2>&1");
$diditwork = exec_pg_command("psql -d axyl_core", $curdb->user, $curdb->host, $curdb->port, "-f $AXYL_HOME/db/postgres/axyl_trig.sql >/dev/null 2>&1");
logit(" -- created latest axyl_core database");

// Apply these specific customisations..
logit(" -- ax_block 'columns' to 'cols' rename..");
$q  = "SELECT pga.attname as fieldname";
$q .= "  FROM pg_class pgc,pg_attribute pga";
$q .= " WHERE pgc.relname='ax_block'";
$q .= "   AND pga.attrelid=pgc.oid";
$q .= "   AND pga.attname='columns'";
$axblock = dbrecordset($q);
if ($axblock->hasdata) {
  $res = dbcommand("ALTER TABLE ax_block RENAME COLUMN columns TO cols");
  if ($res) {
    logit(" -- successfully renamed axblock.columns --> axblock.cols");
  }
  else {
    logit(" -- ERROR: failed to rename axblock.columns --> axblock.cols");
  }
}
else {
  $q  = "SELECT pga.attname as fieldname";
  $q .= "  FROM pg_class pgc,pg_attribute pga";
  $q .= " WHERE pgc.relname='ax_block'";
  $q .= "   AND pga.attrelid=pgc.oid";
  $q .= "   AND pga.attname='cols'";
  $axblock = dbrecordset($q);
  if ($axblock->hasdata) {
    logit(" -- axblock.cols already renamed - good");
  }
  else {
    logit(" -- ERROR: neither axblock.cols nor axblock.columns was found!");
  }
}

$res = create_new_field("ax_block", "blocklet_sep", "INT4");
if ($res) {
  $res = dbcommand("ALTER TABLE ax_block ALTER COLUMN blocklet_sep SET DEFAULT 0");
  if ($res) {
    logit(" -- assigned default to zero");
    $res = dbcommand("UPDATE ax_block SET blocklet_sep=0");
    if ($res) {
      logit(" -- initialised all blocklet_sep values to zero");
    }
  }
}

logit(" -- multi-language extensions");
$q  = "SELECT relname FROM pg_class WHERE relname='ax_language'";
$langtable = dbrecordset($q);
if ($langtable->hasdata) {
  logit(" -- ax_language table already present - good");
}
else {
  logit(" -- creating ax_language table..");
  $q  = "create table ax_language (";
  $q .= " lang_id INT4 not null,";
  $q .= " lang_desc TEXT not null,";
  $q .= " char_encoding TEXT not null,";
  $q .= " direction TEXT not null default 'LTR' constraint CKC_DIRECTION_AX_LANGU check (direction in ('LTR','RTL')),";
  $q .= " is_default BOOL not null default false,";
  $q .= " display_order INT4 not null default 999,";
  $q .= " enabled BOOL not null default true,";
  $q .= " constraint PK_AX_LANGUAGE primary key (lang_id)";
  $q .= ");";
  $res = dbcommand($q);
  if ($res) {
    logit(" -- creating default language record..");
    $q  = "INSERT INTO ax_language (lang_id,lang_desc,char_encoding,is_default)";
    $q .= " VALUES (0,'default','','t')";
    $res = dbcommand($q);
    if ($res) logit(" -- done.");
    else logit( "-- FAILED");
  }
}

$res = create_new_field("ax_wwwsession", "lang_id", "INT4");
if ($res) {
  $res = dbcommand("UPDATE ax_wwwsession SET lang_id=0");
  if ($res) {
    logit(" -- initialised all lang_id values to zero");
  }
}
$res = create_new_field("ax_layout", "lang_id", "INT4");
if ($res) {
  $res = dbcommand("UPDATE ax_layout SET lang_id=0");
  if ($res) {
    logit(" -- initialised all lang_id values to zero");
  }
}
$res = create_new_field("ax_block", "lang_id", "INT4");
if ($res) {
  $res = dbcommand("UPDATE ax_block SET lang_id=0");
  if ($res) {
    logit(" -- initialised all lang_id values to zero");
  }
}

// Make sure we start sessions with a clean slate..
if ($do_menu_changes) {
  $res = dbcommand("UPDATE ax_wwwsession SET menuoption_id=NULL, menu_status=NULL");
  if ($res) {
    logit(" -- menu structure: nulling session menu option FK references");
  }
  else {
    logit(" -- menu structure: failed to null session menu option FK references");
  }
}

logit(" -- menu structure: change from text to integer ax_menu key");
$do_menu_changes = true;
$q  = "SELECT pga.attname as fieldname";
$q .= "  FROM pg_class pgc,pg_attribute pga";
$q .= " WHERE pgc.relname='ax_menu'";
$q .= "   AND pga.attrelid=pgc.oid";
$q .= "   AND pga.attname='menu_name'";
$axblock = dbrecordset($q);
if ($axblock->hasdata) {
  logit(" -- menu structure: already updated - good");
  $do_menu_changes = false;
}
else {
  $res1 = dbcommand("create table ax_menu_old as select * from ax_menu");
  if ($res1) {
    $res2 = dbcommand("create table ax_menuoption_old as select * from ax_menuoption");
  }
  if ($res1 && $res2) {
    logit(" -- menu structure: renamed ax_menu to ax_menu_old");
    $res1 = dbcommand("drop table ax_menuoption");
    if ($res1) {
      $res2 = dbcommand("drop table ax_menu");
    }
    if ($res1 && $res2) {
      logit(" -- menu structure: successfully dropped ax_menu and ax_menuoption tables");
    }
    else {
      logit(" -- menu structure: failed to drop ax_menu and ax_menuoption tables!");
      $do_menu_changes = false;
    }
  }
  else {
    logit(" -- menu structure: failed to create ax_menu_old and ax_menuoption_old tables!");
    $do_menu_changes = false;
  }
}

// Note that we loop because sometimes the diff have problems with
// ordering which requires them to be iterated a few times..
$upgrading_schema = true;
while ($upgrading_schema) {
  logit("\n -- syncing database '$curdb->name' with Axyl core schema");
  logit(" -- using pgdiff to determine upgrade SQL");
  if ($do_menu_changes) {
    logit(" -- NB: re-creating the menu structure. Database diff will notify you");
    logit(" -- that ax_menu and ax_menuoption will need creating.");
  }
  $outputlines = array();
  exec("$AXYL_HOME/scripts/pgdiff.php --no-drops --target=" . $curdb->name . " --ref=axyl_core", $outputlines, $ret);
  if ($ret == 0) {
    logit("\n -- pgdiff output begins");
    $diffs = array();
    foreach($outputlines as $line) {
      if (trim($line) != "" && substr($line, 0, 2) != "--") {
        logit($line);
        $line = preg_replace("/;$/", "^!^", $line);
        $diffs[] = trim($line);
      }
    }
    logit(" -- pgdiff output ends\n");
    $diffsql = trim(implode(" ", $diffs));
    if ($diffsql != "") {
      echo "\napply this SQL to upgrade the database (nothing is dropped) ? Y or N [Y] :";
      $ans = rtrim(fgets($FH, 256));
      if ($ans == "" || $ans == "Y" || $ans == "y") {
        logit(" -- applying diff SQL to current database now");
        $ok = true;
        $upg = new dbquery();
        $diffs = explode("^!^", $diffsql);
        foreach ($diffs as $diff) {
          if ($diff != "") {
            $upg->set_sql("$diff;");
            if (!$upg->execute()) {
              $ok = false;
              logit(" -- QFAIL: $diff;");
              $dberr = $RESPONSE->datasource->errormessage();
              if (trim($dberr != "")) {
                logit(" -- backend DB Error: $dberr");
              }
            }
          }
        } // foreach

        // Final check that diff is now reporting identical schemas..
        if ($ok) {
          exec("$AXYL_HOME/scripts/pgdiff.php --no-drops --target=" . $curdb->name . " --ref=axyl_core", $outputlines);
          foreach($outputlines as $line) {
            if (trim($line) != "" && substr($line, 0, 2) != "--") {
              $ok = false;
              break;
            }
          }
        }

        // Report results..
        if ($ok) {
          logit(" -- done - current database is now up to date");
          logit(" -- applying some harmless data fixups..");
          dbcommand("UPDATE ax_block SET block_type='b' WHERE block_type IS NULL OR block_type=''");
          $upgrading_schema = false;
          logit(" -- done");
        }
        else {
          logit(" -- failed - see any messages above for clues");
          logit(" -- NB: failure might just be a case of statement ordering problems so");
          logit(" -- please try again at least once, or until SQL diffs don't change");
        }
      }
      else {
        logit(" -- upgrade SQL was aborted (user choice)");
        logit(" -- warning: not upgrading the database compromises the upgrade");
        $upgrading_schema = false;
      }
    }
    else {
      logit(" -- identical databases - no (further) upgrade required");
      $upgrading_schema = false;
    }
  }
  else {
    abort(
      "Unfortunately, the database upgrade procedure failed. Check that\n"
    . "$AXYL_HOME/scripts/pgdiff.php exists and is executable.\n"
    . "Since this has occurred part-way through the secutiry update, you\n"
    . "will need to restore website & DB from backup, fix this problem and\n"
    . "then re-run the security update."
    );
  }
} // while

// Now apply changes, by getting old menus and populating the new
// menu tables structure with them..
if ($do_menu_changes) {
  $menuid_lookup = array();
  $oldmenu = dbrecordset("SELECT * FROM ax_menu_old");
  if ($oldmenu->hasdata) {
    do {
      $menu_id   = next_sequencevalue("seq_menu_id");
      $menu_name = $oldmenu->field("menu_id");
      $menu_desc = $oldmenu->field("menu_desc");
      $ugroups   = $oldmenu->field("menu_user_groups");
      $active    = $oldmenu->istrue("active");

      $min = new dbinsert("ax_menu");
      $min->set("menu_id",          $menu_id);
      $min->set("menu_name",        $menu_name);
      $min->set("lang_id",          0);
      $min->set("menu_desc",        $menu_desc);
      $min->set("menu_user_groups", $ugroups);
      $min->set("active",           $active);
      $min->execute();
      $menuid_lookup[$menu_name] = $menu_id;
      logit(" -- menus update: ax_menu: $menu_name");
    } while ($oldmenu->get_next());
  }

  $oldmenuop = dbrecordset("SELECT * FROM ax_menuoption_old");
  if ($oldmenuop->hasdata) {
    do {
      $menuop_id = $oldmenuop->field("menuoption_id");
      $menu_name = $oldmenuop->field("menu_id");
      $label     = $oldmenuop->field("label");

      $menu_id   = $menuid_lookup[$menu_name];

      $moin = new dbinsert("ax_menuoption");
      $moin->set("menuoption_id",  $menuop_id);
      $moin->set("menu_id",        $menu_id);
      $moin->set("parent_id",      $oldmenuop->field("parent_id"));
      $moin->set("user_groups",    $oldmenuop->field("user_groups"));
      $moin->set("user_type",      $oldmenuop->field("user_type"));
      $moin->set("menu_level",     $oldmenuop->field("level"));
      $moin->set("label",          $label);
      $moin->set("description",    $oldmenuop->field("description"));
      $moin->set("display_order",  $oldmenuop->field("display_order"));
      $moin->set("action",         $oldmenuop->field("action"));
      $moin->set("sitepage",       $oldmenuop->field("sitepage"));
      $moin->set("sitepage_parms", $oldmenuop->field("sitepage_parms"));
      $moin->set("auth_code",      $oldmenuop->field("auth_code"));
      $moin->set("width",          $oldmenuop->field("width"));
      $moin->set("height",         $oldmenuop->field("height"));
      $moin->set("active",         $oldmenuop->istrue("active"));
      $moin->set("is_parent",     ($oldmenuop->field("action") == ""));
      $moin->execute();
      logit(" -- menus update: ax_menuoption: $menu_name: $label");
    } while ($oldmenuop->get_next());
  }

  logit(" -- dropping old ax_menu table");
  dbcommand("DROP TABLE ax_menuoption_old");
  logit(" -- dropping old ax_menuoption table");
  dbcommand("DROP TABLE ax_menu_old");

}

// ----------------------------------------------------------------------
// NEW SUB-DIRECTORIES
logit("\nupdating directory structures..");
// Managed content directory..
if (!makepath("./var/cm")) {
  abort(" -- there was a problem creating directory ./var/cm");
}
else {
  logit(" -- created new directory ./var/cm");
}

// Move existing catalog sub-directory to new home, or
// just create the new catalog directory..
if (is_dir($CATALOGDIR)) {
  if (!move($CATALOGDIR, "./var/catalog")) {
    abort(" -- there was a problem moving $CATALOGDIR to ./var/catalog");
  }
  else {
    logit(" -- moved $CATALOGDIR to ./var/catalog");
  }
}
else {
  if (!makepath("./var/catalog")) {
    abort(" -- there was a problem creating required directory ./var/catalog");
  }
  else {
    logit(" -- created new directory ./var/catalog");
  }
}

// Move existing cache sub-directory to new home, or
// just create the new cache directory.
if (is_dir($CACHEDIR)) {
  if (!move($CACHEDIR, "./var/cache")) {
    abort(" -- there was a problem moving $CACHEDIR to ./var/cache");
  }
  else {
    logit(" -- moved $CACHEDIR to ./var/cache");
  }
}
else {
  if (!makepath("./var/cache")) {
    abort(" -- there was a problem creating required directory ./var/cache");
  }
  else {
    logit(" -- created new directory ./var/cache");
  }
}

// Make sure 'inc' directory exists. It might or might not
// depending on the last time this site was tended to..
if ($INCDIR != "inc" && $INCDIR != "./inc" && is_dir($INCDIR)) {
  if (!move($INCDIR, "./inc")) {
    abort(" -- there was a problem moving $INCDIR to ./inc");
  }
  else {
    logit(" -- moved $INCDIR to ./inc");
  }
}
else {
  if (!makepath("inc")) {
    abort(" -- there was a problem creating required directory ./inc");
  }
  else {
    logit(" -- include directory ./inc exists");
  }
}
// Make sure all our latest templates are in place..
copy("$AXYL_HOME/www/inc/axyl-template.inc", "./inc/axyl-template.inc");
copy("$AXYL_HOME/www/inc/cm-formats.inc", "./inc/cm-formats.inc");
copy("$AXYL_HOME/www/inc/cm-template.inc", "./inc/cm-template.inc");
logit(" -- copied latest templates to ./inc");

// ----------------------------------------------------------------------
// APPLICATION XML CONFIG
logit("\nconverting/upgrading application.xml..");
#include_once("application-defs.php");

// Make sure that the local application XML file has all the
// required defs, globals & settings..
$app = new application();
$defaultapp = new application("$LIBDIR/default-application.xml");
$app->synchronize($defaultapp);
logit(" -- synchronized application.xml");
$app->globals["CACHEDIR"] = "var/cache";
$app->globals["CATALOGDIR"] = "var/catalog";
$app->globals["CMDIR"] = "var/cm";
$app->globals["INCDIR"] = "inc";
logit(" -- updated catalog, cache and cm paths");
$app->save();
logit(" -- saved application.xml");

// ----------------------------------------------------------------------
// PHP FILES WITH $LIBDIR REFERENCES
// Fix up any scripts in the top directory which have references to the
// old $LIBDIR. These comprise references to library images, and also
// include_once() calls, which need a physical directory.
// 1) for the image references, we need to replace $LIBDIR with $LIBDIR/img
// 2) for include_once() we need to strip the "" prefix, since
//    the PHP include_path will do the trick for these now.
logit("\nfixing up Php files with LIBDIR references..");
$dir = ".";
$dh  = opendir($dir);
while (false !== ($filename = readdir($dh))) {
  if (substr($filename, -4) == ".php" || substr($filename, -4) == ".inc") {
    $in  = new inputfile($filename);
    $out = new outputfile($filename);
    $updated = false;
    if ($in->opened && $out->opened) {
      $end_of_file = false;
      while (!$end_of_file) {
        $line = $in->readln();
        if ($line === false) {
          $end_of_file = true;
        }
        else {
          if (preg_match("/LIBDIR\/_/", $line)) {
            $line = str_replace("LIBDIR", "LIBDIR/img", $line);
            $updated = true;
          }
          if (preg_match("/include_once\(|include\(|require_once\(|require\(/", $line)) {
            $replacepat = "/\\\$LIBDIR\//";
            $line = preg_replace($replacepat, "", $line);
            $updated = true;
          }
          $out->writeln($line);
        }
      } // while
      $in->closefile();
      $out->closefile();
      if ($updated) {
        logit(" -- fixed up LIBDIR refs in $filename");
      }
    }
  }
}
closedir($dh);

// ----------------------------------------------------------------------
// FIX HARD-CODED IMAGE REFERENCES
// Fix up any scripts in the tree which look like they have hard-coded
// image references to places within the website. Obviously ignore anything
// which refers to remote locations (prefixed with 'http').
logit("\nfix up of hard-coded image references..");
echo "\nIf you have been unwise enough to hard-code paths to either library\n"
   . "images, or to images in the local website images directory, now is your\n"
   . "chance to rectify this sad mistake. Only code files with the extensions\n"
   . ".php or .inc will be considered.\n"
   . "\n";
echo "Fix up all hard-coded image references? Y or N [Y] :";
$ans = rtrim(fgets($FH, 256));
if ($ans == "y" || $ans == "Y" || $ans == "") {
  logit(" -- user chose to update hard-coded image references..");
  $dir = ".";
  $dh  = opendir($dir);
  while (false !== ($filename = readdir($dh))) {
    if (substr($filename, -4) == ".php"
    || substr($filename, -4) == ".inc") {
      $in  = new inputfile($filename);
      $out = new outputfile($filename);
      $updated = 0;
      $end_of_file = false;
      $plugin_inline = false;
      if ($in->opened && $out->opened) {
        while (!$end_of_file) {
          $line = $in->readln();
          if ($line === false) {
            $end_of_file = true;
          }
          else {
            if (preg_match("/RESPONSE->plugin_inline/i", $line)) {
              $plugin_inline = !$plugin_inline;
            }
            // Avoid inline code, where hard-coding is used..
            if (!$plugin_inline) {
              // Standard image references..
              $newline = preg_replace(
                      "/($IMAGESDIR\/)(.+?)(\.gif)/ie",
                      "'\$IMAGESDIR/' . basename('\\2') . '.gif'",
                      $line
                      );
              // Library image references..
              $newline = preg_replace(
                      "/(lib\/_)(.+?)(\.gif)/ie",
                      "'\$LIBDIR/img/_' . basename('\\2') . '.gif'",
                      $newline
                      );
              if ($newline != $line) {
                $updated += 1;
              }
              $line = $newline;
            }
            // Spit it out..
            $out->writeln($line);
          }
        } // while
        // Wrap it up..
        $in->closefile();
        if ($updated > 0) {
          $out->closefile();
          logit(" -- $updated hard-coded image references were fixed in $filename");
        }
        else {
          $out->closefile(ABANDON_FILE);
        }
      }
    }
  }
  closedir($dh);
}
else {
  logit(" -- user chose to NOT update hard-coded image references.");
}

// ----------------------------------------------------------------------
// THE APPLICATION.PHP File..
// The test for the library requires changing to fit in with the way we
// now confitgure the library Php content as ex-website-directory..
logit("\nupgrading application library detection..");
$appfile = "application.php";
$in  = new inputfile($appfile);
$out = new outputfile($appfile);
$updated = false;
if ($in->opened && $out->opened) {
  $end_of_file = false;
  $done = false;
  $abandoned = false;
  $sanitychecked = false;
  while (!$end_of_file) {
    $line = $in->readln();
    if ($line === false) {
      $end_of_file = true;
    }
    elseif (!$done) {
      if (preg_match("/THE AXYL LIBRARY DIRECTORY/", $line)) {
        logit(" -- passed check 1");
        $out->writeln($line);
        // Skip the original lines..
        $skipping = true;
        while ($skipping && !$end_of_file) {
          $line = $in->readln();
          if ($line === false) {
            $end_of_file = true;
          }
          else {
            if (preg_match("/THE RESPONSE OBJECT/", $line)) {
              $s  = "*\n"
                  . "* You can either leave this commented out, and let Axyl find it with\n"
                  . "* the code just below, or force it here. The default is 'lib'.\n"
                  . "* NOTE: The Axyl library is no longer resident in the website root. It\n"
                  . "*       is now accessed via the Php Include Path. This local 'library'\n"
                  . "*       only contains the library images, and the library javascript.\n"
                  . "*/\n"
                  . "//\$LIBDIR = \"lib\";\n"
                  . "\n"
                  . "// -----------------------------------------------------------------------\n"
                  . "// Document root is undefined for command-line scripts..\n"
                  . "if (!isset(\$DOCUMENT_ROOT)) {\n"
                  . "  \$DOCUMENT_ROOT = getcwd();\n"
                  . "}\n"
                  . "// -----------------------------------------------------------------------\n"
                  . "/**\n"
                  . "* AXYL LIBRARY LOCATOR\n"
                  . "* If not defined above, sniff the Axyl library. We only allow for the\n"
                  . "* basic possibilities, as listed in the below possibles array. These\n"
                  . "* are always relative to the website root directory.\n"
                  . "*/\n"
                  . "if (!isset(\$LIBDIR)) {\n"
                  . "  \$LIBDIR_POSSIBLES = array(\"lib\", \"library\", \"axyl\");\n"
                  . "  foreach (\$LIBDIR_POSSIBLES as \$testdir) {\n"
                  . "    if (is_dir(\"\$DOCUMENT_ROOT/\$testdir\")) {\n"
                  . "      \$LIBDIR = \"/\$testdir\";\n"
                  . "      break;\n"
                  . "    }\n"
                  . "  }\n"
                  . "  // Failure is not an option..\n"
                  . "  if (!isset(\$LIBDIR)) {\n"
                  . "    die(\"could not locate Axyl library in document root \$DOCUMENT_ROOT/\");\n"
                  . "  }\n"
                  . "}\n"
                  . "\n"
                  . "// -----------------------------------------------------------------------\n"
                  . "/*\n"
                  ;
              $out->write($s);
              $sanitychecked = true;
              $skipping = false;
              logit(" -- passed check 2 - writing changes");
            }
          }
        } // while
        // We only do this once..
        $done = true;
      }
    }
    $out->writeln($line);
  } // while

  // Wrap it up..
  $in->closefile();
  if ($sanitychecked && $done) {
    $out->closefile();
    logit(" -- application.php library detection has been re-written");
  }
  else {
    $out->closefile(ABANDON_FILE);
    logit(" -- application.php library detection NOT re-written - file check failed");
  }
}

// ----------------------------------------------------------------------
// OPTIONAL LINKED AXYL-* MAINT SCRIPTS
logit("\nAxyl core scripts..");
echo "\nNOTE: The latest Axyl core scripts are named with an 'axyl-' prefix and\n";
echo "will therefore NOT over-write any existing scripts on an older Axyl site.\n";
echo "These are the scripts which maintain users, groups, web-pages, menus etc.\n";
echo "This security update can optionally install the latest versions of the core\n";
echo "scripts, (they are scripts with names starting 'axyl-') either by copying or\n";
echo "by using soft-links. Soft-links will allow you to get the latest updates\n";
echo "each time Axyl is upgraded. Copying allows you to customize the scripts\n";
echo "locally as and when you require. You will be asked which way you want to\n";
echo "install them if you answer 'y' to the following:\n";
echo "\nInstall latest Axyl core scripts? Y or N [Y] :";
$ans = rtrim(fgets($FH, 256));
if ($ans == "y" || $ans == "Y" || $ans == "") {
  logit(" -- installing Axyl axyl-*.php scripts (user choice)..");
  echo "\nCopy the scripts, or Soft-link them? C or S [C] :";
  $ans = rtrim(fgets($FH, 256));
  if ($ans == "s" || $ans == "S") {
    $method = "symlink";
    logit(" -- user opted to sym-link");
  }
  else {
    $method = "copy";
    logit(" -- user opted to copy");
  }
  $n = 0;
  $dir = "$AXYL_HOME/www";
  $dh  = opendir($dir);
  while (false !== ($filename = readdir($dh))) {
    if (substr($filename, 0, 5) == "axyl-") {
      if (file_exists("./$filename")) {
        unlink($filename);
      }
      if ($method == "symlink") {
        symlink("$AXYL_HOME/www/$filename", $filename);
        logit(" -- symlinked $filename --> $AXYL_HOME/www/$filename");
      }
      else {
        copy("$AXYL_HOME/www/$filename", $filename);
        logit(" -- copied $filename --> $AXYL_HOME/www/$filename");
      }
      $n += 1;
    }
  }
  logit(" -- installed $n files");
}
else {
  logit(" -- leaving Axyl maintenance files just as they are");
}

// ----------------------------------------------------------------------
// MANAGED CONTENT
$CMDIR = "var/cm";
logit("\nmoving content-managed content to $CMDIR..");
if (is_dir($CMDIR)) {
  if (is_writeable($CMDIR)) {
    $managed_content = dbrecordset("SELECT * FROM ax_sitepage WHERE managed");
    if ($managed_content->hasdata) {
      $n = 0;
      do {
        $pgid = $managed_content->field("page_id");
        $oldpath = $managed_content->field("page_path");

        // Has database been changed already? If so then they are probably
        // re-running this script. We can possibly deal to that in a sane
        // way, to allow them to re-run multiple times - as you do.
        if (!file_exists($oldpath) && strstr($oldpath, $CMDIR)) {
          if (file_exists(basename($oldpath))) {
            $new_oldpath = basename($oldpath);
            logit(" -- re-run detected, $oldpath morphed to $new_oldpath");
            $oldpath = $new_oldpath;
          }
        }

        // We only do anything if the path doesn't physically exist..
        $file_to_move = file_exists($oldpath);
        if (!$file_to_move) {
          logit(" -- warning: file $oldpath does not exist (ax_sitepage.page_id=$pgid)");
        }
        $subdir = dirname($oldpath);
        if ($subdir != "") {
          makepath("$CMDIR/$subdir");
        }
        if ($file_to_move && !rename($oldpath, "$CMDIR/$oldpath")) {
          abort(" -- rename/move of $oldpath --> $CMDIR/$oldpath failed.");
        }
        else {
          // Double-check it is where it should be..
          clearstatcache();
          if ($file_to_move) {
            if (file_exists("$CMDIR/$oldpath")) {
              logit(" -- moved $oldpath --> $CMDIR/$oldpath");
            }
            else {
              logit(" -- attempt to move $oldpath --> $CMDIR/$oldpath failed.");
            }
          }
          // Update sitepage record..
          $upcm = new dbupdate("ax_sitepage");
          $upcm->set("page_path", "/$CMDIR/$oldpath");
          $upcm->where("page_id=$pgid");
          $upcm->execute();
          logit(" -- sitepage record updated");

          // Update any menuoptions..
          $q  = "SELECT * FROM ax_menuoption";
          $q .= " WHERE action LIKE '%$oldpath%'";
          $q .= "    OR sitepage LIKE '%$oldpath%'";
          $mops = dbrecordset($q);
          if ($mops->hasdata) {
            do {
              $mopid = $mops->field("menuoption_id");
              $moplabel = $mops->field("label");
              $mup = new dbupdate("ax_menuoption");
              $mup->set("action", "/$CMDIR/$oldpath");
              $mup->set("sitepage", "/$CMDIR/$oldpath");
              $mup->where("menuoption_id=$mopid");
              $mup->execute();
              logit(" -- menuoption '$moplabel' ($mopid) record updated");
            } while ($mops->get_next());
          }
          else {
            logit(" -- no menuoption records to update");
          }
          $mops->tidyup();
          unset($mops);

          // Update blocklet HREF's, if any..
          $q  = "SELECT * FROM ax_blocklet";
          $q .= " WHERE content LIKE '%$oldpath%'";
          $blocklets = dbrecordset($q);
          if ($blocklets->hasdata) {
            do {
              $bktid = $blocklets->field("blocklet_id");
              $oldcontent = $blocklets->field("content");
              $newcontent = str_replace("HREF=\"$oldpath\"", "HREF=\"/$CMDIR/$oldpath\"", $oldcontent);
              if ($newcontent != $oldcontent) {
                $bup = new dbupdate("ax_blocklet");
                $bup->set("content", $newcontent);
                $bup->where("blocklet_id=$bktid");
                $bup->execute();
                logit(" -- HREF update in blocklet $bktid");
              }
            } while ($blocklets->get_next());
          }
          $blocklets->tidyup();
          unset($blocklets);

          // Keep count..
          $n += 1;
        }
      } while ($managed_content->get_next());
      logit(" -- moved $n content-managed files");
    }
  }
  else {
    abort("$CMDIR is not writeable.");
  }
}
else {
  abort(
    "$CMDIR does not exist. This is an unusual error, since it should\n"
  . "have been created earlier. Please investigate, and re-try."
  );
}

// ----------------------------------------------------------------------
// MENU OPTION RELATIVE PATHS FIXUP
logit("\nfixing up menuoption target references..");
$mops = dbrecordset("SELECT * FROM ax_menuoption");
if ($mops->hasdata) {
  do {
    $mopid = $mops->field("menuoption_id");
    $moplabel = $mops->field("label");
    $mopaction = $mops->field("action");
    if ($mopaction != "") {
      if (substr($mopaction, 0, 1) != "/") {
        $pathbits = explode("?", $mopaction);
        $target = "./" . $pathbits[0];
        if (file_exists($target)) {
          $new_mopaction = "/" . $mopaction;
          $mup = new dbupdate("ax_menuoption");
          $mup->set("action", $new_mopaction);
          $mup->where("menuoption_id=$mopid");
          $mup->execute();
          logit(" -- menus: menuoption action '$moplabel' ($mopid) --> $new_mopaction");
        }
        else {
          logit(" -- menus: menuoption action '$moplabel' ($mopid) target '$target' not found");
        }
      }
    }
  } while ($mops->get_next());
}
$mops->tidyup();
unset($mops);

// ----------------------------------------------------------------------
// MANAGED CONTENT: MULTI-MEDIA/DATA REFERENCES & HARD_CODED IMAGES FIXUP
$bkt = dbrecordset("SELECT * FROM ax_blocklet");
if ($bkt->hasdata) {
  logit("\ntranslating <IMAGE>, <DOCUMENT>, <MEDIA> and <DATA> to <object> tags");
  do {
    $bktid = $bkt->field("blocklet_id");
    $content = $bkt->field("content");
    $updated = false;
    $new_content = $content;

    // Multimedia objects..
    $datapat = "/(<MEDIA (.*?)>|<IMAGE (.*?)>|<DATA (.*?)>|<DOCUMENT (.*?)>)/i";
    if ( preg_match($datapat, $new_content, $matches) ) {
      $new_content = preg_replace("/<IMAGE /i", "<object type=\"axyl/embedded-media\" codetype=\"image\" ", $new_content);
      $new_content = preg_replace("/<MEDIA /i", "<object type=\"axyl/embedded-media\" codetype=\"media\" ", $new_content);
      $new_content = preg_replace("/<DOCUMENT /i", "<object type=\"axyl/embedded-media\" codetype=\"document\" ", $new_content);
      $new_content = preg_replace("/<DATA /i", "<object type=\"axyl/embedded-media\" codetype=\"data\" ", $new_content);
      $new_content = preg_replace("/img_align/i", "align", $new_content);
      $new_content = preg_replace("/img_pad/i", "pad", $new_content);
      $new_content = preg_replace("/mm_width/i", "width", $new_content);
      $new_content = preg_replace("/mm_height/i", "height", $new_content);
      $new_content = preg_replace("/mm_display/i", "display", $new_content);
      $new_content = preg_replace("/doc_width/i", "width", $new_content);
      $new_content = preg_replace("/doc_height/i", "height", $new_content);
      $new_content = preg_replace("/doc_display/i", "display", $new_content);
      $new_content = preg_replace("/data_format/i", "format", $new_content);
      $new_content = str_replace("desc=\"", "title=\"", $new_content);
      if ($new_content != $content) {
        logit(" -- object(s) found and translated in blocklet ID: $bktid");
        $content = $new_content;
        $updated = true;
      }
    }

    // Adjust hard-coded image references..
    $new_content = preg_replace(
            "/($IMAGESDIR\/)(.+?)(\.gif)/ie",
            "'/$IMAGESDIR/' . basename('\\2') . '.gif'",
            $content
            );
    $new_content = preg_replace(
            "/($LIBDIR\/_)(.+?)(\.gif)/ie",
            "'/$LIBDIR/img/_' . basename('\\2') . '.gif'",
            $new_content
            );
    if ($new_content != $content) {
      logit(" -- hard-coded image(s) found and fixed in blocklet ID: $bktid");
      $content = $new_content;
      $updated = true;
    }

    if ($updated) {
      $bktup = new dbupdate("ax_blocklet");
      $bktup->set("content", $new_content);
      $bktup->where("blocklet_id=$bktid");
      $bktup->execute();
    }

  } while ($bkt->get_next());
}

// ----------------------------------------------------------------------
// CATALOG RECORDS FIXUP
logit("\nfixing up catalog item record paths..");
$CATDIR="var/catalog";
$catitem = dbrecordset("SELECT * FROM ax_catalog");
if ($catitem->hasdata) {
  do {
    $catid = $catitem->field("cat_id");
    $oldfilepath = $catitem->field("filepath");
    $file = basename($oldfilepath);
    $newfilepath = "/$CATDIR/$file";
    if ($newfilepath != $oldfilepath) {
      $catup = new dbupdate("ax_catalog");
      $catup->set("filepath", $newfilepath);
      $catup->where("cat_id=$catid");
      $catup->execute();
      logit(" -- catalog item ($catid) filepath updated: $newfilepath");
    }
  } while ($catitem->get_next());
}

// ----------------------------------------------------------------------
// APACHE VIRTUAL HOSTS
// First find the file(s) with our VirtualHost in it..
$VHservernames = array();
logit("\nApache virtual hosts");
if ($VHSERVERNAMES != "") {
  logit(" -- virtual host server names used to serve this website: $VHSERVERNAMES");
  $VHservernames = explode(" ", $VHSERVERNAMES);
  $VHpaths = array();
  foreach ($VHservernames as $VHservername) {
    exec("find $APACHE_CONFDIR -type f -exec grep -il \"ServerName $VHservername\" {} \;", $paths);
    logit(" -- there were ".count($paths)." paths found!");
    $vhcnt = 0;
    foreach($paths as $path) {
      if (!preg_match("/(\.saved)|([-\.]old)|(\.bak)/i", $path)) {
        $VHpaths[] = $path;
        logit(" -- examining VH config in: $path");
        $vhcnt += 1;
      }
    }
    if ($vhcnt == 0) {
      logit(" -- no applicable VH configs found.");
    }
  }
  // Directive to insert into Apache VirtualHost config..
  $php_include_path = "php_value include_path \".:$AXYL_HOME/lib:$WWW_PATH\"";
  foreach ($VHpaths as $VHpath) {
    $in = new inputfile($VHpath);
    if ($in->opened) {
      $out = new outputfile($VHpath);
      if ($out->opened) {
        logit(" ++ processing $VHpath");
        $do_update = false;
        $inVH = false;
        $VHlines = array();
        $currVHservername = "";
        $end_of_file = false;
        while (!$end_of_file) {
          $line = $in->readln();
          if ($line === false) {
            $end_of_file = true;
          }
          if (preg_match("/^[\s]*<VirtualHost/i", $line)) {
            $inVH = true;
            $hasDir = false;
            $hasOptSym = false;
            $hasDirIx = false;
          }
          if ($inVH) {
            $VHlines[] = $line;
            if (preg_match("/^([\s]*)(ServerName[\s]+)(.*?)$/i", $line, $matches)) {
              $currVHservername = $matches[3];
            }
            if (preg_match("/^[\s]*<Directory/i", $line)) {
              $hasDir = true;
            }
            if (preg_match("/^[\s]*Options.*?FollowSymLinks/i", $line)) {
              $hasOptSym = true;
            }
            if (preg_match("/^[\s]*DirectoryIndex/i", $line)) {
              $hasDirIx = true;
            }
          }
          else {
            $out->writeln($line);
          }
          if ($inVH && preg_match("/^[\s]*<\/VirtualHost/i", $line)) {
            $VHfound = false;
            foreach ($VHservernames as $VHservername) {
              if (strcasecmp($VHservername, $currVHservername) == 0) {
                $VHfound = true;
                break;
              }
            }
            // Write the virtualhost entry out now..
            foreach ($VHlines as $VHline) {
              if ($VHfound && preg_match("/^[\s]*<\/VirtualHost>/i", $VHline)) {
                $out->writeln("    $php_include_path");
                logit(" -- VH re-write: inserting php inlude path");
                if (!$hasDir) {
                  $out->writeln("    <Directory $WWW_PATH>");
                  $out->writeln("      Order Deny,Allow");
                  $out->writeln("      Allow from All");
                  $out->writeln("      Options FollowSymLinks");
                  $out->writeln("      AllowOverride All");
                  $out->writeln("    </Directory>");
                  logit(" -- VH re-write: inserting <Directory> directive");
                }
                if (!$hasOptSym) {
                  $out->writeln("    Options FollowSymLinks");
                  logit(" -- VH re-write: inserting Options directive");
                }
                if (!$hasDirIx) {
                  $out->writeln("    DirectoryIndex index.php");
                  logit(" -- VH re-write: inserting DirectoryIndex directive");
                }
                $do_update = true;
                copy($VHpath, "$VHpath.axyl-security-update.saved");
                logit(" -- updated Apache Virtual Host configuration in $VHpath for $currVHservername");
                logit(" -- backup saved in $VHpath.axyl-security-update.saved");
              }
              // Always output the line if a foreign VH, or if it's our VH and NOT
              // the php_value line, which we will be re-writing above..
              if (!$VHfound || ($VHfound && !preg_match("/php_value /", $VHline))) {
                $out->writeln($VHline);
              }
            }
            $inVH = false;
            $currVHservername = "";
            $VHlines = array();
          }
        } // while

        // Close files..
        $in->closefile();
        if ($do_update) {
          // Will move the new file to the right place..
          $out->closefile();
          // And apache should be reloaded..
          if (file_exists("/usr/sbin/apache")) {
            logit(" -- apache configuration reload..");
            exec("invoke-rc.d apache reload");
          }
          if (file_exists("/usr/sbin/apache-ssl")) {
            logit(" -- apache-ssl configuration reload..");
            exec("invoke-rc.d apache-ssl reload");
          }
        }
        else {
          // Close without saving..
          $out->closefile(ABANDON_FILE);
          logit(" -- nothing to do in $VHpath");
        }
      } // out opened
      else {
        logit("oops: apache config file $VHpath not re-writable");
      }
    } // in opened
    else {
      logit("oops: apache config file $VHpath not accessible");
    }
  } // foreach
}
else {
  logit(" -- no virtual hosts specified.");
}

// ----------------------------------------------------------------------
// SUNDRIES..
logit("\nsecurity checks..");
if (file_exists("control-panel.php")) {
  logit(" -- removing control-panel.php - this is a security risk");
  unlink("control-panel.php");
}

if (file_exists("phpinfo.php")) {
  logit(" -- removing phpinfo.php - this is a security risk");
  unlink("phpinfo.php");
}

if (!file_exists(".htaccess")) {
  logit(" -- installing missing .htaccess file to prevent unauthorised access");
  if (file_exists("$AXYL_HOME/lib/.htaccess")) {
    copy("$AXYL_HOME/lib/.htaccess", ".htaccess");
    logit(" -- done");
  }
  else {
    logit(" -- failed: .htaccess not found in $AXYL_HOME/lib");
  }
}
logit(" -- done.");

// ----------------------------------------------------------------------
// Do these last messages early, since we have to close the logfile before
// we remove and replace the Axyl library..
logit("\nFinal touches..");
logit(" -- old Axyl library removed, img and js links created in new one.");
logit(" -- all permissions in the website tree have been properly set.");
logit("\nAxyl security update completed.");
echo "tidying up..\n";
$LOG->closefile();

// ----------------------------------------------------------------------
// REMOVAL OF LOCAL AXYL LIBRARY
// A soft-link to the full Axyl library, or a physical copy of the full
// Axyl library in the website root is now deprecated - see the notes at
// the top of this script.

if (is_link($LIBDIR)) {
  exec("rm -f $LIBDIR");
}
elseif (is_dir($LIBDIR)) {
  exec("rm -rf $LIBDIR");
}
// Re-create library dir afresh..
makepath("./lib");
// Make links to library images and javascript..
exec("ln -s $AXYL_HOME/lib/img $LIBDIR/img");
exec("ln -s $AXYL_HOME/lib/js $LIBDIR/js");

// ----------------------------------------------------------------------
// SET PERMISSIONS

// Brute force approach..
$THISDIR = realpath(".");
if ($THISDIR != "" && $THISDIR != "/") {
  exec("$AXYL_HOME/install/set-axyl-website-perms.sh");
}
?>



