# Copyright (C) 1999, 2000 Jay Beale
# Copyright (C) 2001-2003 Hewlett Packard Corporation
# Licensed under the GNU General Public License
package Bastille::IOLoader;
use lib "/usr/lib";

require Bastille::API;
import Bastille::API;

use Bastille::TestAPI;

use Exporter;
@ISA = qw ( Exporter );
@EXPORT = qw( Load_Questions compareQandA validateAnswer ReadConfig
	      validateAnswers getRegExp checkQtree outputConfig partialSave
	      %Question %moduleHead
	      );

@ENV="";
$ENV{PATH}="";
$ENV{CDPATH}=".";
$ENV{BASH_ENV}="";

my %deletedQText;  # variable used to store the question text from
                  # questions that will be deleted but are distro appropriate.
                  #  Used in Load_Questions and outputConfig

###############################################################################
# &Load_Questions does:
#
# 1) Create a question record for the Title screen: no question, no default
#    answer, toggle_yn=0, just a Short Explanation=Title Screen
# 2) Load in each question, one by one, by grabbing the expected records one 
#    by one from a file.  Records, within the file, are described below.  
# 3) Append a "Done now--gonna run the script" screen as Y/N question...  Needs
#    to have Yes-Child to be Bastille_Finish  QuestionName:  End_Screen
#
# Record format within Questions file:
# - A record is terminated by a blank line which is not part of a quoted
#   string.
# - A new record is began by the LABEL: tag, along with the index for the
#   question.
# - Otherwise, the entries within the record can be in any order at all.  
# - Multi-line fields must be quoted in double-quotes..
# - Double quotes are allowed inside a string, but must be escaped, like \".
#
# WEIRD CASES:
#
#  1 If a record has no short_exp explanation, none is shown.  This is bad?
#  2 If a record has no question, no question is asked, but the explanation
#    is still displayed.  If this is the case, the default_answer is still
#    entered into the configuration, if it exists.
#  3 If a question has no answer, it doesn't create any blank lines or such
#    in the output file, as it will be skipped in &Output_Config_Files.  For
#    this reason, &Prompt_Input, which is only called when the record contains
#    a user-answerable question, pads a space to the end of any 0-length input.
#    Not to worry: Output_Config_Files replaces said space with 0-length input
#    NOTE: we couldn't just only print the answer field when a real question
#          existed -- this would improperly handle case 2.
#    
###############################################################################

sub Load_Questions($) {
# sub Load_Questions creates a data structure called %%Questions
  my $UseRequiresRules = $_[0];

  my ($current_module_number,$first_question) = &parse_questions();
  $first_question = &prune_questions($UseRequiresRules,$first_question);
  &validate_questions();

  return ($current_module_number, $first_question);
}

sub parse_questions() {

    # Line loaded in from file and it's non-tag data
    my ($line,$data);
    
    # Line number (within the disk file)
    my $line_number=0;
    
    # Module to which the current question being loaded in belongs and
    # the order that Load_Questions loads it in
    my $current_module;
    my $current_module_number=0;
    
    # Question (record) that we're in, listed by index(LABEL) 
    my $current_index;
    
    # The first and last questions -- used for Title_Screen and End_Screen 
    my $first_question="";
    my $previous_question="";
    
    # Field we're in
    my $current_field;
    
    
    # OK, so here's how this goes.  The config file looks has a series of
    # records, which have a series of  TAG: value    lines.  A value may span
    # multiple lines if it was begun with a  "  mark, but is usually expected
    # to be a string.  " marks can occur inside records, if escaped:  \"
    # Records end with a blank line and begin with a LABEL: tag.  We get module
    # names from FILE: somename.pm  lines, which should have whitespace around
    # them...
    
    unless (open QUESTIONS,&getGlobal('FILE',"Questions")) {
       &ErrorLog("ERROR:   Can't open ./Questions.txt data file.\n");
       exit(1);
    }

    while ($line=<QUESTIONS>) {
	
	# Increment line number
	$line_number++;
	
	# If we're currently in a question record...
	if ($current_index) {
	    
	    # If we're currently in a __multi-line__ record (a quoted string),
	    if ($current_field) {
		
		# See if it's terminated in a quote (i.e. is end of a string?)
		my $end_of_string=0;

		if ($line =~ /^(.*)\"\s*\n*$/) {
		    # Make sure the terminating quote isn't an escaped quote
		    my $templine=$1;
		    unless ($templine =~ /\\$/) {
			$line=$templine;
			$end_of_string=1;
		    }
		}
		
		#
		### Text Handling
		#
		
		# Convert escaped quotes \" to real quotes "
		$line =~ s/\\\"/\"/g;
		
		# Strip out terminating \n's
		unless ($line =~ /^\s*\n+\s*$/) {
		    if ($line =~ /^(.*)\n$/) {
			$line = $1 . " ";
		    }
		}
		else {
		    $line .= "\n";
		}
		
		# Add the line to the end of the record and ...
		if ($Question{$current_index}{$current_field} =~ /[^\s\n]$/) {
		    $Question{$current_index}{$current_field} .= " ";
		}
		
		$Question{$current_index}{$current_field}.=$line;
		
		# Check if the record is over.
		if ($end_of_string) {
		    $current_field="";
	        }
            }
            else {
                # We're _not_ in a multi-line record
		
		# Did we hit a blank line? Blank lines, not embedded in
		#  " marks, delimit records
		if ($line =~/^\s*$/) {
		    $current_field="";
		    $current_index="";
		}
		else {
		    # Figure out what field to put this data in...
		    
		    if ($line =~ /^SHORT_EXP:(.*)$/) {
			$current_field="short_exp";
		    }
		    elsif ($line =~ /^LONG_EXP:(.*)$/) { 
			$current_field="long_exp";
		    }
		    elsif ($line =~ /^QUESTION:(.*)$/) {
			$current_field="question";
		    }
		    elsif ($line =~ /^DEFAULT_ANSWER:(.*)$/) {
			$current_field="answer";
		    }
		    # What I wouldn't give for a case/switch in Perl...
		    elsif ($line =~ /^YES_EXP:(.*)$/) {
			$current_field="yes_epilogue";
		    }
		    elsif ($line =~ /^NO_EXP:(.*)$/) {
			$current_field="no_epilogue";
		    }
		    elsif ($line =~ /^CONFIRM_TEXT:(.*)$/) {
			$current_field="confirm_text";
		    }
		    elsif ($line =~ /^REQUIRE_IS_SUID:(.*)$/) {
			$current_field="require_is_suid";
		    }
		    elsif ($line =~ /^REQUIRE_FILE_EXISTS:(.*)$/) {
			$current_field="require_file_exists";
		    }
		    elsif ($line =~ /^REQUIRE_DISTRO:(.*)$/) {
			$current_field="require_distro";
		    }
		    elsif ($line =~ /^YN_TOGGLE:(.*)$/) {
			$current_field="toggle_yn";
		    }
		    elsif ($line =~ /^YES_CHILD:(.*)$/) {
			$current_field="yes_child";
		    }
		    elsif ($line =~ /^NO_CHILD:(.*)$/) {
			$current_field="no_child";
		    }
		    elsif ($line =~ /^SKIP_CHILD:(.*)$/) {
			$current_field="skip_child";
		    }
		    elsif ($line =~ /^PROPER_PARENT:(.*)$/) {
			$current_field="proper_parent";
		    }
		    elsif ($line =~ /^REG_EXP:(.*)$/) {
			$current_field="reg_exp";
		    }
		    elsif ($line =~ /^EXPL_ANS:(.*)$/) {
			$current_field="expl_ans";
		    }
		    else {
			# UH OH!!! We've found a line (inside a record)
			# that isn't recognized
			&ErrorLog("ERROR:   The following line (inside " .
			"a record is not recognized.\n\n$line\n\n".
			"          Line number: $line_number\n");
                        exit(1);
		    }
		    
		    # OK, we've know out what field to assign this data to...
		    $data =$1;
		    
		    # Strip off any initiating white space
		    if ($data =~ /^\s+(.*)$/) {
			$data=$1;
		    }
		    
		    # Strip off any terminating white space...
		    if ($data =~ /^(.*)\s+$/) {
			$data=$1;
		    }
		    
		    # If the data isn't quoted, just finish up...
		    unless ( ($data =~ /^\"/ ) or ($data =~ /[^\\]\"$/) ){
			
			# Convert escaped quote marks
			$data =~ s/\\\"/\"/g;
			
			
			# If this is the REQUIRE_DISTRO field, expand any macros
			if ($current_field eq 'require_distro') {
			    
			    # Replace macros with their (by design) hardcoded values,
			    # making sure to respect recursively defined macros.
			    
			    if ($data =~ /\bLINUX\b/) {
				my $supported_distros = 'RH MN';
				$data =~ s/\bLINUX\b/$supported_distros/;
			    }
			    if ($data =~ /\bRH\b/) {
				my $supported_versions = 'RH6.0 RH6.1 RH6.2 RH7.0 RH7.1 RH7.2 RH7.3 RH8.0 RH9.0 RH9';
				$data =~ s/\bRH\b/$supported_versions/;
			    }
			    if ($data =~ /\bMN\b/) {
				my $supported_versions = 'MN6.0 MN6.1 MN6.2 MN7.0 MN7.1 MN7.2 MN8.0 MN8.1 MN8.2';
				$data =~ s/\bMN\b/$supported_versions/;
			    }
			    if ($data =~ /\bDB\b/) {
				my $supported_versions = 'DB2.2 DB3.0';
				$data =~ s/\bDB\b/$supported_versions/;
			    } 
			    if ($data =~ /\bSE\b/) {
                                my $supported_versions = 'SE7.2 SE7.3 SE8.0';
                                $data =~ s/\bSE\b/$supported_versions/;
                            }
                            if ($data =~ /\bTB\b/) {
                                my $supported_versions = 'TB7.0';
                                $data =~ s/\bTB\b/$supported_versions/;
                            }
			    if ($data =~ /\bHP-UX\b/) {
				my $supported_versions = 'HP-UX11.00 HP-UX11.11 HP-UX11.22 HP-UX11.23';
				$data =~ s/\bHP-UX\b/$supported_versions/;
			    }
			    if ($data =~ /\bOSX\b/) {
				my $supported_versions = 'OSX10.2.0 OSX10.2.1 OSX10.2.2 OSX10.2.3 OSX10.2.4';
				$data =~ s/\bOSX\b/$supported_versions/;
			    }
			}
			
			$Question{$current_index}{$current_field}=$data;
			
			$current_field="";
		    }
		    else {
			# Make sure this looks like a real quoted string
			if ($data !~ /^\s*\"/) {
			    &ErrorLog("ERROR:   Mis-quoted line\n\n$line\n\n" .
			              "         Line number $line_number\n");
                            exit(1);
			}
			
			# Strip off initiating quote mark
		        if ($data =~ /^\s*\"(.*)$/) {
			    $data=$1;
			}
			
			# If this thing has a terminating quote mark, it is a
			# single-line quoted record, probably used to preserve
			# leading or trailling whitespace.
			if ($data =~ /[^\\]\"$/) {
			    
			    if ($data =~ /^(.*)([^\\])\"$/) {
				$data=$1 . $2;
				
				# Convert escaped quote marks
				$data =~ s/\\\"/\"/g;
				
				$Question{$current_index}{$current_field}=$data;
				$current_field="";
			    }
			}
			# Otherwise, it is the beginning of a multi-line record
			else {
			    
			    # Strip off end \n's
			    unless ($data =~ /^\s*\n+\s*$/) {
				if ($data =~ /^(.*)\n$/) {
				    $data=$1 . " ";
				}
			    }
			    
			    # Convert escaped quote marks
			    $data =~ s/\\\"/\"/g;
			    			
			    # Now, actually copy the data in
			    $Question{$current_index}{$current_field}=$data;
			}

		    }		 
		
		}
	    }
	}
	# OK, so we're not in a record (Question) at all...
	else {
	    
	    # Are we starting a new one, on another blank line, or getting a
	    # module name?
	    if ($line =~ /^LABEL:\s*(.*)$/) {
		
		# We have found a new record...
		$current_index=$1;
		
		# Prune whitespace from the name
		if ($current_index=~/^(.*)\s+$/) {
		    $current_index =$1;
		}
		
		$Question{$current_index}{"module"}=$current_module;
		$current_field="";  # This is not a multi-line record
		
                # Record the name of the record so sanity checks can be done
		# later
		$recordnames[@recordnames]=$current_index;
		
		# If this is the first record (question), make the necessary
		# link from Title_Screen record
		unless ( $first_question ) {
		    $first_question=$current_index;
		    $Question{$current_index}{"proper_parent"} = $current_index;
		}
		
		# Save the value of the current index so the End_Screen can
		# find the right parent...
		$previous_question=$current_index;
		
	    }
	    elsif ($line =~ /^FILE:\s*(.*)$/) {
		# Started a new module name...
		$current_module_number++;
		$current_module=$1 . " Module $current_module_number";
	    }
	    elsif ($line =~ /^\s*$/) {
		# A blank line
		# do nothing
	    }
	    else {
                &ErrorLog("ERROR:   Invalid question record found at line $line_number\n" .
                          "         of Questions.txt.  Expecting FILE: or LABEL:\n" .
                          "         Found \n\n$line\n" .
                          "         instead.  This is a fatal error, exiting...\n");
                exit(1);
	    }
	}
    }
    close QUESTIONS;
    
    return ($current_module_number, $first_question);
}

sub prune_questions ($$) {
    my $UseRequiresRules = $_[0];
    my $first_question = $_[1];
    
    ##############################################
    # Walk through $Question hash, eliminating   #
    # questions that don't apply to this system. #
    #                                            #
    # Use the new requires- fields to figure out #
    # which questions to prune.  Prune by simply #
    # moving the parent/child pointers to skip   #
    # around the question.                       #
    #                        - JJB 3/2001        #
    ##############################################

    foreach my $key (@recordnames) {

      # Test all requires 
      #
      # If the distro is correct and any of the other tests pass, 
      # show the question.  Otherwise, skip to the skip_child
      #
      # Example:
      # REQUIRE_DISTRO: RH6.0 HP-UX11.00
      # REQUIRE_IS_SUID: foo bar
      # REQUIRE_FILE_EXISTS: alpha beta gamma
      #
      # should return true iff
      # (we are on a RH6.0 or HP-UX11.00 machine) and
      # (either foo or bar is SUID) or
      # (any of alpha, beta, or gamma exist) or
      # (the internal require test for the question returns 0 or undef)
      #
      # we shortcut the rest of the tests if the DISTRO is wrong
      # for efficiency and because the appropriate hash values
      # may not be defined.

      my $skip_this_question=0;

      my %require_tests;
      
      #print "Key: $key\n";
      
      ####### CHECK OS VERSION TO SEE IF THE QUESTION IS APPLICABLE
      my $require_distro=$Question{$key}{"require_distro"};
      my @require_distro_array;
      (@require_distro_array) = split(/\s+/,$require_distro);
      
      my $distro_is_appropriate=0;
      
      foreach my $distro ( @require_distro_array ) {
	  #print "Testing for distro $distro\n";
	  if ($distro eq &GetDistro ) {
	      $distro_is_appropriate=1;
	  }
      }
      
      if ($distro_is_appropriate) {
	  # Note: UseRequiresRules doesn't work exactly like you'd expect
	  #       because in some cases "not" is implemented
	  #       using the SKIP_CHILD.  Hence, some questions are never
	  #       reached unless you SKIP another question
	  #       When this happens, change the question to use a negated
	  #       test instead of using SKIP.
	  if ( $UseRequiresRules eq 'Y') {
	      
            if (defined $Question{$key}{"require_is_suid"} ) {
	      my $require_is_suid=$Question{$key}{"require_is_suid"};
	      #print "Parsing suid requirement tag: $require_is_suid\n";
	      
	      # NOTE: the anonymous subroutine stuff can get a little
	      # weird.  What we're doing here is defining a code-block
	      # that will be run a little later on.  The value of 
	      # $require_is_suid will be determined at the time the
	      # code block gets run, which is run later on.  It's
	      # still in the same scope, so it will use the same value. 
	      
	      # The "return"s will return out of the anonymous sub, but not
	      # out of the current subroutine
	      
	      $require_tests{"suidbin"} = sub {
		  # we only return 1 (skip) if the REQUIRE_IS_SUID tag has
		  # values and one of them exists.
		  my $retval=0; # no values
		  foreach my $suidbin ( split(/\s+/,$require_is_suid) ) {
		      if ( -u &getGlobal('BIN',$suidbin)) {
			  return 0; # ask question if any suid bits are set
		      } else {
			  $retval=1; # now default to skipping; we've been here
		      }
		  }
		  return $retval;# will skip unless we never looped
	      };
	    }  

            if (defined $Question{$key}{"require_file_exists"} ) {
	      my $require_file_exists=$Question{$key}{"require_file_exists"};
	      #print "Parsing requires file tag: $require_file_exists\n";
	      
	      # NOTE: here's more weird anonymous subroutine stuff.  See
	      # the note above.  Same structure here as the SUID stuff.
	      $require_tests{"file"} = sub { 
		  my $retval=0; # no values
		  foreach my $file ( split(/\s+/,$require_file_exists) ) {
		      if ( -e &getGlobal('FILE',$file)) {
			  return 0; # ask question if any of the files exists
		      } else {
			  $retval=1; # now default to skipping; we've been here
		      }
		  }
		  return $retval; # will skip unless we never looped
	      };
	    }  

	    # TODO: replace all references to REQUIRE_IS_SUID and 
	    # REQUIRE_FILE_EXISTS with internal tests.  Then, just use 
            # B_run_test directly.  For now, we'll make a subroutine 
            # in the hash of tests to call it
	      my $module=$Question{$key}{"module"};
	      # module has a bunch of extra stuff (number, etc.) appended to it
	      my ($shortmod, $rest) = split(/.pm/, $module);
	      
            # Here we use the anonymous subroutines defined for each individual
            # question, which are much more flexible than just the REQUIRE_FILE
            # and REQUIRE_SUID routines.
	      $require_tests{"internaltest"} = sub { &B_run_test($shortmod,$key); };
	      
	      # run non-distro tests.  If the question does not fit, then
	      # juggle the pointers.
	      foreach my $test (keys %require_tests) {
		  #print "running test -- $test --\n";
		  # NOTE: here is where we actually run those anonymous
		  # subroutines defined above.
		  if ( &{$require_tests{"$test"}}) {
                      &ActionLog("Question $shortmod.$key will be skipped " .
                                 "because of the $test test\n");
		      $skip_this_question=1;
		      last;
		  }
	      }
	  }       
      } else {
	  $skip_this_question=1;
      }
	
      # OK, if we didn't meet all the requirements, skip this question.
      # 
      # This is rudimentary pointer mangling.  There are serious speed-ups
      # that we can make by thinking more about tree traversals -- this 
      # is the "simple" implementation intended to introduce the 
      # functionality.  Let's speed it up later, for 1.2.x, x>0.
      #
      # - JJB 3/2001
	
      #print "skip_this_question=$skip_this_question\n";
      if ($skip_this_question) {
	  
	  my $parent=$Question{$key}{"proper_parent"};
	  my $child;
	  
	  # Choose the next question to go to carefully
	  if ($Question{$key}{"yes_child"} eq $Question{$key}{"no_child"}) {
	      $child = $Question{$key}{"yes_child"};
	  }
	  
	  # if there is a skip child, use it
	  if (defined $Question{$key}{"skip_child"}) {
	      $child = $Question{$key}{"skip_child"};
	  }
	  
	  #Now do the pruning.
	  if ($child) {
	      # insure that first question is a valid question
	      if("$key" eq "$first_question"){
		  $first_question = $child;
	      }
	      my $loop_over_key;
	      #print "Pruning $key\n";
	      foreach $loop_over_key (keys(%Question)) {
		  
		  # Any questions which have the phantom question as a child
		  # should now point to the phantom's child instead.
		  if ($Question{$loop_over_key}{"yes_child"} eq $key) {
		      $Question{$loop_over_key}{"yes_child"}=$child;
		  }
		  if ($Question{$loop_over_key}{"no_child"} eq $key) {
		      $Question{$loop_over_key}{"no_child"}=$child;
		  }
		  # This gets tricky...think about this one deeply before
		    # emailling me on this.  - JJB
		  if ($Question{$loop_over_key}{"proper_parent"} eq $key) {
		      
		      $Question{$loop_over_key}{"proper_parent"}=$parent;
		  }
	      }
	      if($distro_is_appropriate) {
		  $deletedQText{$key} = $Question{$key}{'question'};
	      }
	      
	      $Question{$key}{'deleteme'} = "Y";
	      
	  }
	  else {
	      &ErrorLog("ERROR:   Question $key couldn't be skipped because Bastille\n" .
                        "         couldn't figure out which question to skip to!\n");
	  }
      }
  }
    
    ##############################################
    #   Delete irrelevant questions.             #
    ##############################################
    foreach my $key (keys %Question) {
	
	if($Question{$key}{'deleteme'} eq "Y"){
	    delete $Question{$key};
	}
	else {
	    $Question{$key}{"default_answer"}=$Question{$key}{'answer'};
	}

    }
    return $first_question;
}

sub validate_questions () {
    ##############################################
    #   Run sanity checks on questions database  #
    ##############################################
    
    foreach my $key (keys %Question) {
	
	my ($parent,$yes_child,$no_child);

	$parent=$Question{$key}{"proper_parent"};
	$yes_child=$Question{$key}{"yes_child"};

	my ($current_module,$leftover) = split /.pm/, $Question{$key}{'module'};
	my ($parent_module,$pleftover) =  split /.pm/, $Question{$parent}{'module'};

        my $no_child_to_print="";

	if ($Question{$key}{"toggle_yn"}) {
	    $no_child=$Question{$key}{"no_child"};
            my $no_child_to_print=$no_child;
	}

	&DebugLog("LABEL: $key\n".
	          "Yes-child: $yes_child\n".	    
	          "No-child:  $no_child_to_print\n".
	          "Parent:    $parent\n".
	          "Short expression:\n".
	          $Question{$key}{"short_exp"}.
	          "Long expression:\n".
	          $Question{$key}{"long_exp"}.
	          "Question:\n".
	          $Question{$key}{"question"}.
	          "\nDefault: ". $Question{$key}{"default_answer"}."\n\n");

        my $problemfound=0;
        unless ($parent) {
	    &ErrorLog("ERROR:   Problem found in Question database. $key doesn't have a parent!\n" .
                      "         This is likely to cause problems later.\n");
            $problemfound=1;
        }

        unless (exists ($Question{$parent})) {
	    &ErrorLog("ERROR:   Problem found in Question database. $key\'s parent \"$parent\"\n" .
                      "         does not exist!  This is likely to cause problems later.\n");
            $problemfound=1;
        }

	# Allows for header/footer question wrap to come later. ie Title_Screen End_Screen
	if(exists $Question{$key} && $Question{$key}{"yes_child"} !~ "End_Screen"){
            unless (exists ($Question{$yes_child})) {
	        &ErrorLog("ERROR:   Problem found in Question database. $key\'s yes_child \"$yes_child\"\n" .
                          "         does not exist!  This is likely to cause problems later.\n");
                $problemfound=1;
            }
	}

        unless ($yes_child) {
	        &ErrorLog("ERROR:   Problem found in Question database. $key has no yes child.\n" .
                          "         This is likely to cause problems later.\n");
                $problemfound=1;
        }

	if (exists $Question{$key} && $Question{$key}{"toggle_yn"}) {
            unless ($no_child) {
	        &ErrorLog("ERROR:   Problem found in Question database. $key has no no_child.\n" .
                          "         This is likely to cause problems later.\n");
                $problemfound=1;
            }

	    # Allows for header/footer question wrap to come later. ie Title_Screen End_Screen
	    if(exists $Question{$key} && $Question{$key}{"no_child"} !~ "End_Screen"){
                unless (exists ($Question{$no_child})) {
	            &ErrorLog("ERROR:   Problem found in Question database. $key\'s no_child \"$no_child\"\n" .
                              "         does not exist!  This is likely to cause problems later.\n");
                    $problemfound=1;
                }
	    }

            unless ( $Question{$key}{"question"} ) {
	            &ErrorLog("ERROR:   Problem found in Question database. y/n question $key\n" .
                              "         has no Question!  This is likely to cause problems later.\n");
                    $problemfound=1;
            }
	    
	}

        if ($problemfound) {
           &ErrorLog("ERROR:   Earlier problems are preventing correct Bastille execution.  Exiting.\n");
           exit(1);
        }

	# finds the first question in each module.
	if($parent_module ne $current_module){
	    # moduleHead is a global that will be sent to Interactive for progress indication
	    $moduleHead{$current_module} = $key;
	}

    }
    
#    if ($TEST_ONLY) {
#	exit;
#    }      
    
    # Return number of modules loaded in and the index of the first questions.
}

###########################################################################
# &ReadConfig reads in the user's choices from the TUI, stored in the file
# $GLOBAL_BFILE{"config"}.  We were using AppConfig here at first, but it was
# just such a pain in the, ermm, keyboard...
#
###########################################################################

sub ReadConfig {

    if (open CONFIG, &getGlobal('BFILE', "config")) {
	while (my $line = <CONFIG>) {
	    chomp $line;
	    # Skip commented lines...
	    unless ($line =~ /^\s*\#/) {
		if ($line =~ /^\s*(\w+).(\w+)\s*=\s*\"(.*)\"/ ) {
		    $GLOBAL_CONFIG{$1}{$2}=$3;
		    
		    if (exists $Question{$2}) {
			# This is only used by the front end to populate the 
			# "defaults".  It will cause problems with the backend
			# if we accidentally create a %Question entry based on
			# the config file for a question that didn't exist
			$Question{$2}{'answer'} = $3;
		    }
		}  # if the line contains non-whitespace
		elsif($line !~ /^\s*$/) {
		    &ErrorLog("\n" .
			      "WARNING: The following line in the configuration file is invalid:\n" . 
			      "         $line\n" .
			      "         The line will be disregarded.\n\n");
		}
	    }
	}
	close CONFIG;
      return 1;
    }

    # Failed to open config
    return 0;
}



######################################################################
# compareQandA($first_question)
#      This subroutine takes the pruned questions hash and the
#      GLOBAL_CONFIG hashes and does an index compare of the two
#      This program is meant to be run just before the backend
#      It is designed to insure multi-system support.  That is,
#      it tests the config file for question validity on the current
#      machine before the backend will run.
#
#      This function returns:
#        0 for WARNING questions were not answered or questions were
#          answered that do not apply to the current system.
#        1 for correct match of questions and answers.
#
#      REQUIRES %Question
#      REQUIRES %GLOBAL_CONFIG
#      REQUIRES &ActionLog
#      REQUIRES &ErrorLog
#
######################################################################

sub compareQandA($$) {
###
    my $first_question = $_[0];
    my $force = $_[1];
    my $returnValue = "";
    my $sumNotAsked = 0;
    my $warnFlag = 0;

    # this checks to see if any questions were not answered that should
    # have been.
    my ($moduleNotAnswered,$questionNotAnswered) = &checkQtree($first_question);

    
    # if checkQtree returns a question that has not been answered
    if ($questionNotAnswered ne "" && ! $force) {
	&ErrorLog("ERROR:   A fatal error has occurred. Not all of the questions\n" .
		  "         that pertain to this system have been answered.  Rerun\n" .
		  "         the interactive portion of bastille on this system.\n" . 
		  "         MODULE.QUESTION=$moduleNotAnswered.$questionNotAnswered\n");
	exit(1);
    }

    # This section checks to see if a question was answered that does
    # not make sense on this machine.
    for my $module ( keys %GLOBAL_CONFIG ) {
	for my $key (keys %{$GLOBAL_CONFIG{$module}}){
	    # check to see if the question should be answered
	    if( (!(exists $Question{$key}) ) || ($Question{$key}{"mark"} ne "OK") ){
		# This prunes the answer out if the question should
		# not have been answered
		my $parent = $Question{$key}{'proper_parent'};
		my ($parentMod, $junk) = split /.pm/, $Question{$parent}{'module'};

#		print "key=$key\n" . "parent=$parent\n" . 
#		      "toggle_yn=$Question{$parent}{'toggle_yn'}\n" . 
#		      "parent no_child=$Question{$parent}{'no_child'}\n" .
#		      "parent yes_child=$Question{$parent}{'yes_child'}\n" .
#		      "parent answer=$GLOBAL_CONFIG{$parentMod}{$parent}\n";
		# This logic tells us if other values in the config will be affected by removing this answer
		if($Question{$parent}{'toggle_yn'} eq "1")  {
			  
		    if($Question{$parent}{'no_child'} ne $Question{$parent}{'yes_child'}) {
			
			if($Question{$parent}{'no_child'} eq $key && $GLOBAL_CONFIG{$parentMod}{$parent} eq "Y"){
			    $warnFlag = 1;
			}
			elsif($Question{$parent}{'yes_child'} eq $key && $GLOBAL_CONFIG{$parentMod}{$parent} eq "N"){
			    $warnFlag = 1;
			}
		    }
		}
		if(! $force) {
		    delete $GLOBAL_CONFIG{$module}{$key};
		    # checking to see if this answer is appropriate for this OS.
		    if(! exists $deletedQText{$key} ) {
			# Warn the user that this question will not run on 
			# their system as it is was not designed for their OS.
			&ErrorLog("\nWARNING: $module\.$key was removed (not applicable).\n");
		    }

		    $sumNotAsked++;
		}
	    }

	}
    }

    # Logging this subroutines actions.

    if($sumNotAsked > 0){

	&ActionLog("\nNOTE:    $sumNotAsked question(s) were answered that do not pertain to this system.\n" .
		   "         Answered questions that do not pertain to this machine have\n" . 
		   "         been removed.\n");

	if($warnFlag){
	    &ErrorLog("\nWARNING: The configuration file appears to contain invalid entries.\n" . 
		      "         Bastille will continue but you should rerun the interactive\n" . 
		      "         portion of Bastille to correct the invalid portions of the\n" .
		      "         configuration file.\n\n");
	    &ActionLog("\nWARNING: The configuration file appears to contain invalid entries.\n" . 
		      "         Bastille will continue but you should rerun the interactive\n" . 
		      "         portion of Bastille to correct the invalid portions of the\n" .
		      "         configuration file.\n\n");
	}

	$returnValue = 0;
    }

    # return 1 for success and 0 for Warnings that were reported.
    return $returnValue;
}

######################################################################
#  &validateAnswer($question,$answer)
#     This subroutine takes the in the LABEL of a question and the
#     answer that is being proposed.  Both in string form
#     It then checks the proposed answer against a regular expression
#     that is listed in Questions.txt as REG_EXP and in the Question
#     hash as $Question{$question}{"reg_exp"}.
#     If the reg_exp matches the proposed question then 1 is returned
#     otherwise 0 is returned.
#     An exception to this rule is if the reg_exp field is not present
#     then an 1 is returned suggesting that any answer will do.
#
#     REQUIRES %Questions
#     REQUIRES &ErrorLog
#     REQUIRES &getRegExp
#
######################################################################
sub validateAnswer($$) {

    my $question = $_[0];
    my $answer = $_[1];

    if( defined &getRegExp($question)){
	
	$pattern = &getRegExp($question);
	if( $answer =~ /$pattern/ ){
	    return 1;
	}
	else {
	    return 0;
	}
    }
    elsif( exists $Question{$question} ) {
	return 1;
    }
    else {
	&ErrorLog("Could not find \"$question\" in the Questions hash\n");
	return 0;
    }


}

######################################################################
#  &validateAnswers
#     This subroutine checks the proposed answers against a regular 
#     expressions that are listed in Questions.txt as REG_EXP and in 
#     the Question hash as $Question{$question}{"reg_exp"}.
#     If the reg_exp matches for all the proposed answers then 1 is 
#     returned otherwise a non-zero exit is performed and the user
#     is asked to rerun Interactive Bastille.
#
#     This subroutine is to be used in the backend as a qualifier to
#     running the code.
#
#     REQUIRES %GLOBAL_CONFIG
#     REQUIRES %Questions
#     REQUIRES &validateAnswer
#     REQUIRES &ActionLog
#     REQUIRES &ErrorLog
#
######################################################################

sub validateAnswers {
    
    for my $module ( keys %GLOBAL_CONFIG ){
	for my $question (keys %{ $GLOBAL_CONFIG{$module} } ){
	    
	    my $answer = $GLOBAL_CONFIG{$module}{$question};
	    if(! &validateAnswer($question,$answer)){
		my $error = "ERROR: A fatal error has occured. On the following\n" . 
		      "line of Bastilles config, the specified answer does\n" .
		      "not match the following perl regular expression.\n" . 
		      "config: $module.$question=$answer\n" . 
		      "Regular Expression: \"". &getRegExp($question) . "\"\n" . 
		      "Please run the interactive portion of Bastille again\n" .
		      " and fix the error.\n";
		&ErrorLog( $error );
		exit(1);  
	    }
	}
    }
 
   &ActionLog("Validated config file input\n");
    return 1;
}

######################################################################
#  &getRegExp($question)
#     This subroutine is a lookup function that for a given question
#     label will return a regular expression that is defined.
#     If no regular expression is defined for that question then
#     this subroutine will return undefined.
#
#     REQUIRES: %Questions
#    
######################################################################

sub getRegExp($) {

    my $question = $_[0];
    
    if( exists $Question{$question}{"reg_exp"} ) {
	return $Question{$question}{"reg_exp"};
    } 
    else {
	return undef;
    }
}

######################################################################
#  &checkQtree($first_question);
#    This subroutine checks to see if all applicable Questions
#    have been asked on this system.  If it finds a discontinuity
#    in the pruned questions tree vs the GLOBAL_CONFIG hash it will
#    return the ($offending_module,$offending_key).  Otherwise it
#    will return NULL stings.  i.e. ("","")
#
#    This subroutine also marks the questions that have answers in
#    the GLOBAL_HASH.  This allows &compareQandA to activly delete
#    GLOBAL_HASH keys if they are not apropriate for the current
#    machine.
#
#    REQUIRES: %Questions
#    REQUIRES: %GLOBAL_CONFIG
#
######################################################################

sub checkQtree($) {

    my $first_question = $_[0];
    my $current_question = $first_question;

    while( $current_question ne "End_Screen" ) {
	my ($module,$ext) = split /\.pm/, $Question{$current_question}{"module"};

	# check and see if this record is a question...
	if( $Question{$current_question}{"question"} ne "" ) {
	    # This question should have an answer...
	    if( ! (exists $GLOBAL_CONFIG{$module}{$current_question})){
		# This question has no answer and should...
		return ($module,$current_question);
	    }
	    elsif($Question{$current_question}{"toggle_yn"} == 1) {
		# this is a yes or no question
		if($GLOBAL_CONFIG{$module}{$current_question} eq "Y"){
		    $Question{$current_question}{"mark"} = "OK";
		    $current_question=$Question{$current_question}{"yes_child"};
		}
		else {
		    $Question{$current_question}{"mark"} = "OK";
		    $current_question=$Question{$current_question}{"no_child"};
		}
		
	    }
	    else {
		$Question{$current_question}{"mark"} = "OK";
		$current_question=$Question{$current_question}{"yes_child"};
	    }
	}
	else {
	    $current_question=$Question{$current_question}{"yes_child"};
	}

    }
    # all of the questions that should be answered are.
    return ("","");
}


######################################################################
#  &outputConfig;
#
#    This subroutine writes out a configuration
#    file.  It uses Global_Config as a data source and will write
#    out all defined values excepting End_Screen.
#
#    REQUIRES: %GLOBAL_CONFIG
#    REQUIRES: %Question
#    REQUIRES: %deletedQText
#
######################################################################

# When does a previously answered question get written out to the
# config file?

#Always write out answers to questions which the user just answered.
#
#For answers which were retrieved from the config file, there are the
#following cases:
#
#Case                        write    GUI behavior       backend behavior
#                            answer?  (questions)        (if answer is missing)
#----------------------------------------------------------------------------
# Pruned (can't get to in GUI):
#  Configured Securely              Y      don't ask          don't warn
#  Missing software
#    - Security related             Y      ask different Q    ensure other Q 
#                                          (install foo?)     is answered   
#    - non-security related         Y      don't ask          don't warn
#
#  distro not applicable            Y      warn (not asking   warn (not doing 
#                                          foo)               foo)
# Not pruned:
#  Question depends on Y/N
#    from another question          N      ask Q if user      warn (invalid 
#                                          changes answer     config)

sub outputConfig {

    my %CONFIG;

    my $config = &getGlobal('BFILE', "config");

#   Needs to use a tree traversal as well as the proper distro deletion items
    my $index="Title_Screen";

    while ($index ne "End_Screen") {

	if ($Question{$index}{"question"} ne "" && exists $Question{$index}{"answer"}) {

	    # If the answer is just a space (the way the &Prompt_Input sub
	    # designates a blank line, strip it.	    
	    if ($Question{$index}{"answer"} =~ /^\s+$/) {
		$Question{$index}{"answer"} = "";
	    }

	    my $module = $Question{$index}{"module"};
	    if ($module =~ /^([^.]+).pm/) {
		$module =$1;
	    }
	    # adding this question to the config hash which will be written out
	    $CONFIG{$module}{$index} = $Question{$index}{"answer"};

	    
	}
	if ($Question{$index}{"toggle_yn"} == 0) {
	    $index=$Question{$index}{"yes_child"};
	}
	else {
	    if ($Question{$index}{"answer"} =~ /^\s*Y/i) {
		$index=$Question{$index}{"yes_child"};
	    }
	    elsif ($Question{$index}{"answer"} =~ /^\s*N/i) {
		$index=$Question{$index}{"no_child"};
	    }
	    else {
                &ErrorLog("ERROR:  Internal Bastille error on question $index.  Answer\n" .
                          "        to y/n question is not 'Y' or 'N'.\n");
	    }
	}
    }

    # We already got the answers which the user just put in, so now we start
    # looping through the GLOBAL_CONFIG looking for deleted questions that 
    # have been answered (possibly due to an OS switch or the action  
    # already having been performed and it does not make sense to attempt to 
    # perform the action) i.e. the configurable software is not installed. 
    foreach my $module (keys %GLOBAL_CONFIG) {
	foreach my $question (keys %{$GLOBAL_CONFIG{$module}}) {
	    if((defined $GLOBAL_CONFIG{$module}{$question}) && ($module ne "End")){
		# if the question is defined in the deletedQText hash then 
                # it is distro appropriate and therefore should be saved 
                # to maintain state across Bastille backend/frontend runs.
		if( defined $deletedQText{$question} ){ 
		    $CONFIG{$module}{$question} = $GLOBAL_CONFIG{$module}{$question};
		}
	    }
	}
    }
    

    # it is finally time to print the config file out.
    unless (open FORMATTED_CONFIG,"> $config") {
        &ErrorLog("ERROR:   Couldn't not write to " . $config  ."\n");
        exit(1);
    }

    foreach my $module (sort keys %CONFIG) {
	foreach my $question (sort keys %{$CONFIG{$module}}) {
	    if((defined $CONFIG{$module}{$question}) && ($module ne "End")){
		# if the question is defined in the Question hash then 
		if( defined $Question{$question}{'question'} ) {
		    print FORMATTED_CONFIG "# Q:  $Question{$question}{question}\n";
		    print FORMATTED_CONFIG "$module\.$question=\"$GLOBAL_CONFIG{$module}{$question}\"\n";
		}
		# if the question is defined in the deletedQText hash then 
                # it is distro appropriate and therefore should be saved 
                # to maintain state across Bastille backend/frontend runs.
		elsif( defined $deletedQText{$question} ){ 
		    print FORMATTED_CONFIG "# Q:  $deletedQText{$question}\n";
		    print FORMATTED_CONFIG "$module\.$question=\"$GLOBAL_CONFIG{$module}{$question}\"\n";
		}
		    
	    }
	}
    }

    close(FORMATTED_CONFIG);
    
    
}


######################################################################
#  &partialSave;
#
#    This subroutine writes out an incomplete configuration 
#    file.  It uses Global_Config as a data source and will write
#    out all defined values excepting End_Screen.
#
#    REQUIRES: %GLOBAL_CONFIG
#    REQUIRES: %Question
#    REQUIRES: %deletedQText
#
######################################################################

sub partialSave {
    my $config = &getGlobal('BFILE', "config");
    unless (open FORMATTED_CONFIG,"> $config") {
        &ErrorLog("ERROR:   Couldn't not write to " . $config  ."\n");
        exit(1);
    }

    foreach my $module (sort keys %GLOBAL_CONFIG) {
	foreach my $question (sort keys %{$GLOBAL_CONFIG{$module}}) {
	    if((defined $GLOBAL_CONFIG{$module}{$question}) && ($module ne "End")){
		# if the question is defined in the Question hash then 
		if( defined $Question{$question}{'question'} ) {
		    print FORMATTED_CONFIG "# Q:  $Question{$question}{question}\n";
		    print FORMATTED_CONFIG "$module\.$question=\"$GLOBAL_CONFIG{$module}{$question}\"\n";
		}
		# if the question is defined in the deletedQText hash 
                # then it is distro appropriate and therefore should be 
                # saved to maintain state across Bastille backend/frontend runs.
		elsif( defined $deletedQText{$question} ){ 
		    print FORMATTED_CONFIG "# Q:  $deletedQText{$question}\n";
		    print FORMATTED_CONFIG "$module\.$question=\"$GLOBAL_CONFIG{$module}{$question}\"\n";
		}
	    }
	}
    }

    close(FORMATTED_CONFIG);
    
}



1;



