#ifndef _PART_H_
#define _PART_H_

#include "compound.h"
#include "position.h"

class Reference;
class Event;
class Table;
class PrPart;
class Track;
class Symbol;

/** A Part is thought of as a container for events (like Notes, MasterEvents, etc. An Event is
 * an Atom which has a position and a length). These events have to appear ordered in their
 * position property. More than one event at the same position are admitted.
 */
class Part : public Compound
{

 private:
  Position _pos;
  int _key;
  int _clef;
  int _meter0;
  int _meter1;
  int _program;
  Part * _ghost_of;
  int _ghosts;

  Track * _track;

 protected:
  PrPart * _pr;

 public:

  /**
   ** The Part's Constructors
   ** -----------------------
   **/

  /** Constructs an empty part with an empty memory */
  Part();

  /** Constructs a part from another part. Caution: Memory is not duplicated! */
  Part(const Part&);

  /** Constructs a part and remembers the track it belongs to (important for presentations) */
  Part(Track*);

  /** Destructor */
  ~Part();

  /**
   ** Manipulationg the Part's Properties
   ** -----------------------------------
   **/

  /** returns the start position of the part */
  Position start() const { return _pos; }

  /** returns the (absolute) start position of the event (within the track) */
  Position start(Event * ev) const;

  /** returns the (absolute) end position of the event (within the track) */
  long end(Event * ev) const;

  /** returns the key */
  int key() const { return _key; }

  /** returns the clef */
  int clef() const { return _clef; }

  /** returns the numerator of the measure */
  int meter0();

  /** returns the denominator of the measure */
  int meter1();

  /** returns the midi program */
  int program() const { return _program; }

  /** returns the track the part belongs to, or 0 if this informaition does not exist! */
  Track * track() const { return _track; }

  /** sets the start position of the part */
  void setStart(Position);

  /** sets the (absolute) start position p of the event (relative to the part) */
  void setStart(Event * ev, Position pos);

  /** sets the key */
  void setKey(int key);

  /** sets the clef */
  void setClef(int clef);

  /** sets the numerator and denominator of the measure */
  void setMeter(int numerator,int denominator);

  /** sets the numerator of the measure */
  void setMeter0(int numerator);

  /** sets the denominator of the measure */
  void setMeter1(int denominator);

  /** sets the midi program */
  void setProgram(int program);

  /** sets the track */
  void setTrack(Track *);


  bool eventEndsAt(Event * ev, long p);
  bool eventStartsAt(Event * ev, long p);
  bool eventStartsAfter(Event * ev, long p);


  /**
   ** Concerning the Presentation
   ** ---------------------------
   */

  /** This funktion returns a pointer to its presentation
   */
  PrPart * presentation() const { return _pr; }

  void hide();

  void show();

  /**
   ** Other Functionality
   ** -------------------
   */

  /*
   */
  Event * grabEvent(Position p);

  /** The parts have to be ordered with increasing position. This method ensures consistency.
   */

  /** returns a list of references containing events between left and right, where notes also
   * have their pitch to be between lowPitch and highPitch
   */
  Reference * makeRefs(int lowPitch, int highPitch, long left, long right);
  void reorder();

  void partCopy();

  /** glues two parts. Returns the 'lost' part which should not be deleted if the operation is to be undone
   */
  Part * partGlue();

  /** splits the part at position p and returns the event after p (or null)
   */
  Event * partSplit(Position p);

  /** this add method differs from the compounds add method by inserting the new element not at the end of the part, but
   * at the correct position
   */
  virtual void add(Element *);

  Symbol * setSymbol(Position p, int sym, int off, int par=-1);

  /**
   ** Virtual operations, each element has to implement
   ** -------------------------------------------------
   **/

  virtual ostream & print(int,ostream&) const;
  virtual void flush(const char*) const;
  virtual Element * copy() const;

  Element * ghostcopy();
  Part * ghostOf() const { return _ghost_of; }
  int ghosts() const { return _ghosts; }
  void incGhosts();
  void decGhosts();

  static Element * load(char*,ifstream*&,Element*);

};

#endif
