#ifndef _PLAYER_CPP_
#define _PLAYER_CPP_

#include <iostream.h>
#include <string.h>

#define protected public // HACK! makes it compile with standard libkmid (stw)
#include <libkmid/midiout.h>
#include <libkmid/deviceman.h>
#include <libkmid/fmout.h>
#undef protected

#include "config.h"

#include "player.h"
#include "reference.h"
#include "part.h"
#include "track.h"
#include "note.h"
#include "midiEvent.h"
#include "masterEvent.h"
#include "audioEvent.h"
#include "song.h"
#include "table.h"
#include "element.h"
#include "scoreTrack.h"
#include "drumTrack.h"
#include "prProgress.h"
#include "prMainEditor.h"
#include "songIterator.h"
#include "addElement.h"

#include "artsmidi.h"
#include "alsaManager.h"
#include "soundserver.h"
#include "BrahmsArts.h"

extern Table * selectioN;

using namespace Arts;

struct PlayerState {
  TimeStamp playTime;
  TimeStamp startTime;
  MidiManager midiManager;
  MidiPort midiPort;
  MidiClient midiClient;
  SoundServer soundServer;
  BrahmsMidiRecorder midiRecorder;
  MidiClient midiClientRec;
  Part * record_part;
  int record_track;
  long pp;
  int ntracks;
  SongIterator * iterator;
  long left;
  long right;
  double interval;
  // unsigned long volume;
};


extern int output;
extern PrMainEditor * mainEditor;


Player::Player() : _song(0), devnum(0), _playing(false) {
  init();
}

Player::Player(Song * s) : _song(s), devnum(0), _playing(false) {
  init();
}

Player::~Player() {
  freePlayer();
  // TODO: delete devman; // causes a segfault invalid filedescriptor
}



void Player::init() {
  if (output==KMID) {
    //
    // KMID
    //
    char * map_path = NULL;
    char *kdedir = getenv("KDEDIR");
    if (kdedir) {
      string path = kdedir;
      path += "/share/apps/kmid/fm";
      FMOut::setFMPatchesDirectory(path.c_str());
    }
    else // fallback
      FMOut::setFMPatchesDirectory("/opt/kde/share/apps/kmid/fm");
    devman = new DeviceManager(0);
    devman->initManager();
    MidiMapper *map=new MidiMapper(map_path);
    devman->setMidiMap(map);
    char * line;
    devnum = devman->synthDevices() + devman->midiPorts(); // my AWE has 2 synths and 1 MIDI
    cout << " devnum: " << devnum << endl;
    for (int i=0; i<devnum && i<MAXDEVNUM; i++) { devices[i] = strdup(devman->name(i)); line = devices[i]; line+=9; line[0]=0; }
  } else if (output==ARTS) {
    //
    // ARTS
    //
    initPlayer();
  } else if (output==ALSA) {
    //
    // ALSA
    //
    initPlayer();
  } else {
    cout << "output type " << output << " is not supported!" << endl;
  }
}

const char * Player::device(int i) {
  const char * dev = 0;
  if (output==KMID) {
    //
    // KMID
    //
    dev = devices[i];
  } else if (output==ARTS) {
    //
    // ARTS
    //
    dev = "aRts";
  } else if (output==ALSA) {
    //
    // ALSA
    //
#ifdef HAVE_ALSA
    dev = alsa->device(i);
#endif
  }
  return dev;
}

int Player::devicenum() {
  int ret = 0;
  if (output==KMID) {
    //
    // KMID
    //
    ret = devnum;
  } else if (output==ARTS) {
    //
    // ARTS
    //
    ret = 1;
  } else if (output==ALSA) {
    //
    // ALSA
    //
#ifdef HAVE_ALSA
    ret = alsa->devnum();
#endif
  }
  return ret;
}

void Player::play(PrProgress * progress, Part * part, long left, long right, bool cycl, int record_track) {

  if (output==KMID) {
    //
    // KMID
    //
    if (!_playing) {
      _playing = true;

      _playing = false;
    }
  } else if (output==ARTS) {
    //
    // ARTS
    //
    ps->record_track = record_track;
    if (record_track!=-1)
      ps->record_part = (Part*) ((::Reference*)selectioN->first())->getValue();
    else
      ps->record_part = 0;
    playStart(progress, part, left, right);
    mainEditor->startTimer();

  } else if (output==ALSA) {
    //
    // ALSA
    //
    playStart(progress, part, left, right);
    mainEditor->startTimer();

  }
}


void Player::wait(long timePos, long offset) {
  if (output==KMID) {
      //
      // KMID
      //
    if (oldTimePos!=timePos) {
      devman->wait(timePos-offset);
      oldTimePos=timePos;
    }
    needSync=true;
  }
}



void Player::hit(int out, int ch, int freq, int vol) {
  if (!_playing) {
    _playing = true;
    if (output==KMID) {
      //
      // KMID
      //
      if (devnum>out) {
	devman->chn2dev[ch] = out;
	devman->openDev();
	devman->initDev();
	if (freq!=0) {
	  devman->tmrSetTempo(100);
	  devman->setVolumePercentage(100);
	  // devman->chnPatchChange(ch,0);
	  devman->wait(0);
	  devman->noteOn(ch,freq,vol);
	  devman->sync();
	  devman->wait(200);
	  devman->noteOff(ch,freq,vol);
	  devman->sync();
	  devman->tmrStop();
	}
	devman->closeDev();
      }
    } else if (output==ARTS) {
      //
      // ARTS
      //
      ps->playTime = ps->midiPort.time();
      // ps->midiPort.processCommand(MidiCommand(mcsProgram | ch, 0 /*=prg*/, 0));
      Arts::MidiEvent ev;
      ev.time = ps->playTime;
      ev.command = MidiCommand(mcsNoteOn|ch, freq, vol);
      ps->midiPort.processEvent(ev);

      ps->playTime.usec += 60*1000000/_song->tempo(); // 1/4 note
      ps->playTime.sec += ps->playTime.usec / 1000000;
      ps->playTime.usec %= 1000000;

      ev.time = ps->playTime;
      ev.command = MidiCommand(mcsNoteOff|ch, freq, 0);
      ps->midiPort.processEvent(ev);
    } else if (output==ALSA) {
      //
      // ALSA
      //
#ifdef HAVE_ALSA
      alsa->hit(out, ch, freq, vol);
#endif
    }
    _playing = false;
  }
}

void Player::quad(int out, int ch, int freq1, int freq2, int freq3, int freq4, int vol) {
  if (!_playing) {
    _playing = true;
    if (output==KMID) {
      //
      // KMID
      //
      if (devnum>out) {
	devman->chn2dev[ch] = out;
	devman->openDev();
	devman->initDev();
	devman->tmrSetTempo(100);
	devman->setVolumePercentage(100);
	//devman->chnPatchChange(ch,0);
	devman->wait(0);
	if (freq1!=0) devman->noteOn(ch,freq1,vol);
	if (freq2!=0) devman->noteOn(ch,freq2,vol);
	if (freq3!=0) devman->noteOn(ch,freq3,vol);
	if (freq4!=0) devman->noteOn(ch,freq4,vol);
	devman->sync();
	devman->wait(600);
	if (freq1!=0) devman->noteOff(ch,freq1,vol);
	if (freq2!=0) devman->noteOff(ch,freq2,vol);
	if (freq3!=0) devman->noteOff(ch,freq3,vol);
	if (freq4!=0) devman->noteOff(ch,freq4,vol);
	devman->sync();
	devman->tmrStop();
      }
      devman->closeDev();

    } else if (output==ARTS) {
      //
      // ARTS
      //
      cout << "TODO: ARTS: Player::quad" << endl;
    } else if (output==ALSA) {
      //
      // ALSA
      //
      cout << "TODO: ALSA: Player::quad" << endl;
    }
    _playing = false;
  }
}



// #########################################
// ##                                     ##
// ##                                     ##
// ##             ARTS/ALSA               ##
// ##                                     ##
// ##                                     ##
// #########################################




static float deltaT(TimeStamp a, TimeStamp b)
{
  return ((float)(a.sec - b.sec) + (float)(a.usec - b.usec)/1000000.0);
}


double Player::time() {
  if (output==ARTS)
    return ((double)(ps->midiPort.time().sec - ps->startTime.sec) + (double)(ps->midiPort.time().usec - ps->startTime.usec)/1000000.0);
  else if (output==ALSA) {
#ifdef HAVE_ALSA
    return alsa->time();
#endif
  }
  return 0.0;
}
  
long Player::ticks() {
  if (output==ARTS) {
    return long(_song->tempo()*384*time()/60);
  } else if (output==ALSA) {
#ifdef HAVE_ALSA
    return alsa->ticks();
#endif
  }
  return 0L;
}

long Player::volume() {
  return 0; //ps->volume;
}

void Player::initPlayer() {
  if (output==ARTS) {
    //
    // ARTS
    //
    ps = new PlayerState;
    ps->iterator = 0;
    bool veto = false;

    ps->soundServer = Arts::Reference("global:Arts_SoundServer");
    if (ps->soundServer.isNull()) {
      mainEditor->fatalError(PrMainEditor::errorArtsConnect);
      cout << "soundServer is null" << endl; veto = true;
    }

    ps->midiManager = Arts::Reference("global:Arts_MidiManager");
    if (ps->midiManager.isNull()) {
      mainEditor->fatalError(PrMainEditor::errorArtsMidi);
      cout << "midiManager is null" << endl; veto = true;
    }

    ps->midiClient = ps->midiManager.addClient(mcdPlay, mctApplication, "Brahms", "brahms");
    if (ps->midiClient.isNull()) { cout << "midiClient is null" << endl; veto = true; }

    ps->midiPort = ps->midiClient.addOutputPort();
    if (ps->midiPort.isNull()) { cout << "midiOutputPort is null" << endl; veto = true; }

    ps->midiRecorder = Arts::DynamicCast(ps->soundServer.createObject("BrahmsMidiRecorder"));
    if (ps->midiRecorder.isNull()) { 
      mainEditor->fatalError(PrMainEditor::errorArtsMidiRecord);
      cout << "midiRecorder is null" << endl; veto = true;
    }

    ps->midiClientRec = ps->midiManager.addClient(mcdRecord, mctApplication, "Brahms", "brahms");
    if (ps->midiClientRec.isNull()) { cout << "midiClientRec is null" << endl; veto = true; }

    ps->midiClientRec.addInputPort(ps->midiRecorder);
    ps->midiRecorder.setMidiThru(ps->midiPort);

  } else if (output==ALSA) {
    //
    // ALSA
    //
#ifdef HAVE_ALSA
    alsa = new AlsaManager();
    setTempo(_song->tempo());
#endif
  }
}

void Player::freePlayer()
{
  if (output == ARTS)
  {
    delete ps;
  } // TODO: other outputs
}

bool Player::playStart(PrProgress * progress, Part * part, long lft, long rgt) {
  if (!_playing) {
    _playing = true;

    if (output==ARTS) {
      //
      // ARTS
      //
      if (ps->record_track>=0) ps->midiRecorder.record(true);
      ps->ntracks = _song->size();
      ps->left = lft;
      ps->right = rgt;
      ps->iterator = new SongIterator(_song, lft, rgt);
      ps->interval = TIMERINTERVALL * 0.001;
      bool veto = false;
      // ps->volume = 0;

      // TODO: handle track->output()

#ifdef HAVE_ARTS_ENVIRONMENT
      Track * tr = ps->iterator->track(0);
      if( tr )
	ps->midiPort = tr->envInstrument().port();
#endif


      ps->playTime = ps->midiPort.time(); // + xy in order to avoid neglecting the first events...
      //ps->playTime.usec += 200000;
      //ps->playTime.sec += ps->playTime.usec / 1000000;
      ps->startTime = ps->playTime; // remember start time

      if (!veto) {
	
	for (int t=0; t<ps->ntracks; t++) {
	  if (ps->iterator->channel(t)!=-1) {
#if HAVE_ARTS_ENVIRONMENT
	    ps->iterator->track(t)->envInstrument().port().processCommand(MidiCommand(mcsProgram | ps->iterator->channel(t), ps->iterator->program(t), 0));
#else
	    ps->midiPort.processCommand(MidiCommand(mcsProgram | ps->iterator->channel(t), ps->iterator->program(t), 0));
#endif
	  }
	}
	
	ps->pp = ps->left;
	
	
	/* this is to adjust the start-position into the past, in order to get a correct value for time(),
	 * which is displayed in the gui!
	 */
	long delta_usec = long(ps->pp*60*1000000.0/(384.0*_song->tempo()));
	long delta_sec  = long(delta_usec * 1.0 / 1000000.0);
	delta_usec %= 1000000;
	ps->startTime.sec -= delta_sec;
	ps->startTime.usec -= delta_usec;
	if (ps->startTime.usec<0) {
	  ps->startTime.usec += 1000000;
	  ps->startTime.sec -= 1;
	}
	
      }
      // we are still playing! (was: _playing = false;)
    } else   if (output==ALSA) {
      //
      // ALSA
      //
#ifdef HAVE_ALSA
      ntracks = _song->size();
      setTempo(_song->tempo());
      aiterator = new SongIterator(_song, lft, rgt);
      for (int t=0; t<ntracks; t++) {
	if (aiterator->channel(t)!=-1) alsa->prgchange(0, aiterator->channel(t), aiterator->program(t));
      }
      pp = lft;
      left = pp;
      alsa->playStart(left);
#endif
    }
  }
  return true;
}

bool Player::playQuantum(bool cycl) {
  bool ret = true;
  if (output==ARTS) {
    //
    // ARTS
    //
    TimeStamp midiTime = ps->midiPort.time();
    Arts::MidiEvent ev;
    Event * event;
    Note * note;
    Track * tr;
    int chan;
    long num;
    ::MidiEvent * mev;

    while((deltaT(ps->playTime,midiTime) < ps->interval)
       && (ps->pp < ps->right || ps->right == 0)) {
      ev.time = ps->playTime;

      if (ps->record_part!=0) metronome(ps->pp);

      if (!ps->iterator->done()) {
	for (int t=0; t<ps->ntracks; t++) {
	  tr = ps->iterator->track(t);
	  while (ps->iterator->startsAt(t,ps->pp)) {
	    if (!ps->iterator->mute(t)) {
	      event = (Event*) (*ps->iterator)[t];
	      if (tr->delay()!=0) {
		ev.time.usec += tr->delay()*1000000/(384*_song->tempo()); // 60000;   // 60ms
		ev.time.sec += ev.time.usec / 1000000;
		ev.time.usec %= 1000000;
	      }
	      if (event->isA()==NOTE) {
		note = (Note*) event;
		// cout << ps->pp << " on " << note->pitch() << endl;
		ev.command = MidiCommand(mcsNoteOn|ps->iterator->channel(t), tr->pitch(note), tr->vel(note));
#ifdef HAVE_ARTS_ENVIRONMENT
		tr->envInstrument().port().processEvent(ev);
#else
		ps->midiPort.processEvent(ev);
#endif
		ps->iterator->activate(t);
		// ps->volume = ps->volume|(1<<t);
		// cout << "* " << ps->volume << endl;
	      } else if (event->isA()==MASTEREVENT) {
		// TODO!
	      } else if (event->isA()==MIDIEVENT) {
		mev = (::MidiEvent*) event;
		if(mev->code() >= 10 && mev->code() <= 14) {
		  ev.command = MidiCommand((mev->code() << 4) | mev->channel(), mev->value1(), mev->value2());
#ifdef HAVE_ARTS_ENVIRONMENT
		tr->envInstrument().port().processEvent(ev);
#else
		  ps->midiPort.processEvent(ev);
#endif
		}
	      } else if (event->isA()==AUDIOEVENT) {
		cout << "play " << ((AudioEvent*)event)->fileName() << endl;
	      } else {
		cout << "track " << t << *event << endl;
	      }
	      if (tr->delay()!=0) {
		ev.time = ps->playTime;
	      }
	    }
	    ps->iterator->increment(t); // TODO: new program settings?
	  }
	}
      }
      ps->pp++;

      ps->playTime.usec += 60*1000000/(384*_song->tempo()); // 60000;   // 60ms
      ps->playTime.sec += ps->playTime.usec / 1000000;
      ps->playTime.usec %= 1000000;

      if (!ps->iterator->done()) {
	while (ps->iterator->activeEndsAt(ps->pp)) {
	  note = ps->iterator->activeNote();
	  chan = ps->iterator->activeChannel();
	  num  = ps->iterator->activeTrack();
	  tr   = ps->iterator->track(num);
	  // cout << ps->pp << " off " << note->pitch() << endl;
	  ev.command = MidiCommand(mcsNoteOff|chan, tr->pitch(note), tr->vel(note));
#ifdef HAVE_ARTS_ENVIRONMENT
		tr->envInstrument().port().processEvent(ev);
#else
	  ps->midiPort.processEvent(ev);
#endif
	  ps->iterator->cutActive();
	  // ps->volume = ps->volume&((-1)-(1<<num));
	  // cout << "- " << ps->volume << endl;
	}
      }

      if (cycl && ps->iterator->done() && ps->pp == ps->right) {
	// looping over the end of an event will not work correctly, i.e.
	// if your song is
	//     left           right
	//
	//      |[---]          |
	//      |-]   [---]   [-|-]
	//      |     [---]   [-|-]
	//
	// then the events that cross the left/right mark should be stopped/
	// restarted - we don't do this yet

	ps->iterator = new SongIterator(_song, ps->left, ps->right);
	ps->pp = ps->left;
	ps->startTime = ps->playTime; // remember start time
	// TODO: new program settings
      }
    }


    /*
     * to give the caller a realistic idea when we are "really done", i.e. when
     * no more events are in the play queue, check that the play queue is empty
     * before saying "there is nothing more to play"
     */
    // bool reallyDone=(ps->iterator->done() && deltaT(ps->playTime, midiTime) < 0);
    bool reallyDone=(ps->pp>=ps->right && deltaT(ps->playTime, midiTime) < 0);
    ret= !reallyDone;
  } else if (output==ALSA) {
    //
    // ALSA
    //
#ifdef HAVE_ALSA
    Event * event = 0;
    Note * note   = 0;
    Track * tr    = 0;
    ::MidiEvent * mev;
    int temp = 0;
    while ((pp < alsa->ticks() + 768) && !aiterator->done()) {
      for (int t=0; t<ntracks; t++) {
	tr = aiterator->track(t);
	while (aiterator->startsAt(t,pp)) {
	  if (!aiterator->mute(t)) {
	    event = (Event*) (*aiterator)[t];
	    if (event->isA()==NOTE) {
	      note = (Note*) event;
	      alsa->playNote(0, aiterator->channel(t), tr->pitch(note), tr->vel(note), pp + tr->delay(), note->duration());
	    } else if (event->isA()==MASTEREVENT) {
	      temp = ((MasterEvent*) event)->tempo();
	      if (temp>0) {
		setTempo(temp, pp);
		_song->setTempo(temp);
	      } else {
		_song->setMeter0(((MasterEvent*) event)->meter0());
		_song->setMeter1(((MasterEvent*) event)->meter1());
	      }
	    } else if (event->isA()==MIDIEVENT) {
	      mev = (::MidiEvent*) event;
	      /* if(mev->code() >= 10 && mev->code() <= 14) {
		ev.command = MidiCommand((mev->code() << 4) | mev->channel(), mev->value1(), mev->value2());
		ps->midiPort.processEvent(ev);
		}*/
	      alsa->event(mev->code(), mev->channel(), mev->value1(), mev->value2(), pp + tr->delay());
	
	    } else if (event->isA()==AUDIOEVENT) {
	      cout << "play " << ((AudioEvent*)event)->fileName() << endl;
	    } else {
	      cout << "track " << t << *event << endl;
	    }
	  }
	  aiterator->increment(t); // TODO: new program settings?
	}
      }
      pp++;
    }
    ret = !(aiterator->done() && pp>=alsa->ticks());
#endif
  }
  return ret;
}

void Player::killAllNotes() {
  if (output==ARTS) {
    //
    // ARTS
    //
    ps->midiRecorder.record(false);
    Arts::MidiEvent ev;
    ev.time = ps->playTime;
    for (int ch=0; ch<16; ch++) {
      // TODO: after KDE2.2 has been released, switch to
      // MidiCommand(mcsParameter|ch, mcpAllNotesOff, 0);
      // but as this will not compile for KDE2.1 users, hardcode it:
      ev.command = MidiCommand(mcsParameter|ch, 0x7b, 0);
#ifdef HAVE_ARTS_ENVIRONMENT
      if (ps->iterator!=0)
	for (int t=0; t<ps->ntracks; t++)
	  ps->iterator->track(t)->envInstrument().port().processEvent(ev);
#else
      ps->midiPort.processEvent(ev);
#endif
    }
    if (ps->iterator!=0) delete ps->iterator;
    ps->iterator = 0;

    mergeQueue();

  } else   if (output==ALSA) {
    //
    // ALSA
    //
#ifdef HAVE_ALSA
    alsa->playEnd();
#endif
  }
  _playing = false;
}

void Player::setTempo(int t, long pos) {
  if (output==ALSA) {
#ifdef HAVE_ALSA
    alsa->tempo(t);
#endif
  }
}


#define MAXSUS 512


void Player::mergeQueue() {
  if (ps->record_part!=0) {
    std::vector<Arts::MidiEvent> * q = ps->midiRecorder.getQueue();
    cout << "merge " << endl;
    Note* sus[MAXSUS];
    Note* newNote;
    int susp=0;

    int remaining = q->size();
    /* for (std::vector<Arts::MidiEvent>::iterator i=q->begin(); i!=q->end(); i++) {
       cout << i->time.sec << ":" << i->time.usec << ", " << i->command.status << endl;
       }*/
    std::vector<Arts::MidiEvent>::iterator qiter = q->begin();

    if (ps->iterator!=0) delete ps->iterator;
    ps->iterator = new SongIterator(_song, ps->left, ps->right);
    ps->pp = ps->left;
    ps->playTime = ps->startTime;
    int ch = 0;
    int cmd = 0;
    int ii = 0;
    MidiCommand mc;

    for (; ps->pp < ps->right && remaining!=0;) {
      if (deltaT(qiter->time, ps->playTime) <=0) {
	mc = qiter->command;
	ch = mc.status&0x0f;
	cmd = mc.status&0xf0;
	if (cmd==0x90 && mc.data2==0) cmd=0x80;
	// cout << "add: " << ps->pp << ", cmd: " << cmd << ", ch: " << ch << ", note:" << int(mc.data1) << ", " << int(mc.data2) << endl;
	switch (cmd) {
	case 0x80: // NOTEOFF
	  ii=0;
	  for (int i=0; i<susp; i++)
	    if ((mc.data1 == sus[i]->pitch()))
	      {
		sus[i]->setDuration(ps->pp - ps->record_part->start(sus[i]).ticks());
		ii=i;
		i=susp;
	      }
	  for (int i=ii; i<susp-1; i++) sus[i]=sus[i+1];
	  if (susp>0) susp--;
	  break;
	case 0x90: // NOTEON
	  newNote = new Note(mc.data1, mc.data2, 191, ps->pp-ps->record_part->start(), 0, ch);
	  // ps->record_part->add(newNote);
	  _song->doo(new AddElement(newNote,ps->record_part));
	  if (susp<MAXSUS) sus[susp++] = newNote;
	  break;
	}

	qiter++;
	remaining--;
      }

      // TODO: cycles!
      ps->pp++;
      ps->playTime.usec += 60*1000000/(384*_song->tempo()); // 60000;   // 60ms
      ps->playTime.sec += ps->playTime.usec / 1000000;
      ps->playTime.usec %= 1000000;
    }
  }
  if (ps->iterator!=0) delete ps->iterator;
  ps->iterator = 0;
}


void Player::programChange(int t) {
  Track * tr = (Track*) _song->get(t);
  programChange(tr);
}

void Player::programChange(Track * tr) {
  int ch  = tr->channel();
  int prg = tr->program();
  if (output==ARTS) {
    ps->midiPort.processCommand(MidiCommand(mcsProgram | ch, prg, 0));
  } else if (output==ALSA) {
#ifdef HAVE_ALSA
    alsa->prgchange(0, ch, prg);
#endif
  }
}

int met_chn = 9;
int met_pitch = 32;
int met_vel = 127;

void Player::metronome(int p) {
  if (output==ARTS) {
    if (p%384==0) {
      Arts::MidiEvent ev;
      ev.time = ps->playTime;
      ev.command = MidiCommand(mcsNoteOn|met_chn, met_pitch, met_vel);
      ps->midiPort.processEvent(ev);
      ev.time = ps->playTime;
      ev.time.usec += 6000;
      ev.command = MidiCommand(mcsNoteOff|met_chn, met_pitch, met_vel);
      ps->midiPort.processEvent(ev);
    }
  }
}

// vim:ts=8:sw=2:sts=2

#endif
