/*
 * Cheops Network User Interface
 *
 * Copyright (C) 1999, Adtran, Inc.
 * 
 * Distributed under the terms of the GNU GPL
 *
 */

#include <gtk/gtk.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>
#include <stdio.h>
#include "cheops.h"

static GList *eventq=NULL;

static GList *windowl=NULL;

static FILE *filep=NULL;

static char *lev2str(int lev)
{
	switch(lev) {
	case MONITOR_NOT_MONITORED:
	return "Not Monitored";
	case MONITOR_MONITORED:
	return "Monitored";
	case MONITOR_NOMINAL:
	return "Nominal";
	case MONITOR_WARNING:
	return "Warning";
	case MONITOR_CRITICAL:
	return "Critical";
	default:
	return "Unknown";
	}
}

void close_logfile(void)
{
	if (filep) {
		fclose(filep);
		filep=NULL;
	}
}

static void init_log(FILE *f)
{
	log_event(MONITOR_NOMINAL, "logger", "", "*", "Log begins");
}

static FILE *get_logfile(char *how)
{
	if (!option_need_logfile) {
		close_logfile();
		return NULL;
	}
	if (!filep) {
		if (!strlen(logfile))
			make_logfile();
		if (strlen(logfile))
			filep = fopen(logfile, how);
		if (filep) {
			setlinebuf(filep);
			if (!ftell(filep)) 
				init_log(filep);
		}
	}
	return filep;
}

void clear_logfile(void)
{
	FILE *f;
	close_logfile();
	f = get_logfile("w");
}

static void add_entry(struct event *e, struct event_window *w)
{
	static GdkColor *yellow = NULL;
	static GdkColor *green = NULL;
	static GdkColor *red = NULL;
	static GdkColor *blue = NULL;
	
	int id;
	
	char *args[6];
	
	args[0] = e->date;
	args[1] = lev2str(e->lev);
	args[2] = e->who;
	args[3] = e->host;
	args[4] = e->svc;
	args[5] = e->msg; 
	
	id = gtk_clist_append(GTK_CLIST(w->clist), args);

	gtk_clist_set_row_data(GTK_CLIST(w->clist), id, e);
	
	if (!blue) {
		blue = g_new0(GdkColor, 1);
		blue->red = 108 * 256;
		blue->green = 166 * 256;
		blue->blue = 205 * 256;
		gdk_color_alloc(gdk_window_get_colormap(w->window->window), blue);

	}
	
	if (!yellow) {
		yellow = g_new0(GdkColor, 1);
		yellow->red = 65535;
		yellow->green = 65535;
		yellow->blue = 0;
		gdk_color_alloc(gdk_window_get_colormap(w->window->window), yellow);
	};
	
	if (!green) {
		green = g_new0(GdkColor, 1);
		green->red = 0;
		green->green = 205 * 256;
		green->blue = 0;
		gdk_color_alloc(gdk_window_get_colormap(w->window->window), green);
	};
	if (!red) {
		red = g_new0(GdkColor, 1);
		red->red = 205 * 256;
		red->green = 0;
		red->blue = 0;
		gdk_color_alloc(gdk_window_get_colormap(w->window->window), red);
	};

	/* Need a way to show this */
	switch(e->lev) {
	case MONITOR_NOT_MONITORED:
		gtk_clist_set_background(GTK_CLIST(w->clist), id, NULL);
		break;
	case MONITOR_MONITORED:
		gtk_clist_set_background(GTK_CLIST(w->clist), id, blue);
		break;
	case MONITOR_NOMINAL:
		gtk_clist_set_background(GTK_CLIST(w->clist), id, green);
		break;
	case MONITOR_WARNING:
		gtk_clist_set_background(GTK_CLIST(w->clist), id, yellow);
		break;
	case MONITOR_CRITICAL:
		gtk_clist_set_background(GTK_CLIST(w->clist), id, red);
		break;
	default:
	}
	
}

static void update_windows_add(struct event *e)
{
	GList *l = windowl;
	struct event_window *w;
	while(l) {
		w = (struct event_window *)(l->data);
		if (!w->hostname || !strcmp(w->hostname, e->host))
			add_entry(e, w);
		l = g_list_next(l);
	}
}

static void add_outstanding(struct event_window *w)
{
	GList *l;
	struct event *e;
	l = eventq;
	while(l) {
		e = (struct event *)(l->data);
		if (!w->hostname || !strcmp(e->host, w->hostname))
			add_entry(e, w);
		l = g_list_next(l);
	}
}

static void close_event_window(GtkWidget *w)
{
	struct event_window *e = (struct event_window *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	gtk_widget_destroy(e->window);
	windowl = g_list_remove(windowl, e);
	if (e->hostname)
		g_free(e->hostname);
	g_free(e);
}

static void event_ack(struct event *e)
{
	GList *l;
	struct event_window *w;
	int id;
	eventq = g_list_remove(eventq, e);
	l = windowl;
	while(l) {
		w = (struct event_window *)(l->data);
		id = gtk_clist_find_row_from_data(GTK_CLIST(w->clist), e);
		if (id > -1) 
			gtk_clist_remove(GTK_CLIST(w->clist), id);
		l = g_list_next(l);
	}
	g_free(e);
}

static void handle_ack_all(GtkWidget *w)
{
	struct event_window *ew = (struct event_window *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	struct event *e;
	while((e = (struct event *)gtk_clist_get_row_data(GTK_CLIST(ew->clist), 0)))
		event_ack(e);
}

static void handle_ack(GtkWidget *w)
{
	struct event_window *e = (struct event_window *)
		gtk_object_get_user_data(GTK_OBJECT(w));
	GList *l;
	
	l = GTK_CLIST(e->clist)->selection;
	while(l) {
		event_ack((struct event *)gtk_clist_get_row_data(GTK_CLIST(e->clist), GPOINTER_TO_INT(l->data)));
		l = GTK_CLIST(e->clist)->selection;
	}
}

static void new_event_window(char *hostname)
{
	struct event_window *w;
	char buf[256];
	GtkWidget *vbox;
	GtkWidget *bbox;
	GtkWidget *close;
	GtkWidget *ack;
	GtkWidget *acka;
	static char *titles[] = { "Time", "Level", "Type","Host", "Service", "Message" };
#if (GTK_MINOR_VERSION > 1) || ((GTK_MINOR_VERSION > 0) && (GTK_MICRO_VERSION > 4))
	GtkWidget *sw;
	sw = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(sw);
#endif
	w = g_new0(struct event_window, 1);
	w->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	w->hostname = (hostname ? g_strdup(hostname) : NULL);
	gtk_widget_realize(w->window);
	gtk_object_set_user_data(GTK_OBJECT(w->window), w);
	fix_icon(w->window->window);
	vbox = gtk_vbox_new(FALSE, 5);
	gtk_widget_show(vbox);
	w->clist = gtk_clist_new_with_titles(6, titles);
#if ((GTK_MINOR_VERSION > 0) && (GTK_MICRO_VERSION > 2))  || (GTK_MAJOR_VERSION > 1)
	gtk_clist_set_column_auto_resize(GTK_CLIST(w->clist), 0, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(w->clist), 1, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(w->clist), 2, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(w->clist), 3, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(w->clist), 4, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(w->clist), 5, TRUE);
#else
	gtk_clist_set_column_width(GTK_CLIST(w->clist), 0,150);
	gtk_clist_set_column_width(GTK_CLIST(w->clist), 1,50);
	gtk_clist_set_column_width(GTK_CLIST(w->clist), 2,50);
	gtk_clist_set_column_width(GTK_CLIST(w->clist), 3,100);
	gtk_clist_set_column_width(GTK_CLIST(w->clist), 4,50);
	gtk_clist_set_column_width(GTK_CLIST(w->clist), 5,400);
#endif	
	gtk_clist_set_selection_mode(GTK_CLIST(w->clist), GTK_SELECTION_MULTIPLE);
	gtk_widget_show(w->clist);
	gtk_widget_set_usize(w->window, 700, 200);
	ack = gtk_button_new_with_label("Acknowledge");
	gtk_object_set_user_data(GTK_OBJECT(ack), w);
	gtk_signal_connect(GTK_OBJECT(ack), "clicked", GTK_SIGNAL_FUNC(handle_ack), NULL);
	gtk_widget_show(ack);
	acka = gtk_button_new_with_label("Acknowledge All");
	gtk_object_set_user_data(GTK_OBJECT(acka), w);
	gtk_signal_connect(GTK_OBJECT(acka), "clicked", GTK_SIGNAL_FUNC(handle_ack_all), NULL);
	gtk_widget_show(acka);
	close = gtk_button_new_with_label("  Close  ");
	gtk_object_set_user_data(GTK_OBJECT(close), w);
	gtk_signal_connect(GTK_OBJECT(close), "clicked", GTK_SIGNAL_FUNC(close_event_window), NULL);
	gtk_widget_show(close);
	bbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show(bbox);
	gtk_box_pack_end(GTK_BOX(bbox), close, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(bbox), ack, FALSE, FALSE, 5);
	gtk_box_pack_end(GTK_BOX(bbox), acka, FALSE, FALSE, 5);
#if (GTK_MINOR_VERSION > 1) || ((GTK_MINOR_VERSION > 0) && (GTK_MICRO_VERSION > 4))
	gtk_container_add(GTK_CONTAINER(sw), w->clist);
	gtk_box_pack_start(GTK_BOX(vbox), sw, TRUE, TRUE, 5);
#else	
	gtk_box_pack_start(GTK_BOX(vbox), w->clist, TRUE, TRUE, 5);
#endif
	gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 5);
	if (hostname) 
		snprintf(buf, sizeof(buf), "Event log for '%s'", w->hostname);
	else
		strncpy(buf, "General Event Log", sizeof(buf));
	gtk_window_set_title(GTK_WINDOW(w->window), buf);
	gtk_container_add(GTK_CONTAINER(w->window), vbox);
	gtk_signal_connect(GTK_OBJECT(w->window), "delete_event", GTK_SIGNAL_FUNC(close_event_window), NULL);
	gtk_widget_grab_focus(close);
	gtk_widget_show(w->window);
	windowl = g_list_append(windowl, w);
	add_outstanding(w);
}

void show_event_window(char *hostname)
{
	GList *l;
	struct event_window *w;
	l = windowl;
	while(l) {
		w = (struct event_window *)(l->data);
		if ((w->hostname && !strcmp(w->hostname, hostname)) ||
		     (!w->hostname && !hostname)) {
			gtk_widget_show(w->window);
			break;
		}
		l = g_list_next(l);
	}
	if (!l)
		new_event_window(hostname);
}

void log_event(int lev, char *who, char *svc, char *host, char *msg)
{
	struct event *e;
	char buf[256];
	struct tm *tm;
	time_t t;
	FILE *f;
	t = time(NULL);
	tm = localtime(&t);
	e = g_new0(struct event, 1);
	e->lev = lev;
	e->who = who;
	e->host = g_strdup(host);
	e->svc = svc;
	e->msg = g_strdup(msg);
	eventq = g_list_append(eventq, e);
	strftime(buf, sizeof(buf), "%c", tm);
	e->date = g_strdup(buf);
	if ((f = get_logfile("a"))) {
		fprintf(f, "%s %s: %s %s on %s reports '%s'\n", e->date,lev2str(e->lev), 
					   e->who,  e->svc, e->host, e->msg);
	}
	if (option_send_email) {
		snprintf(buf, sizeof(buf), "mail -s '%s event on %s' %s", lev2str(e->lev), e->host, email);
		if ((f=popen(buf, "w"))) {
			fprintf(f, "%s %s: %s %s on %s reports '%s'\n", e->date,lev2str(e->lev), 
					   e->who,  e->svc, e->host, e->msg);
			pclose(f);
		}
	}
	update_windows_add(e);
}

void event_log()
{
	/* Show general event logger */
	show_event_window(NULL);
}

int events_exist(char *hostname)
{
	GList *l;
	l = eventq;
	while(l) {
		if (!strcmp(hostname, ((struct event *)(l->data))->host))
			return -1;
		l = g_list_next(l);
	}
	return 0;
}

