/**************************************************************************/
/*                                                                        */
/*  cpbk - a mirroring utility for backing up your files                  */
/*  Copyright (C) 1998 Kevin Lindsay <klindsay@mkintraweb.com>            */
/*  Copyright (C) 2001 Yuuki NINOMIYA <gm@debian.or.jp>                   */
/*                                                                        */
/*  This program is free software; you can redistribute it and/or modify  */
/*  it under the terms of the GNU General Public License as published by  */
/*  the Free Software Foundation; either version 2, or (at your option)   */
/*  any later version.                                                    */
/*                                                                        */
/*  This program is distributed in the hope that it will be useful,       */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/*  GNU General Public License for more details.                          */
/*                                                                        */
/*  You should have received a copy of the GNU General Public License     */
/*  along with this program; if not, write to the                         */
/*  Free Software Foundation, Inc., 59 Temple Place - Suite 330,          */
/*  Boston, MA 02111-1307, USA.                                           */
/*                                                                        */
/**************************************************************************/

/* $Id: copyproc.c,v 1.5 2001/03/30 13:16:36 gm Exp $ */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdlib.h>
#include <unistd.h>
#include <utime.h>
#include <sys/types.h>

#include "intl.h"
#include "strlib.h"
#include "variable.h"
#include "proto.h"


static void execute_order(Order *order);


/* --- PUBLIC FUNCTIONS --- */

void prepare_source_dir(void)
{
	config.source_dir = canonicalize_path(config.source_dir, _("source"));
	check_valid_dir(config.source_dir, _("source"));
}


void prepare_dest_dir(void)
{
	struct stat status;
	struct utimbuf newtime;

	if (!config.simulate) {
		mkdir_parents(config.dest_dir, 0755);
	}

	config.dest_dir = canonicalize_path(config.dest_dir, _("destination"));

	if (config.simulate) {
		return;
	}

	if (lstat(config.dest_dir, &status) == -1) {
                err_printf(CAN_PROCEED, "lstat", _("Cannot get file status: %s\n"), config.dest_dir);
                return;
        }

	if (am_i_root()) {
		if (lchown(config.dest_dir, status.st_uid, status.st_gid) == -1) {
			err_printf(CAN_PROCEED, "lchown", _("Cannot change ownership of directory `%s'.\n"), config.dest_dir);
		}	
	}

	/* Set time on directory to same as source directory */
	newtime.actime = status.st_atime;
	newtime.modtime = status.st_mtime;

	if (utime(config.dest_dir, &newtime) == -1) {
		err_printf(CAN_PROCEED, "utime", _("Cannot change access and modification times of an inode: %s.\n"), config.dest_dir);
	}	
}



void copy_procedure(void)
{
	FileData source_file_data;
	FileData dest_file_data;
	Order *order;

	if (!config.quiet) {
		printf(_("Searching source files and building lists... "));
		fflush(stdout);
	}
	get_file_data(&source_file_data, config.source_dir);
	if (!config.quiet) {
		printf("\b ");
		printf(_("Done\n"));
	}

	if (!config.quiet) {
		printf(_("Searching destination files and building lists... "));
		fflush(stdout);
	}
	get_file_data(&dest_file_data, config.dest_dir);
	if (!config.quiet) {
		printf("\b ");
		printf(_("Done\n"));
	}

	if (!config.quiet) {
		printf(_("Comparing files and generating order lists... "));
		fflush(stdout);
	}
	order = compare_files_and_generate_order(&source_file_data, &dest_file_data, config.source_dir, config.dest_dir);
	if (!config.quiet) {
		printf(_("Done\n"));
		printf("\n");
	}

	if (command_line_option.list) {
		put_listing_of_updated_file(order);
		exit(0);
	}
	if (command_line_option.nlist) {
		put_the_number_of_updated_file(order);
		exit(0);
	}

	execute_order(order);

	free_order(order);
}


/* --- PRIVATE FUNCTIONS --- */

static void execute_order(Order *order)
{
	int i;
	char *src_path_file;
	char *dest_path_file;

	init_progress_info(order);

	for (i = 0; order[i].action != END; i++) {
		src_path_file = str_concat(order[i].src_dir, "/", order[i].file, NULL);
		dest_path_file = str_concat(order[i].dest_dir, "/", order[i].file, NULL);

		switch (order[i].action) {
		case UPDATE:
			trash_it(UPDATE, dest_path_file);
		case NEW:
			output_progress_info(NEW, src_path_file, dest_path_file, order + i);
			copy_file(order + i, src_path_file, dest_path_file);
			break;
		case REMOVE:
			output_progress_info(REMOVE, dest_path_file, NULL, order + i);
			trash_it(REMOVE, dest_path_file);
			remove_file(order + i, dest_path_file);
			break;
		case MKDIR:
			output_progress_info(MKDIR, dest_path_file, NULL, order + i);
			make_dir(order + i, dest_path_file);
			break;
		case RMDIR:
			output_progress_info(RMDIR, dest_path_file, NULL, order + i);
			trash_it(RMDIR, dest_path_file);
			remove_dir(order + i, dest_path_file);
			break;
		case CHMOD:
			output_progress_info(CHMOD, dest_path_file, NULL, order + i);
			change_mode(order + i, dest_path_file);
			break;
		case ENTER:
		case LEAVE:
			break;
		default:
			internal_error(__FILE__, __LINE__);
		}

		free(src_path_file);
		free(dest_path_file);
	}
	if (!config.quiet) {
		printf("\x1b[2K");
		fflush(stdout);
	}
}
