/* 
 * crafteddialog.c
 *
 * crafted - a pud editor for the freecraft project.
 * 
 * Copyright (C) 2001-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "crafteddialog.h"

/*************************************************************************
*  crafted_dialog_delete_callback
**************************************************************************/
static gint crafted_dialog_delete_callback(GtkWidget *widget,
                                           GdkEvent  *event,
                                           gpointer   data)
{
  GtkSignalFunc  cancel_callback;
  GtkWidget     *cancel_widget;

  cancel_callback = (GtkSignalFunc)gtk_object_get_data(GTK_OBJECT(widget),
                                                       "crafted_dialog_cancel_callback");
  cancel_widget = (GtkWidget*)gtk_object_get_data(GTK_OBJECT(widget),
                                                  "crafted_dialog_cancel_widget");

  /*  the cancel callback has to destroy the dialog  */
  if (cancel_callback)
    (* cancel_callback)(cancel_widget, data);

  return TRUE;
}

/*************************************************************************
*  crafted_dialog_create_action_areav
**************************************************************************/
static void crafted_dialog_create_action_areav(GtkDialog *dialog,
                                               va_list    args)
{
  GtkWidget *hbbox = NULL;
  GtkWidget *button;

  /*  action area variables  */
  const gchar    *label;
  GtkSignalFunc   callback;
  gpointer        data;
  GtkObject      *slot_object;
  GtkWidget     **widget_ptr;
  gboolean        default_action;
  gboolean        connect_delete;

  gboolean delete_connected = FALSE;

  g_return_if_fail(dialog != NULL);
  g_return_if_fail(GTK_IS_DIALOG (dialog));

  label = va_arg(args, const gchar *);

  /*  the action_area buttons  */
  while (label)
  {
    callback       = va_arg(args, GtkSignalFunc);
    data           = va_arg(args, gpointer);
    slot_object    = va_arg(args, GtkObject *);
    widget_ptr     = va_arg(args, GtkWidget **);
    default_action = va_arg(args, gboolean);
    connect_delete = va_arg(args, gboolean);

    if (slot_object == (GtkObject *)1)
      slot_object = GTK_OBJECT(dialog);

    if (data == NULL)
      data = dialog;

    /*
     * Dont create a button if the label is "_delete_event_" --
     * some dialogs just need to connect to the delete_event from
     * the window...
     */
    if (connect_delete &&
        callback &&
        !delete_connected &&
        strcmp(label, "_delete_event_") == 0)
    {
      if (widget_ptr)
        *widget_ptr = GTK_WIDGET(dialog);

      gtk_object_set_data(GTK_OBJECT(dialog),
                          "crafted_dialog_cancel_callback",
                          callback);
      gtk_object_set_data(GTK_OBJECT(dialog),
                          "crafted_dialog_cancel_widget",
                          slot_object ? slot_object : GTK_OBJECT(dialog));

      /*  catch the WM delete event  */
      gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                         (GtkSignalFunc)crafted_dialog_delete_callback,
                         data);

      delete_connected = TRUE;
    }

    /* otherwise just create the requested button. */
    else
    {
      if (hbbox == NULL)
      {
        /*  prepare the action_area  */
        gtk_container_set_border_width(GTK_CONTAINER(dialog->action_area), 2);
        gtk_box_set_homogeneous(GTK_BOX(dialog->action_area), FALSE);

        hbbox = gtk_hbutton_box_new();
        gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbbox), 4);
        gtk_box_pack_end(GTK_BOX(dialog->action_area), hbbox, FALSE, FALSE, 0);
        gtk_widget_show(hbbox);
      }

      button = gtk_button_new_with_label(label);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      gtk_box_pack_start(GTK_BOX(hbbox), button, FALSE, FALSE, 0);

      if (callback)
      {
        if (slot_object)
          gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                                    (GtkSignalFunc)callback,
                                    slot_object);
        else
          gtk_signal_connect(GTK_OBJECT(button), "clicked",
                             (GtkSignalFunc)callback, data);
      }


      if (widget_ptr)
        *widget_ptr = button;
      if (connect_delete && callback && !delete_connected)
      {
        gtk_object_set_data(GTK_OBJECT(dialog),
                            "crafted_dialog_cancel_callback",
                            callback);
        gtk_object_set_data(GTK_OBJECT(dialog),
                            "crafted_dialog_cancel_widget",
                            slot_object ? slot_object : GTK_OBJECT(button));

        /*  catch the WM delete event  */
        gtk_signal_connect(GTK_OBJECT(dialog), "delete_event",
                           (GtkSignalFunc)crafted_dialog_delete_callback,
                           data);

        delete_connected = TRUE;
      }

      if (default_action)
        gtk_widget_grab_default(button);
      gtk_widget_show(button);
    }
    label = va_arg(args, gchar *);
  }
}

/*************************************************************************
*  crafted_dialog_newv
**************************************************************************/
static GtkWidget *crafted_dialog_newv(const gchar       *title,
                                      const gchar       *wmclass_name,
                                      GtkWindowPosition  position,
                                      gboolean           allow_shrink,
                                      gboolean           allow_grow,
                                      gboolean           auto_shrink,
                                      va_list            args)
{
  GtkWidget *dialog;

  g_return_val_if_fail(title != NULL, NULL);
  g_return_val_if_fail(wmclass_name != NULL, NULL);

  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), title);
  gtk_window_set_wmclass(GTK_WINDOW(dialog), wmclass_name, "Crafted");
  gtk_window_set_position(GTK_WINDOW(dialog), position);
  gtk_window_set_policy(GTK_WINDOW(dialog),
                        allow_shrink, allow_grow, auto_shrink);

  /*  prepare the action_area  */
  crafted_dialog_create_action_areav(GTK_DIALOG(dialog), args);

  return dialog;
}

/*************************************************************************
*  crafted_dialog_new
**************************************************************************/
GtkWidget *crafted_dialog_new(const gchar       *title,
                              const gchar       *wmclass_name,
                              GtkWindowPosition  position,
                              gboolean           allow_shrink,
                              gboolean           allow_grow,
                              gboolean           auto_shrink,
                              /* specify action area buttons as va_list:
                               *  const gchar    *label,
                               *  GtkSignalFunc   callback,
                               *  gpointer        data,
                               *  GtkObject      *slot_object,
                               *  GtkWidget     **widget_ptr,
                               *  gboolean        default_action,
                               *  gboolean        connect_delete,
                               */
                              ...)
{
  GtkWidget *dialog;
  va_list    args;

  va_start(args, auto_shrink);

  dialog = crafted_dialog_newv(title,
                               wmclass_name,
                               position,
                               allow_shrink,
                               allow_grow,
                               auto_shrink,
                               args);

  va_end(args);

  return dialog;
}

